*** Settings ***
Library           JSONSchemaLibrary    schemas/
Resource          environment/variables.txt    # Generic Parameters
Library           OperatingSystem
Library           JSONLibrary
Library           RequestsLibrary
Library           REST    ${NFVO_SCHEMA}://${NFVO_HOST}:${NFVO_PORT}    ssl_verify=false
Library           Process
Library           MockServerLibrary
Library           String
Library           Collections
Library           String
#Variables         descriptors/SOL001/VNFD/vnfd_SOL001.yaml
#Variables         descriptors/SOL006/VNFD/vnfd_SOL006.yaml

*** Keywords ***
Get all VNF Packages
    Log    Trying to get all VNF Packages present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}   

Check HTTP Response Body Does Not Contain softwareImages
    Log    Checking that field element is missing
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Should Be Empty    ${softwareImages}
    Log    Element is empty as expected
    
Check HTTP Response Body Does Not Contain additionalArtifacts
    Log    Checking that field element is missing
    ${additionalArtifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Should Be Empty    ${additionalArtifacts}
    Log    Element is empty as expected
    
Check HTTP Response Body Does Not Contain userDefinedData 
    Log    Checking that field element is missing
    ${userDefinedData}=    Get Value From Json    ${response['body']}    $..userDefinedData
    Should Be Empty    ${userDefinedData}
    Log    Element is empty as expected

Check HTTP Response Body Does Not Contain checksum
    Log    Checking that field element is missing
    ${checksum}=    Get Value From Json    ${response['body']}    $..checksum
    Should Be Empty    ${checksum}
    Log    Element is empty as expected    

GET VNF Packages with attribute-based filter
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using filter params
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?${POS_FILTER}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body VnfPkgsInfo Matches the requested Attribute-Based Filter
    Log    Checking that attribute-based filter is matched
    @{attr} =    Split String    ${POS_FILTER}       ,${VAR_SEPERATOR} 
    @{var_id} =    Split String    @{attr}[0]       ,${SEPERATOR}
    @{var_provider} =    Split String    @{attr}[1]       ,${SEPERATOR}
    Should Be True     "${response['body'][0]['vnfdId']}"=="@{var_id}[1]" and "${response['body'][0]['vnfProvider']}"=="@{var_provider}[1]"
    
GET VNF Packages with invalid attribute-based filter
    Log    Trying to perform a negative get, filtering by the inexistent filter 'nfvId'
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?${NEG_FILTER}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get all VNF Packages with malformed authorization token
    Pass Execution If    ${AUTH_USAGE} == 0    Skipping test as NFVO is not supporting authentication
    Log    The GET method queries using invalid token
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"${AUTHORIZATION_HEADER}": "${BAD_AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get all VNF Packages without authorization token
    Pass Execution If    ${AUTH_USAGE} == 0    Skipping test as it is not supporting authentication
    Log    The GET method queries omitting token
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Log    Execute Query and validate response
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET VNF Packages with all_fields attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using filter params
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?all_fields
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
   
Check HTTP Response Body VnfPkgsInfo Matches the requested all_fields selector
    Log    Trying to validate softwareImages schema
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Validate Json    softwareImage.schema.json    ${softwareImages[0]}
    Log    Validation for softwareImage schema OK
    Log    Trying to validate additionalArtifacts schema
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Validate Json    additionalArtifacts.schema.json    ${additional_artifacts[0]}
    Log    Validation for additionalArtifacts schema OK
    ${links}=    Get Value From Json    ${response['body']}    $.._links
    Validate Json    links.schema.json    ${links[0]}
    Log    Validation for _links schema OK

GET VNF Packages with exclude_default attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using exclude_default filter.
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?exclude_default   
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body VnfPkgsInfo Matches the requested exclude_default selector
    Log    Checking missing information for softwareImages element
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Should Be Empty    ${softwareImages}
    Log    softwareImages element is missing as excepted
    Log    Checking missing information for additionalArtifact element
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Should Be Empty    ${additional_artifacts}
    Log    additionalArtifact element is missing as excepted

GET VNF Packages with fields attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using fields
    Pass Execution If    ${NFVO_FIELDS} == 0    The NFVO is not able to use fields parameter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body vnfPkgsInfo Matches the requested fields selector
    Log    Trying to validate softwareImages schema
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Validate Json    softwareImage.schema.json    ${softwareImages[0]}
    Log    Validation for softwareImage schema OK
    Log    Trying to validate additionalArtifacts schema
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Validate Json    additionalArtifacts.schema.json    ${additional_artifacts[0]}
    Log    Validation for additionalArtifacts schema OK
    
GET VNF Packages with exclude_fields attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using fields
    Pass Execution If    ${NFVO_FIELDS} == 0    The NFVO is not able to use fields parameter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?exclude_fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body vnfPkgsInfo Matches the requested exclude_fields selector
    Log    Checking missing information for softwareImages element
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Should Be Empty    ${softwareImages}
    Log    softwareImages element is missing as excepted
    Log    Checking missing information for additionalArtifact element
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Should Be Empty    ${additional_artifacts}
    Log    additionalArtifact element is missing as excepted

GET all VNF Packages with invalid resource endpoint
    Log    Trying to perform a GET on a erroneous URI
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPkgId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Post Request to create new VNF Package Resource
    Log    Creating a new VNF Package
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${body} =    Get File    jsons/CreateVnfPkgInfoRequest.json
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
    
Check HTTP Response Body vnfPkginfo content against VNF Descriptor
    #${check_descriptors} flag, 1 to check descriptors
    Run Keyword If    ${check_descriptors} == 1    Check Individual VNFPkgInfo Content

Check Individual VNFPkgInfo Content    
    PARSE the Descriptor File
    Match the Response Attributes of vnfPkgInfo with Descriptors
    
PARSE the Descriptor File
    Run Keyword If  '${descriptorType}'=='SOL001'    Fetch Information from SOL001 descriptor file    ELSE    Fetch Information from SOL006 descriptor file
    
Fetch Information from SOL001 descriptor file
    Import Variables    ${CURDIR}/descriptors/SOL001/VNFD/vnfd_SOL001.yaml
    ${descriptor_id}=    Get Variable Value    ${topology_template.node_templates.properties.descriptor_id}
    ${provider}=    Get Variable Value    ${topology_template.node_templates.properties.provider}
    ${product_name}=    Get Variable Value    ${topology_template.node_templates.properties.product_name}
    ${software_version}=    Get Variable Value    ${topology_template.node_templates.properties.software_version}
    ${descriptor_version}=    Get Variable Value    ${topology_template.node_templates.properties.descriptor_version}
    
    Set Global Variable    ${Descriptor_ID}    ${descriptor_id}
    Set Global Variable    ${Provider}    ${provider}
    Set Global Variable    ${Product_Name}    ${product_name}
    Set Global Variable    ${Software_Version}    ${software_version}
    Set Global Variable    ${Descriptor_Version}    ${descriptor_version}

Fetch Information from SOL006 descriptor file
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    ${descriptor_id}=    Get Variable Value    ${nfv.vnfd[0].id}
    ${provider}=    Get Variable Value    ${nfv.vnfd[0].provider}
    ${product_name}=    Get Variable Value    ${nfv.vnfd[0].product-name}
    ${software_version}=    Get Variable Value    ${nfv.vnfd[0].software-version}
    ${descriptor_version}=    Get Variable Value    ${nfv.vnfd[0].version}    
    Set Global Variable    ${Descriptor_ID}    ${descriptor_id}
    Set Global Variable    ${Provider}    ${provider}
    Set Global Variable    ${Product_Name}    ${product_name}
    Set Global Variable    ${Software_Version}    ${software_version}
    Set Global Variable    ${Descriptor_Version}    ${descriptor_version}

Match the Response Attributes of vnfPkgInfo with Descriptors
    #Checking Response attributes with VNF Descriptors
    Should Be Equal As Strings    ${response['body']['vnfdId']}    ${Descriptor_Id}
    Should Be Equal As Strings    ${response['body']['vnfProvider']}    ${Provider}
    Should Be Equal As Strings    ${response['body']['vnfProductName']}    ${Product_Name}
    Should Be Equal As Strings    ${response['body']['vnfSoftwareVersion']}    ${Software_Version}
    Should Be Equal As Strings    ${response['body']['vnfdVersion']}    ${Descriptor_Version}

Check Postcondition VNF Package Resource Exists
    Log    Checking that nsd info exists
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${response['body']['id']}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    vnfPkgInfo

Send PUT Request for all VNF Packages
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PATCH Request for all VNF Packages
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for all VNF Packages
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package
    Log    Trying to get a VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check HTTP Response Body vnfPkgInfo Identifier matches the requested VNF Package
    Log    Going to validate pacakge info retrieved
    Should Be Equal    ${response['body']['id']}    ${vnfPackageId} 
    Log    Pacakge identifier as expected

GET Individual VNF Package with invalid resource identifier
    Log    Trying to perform a negative get, using wrong authorization bearer
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request for individual VNF Package
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual VNF Package
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH to disable Individual VNF Package
    Log    Trying to perform a PATCH. As prerequisite the nsdInfo shall be in enabled operational state
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${body}=    Get File    jsons/VnfPkgInfoModificationsDisabled.json
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check Postcondition VNF Package is in operational state
    [Arguments]    ${status}
    Log    Checking postcondition op status
    Should Be Equal As Strings   ${response['body']['operationalState']}    ${status} 

Send PATCH to enable Individual VNF Package
    Log    Trying to perform a PATCH. As prerequisite the nsdInfo shall be in disabled operational state
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${body}=    Get File    jsons/VnfPkgInfoModifications.json
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual VNF Package
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${disabledVnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check Postcondition VNF Package is Deleted
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${disabledVnfPackageId}
    Integer    response status    404

Send DELETE Request for Individual VNF Package in operational state ENABLED
    Log    Trying to perform a DELETE nsdInfo in ENABLED operational state
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check Postcondition VNF Package Exists
    Log    Checking that vnf pacakge still exists
    GET Individual VNF Package

Send DELETE Request for Individual VNF Package used for instantiated VNF instances
    Log    Trying to perform a DELETE nsdInfo in ENABLED operational state
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${instantiatedVnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Get VNFD in Individual VNF Package in Zip Format
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Get VNFD in Individual VNF Package with security information
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Check HTTP Response Header Content-Type Is Any of
    [Arguments]   ${header1}    ${header2}
    Should Contain Any  ${response['headers']['Content-Type']}    ${header1}    ${header2}

Get multi file VNFD in Individual VNF Package in Plain Format
    Log    Trying to get a negative case performing a get on a VNFD from a given VNF Package present in the NFVO Catalogue. Accept will be text/plain but VNFD is composed my multiple files.
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Get VNFD in Individual VNF Package with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPkgId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Get VNFD in Individual VNF Package with conflict due to onboarding state 
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send POST Request for VNFD in individual VNF Package
    Log    Trying to perform a POST. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send PUT Request for VNFD in individual VNF Package
    Log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Send PATCH Request for VNFD in individual VNF Package
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send DELETE Request for VNFD in individual VNF Package
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

GET Individual VNF Package Content  
    Log    Trying to get a VNF Package Content
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Header Content-Type Is
    [Arguments]   ${header}
    Should Contain    ${response['headers']['Content-Type']}    ${header}

GET Individual VNF Package Content with Range Request
    Log    Trying to get a VNF Package Content using RANGE using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package Content with explicit Range Request
    Log    Trying to get a VNF Package Content using RANGE using an NFVO that can handle it
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}


Check HTTP Response Header Content-Range Is Present and Matches the requested range
    Log    Check Content-Range HTTP Header
    Should Contain    ${response['headers']}    Content-Range
    Should Be Equal As Strings    ${response['headers']['Content-Range']}    ${range}/${full_size}
    Log    Header Content-Range is present
    
Check HTTP Response Header Content-Length Is Present and Matches the requested range length
    Log    Check Content-Length HTTP Header
    Should Contain    ${response['headers']}    Content-Length
    Should Be Equal As Integers    ${response['headers']['Content-Length']}    ${length}
    Log    Header Content-Length is present

GET Individual VNF Package Content with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package Content with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPkgId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Content for VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a VNF Package content present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request for individual VNF Package Content
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual VNF Package Content
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request to upload VNF Package Content
    Log    Trying to perform a PUT. This method upload the content of a NSD
    Create Session      nbi     ${NFVO_SCHEMA}://${NFVO_HOST}:${NFVO_PORT}  headers={"Accept": "${ACCEPT_ZIP}", "Content-Type": "application/zip", "${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}      verify=False
    ${body}=  Get Binary File     ${contentZipVnfPackage}
    ${response}=    Put Request     nbi     ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${newVnfPackageId}/package_content    data=${body}
    Should Be Empty     ${response.text}
    ${response}=    Create Dictionary   status=${response.status_code}
    Set Suite Variable      ${response}     ${response}


Check Postcondition VNF Package Content is uploaded and available in the NFVO
    Log    Trying to get a VNF Package Content
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${newVnfPackageId}/package_content
    Integer    response status    200
 
Send PUT Request to upload VNF Package Content with conflict due to onboarding state
    Log    Trying to perform a PUT. This method upload the content of a NSD
    Create Session      nbi     ${NFVO_SCHEMA}://${NFVO_HOST}:${NFVO_PORT}  headers={"Accept": "${ACCEPT_ZIP}", "Content-Type": "application/zip", "${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}      verify=False
    ${body}=  Get Binary File     ${contentZipVnfPackage}
    ${response}=    Put Request     nbi     ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${creatingVnfPackageId}/package_content    data=${body}
    Should Be Empty     ${response.text}
    ${response}=    Create Dictionary   status=${response.status_code}
    Set Suite Variable      ${response}     ${response}
 
Send PATCH Request for individual VNF Package Content
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual VNF Package Content
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual VNF Package Artifact
    Log    Trying to get a VNF Package Artifact
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package Artifact in octet stream format  
    Log    Trying to get a VNF Package Artifact
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageOctetStreamId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package Artifact with Range Request
    Log    Trying to get an Artifact using RANGE Header and using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual VNF Package Artifact with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
GET Individual VNF Package Artifact with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPkgId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Artifact for VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a VNF Package artifact present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request for individual VNF Package Artifact
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual VNF Package Artifact
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for individual VNF Package Artifact
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual VNF Package Artifact
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request to upload VNF Package Content from URI
    Log    Trying to perform a POST. The POST method provides the information for the NFVO to get the content of a VNF package.
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/UploadVnfPkgFromUriRequest.json
    ${body}=        Format String   ${template}     vnfPkgUri=${vnfPkgUri}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgViaUriId}/package_content/upload_from_uri    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request to upload VNF Package Content from URI with conflict due to onboarding state
    Log    Trying to perform a POST. The POST method provides the information for the NFVO to get the content of a VNF package.
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/UploadVnfPkgFromUriRequest.json
    ${body}=        Format String   ${template}     vnfPkgUri=${vnfPkgUri}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${creatingVnfPkgViaUriId}/package_content/upload_from_uri    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send GET Request for individual VNF Package Content from URI
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgViaUriId}/package_content/upload_from_uri
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual VNF Package Content from URI
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgViaUriId}/package_content/upload_from_uri
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PATCH Request for individual VNF Package Content from URI
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgViaUriId}/package_content/upload_from_uri
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual VNF Package Content from URI
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgViaUriId}/package_content/upload_from_uri
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get all VNF Package Subscriptions
    Log    Trying to get the list of subscriptions
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get VNF Package Subscriptions with attribute-based filters
    Log    Trying to get the list of subscriptions using filters
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${filter_ok}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}    

Get VNF Package Subscriptions with invalid attribute-based filters
    Log    Trying to get the list of subscriptions using filters with wrong attribute name
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${filter_ko}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Get VNF Package Subscriptions with invalid resource endpoint
    Log    Trying to perform a request on a Uri which doesn't exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscription
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send Post Request for VNF Package Subscription
    Log    Trying to create a new subscription
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}      vnfPkgId=${vnfPkgId}    vnfdId=${Descriptor_ID}    vnfProvider=${Provider}    vnfProductName=${Product_Name}    vnfSoftwareVersion=${Software_Version}    vnfdVersions=${Descriptor_Version}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    
	${output}=    Output    response
	Set Global Variable    ${response}    ${output}
	Clear Requests    ${callback_endpoint}
    #Run Keyword If    ${NFVO_CHECKS_NOTIF_ENDPOINT} == 1    #removed as the check of notif endpoint is done by NFVO while processing subscription reques
    #...    Check Notification Endpoint 

Send Post Request for Duplicated VNF Package Subscription
    Log    Trying to create a subscription with an already created content
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}      vnfPkgId=${vnfPkgId}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    
	${output}=    Output    response
	Set Global Variable    ${response}    ${output}
	Clear Requests    ${callback_endpoint}
    #Run Keyword If    ${NFVO_CHECKS_NOTIF_ENDPOINT} == 1    #removed as the check of notif endpoint is done by NFVO while processing subscription reques
    #...    Check Notification Endpoint 

Send Put Request for VNF Package Subscriptions
    Log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send Patch Request for VNF Package Subscriptions
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send Delete Request for VNF Package Subscriptions
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}    

Check HTTP Response Body PkgmSubscription Attributes Values Match the Issued Subscription
    Log    Check Response matches subscription
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}      vnfPkgId=${vnfPkgId}
    ${subscription}=    evaluate    json.loads('''${body}''')    json
    Should Be Equal    ${response['body']['callbackUri']}    ${subscription['callbackUri']}


Check Postcondition VNF Package Subscription Is Set
    [Arguments]    ${location}=""
    Log    Check Postcondition subscription exist
    Log    Trying to get the subscription
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Run Keyword If    ${location} == Location
    ...    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${response['body']['id']}
    Run Keyword If    ${location} == Location
    ...    REST.GET    ${response['headers']['Location']}  
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Check HTTP Response Status Code Is    200

Check Postcondition Subscription Resource Returned in Location Header Is Available
    Log    Going to check postcondition
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${response['headers']['Location']}
    Integer    response status    200
    Log    Received a 200 OK as expected
    ${contentType}=    Output    response headers Content-Type
    Should Contain    ${contentType}    application/json
    ${result}=    Output    response body
    Validate Json    PkgmSubscription.schema.json    ${result}
    Log    Validated PkgmSubscription schema
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}      vnfPkgId=${vnfPkgId}
    ${subscription}=    evaluate    json.loads('''${body}''')    json
    Should Be Equal    ${result['callbackUri']}    ${subscription['callbackUri']}
    Log    Validated Issued subscription is same as original

Get Individual VNF Package Subscription
    Log    Trying to get a single subscription identified by subscriptionId
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET individual VNF Package Subscription with invalid resource identifier
    Log    Trying to perform a request on a subscriptionID which doesn't exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${erroneousSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete request for individual VNF Package Subscription
    Log    Trying to perform a DELETE on a subscriptionId
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check Postcondition VNF Package Subscription is Deleted
    Log    Check Postcondition Subscription is deleted
    GET individual VNF Package Subscription
    Check HTTP Response Status Code Is    404 

Send Delete request for individual VNF Package Subscription with invalid resource identifier
    Log    Trying to perform a DELETE on a subscriptionId which doesn't exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${erroneousSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Post request for individual VNF Package Subscription
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${newSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Put request for individual VNF Package Subscription
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Patch request for individual VNF Package Subscription
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
   
Check Postcondition VNF Package Subscription is Unmodified (Implicit)
    Log    Check postconidtion subscription not modified
    GET individual VNF Package Subscription
    Log    Check Response matches original VNF Threshold
    ${subscription}=    evaluate    json.loads('''${response['body']}''')    json
    Should Be Equal    ${origResponse['body']['id']}    ${subscription.id}
    Should Be Equal    ${origResponse['body']['callbackUri']}    ${subscription.callbackUri}

Check Postcondition VNF Package Subscription is not Created
    Log    Trying to get a new subscription
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${newSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Check HTTP Response Status Code Is    404

Create Sessions
    Start Process  java  -jar  ${MOCK_SERVER_JAR}    -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}
    
Check HTTP Response Status Code Is
    [Arguments]    ${expected_status}
    Should Be Equal As Strings    ${response['status']}    ${expected_status}
    Log    Status code validated
    
Check HTTP Response Body Json Schema Is
    [Arguments]    ${input}
    Run Keyword If    '${input}' == 'ProblemDetails'    Should Contain    ${response['headers']['Content-Type']}    application/problem+json
    ...    ELSE    Should Contain    ${response['headers']['Content-Type']}    application/json
    ${schema} =    Catenate    SEPARATOR=    ${input}    .schema.json
    Validate Json    ${schema}    ${response['body']}
    Log    Json Schema Validation OK  
    
Check HTTP Response Body Subscriptions Match the requested Attribute-Based Filter
    Log    Check Response includes VNF Package Management according to filter
    @{words} =  Split String    ${filter_ok}       ,${SEPERATOR} 
    Should Be Equal As Strings    ${response['body'][0]['callbackUri']}    @{words}[1]

Check HTTP Response Body Subscription Identifier matches the requested Subscription
    Log    Trying to check response ID
    Should Be Equal    ${response['body']['id']}    ${subscriptionId} 
    Log    Subscription identifier as expected

Check HTTP Response Body PkgmSubscription content against VNF Descriptor
    #${check_descriptors} flag, 1 to check descriptors
    Run Keyword If    ${check_descriptors} == 1    Check Individual PkgmSubscription Content

Check Individual PkgmSubscription Content
    PARSE the Descriptor File
    Match the Response Attributes of PkgmSubscription with Descriptors
    
Match the Response Attributes of PkgmSubscription with Descriptors
    Should Be Equal As Strings    ${response['body']['filter']['vnfdId']}    ${Descriptor_ID}
    Should Be Equal As Strings    ${response['body']['filter']['vnfProductsFromProviders']['vnfProvider']}    ${Provider}
    Should Be Equal As Strings    ${response['body']['filter']['vnfProductsFromProviders']['vnfProducts']['vnfProductName']}    ${Product_Name}
    Should Be Equal As Strings    ${response['body']['filter']['vnfProductsFromProviders']['vnfProducts']['vnfProductName']['versions']['vnfSoftwareVersion']}    ${Software_Version}
    Should Be Equal As Strings    ${response['body']['filter']['vnfProductsFromProviders']['vnfProducts']['vnfProductName']['versions']['vnfdVersion']}    ${Descriptor_Version}

Check Notification Endpoint
    &{notification_request}=  Create Mock Request Matcher	REST.GET  ${callback_endpoint}    
    &{headers} =	Create Dictionary	Content-Type=application/json
    &{notification_response}=  Create Mock Response	headers=${headers}  status_code=204
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Wait Until Keyword Succeeds    ${total_polling_time}   ${polling_interval}   Verify Mock Expectation    ${notification_request}
    Clear Requests  ${callback_endpoint}

Check HTTP Response Body Matches the Subscription
    Log    Check Response matches subscription
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}      vnfPkgId=${vnfPkgId}
    ${subscription}=    evaluate    json.loads('''${body}''')    json
    Should Be Equal As Strings    ${response['body']['callbackUri']}    ${subscription['callbackUri']}
    
Check HTTP Response Body Is Empty
    Should Be Empty    ${response['body']}    
    Log    No json schema is provided. Validation OK  
    
Check HTTP Response Header Contains
    [Arguments]    ${CONTENT_TYPE}
    Should Contain    ${response['headers']}    ${CONTENT_TYPE}
    Log    Header is present
    
Check HTTP Response Header Contains Location
    ${linkURL}=    Get Value From Json    ${response['headers']}    $..Link
    Should Not Be Empty    ${linkURL}

Get access configuration information for external artifacts
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/ext_artifacts_access
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send PUT Request to access configuration information for external artifacts
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=  Get File     jsons/ExternalArtifactsAccessConfig.json
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/ext_artifacts_access    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check Postcondition individual VNF Package is created
    GET Individual VNF Package
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is   vnfPkgInfo
    Should Be True     '${response['body']['id']}' =='CREATED' or '${response['body']['id']}'=='ERROR'
    
Send POST Request for access configuration for external artifacts
    Log    Trying to perform a POST. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/ext_artifacts_access
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Send PATCH Request for access configuration for external artifacts
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/ext_artifacts_access
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send DELETE Request for Access configuration for external artifacts
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/ext_artifacts_access
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Manifest in Individual VNF Package in plainFormat
        Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Manifest in Individual VNF Package with security information
    Log    Trying to get a manifest from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/manifest/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Manifest in Individual VNF Package with unacceptable request
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/manifest/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send POST Request for manifest in individual VNF Package
    Log    Trying to perform a POST. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send PUT Request for manifest in individual VNF Package
    Log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Send PATCH Request for manifest in individual VNF Package
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send DELETE Request for manifest in individual VNF Package
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Get Artifact in VNF Package
    Log    Trying to get Artifact in VNF Package
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get Artifact in VNF Package with exclude_all_mano_artifacts parameter
    Log    Trying to get Artifact in VNF Package with exclude_all_mano_artifacts
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/exclude_all_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}  
    
Get Artifact in VNF Package with exclude_all_non_mano_artifacts parameter
    Log    Trying to get Artifact in VNF Package with security certificates
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with include_external_artifacts parameter
    Log    Trying to get Artifact in VNF Package with include_external_artifacts
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/include_external_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with select_non_mano_artifact_sets parameter
    Log    Trying to get Artifact in VNF Package with security certificates
    Pass Execution If    ${NFVO_non-MANO_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/select_non_mano_artifact_sets=${non_mano_artifact_sets}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with include_signatures parameter
    Log    Trying to get Artifact in VNF Package with security certificates
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with Range Request
    Log    Trying to get an Artifact using RANGE Header and using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with exclude_all_non_mano_artifacts and select_non_mano_artifact_sets
    Log    Trying to get Artifact in VNF Package 
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts&select_non_mano_artifact_sets=${non_mano_artifact_sets}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with exclude_all_non_mano_artifacts and exclude_all_mano_artifacts
    Log    Trying to get Artifact in VNF Package 
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts&exclude_all_non_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Artifact of VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a VNF Package artifact present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET VNF Package Artifact with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send POST Request for VNF Package Artifact
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for VNF Package Artifact
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for VNF Package Artifact
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for VNF Package Artifact
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    REST.DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
