*** Settings ***
Library           JSONSchemaLibrary    schemas/
Resource          environment/variables.txt    # Generic Parameters
Library           OperatingSystem
Library           JSONLibrary
Library           Process
Library           String
Library           Collections
Library           String
Library           MockServerLibrary
Variables         descriptors/SOL001/VNFD/vnfd_SOL001.yaml
Variables         descriptors/SOL006/VNFD/vnfd_SOL006.yaml
Library           REST    ${None}    ssl_verify=${False}

*** Keywords ***
Initialize System
    Create Sessions

Create Sessions   
    Start Process  java  -jar  ${MOCK_SERVER_JAR}    -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}

GET Individual VNF Package
    [Documentation]    Test ID: 7.3.3.2.1
    ...    Test title: GET Individual VNF Package
    ...    Test objective: The objective is to test the retrieval of an individual VNF package information perform a JSON schema validation of the collected data structure
    ...    Pre-conditions: One or more VNF packages are onboarded in the NFVO.
    ...    Reference: Clause 10.4.3.3.2 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_NFVO
    ...    Applicability: none
    ...    Post-Conditions: none
    GET Individual VNF Package [Keyword]
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    vnfPkgInfo
    Check HTTP Response Body vnfPkgInfo Identifier matches the requested VNF Package
    Check HTTP Response Body of Individual VNF Package content against VNF Descriptor

POST Create a new vnfInstance
    [Documentation]    Test ID: 7.3.1.1.1
    ...    Test title: POST Create a new vnfInstance
    ...    Test objective: The objective is to create a new VNF instance resource
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.2.3.1 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: VNF instance created
    POST Create a new vnfInstance [keyword]
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is    vnfInstance

GET information about multiple VNF instances
    [Documentation]    Test ID: 7.3.1.1.2
    ...    Test title: GET information about multiple VNF instances
    ...    Test objective: The objective is to query information about multiple VNF instances
    ...    Pre-conditions:    none
    ...    Reference: Clause 5.4.2.3.2 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    GET multiple vnfInstances
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    vnfInstances

Get Information about an individual VNF Instance
    [Documentation]    Test ID: 7.3.1.2.2
    ...    Test title: Get Information about an individual VNF Instance
    ...    Test objective: The objective is to create a new VNF instance resource
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.3.3.2 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    GET individual vnfInstance
    Check HTTP Response Header Contains ETag and Last-Modified
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    vnfInstance
    Check HTTP Response Body vnfInstance content against VNF Descriptor

DELETE Individual VNFInstance
    [Documentation]    Test ID: 7.3.1.2.7
    ...    Test title: DELETE Individual VNFInstance
    ...    Test objective: The objective is to delete a VNF instance
    ...    Pre-conditions: the VNF instance resource is existing
    ...    Reference: Clause 5.4.3.3.5 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: VNFInstance deleted
    DELETE individual vnfInstance [Keyword]
    Check HTTP Response Status Code Is    204

PATCH Individual VNFInstance
    [Documentation]    Test ID: 7.3.1.2.4
    ...    Test title: PATCH Individual VNFInstance
    ...    Test objective: This method modifies an individual VNF instance resource
    ...    Pre-conditions: an existing VNF instance resource
    ...    Reference: Clause 5.4.3.3.4 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: VNFInstance modified
    PATCH individual vnfInstance [keyword]
    Check HTTP Response Status Code Is    202
    Check Operation Occurrence Id existence

Instantiate a vnfInstance
    [Documentation]    Test ID: 7.3.1.3.1
    ...    Test title: Post Instantiate Individual VNFInstance
    ...    Test objective: The objective is to instantiate a VNF instance
    ...    Pre-conditions: VNF instance resource is in NOT INSTANTIATED state
    ...    Reference: Clause 5.4.4.4.1 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    POST instantiate individual vnfInstance
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location
    Check Individual VNF LCM operation occurrence operationState is    STARTING

VNF Instantiation
    [Documentation]    Test ID: 7.3.1.26.1
    ...    Test title: VNF Instantiation workflow
    ...    Test objective: The objective is to test the workflow for the instantiation of a VNF instance
    ...    Pre-conditions: VNF instance resources is already created. NFVO is subscribed to VNF LCM Operation Occurrence notifications 
    ...    Reference: Clause 5.4.4 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability:  NFVO is able to receive notifications from VNFM
    ...    Post-Conditions: VNF instance in INSTANTIATED state
    Send VNF Instantiation Request
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location
    Check Operation Occurrence Id existence
    Check Operation Notification For    STARTING
    Check Operation Notification For    PROCESSING
    Check Operation Notification For    COMPLETED
    Check Postcondition VNF Status  INSTANTIATED

POST Terminate a vnfInstance
    [Documentation]    Test ID: 7.3.1.7.1
    ...    Test title: POST Terminate a vnfInstance
    ...    Test objective: The objective is to test that POST method terminate a VNF instance
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.8.3.1 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    POST Terminate VNF
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location
    Check Individual VNF LCM operation occurrence operationState is    STARTING

Terminate a VNF Instance
    [Documentation]    Test ID: 7.3.1.33.1
    ...    Test title: Terminate a VNF Instance
    ...    Test objective: The objective is to terminate a VNF instance.
    ...    Pre-conditions: VNF instance in INSTANTIATED state 
    ...    Reference: Clause 5.3.3 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: NFVO is able to receive notifications from VNFM. 
    ...    Post-Conditions: VNF instance in NOT_INSTANTIATED state 
    Send Terminate VNF Request
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location 
    Check Operation Occurrence Id existence
    Check Operation Notification For    STARTING
    Check Operation Notification For    PROCESSING
    Check Operation Notification For    COMPLETED
    Check Postcondition VNF Terminate

GET status information about multiple VNF LCM Operation OCC 
    [Documentation]    Test ID: 7.3.1.11.2
    ...    Test title: GET status information about multiple VNF LCM Operation OCC
    ...    Test objective: The objective is to test that GET method retrieve Query status information about multiple VNF lifecycle management operation occurrences.
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.12.3.2 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none   
    GET VNF LCM Operation occurrences
    Check HTTP Response Status Code Is    202
    Check HTTP Response Body Json Schema Is    VnfLcmOpOccs 

Get Individual VNF LCM Operation occurrence
    [Documentation]    Test ID: 7.3.1.12.2
    ...    Test title: Get Individual VNF LCM Operation occurrences
    ...    Test objective: The objective is to test that this method retrieve information about a VNF lifecycle management operation occurrence 
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.13.3.2 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none 
    Get Individual VNF LCM Operation occurrence [keyword]
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    VnfLcmOpOcc 
    Check HTTP Response Body of Individual VNF LCM Operation occurrences content against VNF Descriptor

POST Create a new subscription
    [Documentation]    Test ID: 7.3.1.17.1
    ...    Test title: POST Create a new subscription
    ...    Test objective: The POST method creates a new subscription
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.18.3.1 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: in response header Location shall not be null
    Post Create subscription
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is    Subscription

GET Subscriptions
    [Documentation]    Test ID: 7.3.1.17.4
    ...    Test title: GET Subscriptions
    ...    Test objective: The objective is Get the list of active subscriptions
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.18.3.2 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none 
    Get subscriptions [keyword]
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    Subscriptions

GET Individual Subscription
    [Documentation]    Test ID: 7.3.1.18.2
    ...    Test title: GET Individual Subscription
    ...    Test objective: The objective is to test the Get individual subscription
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.19.3.2 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none 
    Get Individual Subscription [keyword]
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    Subscription

DELETE an individual subscription
    [Documentation]    Test ID: 7.3.1.18.5
    ...    Test title: DELETE an individual subscription
    ...    Test objective: The objective is to test that the individual subscription is succesfully deleted
    ...    Pre-conditions: none
    ...    Reference: Clause 5.4.19.3.5 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: Subscription deleted from VNFM
    Delete Individual subscription
	Check HTTP Response Status Code Is    204
    Check Postcondition Subscription Resource is Deleted

VNF Instance Scale Out
    [Documentation]    Test ID: 7.3.1.32.1
    ...    Test title: VNF Instance Scale Out workflow
    ...    Test objective: The objective is to test the workflow for the scaling out a VNF instance
    ...    Pre-conditions: VNF instance in INSTANTIATED state (Test ID 5.4.4.1). NFVO is subscribed to VNF LCM Operation Occurrence notifications (Test ID 5.4.20.1)
    ...    Reference: Clause 5.4.5 - ETSI GS NFV-SOL 003 [1] v4.4.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability:  NFVO is able to receive notifications from VNFM. Scale operation is supported for the VNF (as capability in the VNFD)
    ...    Post-Conditions: VNF instance still in INSTANTIATED state and VNF is scaled out
    Send VNF Scale Out Request
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location 
    Check Operation Occurrence Id existence
    Check Operation Notification For    STARTING
    Check Operation Notification For    PROCESSING
    Check Operation Notification For    COMPLETED
    Check Postcondition VNF    SCALE_OUT

Post Retry operation task
    [Documentation]    Test ID: 7.3.1.13.1
    ...    Test title: Post Retry operation task
    ...    Test objective: The objective is to test that POST method The POST method initiates retrying a VNF lifecycle operation if the operation is in FAILED_TEMP state
    ...    Pre-conditions: the "VNF LCM operation occurrence" resource is in "FAILEDGET status information about multiple VNF LCM Operation OCC_TEMP" state.
    ...    Reference: Clause 5.4.14.4.1 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none 
    Post Retry operation
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location
    Check Individual VNF LCM operation occurrence operationState is    STARTING

Post Rollback operation task
    [Documentation]    Test ID: 7.3.1.14.1
    ...    Test title: Post Rollback operation task
    ...    Test objective: The objective is to test that POST method initiates rollback on a VNF lifecycle operation
    ...    Pre-conditions: the "VNF LCM operation occurrence" resource is in "FAILED_TEMP" state.
    ...    Reference: Clause 5.4.15.3.1 - ETSI GS NFV-SOL 003 [1] v5.1.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none 
    Post Rollback operation
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location
    Check Individual VNF LCM operation occurrence operationState is    STARTING

VNF Instance Scale In
    Send VNF Scale In Request
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location 
    Check Operation Occurrence Id existence
    Check Operation Notification For    STARTING
    Check Operation Notification For    PROCESSING
    Check Operation Notification For    COMPLETED
    Check Postcondition VNF    SCALE_IN

VNF Instance Heal
    Send Heal VNF Request
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location 
    Check Operation Occurrence Id existence
    Check Operation Notification For    STARTING
    Check Operation Notification For    PROCESSING
    Check Operation Notification For    COMPLETED
    Check Postcondition VNF Heal

Send Heal VNF Request
    Log    Trying to heal a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/healVnfRequest.json
    Post    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfid}/heal    ${body}
    ${output}=    Output    response
    Set Global Variable    ${response}    ${output}

Check Postcondition VNF Heal
    Check resource Instantiated

VNF Change External Connectivity
    Send Change Ext Connectivity Request
    Check HTTP Response Status Code Is    202
    Check HTTP Response Header Contains    Location 
    Check Operation Occurrence Id existence
    Check Operation Notification For    STARTING
    Check Operation Notification For    PROCESSING
    Check Operation Notification For    COMPLETED
    Check Postcondition VNF Change Ext Connectivity

Send Change Ext Connectivity Request
    Log    Trying to change the external connectivity of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/changeExtVnfConnectivityRequest.json
    Post    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfInstanceId}/change_ext_conn    ${body}
    ${output}=    Output    response
    Set Global Variable    ${response}    ${output}

Get Vnf Ext Link Id
    [Arguments]    ${vnfInstanceId}
    ${vnfInstance}=    Get Vnf Instance    ${vnfInstanceId}
    [Return]    ${vnfInstance.instantiatedVnfInfo.extVirtualLinkInfo.id}

Get Vnf Instance 
    [Arguments]    ${vnfInstanceId}
    Set Headers    {"Accept":"${ACCEPT}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfInstanceId}
    ${body}=    Output    response body
    [Return]    ${body}

Check Postcondition VNF Change Ext Connectivity
    Check resource instantiated
    ${extVLId}=    Get Vnf Ext Link Id    ${vnfid}
    Should be Equal    ${changeVnfExtConnectivityRequest["extVirtualLinks"][0]["id"]}    ${extVLId}
    
Check Operation Notification For Change Ext Connectivity 
    [Arguments]    ${status}
    Check Operation Notification    VnfLcmOperationOccurrenceNotification   ${status}

GET Individual VNF Package [keyword]
    Log    Trying to get a VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${NFVO}/${NFVO_API_ROOT}/${NFVO_API_NAME}/${NFVO_API_MAJOR_VERSION}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Global Variable    ${response}    ${output}

POST Create a new vnfInstance [keyword]
    Log    Create VNF instance by POST to /vnf_instances
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/createVnfRequest.json
    ${body}=    Format String    ${template}    vnfdId=${vnfdId}
    Post    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}
    Set Global Variable    ${vnfid}    ${outputResponse['body']['id']}

GET multiple vnfInstances
    Log    Query VNF The GET method queries information about multiple VNF instances.
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    GET    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

GET individual vnfInstance
    log    Trying to get information about an individual VNF instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfid}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

DELETE individual vnfInstance [Keyword]
    log    Trying to delete an individual VNF instance
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${response['body']['id']}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

PATCH individual vnfInstance [Keyword]
    log    Trying to modify an individual VNF instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_PATCH}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/patchBodyRequest.json
    ${body}=    Format String    ${template}    vnfdId=${Descriptor_ID}    vnfdVersion= ${Descriptor_Version}
    Patch    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${response['body']['id']}    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

POST instantiate individual vnfInstance
    Log    Trying to Instantiate a vnf Instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/instantiateVnfRequest.json
    Post    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${response['body']['id']}/instantiate    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

POST Terminate VNF
    Log    Trying to terminate a VNF instance.
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/terminateVnfRequest.json
    ${body}=    Format String    ${template}    terminationType=${terminationType}
    Post    ${VNFM}/${VNFM_API_ROOT}/${VNFM_API_ROOT}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${response['body']['id']}/terminate    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}
#-----------------------------------------Individual VNF Package------------------------------

Check HTTP Response Body of Individual VNF Package content against VNF Descriptor
    Run Keyword If    ${check_descriptors} == 1    Check Individual VNF Package Content

Check Individual VNF Package Content
    PARSE the Descriptor File
    Match the VNF Package Response Attributes with Descriptors
#-------------------------------------------Individual Instance------------------------

Check HTTP Response Body vnfInstance content against VNF Descriptor
    Run Keyword If    ${check_descriptors} == 1    Check Individual VNF Instance Content

Check Individual VNF Instance Content
    PARSE the Descriptor File
    Match the VNF Instance Response Attributes with Descriptors
#-------------------------------------------descriptor------------------------

PARSE the Descriptor File
    Run Keyword If    '${descriptorType}'=='SOL001'    Fetch Information from SOL001 descriptor file    ELSE    Fetch Information from SOL006 descriptor file

Fetch Information from SOL001 descriptor file
    Import Variables    ${CURDIR}/descriptors/SOL001/VNFD/vnfd_SOL001.yaml
    @{VDU_labels}=    Create List
    @{VNF_labels}=    Create List
    @{VirtualLink_labels}=    Create List
    @{CP_labels}=    Create List
    @{Storage_labels}=    Create List
    FOR    ${key}    IN    @{node_types.keys()}
        ${node_type}=    Get Variable Value    ${node_types['${key}']['derived_from']}
        ${is_VNF}=    Run Keyword And Return Status    Should Be Equal As Strings    ${node_type}    ${tosca_type_VNF}
        Run Keyword If    ${is_VNF}    Set Global Variable    ${tosca_type_derived_from_VNF}    ${key}
    END
    ${derived_type_is_used}=    Run Keyword And Return Status    Should not be empty    ${tosca_type_derived_from_VNF}
    Run Keyword If    ${derived_type_is_used}    Set Global Variable    ${tosca_type_VNF}    ${tosca_type_derived_from_VNF}
    FOR    ${key}    IN    @{topology_template.node_templates.keys()}
        ${key_type}=    Get Variable Value    ${topology_template.node_templates['${key}'].type}
        ${VDU_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_VDU_compute}
        ${VNF_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_VNF}
        ${Link_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_virtual_link}
        ${VDUcp_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_VDU_cp}
        ${Storage_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_storage}
        Run Keyword If    ${VDU_check}    Append To List    ${VDU_labels}    ${key}
        Run Keyword If    ${VNF_check}    Append To List    ${VNF_labels}    ${key}
        Run Keyword If    ${VNF_check}    Get VNF Attributes from SOL001    ${key}
        Run Keyword If    ${Link_check}    Append To List    ${VirtualLink_labels}    ${key}
        Run Keyword If    ${VDUcp_check}    Append To List    ${CP_labels}    ${key}
        Run Keyword If    ${Storage_check}    Append To List    ${Storage_labels}    ${key}
    END
    Set Global Variable    @{VDU_IDs}    @{VDU_labels}
    Set Global Variable    @{VNF_IDs}    @{VNF_labels}
    Set Global Variable    @{VirtualLink_IDs}    @{VirtualLink_labels}
    Set Global Variable    @{CP_IDs}    @{CP_labels}
    Set Global Variable    @{Storage_IDs}    @{Storage_labels}

Get VNF Attributes from SOL001
    [Arguments]    ${VNF_label}
    Import Variables    ${CURDIR}/descriptors/SOL001/VNFD/vnfd_SOL001.yaml
    ${descriptor_id}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.descriptor_id}
    ${provider}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.provider}
    ${product_name}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.product_name}
    ${software_version}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.software_version}
    ${descriptor_version}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.descriptor_version}
    ${flavour_id}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.flavour_id}
    Set Global Variable    ${Descriptor_ID}    ${descriptor_id}
    Set Global Variable    ${Provider}    ${provider}
    Set Global Variable    ${Product_Name}    ${product_name}
    Set Global Variable    ${Software_Version}    ${software_version}
    Set Global Variable    ${Descriptor_Version}    ${descriptor_version}
    Set Global Variable    ${Flavour_ID}    ${flavour_id}

Fetch Information from SOL006 descriptor file
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    ${descriptor_id}=    Get Variable Value    ${nfv.vnfd[0].id}
    ${provider}=    Get Variable Value    ${nfv.vnfd[0].provider}
    ${product_name}=    Get Variable Value    ${nfv.vnfd[0]['product-name']}
    ${software_version}=    Get Variable Value    ${nfv.vnfd[0]['software-version']}
    ${descriptor_version}=    Get Variable Value    ${nfv.vnfd[0].version}
    ${VDUcount}=    Get Length    ${nfv.vnfd[0].vdu}
    ${extCP_count}=    Get Length    ${nfv.vnfd[0]['ext-cpd']}
    ${virtualLink_count}=    Get length    ${nfv.vnfd[0]['int-virtual-link-desc']}
    ${instantiation_levels}=    Get Length    ${nfv.vnfd[0].df['instantiation-level']}
    FOR    ${key}    IN    @{nfv.vnfd[0].keys()}
        ${VDU_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    vdu
        Run Keyword If    ${VDU_check}    Get VDU IDs    ${VDUcount}
        ${extCP_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    ext-cpd
        Run Keyword If    ${extCP_check}    Get External CP IDs    ${extCP_count}
        ${virtualLink_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    int-virtual-link-desc
        Run Keyword If    ${virtualLink_check}    Get Virtual Link IDs    ${virtualLink_count}
        ${DF_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    df
        Run Keyword If    ${DF_check}    Get Instantiation Levels    ${instantiation_levels}
    END
    Set Global Variable    ${Descriptor_ID}    ${descriptor_id}
    Set Global Variable    ${Provider}    ${provider}
    Set Global Variable    ${Product_Name}    ${product_name}
    Set Global Variable    ${Software_Version}    ${software_version}
    Set Global Variable    ${Descriptor_Version}    ${descriptor_version}

Get VDU IDs
    [Arguments]    ${count}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    @{VDU_labels}=    Create List
    ${Storage_labels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${VDU_labels}    ${nfv.vnfd[0].vdu[${i}]['id']}
        Append To List    ${Storage_labels}    ${nfv.vnfd[0].vdu[${i}]['virtual-storage-desc']}
        Get Internal CPs for each VDU    ${i}
    END
    Set Global Variable    ${VDU_IDs}    ${VDU_labels}
    Set Global Variable    ${Storage_IDs}    ${Storage_labels}

Get Internal CPs for each VDU
    [Arguments]    ${vdu}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    ${count}=    Get Length    ${nfv.vnfd[0].vdu[${vdu}]['int-cpd']}
    ${internal_CPs}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${internal_CPs}    ${nfv.vnfd[0].vdu[${vdu}]['int-cpd'][${i}]['id']}
    END
    Set Global Variable    ${internalCP_IDs}    ${internal_CPs}

Get External CP IDs
    [Arguments]    ${count}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    @{external_CPs}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${external_CPs}    ${nfv.vnfd[0].ext_cpd[${i}].id}
    END
    Set Global Variable    ${externalCP_IDs}    ${external_CPs}

Get Virtual Link IDs
    [Arguments]    ${count}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    ${VirtualLink_labels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${VirtualLink_labels}    ${nfv.vnfd[0]['int-virtual-link-desc'][${i}]['id']}
    END
    Set Global Variable    ${VirtualLink_IDs}    ${VirtualLink_labels}

Get Instantiation Levels
    [Arguments]    ${count}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    @{Instantiation_Levels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${Instantiation_Levels}    ${nfv.vnfd[0].df['instantiation-level'][${i}]['id']}
    END
    Set Global Variable    ${InstantiationLevel_IDs}    ${Instantiation_Levels}

Match the VNF Package Response Attributes with Descriptors
    #Checking Response attributes with VNF Descriptors
    Should Be Equal As Strings    ${response['body']['vnfdId']}    ${Descriptor_ID}
    Should Be Equal As Strings    ${response['body']['vnfProvider']}    ${Provider}
    Should Be Equal As Strings    ${response['body']['vnfProductName']}    ${Product_Name}
    Should Be Equal As Strings    ${response['body']['vnfSoftwareVersion']}    ${Software_Version}
    Should Be Equal As Strings    ${response['body']['vnfdVersion']}    ${Descriptor_Version}

Match the VNF Instance Response Attributes with Descriptors
    #Checking Response attributes with VNF Descriptors
    Should Be Equal As Strings    ${response['body']['vnfdId']}    ${Descriptor_ID}
    Should Be Equal As Strings    ${response['body']['vnfProvider']}    ${Provider}
    Should Be Equal As Strings    ${response['body']['vnfProductName']}    ${Product_Name}
    Should Be Equal As Strings    ${response['body']['vnfSoftwareVersion']}    ${Software_Version}
    Should Be Equal As Strings    ${response['body']['vnfdVersion']}    ${Descriptor_Version}
    Should Be Equal As Strings    ${response['body']['instantiatedVnfInfo']['flavourId']}    ${Flavour_ID}
    Run Keyword If    '${descriptorType}'=='SOL006'List Should Contain Value    ${VDU_IDs}    ${response['body']['instantiatedVnfInfo']['vnfcResourceInfo']['vduId']}
    Run Keyword If    '${descriptorType}'=='SOL006'    List Should Contain Value    ${externalCP_IDs}    ${response['body']['instantiatedVnfInfo']['extCpInfo'][0]['cpdId']}
    Run Keyword If    '${descriptorType}'=='SOL006'    List Should Contain Value    ${internalCP_IDs}    ${response['body']['instantiatedVnfInfo']['vnfcResourceInfo'][0]['vnfcCpInfo'][0]['cpdId']}
    Run Keyword If    '${descriptorType}'=='SOL001'    List Should Contain Value    @{CP_IDs}    ${response['body']['instantiatedVnfInfo']['extCpInfo'][0]['cpdId']}
    Run Keyword If    '${descriptorType}'=='SOL001'    List Should Contain Value    @{CP_IDs}    ${response['body']['instantiatedVnfInfo']['vnfcResourceInfo'][0]['vnfcCpInfo'][0]['cpdId']}
    Run Keyword If    '${descriptorType}'=='SOL006'List Should Contain value    ${Storage_IDs}    ${response['body']['instantiatedVnfInfo']['virtualStorageResourceInfo'][0]['virtualStorageDescId']}
    Run Keyword If    '${descriptorType}'=='SOL006'List Should Contain Value    ${VirtualLink_IDs}    ${response['body']['instantiatedVnfInfo']['extManagedVirtualLinkInfo'][0]['vnfVirtualLinkDescId']}
#-----------------------------------------Common------------------------------

Check HTTP Response Header Contains ETag and Last-Modified
    Pass Execution If    ${HEADER_TEST_SUPPORTED} == 0    Headers testing not supported. Skipping the Test.
    Log    ${response['headers']}
    Should Contain    ${response['headers']}    Etag
    Should Contain    ${response['headers']}    Last-Modified
    Log    Etag and Last-Modified Headers are present

Check HTTP Response Body vnfPkgInfo Identifier matches the requested VNF Package
    Log    Going to validate pacakge info retrieved
    Should Be Equal    ${response['body']['id']}    ${vnfPackageId}
    Log    Pacakge identifier as expected

Check HTTP Response Status Code Is
    [Arguments]    ${expected_status}
    Should Be Equal As Strings    ${response['status']}    ${expected_status}
    Log    Status code validated

Check HTTP Response Body Json Schema Is
    [Arguments]    ${input}
    Run Keyword If    '${input}' == 'ProblemDetails'    Should Contain    ${response['headers']['Content-Type']}    application/problem+json
    ...    ELSE    Should Contain    ${response['headers']['Content-Type']}    application/json
    ${schema} =    Catenate    SEPARATOR=    ${input}    .schema.json
    Validate Json    ${schema}    ${response['body']}
    Log    Json Schema Validation OK

Check Operation Occurrence Id existence
    ${vnfLcmOpOccId}=    Get Value From Json    ${response['headers']}    $..Location
    Should Not Be Empty    ${vnfLcmOpOccId}
    Set Global Variable    ${globalVnfLcmOpOccId}    ${vnfLcmOpOccId}

Check HTTP Response Header Contains
    [Arguments]    ${CONTENT_TYPE}
    Should Contain    ${response['headers']}    ${CONTENT_TYPE}
    Log    Header is present

Check Individual VNF LCM operation occurrence operationState is
    [Arguments]    ${status}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${opOccResponse}=    Get    ${response['headers']['Location']}
    Log    Validate operationState
    Should Be Equal as Strings    ${opOccResponse['body']['operationState']}    ${status}
    Log    operationState validated

Send VNF Instantiation Request
    Log    Instantiate a VNF Instance
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/instantiateVnfRequest.json
    Post    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfid}/instantiate    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

Configure Notification Status Handler
    [Arguments]    ${endpoint}    ${status}=""
    ${json}=     Create Dictionary
    ${value}=    Run Keyword And Return Status    Should Not Be Equal As Strings    ${status}        ""
    Run Keyword If   ${value} == True      Set to dictionary    ${json}    operationState    ${status}
    Log  Creating mock request and response to handle ${element}
    ${notification_request}=  Create Mock Request Matcher	POST  ${endpoint}  body_type='JSON'    body=${json}
    &{headers}=  Create Dictionary  Content-Type=application/json
    ${notification_response}=  Create Mock Response	    status_code=${status}    headers=${headers}     
    Create Mock Expectation  ${notification_request}  ${notification_response}
    [Return]    ${notification_request}
    
Configure Notification VNF Instance Handler
    [Arguments]    ${endpoint}    ${instanceId}=""
    ${json}=     Create Dictionary
    ${value}=    Run Keyword And Return Status    Should Not Be Equal As Strings    ${instanceId}        ""
    Run Keyword If   ${value} == True      Set to dictionary    ${json}    vnfInstanceId    ${instanceId}
    Log  Creating mock request and response to handle ${element}
    ${notification_request}=  Create Mock Request Matcher	POST  ${endpoint}    body_type='JSON'  body=${json}
    &{headers}=  Create Dictionary  Content-Type=application/json
    ${notification_response}=  Create Mock Response     204    headers=${headers}    
    Log    ${notification_request}
    Log    ${notification_response}
    Create Mock Expectation  ${notification_request}  ${notification_response}
    [Return]    ${notification_request}

Configure Notification Forward
    [Arguments]    ${element}    ${endpoint}    ${endpoint_fwd}
    ${json}=    Get File    schemas/${element}.schema.json
    Log  Creating mock HTTP forward to handle ${element}
    ${notification_tmp}=  Create Mock Request Matcher	POST  ${endpoint}  body_type='JSON_SCHEMA'    body=${json}
    ${notification_fwd}=  Create Mock Http Forward	${endpoint_fwd}
    Create Mock Expectation With Http Forward  ${notification_tmp}  ${notification_fwd}
    [Return]  ${notification_tmp}

Check Operation Notification
    [Arguments]    ${element}    ${status}=""
    Log    Checking Operation Notification ${element} ${status}
    ${json}=	Get File	schemas/${element}.schema.json
    ${req1}=    Configure Notification Forward    ${element}    ${callback_endpoint}    ${callback_endpoint_fwd}
    ${req2}=    Configure Notification Status Handler    ${callback_endpoint_fwd}    ${status}
    Wait Until Keyword Succeeds    12 sec   3 sec   Verify Mock Expectation    ${req2}
    Clear Requests    ${callback_endpoint}
    Clear Requests    ${callback_endpoint_fwd}
    
Check VNF Instance Operation Notification
    [Arguments]    ${element}   ${instance_id}
    ${json}=	Get File	schemas/${element}.schema.json
    ${req1}=  Configure Notification Forward   ${element}    ${callback_endpoint}    ${callback_endpoint_fwd}
    ${req2}=  Configure Notification VNF Instance Handler    ${callback_endpoint_fwd}    ${instance_id}
    Wait Until Keyword Succeeds    12 sec  3 sec   Verify Mock Expectation    ${req2}
    Clear Requests    ${callback_endpoint}
    Clear Requests    ${callback_endpoint_fwd}

Check Operation Notification For
    [Arguments]    ${status}
    Run Keyword If    ${SKIP_INTERMEDIATE_NOTIFICATION_CHECKS} == 0 or (${SKIP_INTERMEDIATE_NOTIFICATION_CHECKS} == 1 and '${status}' == 'COMPLETED')    Check Operation Notification    VnfLcmOperationOccurrenceNotification   ${status}

Check Postcondition VNF Status
    [Arguments]    ${status}
    Log    Retrieve VNF Instance
    Check VNF Instance    ${vnfid}
    Should Not Be Empty    ${response}
    Check HTTP Response Status Code Is    200
    Should Be Equal    ${response['body']['id']}    ${vnfid}    
    Check HTTP Response Header Contains    Content-Type
    Check HTTP Response Body Json Schema Is    schemas/vnfInstance.schema.json
    Check VNF Status    ${response['body']['instantiationState']}    ${status}

Check VNF Instance
    [Arguments]    ${vnfid}
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfid}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

Check VNF Status
    [Arguments]    ${current}    ${expected}
    Should Be Equal As Strings    ${current}    ${expected}
    Log    VNF Status in the correct status

Send Terminate VNF Request
    Log    Trying to terminate a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/terminateVnfRequest.json
    Post    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfid}/terminate    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

Check Postcondition VNF Terminate
    Check resource not Instantiated

Check resource not Instantiated
    Check VNF Instance    ${vnfid}
    Check HTTP Response Status Code Is    200
    Check VNF Status    ${response['body']['instantiationState']}     NOT_INSTANTIATED

GET VNF LCM Operation occurrences
    Log    Query status information about multiple VNF lifecycle management operation occurrences.
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_lcm_op_occs
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	

Get Individual VNF LCM Operation occurrence [keyword]
    Log    Query status information about multiple VNF lifecycle management operation occurrences.
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_lcm_op_occs/${globalVnfLcmOpOccId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Check HTTP Response Body of Individual VNF LCM Operation occurrences content against VNF Descriptor
    Run Keyword If    ${check_descriptors} == 1    Check Individual VNF LCM Operation Occurence Content
    
Check Individual VNF LCM Operation Occurence Content
    PARSE the Descriptor File
    Match the VNF LCM Operation Occurence Response Attributes with Descriptors
    
Match the VNF LCM Operation Occurence Response Attributes with Descriptors
    #Checking Response attributes with VNF Descriptors
    Should Be Equal As Strings    ${response['body']['changedInfo']['vnfdId']}    ${Descriptor_ID}
    Should Be Equal As Strings    ${response['body']['changedInfo']['vnfdVersion']}    ${Descriptor_Version}
    List Should Contain Value    ${VDU_IDs}    ${response['body']['resourceChanges']['affectedVnfcs'][0]['vduId']}
    List Should Contain value    ${Storage_IDs}    ${response['body']['resourceChanges']['affectedVirtualStorages'][0]['virtualStorageDescId']}
    List Should Contain Value    ${VirtualLink_IDs}    ${response['body']['resourceChanges']['affectedVirtualLinks'][0]['virtualStorageDescId']}

Post Create subscription
    Log    Create subscription instance by POST to ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/subscriptions
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/lccnSubscriptionRequest.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/subscriptions    ${body}    allow_redirects=false
	${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
    Set Global Variable    ${globalSubscriptionId}    ${outputResponse['body']['id']}
	Log    Verify notification endpoint is tested
	Verify Mock Expectation    ${notification_request}
	Clear Requests    ${callback_endpoint}

Get subscriptions [keyword]
    Log    Get the list of active subscriptions
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/subscriptions	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	

Get Individual subscription [keyword]
    log    Trying to get information about an individual subscription
    Set Headers    {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/subscriptions/${globalSubscriptionId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Delete Individual subscription
    log    Try to delete an individual subscription
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/subscriptions/${globalSubscriptionId}  	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Check Postcondition Subscription Resource Is Deleted
    Get Individual subscription [keyword] 
    Integer    response status    404
    Log    Subscription Resource deleted

Send VNF Scale Out Request
    Log    Trying to scale a vnf Instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}  
    ${template}=    Get File    jsons/scaleVnfRequest.json
    ${body}=        Format String   ${template}     scaletype=${scaleOutType}    scaleAspectId=${scaleAspectId}
    Post    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfid}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Send VNF Scale In Request
    Log    Trying to scale a vnf Instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}  
    ${template}=    Get File    jsons/scaleVnfRequest.json
    ${body}=        Format String   ${template}     scaletype=${scaleInType}    scaleAspectId=${scaleAspectId}
    Post    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_instances/${vnfid}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Check Postcondition VNF
    [Arguments]    ${operation}
    Check resource instantiated
    ${newScaleInfo}=    Get Vnf Scale Info    ${vnfInstanceId}
    Compare ScaleInfos    ${operation}    ${scaleInfo}    ${newScaleInfo}

Compare ScaleInfos
    [Arguments]    ${type}    ${old_scaleinfo}    ${new_scaleinfo}
    FOR    ${element}    IN    ${old_scaleinfo}
        ${old_level}=    Set Variable If    ${element.aspectId}==${aspectId}   ${element.scaleLevel}
        ${old_level_value}=    Convert To Integer    ${old_level}
    END
    FOR    ${element}    IN    ${new_scaleinfo}
        ${new_level}=    Set Variable If    ${element.aspectId}==${aspectId}   ${element.scaleLevel}
        ${new_level_value}=    Convert To Integer    ${new_level}
    END
    Run Keyword If    ${type}==SCALE_OUT    Should Be True    ${old_level_value}<${new_level_value}
    ...    ELSE    Should Be True    ${old_level_value}<${new_level_value}

Check resource Instantiated
    Check VNF Instance    ${vnfid}
    Check HTTP Response Status Code Is    200
    Check VNF Status    ${response['body']['instantiationState']}    INSTANTIATED

Get Vnf Scale Info
    [Arguments]    ${vnfInstanceId}
    ${vnfInstance}=    Check VNF Instance    ${vnfid}
    ${scaleInfo}=    Get Value From Json    ${vnfInstance}    $..scaleStatus
    [Return]   ${scaleInfo} 

Check operation resource state is FAILED_TEMP
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_lcm_op_occs/${globalVnfLcmOpOccId}
    ${outputResponse}=    Output    response
    ${state}=    Set Variable    ${outputResponse['body']['instantiationState']}
    Log    Current operation state: ${state}
    Run Keyword If    "${state}" != "FAILED_TEMP"    
    ...    Pass Execution If    True    Skipping test: Operation state must be FAILED_TEMP but found ${state}
    Log    Operation resource state is FAILED_TEMP as expected

Post Retry operation
    Log    Retry a VNF lifecycle operation if that operation has experienced a temporary failure
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post     ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_lcm_op_occs/${globalVnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Post Rollback operation
    Log    Rollback a VNF lifecycle operation if that operation has experienced a temporary failure
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post     ${VNFM_API_ROOT}/${VNFM_API_NAME}/${VNFM_API_MAJOR_VERSION}/vnf_lcm_op_occs/${globalVnfLcmOpOccId}/rollback
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}