*** Settings ***
Library    String
Resource   environment/variables.txt
Library    REST    ${NFVMANO_SCHEMA}://${NFVMANO_HOST}:${NFVMANO_PORT}    ssl_verify=false
Library    MockServerLibrary 
Library    OperatingSystem
Library    BuiltIn
Library    JSONLibrary
Library    Collections
Library    JSONSchemaLibrary    schemas/
Library    Process
Library    jwt

*** Keywords ***
Check HTTP Response Status Code Is
    [Arguments]    ${expected_status}
    ${status}=    Convert To Integer    ${expected_status}    
    Should Be Equal    ${response['status']}    ${status} 
    Log    Status code validated

Check HTTP Response Header Contains
    [Arguments]    ${CONTENT_TYPE}
    Should Contain    ${response['headers']}    ${CONTENT_TYPE}
    Log    Header is present
    Run Keyword If    '${CONTENT_TYPE}' == 'Location'    Check Existence of Location Header Resource
Check Existence of Location Header Resource
    GET    ${response['headers']['Location']}
    Log    Validate Status code
    Integer    response status    200

Check HTTP Response Body Json Schema Is
    [Arguments]    ${input}
    Run Keyword If    '${input}' == 'ProblemDetails'    Should Contain    ${response['headers']['Content-Type']}    application/problem+json
    ...    ELSE    Should Contain    ${response['headers']['Content-Type']}    application/json
    ${schema} =    Catenate    SEPARATOR=    ${input}    .schema.json
    Validate Json    ${schema}    ${response['body']}
    Log    Json Schema Validation OK

POST Create a new logging job	
     Log    Create a logging job
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/createLoggingJobRequest.json
    ${body}=        Format String   ${template}     objectInstanceId=${objectInstanceId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}

POST Create a new logging job with permitted authorization scope
     Log    Create a logging job
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scope=${JOB_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    ${template}=    Get File    jsons/createLoggingJobRequest.json
    ${body}=        Format String   ${template}     objectInstanceId=${objectInstanceId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}

POST Create a new logging job with not permitted authorization scope
     Log    Create a logging job
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scope=${JOB_NOT_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    ${template}=    Get File    jsons/createLoggingJobRequest.json
    ${body}=        Format String   ${template}     objectInstanceId=${objectInstanceId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}	
Check Post-Condition Logging job is created
    GET Logging Jobs
    Check HTTP Response Status Code Is    200
	
GET Logging Jobs
    Log    Query to GET information about logging jobs.
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}
	
GET Logging Jobs with malformed authorization token
    Pass Execution If    ${AUTH_USAGE} == 0    Skipping test as NFVO is not supporting authentication
    Log    The GET method queries using invalid token
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"${AUTHORIZATION_HEADER}": "${BAD_AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Logging Jobs without authorization token
    Pass Execution If    ${AUTH_USAGE} == 0    Skipping test as it is not supporting authentication
    Log    The GET method queries omitting token
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Logging Jobs with expired or revoked authorization token
    Pass Execution If    ${AUTH_USAGE} == 0    Skipping test as it is not supporting authentication
    Log    The GET method queries  using invalid token
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"${AUTHORIZATION_HEADER}": "${NEG_AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Logging Job with invalid filter
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs?attribute_not_exist=some_value
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}	
	
GET Logging Job with invalid selector
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs?fields=wrong_field
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}	
	
GET Logging Job with filter
	Log    Query  information about logging jobs with filters.
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs?${alarm_filter}=${filter_value} 
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}
	
GET Logging jobs with all_fields attribute selector
	Log    Queries information about logging jobs, using all_fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs?exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Logging Jobs with exclude_default attribute selector
    Log    Queries information about logging jobs, using exclude_default
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs?exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Logging Jobs with fields attribute selector
    Log    Queries information about Logging Jobs, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs?fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Logging Jobs with fields and exclude_default attribute selector
    Log    Queries information about Logging Jobs, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs?exclude_default&fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Logging Jobs with exclude_fields attribute selector
    Log    Query NFV-MANO The GET method queries information about multiple alarms, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs?exclude_fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check HTTP Response Header Contains Link
    ${linkURL}=    Get Value From Json    ${response['headers']}    $..Link
    Should Not Be Empty    ${linkURL}

PUT Logging Jobs
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}
PATCH Logging Jobs
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}
DELETE Logging Jobs
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}		
	
Send Post request for individual Logging Job    
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET individual Logging Job
    Log    Trying to get a Logging Job 
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check HTTP Response Body Logging Job Identifier matches the requested Logging Job
    Log    Going to validate Logging Job info retrieved
    Should Be Equal    ${response['body']['id']}    ${logJobId} 
    Log    Logging Job identifier as expected


GET individual Logging Job with invalid resource identifier  
    Log    Trying to perform a negative get, using erroneous logging Job identifier
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Put request for individual Logging Job    
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Patch request for individual Logging Job    
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}


Send Delete request for individual Logging Job
    Log    Trying to delete an existing logging Job
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete request for individual Logging Job with permitted authorization scope
    Log    Trying to delete an existing logging Job
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    ${scopeValue}=    Create Dictionary    scope=${JOB_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete request for individual Logging Job with not permitted authorization scope
    Log    Trying to delete an existing logging Job
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    ${scopeValue}=    Create Dictionary    scope=${JOB_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check Postcondition Logging Job is Deleted
    Log    Check Postcondition
    GET individual Logging Job
    Check HTTP Response Status Code Is    404

Send Delete request for individual Logging Job with invalid resource identifier
    Log    Trying to perform a negative delete, using erroneous Logging Job identifier
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${erroneousLogJobId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Post request for Individual Log Report
    Log    Trying to create new log report
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${newReportId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get Individual Log Report
    Log    Trying to get log report
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${reportId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get Individual Log Report with permitted authorization scope
    Log    Trying to get log report
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    ${scopeValue}=    Create Dictionary    scope=${REPORT_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${reportId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get Individual Log Report with not permitted authorization scope
    Log    Trying to get log report
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    ${scopeValue}=    Create Dictionary    scope=${NOT_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${reportId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get Individual Log Report with invalid resource endpoint
    Log    Trying to get a log report with invalid resource endpoint
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${erroneousReportId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Put request for Individual Log Report
    Log    Trying to update log report
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${reportId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${reportId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Patch request for Individual Log Report
    Log    Trying to update log report
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${reportId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${reportId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete request for Individual Log Report
    Log    Trying to delete log report   
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/log_reports/${reportId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Log data Request in Synchronous mode
    Log    Request to compile the logged data into a file Synchronous mode
    Pass Execution If    ${SYNC_MODE} == 0    The compile process is asynchronous mode. Skipping the test
    Set Headers    {"Accept": "${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/compileLogRequest.json
    ${body}=        Format String   ${template}     objectInstanceId=${objectInstanceId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log    ${body}
    ${body}=    Output    response
    Set Suite Variable    &{response}    ${body}

Send Log data Request in Synchronous mode with permitted authorization scope
    Log    Request to compile the logged data into a file Synchronous mode
    Pass Execution If    ${SYNC_MODE} == 0    The compile process is asynchronous mode. Skipping the test
    Set Headers    {"Accept": "${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scope=${JOB_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    ${template}=    Get File    jsons/compileLogRequest.json
    ${body}=        Format String   ${template}     objectInstanceId=${objectInstanceId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log    ${body}
    ${body}=    Output    response
    Set Suite Variable    &{response}    ${body}

Send Log data Request in Synchronous mode with not permitted authorization scope
    Log    Request to compile the logged data into a file Synchronous mode
    Pass Execution If    ${SYNC_MODE} == 0    The compile process is asynchronous mode. Skipping the test
    Set Headers    {"Accept": "${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scope=${NOT_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    ${template}=    Get File    jsons/compileLogRequest.json
    ${body}=        Format String   ${template}     objectInstanceId=${objectInstanceId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log    ${body}
    ${body}=    Output    response
    Set Suite Variable    &{response}    ${body}

Check HTTP Response Header Contains Location
    ${vnfLcmOpOccId}=    Get Value From Json    ${response['headers']}    $..Location
    Should Not Be Empty    ${vnfLcmOpOccId}

Send Log data Request in Asynchronous mode
    Log    Request to compile the logged data into a file Asynchronous mode
    Pass Execution If    ${SYNC_MODE} == 1    The compile process is synchronous mode. Skipping the test
    Set Headers    {"Accept": "${ACCEPT}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/compileLogRequest.json
    ${body}=        Format String   ${template}     objectInstanceId=${objectInstanceId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log    ${body}
    ${body}=    Output    response
    Set Suite Variable    &{response}    ${body}
    
 Check postcondition Wait for compilation success notification
    Wait Until Keyword Succeeds    ${retry}   ${polling}    Get success notification
    
Get Success notification
    log    Trying to read an compiled file
    Set Headers    {"Accept":"${ACCEPT}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${response['headers']['Location']}
    Log    Validate Status code
    Integer    response status    200
    ${result}    Output    response body
    Validate Json   LogReport.schema.json    ${result}
    Log    Validation OK
    
Send Log data Request
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/createLoggingJobRequest.json
    ${body}=        Format String   ${template}     objectInstanceId=${objectInstanceId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    @{response}    ${outputResponse}

Send Get Compiled Log Data
    Log    Trying to delete log report   
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Put Compiled Log Data
    Log    Trying to update log report
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Patch Compiled Log Data
    Log    Trying to update log report
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete Compiled Log Data
    Log    Trying to delete log report   
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/log_jobs/${logJobId}/compile_log
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Create Sessions
    Pass Execution If    ${NFVMANO_CHECKS_NOTIF_ENDPOINT} == 0   MockServer not necessary to run    
    Start Process  java  -jar  ${MOCK_SERVER_JAR}    -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}
    
Check Notification Endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}    
    &{notification_response}=  Create Mock Response	headers="Content-Type: application/json"  status_code=204
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Wait Until Keyword Succeeds    ${total_polling_time}   ${polling_interval}   Verify Mock Expectation    ${notification_request}
    Clear Requests  ${callback_endpoint}

Get all Subscriptions
    [Documentation]    The api consumer can use this method to query the list of active subscriptions to log management notifications
    Set headers    {"Accept": "application/json"}
    Set headers    {"Content-Type": "application/json"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get all Subscriptions with permitted authorization scope
    [Documentation]    The api consumer can use this method to query the list of active subscriptions to log management notifications
    Set headers    {"Accept": "application/json"}
    Set headers    {"Content-Type": "application/json"}
    ${scopeValue}=    Create Dictionary    scope=${SUBSCRIPTIONS_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get all Subscriptions with not permitted authorization scope
    [Documentation]    The api consumer can use this method to query the list of active subscriptions to log management notifications
    Set headers    {"Accept": "application/json"}
    Set headers    {"Content-Type": "application/json"}
    ${scopeValue}=    Create Dictionary    scope=${NOT_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
 Get Subscriptions with attribute-based filters
    [Documentation]    The api consumer can use this method to query the list of active subscriptions to log management notifications
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${filter_ok}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Subscriptions with invalid attribute-based filters
    [Documentation]    The api consumer can use this method to query the list of active subscriptions to log management notifications
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${filter_ko}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get subscriptions with filter "id"
    Log    Get the list of active subscriptions using a filter "id"
    Set Headers    {"Accept": "${ACCEPT}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?id=${subscription_id}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
Check PostCondition HTTP Response Body Subscription Matches the requested attribute-based filter "id"
    Should Be Equal As Strings    ${response['body']['id']}    ${subscription_id}
	
Get subscriptions with filter "filter_notificationTypes"
    Log    Get the list of active subscriptions using a filter "filter.notificationTypes"
    Set Headers    {"Accept": "${ACCEPT}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?filter.notificationTypes=${notification_type}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
Check PostCondition HTTP Response Body Subscriptions Matches the requested attribute-based filter "filter_notificationTypes"
    FOR   ${item}   IN  @{response['body']}
        Should Be Equal As Strings    ${item['filter']['notificationTypes']}   ${notification_type}
    END

Send Post Request for Subscription
    Set headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}    callback_endpoint=${callback_endpoint}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Post Request for Subscription with unreachable callbackUri
    Set headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${unreachable_callback_uri}    callback_endpoint=${callback_endpoint}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body LogmSubscription Attributes Values Match the Issued Subscription
    Log    Check Response matches subscription
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}    callback_endpoint=${callback_endpoint}
    ${subscription}=    evaluate    json.loads('''${body}''')    json
    Should Be Equal As Strings    ${response['body']['callbackUri']}    ${subscription['callbackUri']}

Check Postcondition Subscription Is Set
    Log    Check Postcondition subscription exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${response['body']['id']}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Check HTTP Response Status Code Is    200
    
Send Post Request for Duplicated Subscription
    Log    POST request for duplicated subscription
    Pass Execution If    ${NFVMANO_ALLOWS_DUPLICATE_SUBS}==0    Skipping test as duplicate subscription is not enabled
    Set headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}    callback_endpoint=${callback_endpoint}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}  

Send Post Request for No Duplicated Subscription
    Log    POST request for duplicated subscription
    Pass Execution If    ${NFVMANO_ALLOWS_DUPLICATE_SUBS}==1    Skipping test as duplicate subscription is enabled
    Set headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}    callback_endpoint=${callback_endpoint}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}  
    
Check HTTP Response Body Is Empty
    Should Be Empty    ${response['body']}    
    Log    No json schema is provided. Validation OK  
    
Check Postcondition Subscription Resource Returned in Location Header Is Available
    Log    Going to check postcondition
    GET    ${response['headers']['Location']}
    Integer    response status    200
    Log    Received a 200 OK as expected
    ${contentType}=    Output    response headers Content-Type
    Should Contain    ${contentType}    application/json
    ${result}=    Output    response body
    Validate Json    LogmSubscription.schema.json    ${result}
    Log    Validated LogmSubscription schema
    
Send Put Request for Subscriptions
    [Documentation]    This method is not supported. When this method is requested on this resource, the NFV-MANO shall return a "405 Method
    ...    Not Allowed" response as defined in clause 4.3.5.4.
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Patch Request for Subscriptions
    [Documentation]    This method is not supported. When this method is requested on this resource, the NFV-MANO shall return a "405 Method
    ...    Not Allowed" response as defined in clause 4.3.5.4.
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete Request for Subscriptions
    [Documentation]    This method is not supported. When this method is requested on this resource, the NFV-MANO shall return a "405 Method
    ...    Not Allowed" response as defined in clause 4.3.5.4.
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Individual Subscription
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get Individual Subscription with permitted authorization scope
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    ${scopeValue}=    Create Dictionary    scope=${SUBSCRIPTIONS_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get Individual Subscription with not permitted authorization scope
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    ${scopeValue}=    Create Dictionary    scope=${NOT_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=''    algorithm=${OAUTH_ENCRIPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Authorization Token: ${authorizationToken}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check HTTP Response Body Subscription Identifier matches the requested Subscription
    Log    Trying to check response ID
    Should Be Equal As Strings    ${response['body']['id']}    ${subscriptionId} 
    Log    Subscription identifier as expected
    
GET individual Subscription with invalid resource identifier
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${erroneousSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete request for individual Subscription
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check Postcondition Subscription is Deleted
    Log    Check Postcondition Subscription is deleted
    GET individual Subscription
    Check HTTP Response Status Code Is    404 

Send Delete request for individual Subscription with invalid resource identifier
    Log    Trying to delete a subscription in the VNFM with invalid id
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${erroneousSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    @{response}    ${output}
    
Send Post request for individual Subscription
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${newSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    @{response}    ${output}

Send Put request for individual Subscription
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    @{response}    ${output}
    
Send Patch request for individual Threshold
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    @{response}    ${output}

Trigger the availability of Log Report (external action) 
    #do nothing
    Log    do nothing

Check Log Report Available Notification Http POST Request Body Json Schema Is    
    [Arguments]    ${element}
    ${schema}=	Get File	schemas/${element}.schema.json
    Configure Notification Forward    ${schema}    ${callback_endpoint}    ${callback_endpoint_fwd}

Check Log Report Available Notification Http POST Request Body notificationType attribute Is
    [Arguments]    ${type}
    Configure Notification Log Report Available Handler    ${callback_endpoint_fwd}    ${type}
    Wait Until Keyword Succeeds    ${total_polling_time}   ${polling_interval}   Verify Mock Expectation    ${notification_request}
    Clear Requests    ${callback_endpoint}
    Clear Requests    ${callback_endpoint_fwd}

Check Threshold Crossed Notification Http POST Request Body Json Schema Is    
    [Arguments]    ${element}
    ${schema}=	Get File	schemas/${element}.schema.json
    Configure Notification Forward    ${schema}    ${callback_endpoint}    ${callback_endpoint_fwd}
    
Configure Notification Log Report Available Handler
    [Arguments]    ${endpoint}    ${type}
    ${json}=    evaluate    {}
    set to dictionary    ${json}    notificationType    ${type}    
    ${BODY}=    evaluate    json.dumps(${json})    json
    Log  Creating mock request and response to handle status notification
    &{req}=  Create Mock Request Matcher	POST  ${endpoint}  body_type="JSON"    body=${BODY}
    Set Global Variable    ${notification_request}    ${req}   
    &{notification_response}=  Create Mock Response    status_code=204
    Create Mock Expectation  ${notification_request}  ${notification_response}

Configure Notification Forward
    [Arguments]    ${schema}    ${endpoint}    ${endpoint_fwd}    
    Log  Creating mock Http POST forward to handle ${schema}
    &{notification_tmp}=  Create Mock Request Matcher	POST  ${endpoint}  body_type="JSON_SCHEMA"    body=${schema}
    &{notification_fwd}=  Create Mock Http Forward	${endpoint_fwd}
    Create Mock Expectation With Http Forward  ${notification_tmp}  ${notification_fwd}

Create Sessions for Notifications
    Start Process  java  -jar  ${MOCK_SERVER_JAR}  -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}

Check resource existence and get CallbackUri
    Set Headers  {"Accept":"${ACCEPT_JSON}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    Integer    response status    200
    Validate Json    response body    LogmSubscription.schema.json
    Set Global Variable    ${callbackResp}    response body callbackUri

Post Log Report Available Notification
    log    Trying to perform a POST to get notification
    Set Headers  {"Accept":"${ACCEPT_JSON}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template} =    Get File    jsons/LogReportAvailableNotification.json  
    ${body}=    Format String    ${template}    subscriptionId=${subscriptionId}    objectInstanceId=${objectInstanceId}
    Post    ${callbackResp}    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST API Version
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
GET API Version
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
PUT API Version
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
PATCH API Version
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
DELETE API Version
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
POST API Version with apiMajorVersion
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion} /api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
GET API Version with apiMajorVersion
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion} /api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
PUT API Version with apiMajorVersion
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion} /api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
PATCH API Version with apiMajorVersion
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion} /api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
DELETE API Version with apiMajorVersion
    Set Headers    {"Accept":"${ACCEPT_JSON}"} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion} /api_versions
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

JWT Encode
    [Arguments]    ${payload}    ${key}    ${algorithm}
    ${encoded}=    Evaluate    jwt.encode(${payload}, ${key}, ${algorithm})
    [Return]    ${encoded} 
