*** Settings ***
Library    String
Resource    environment/variables.txt
Library    MockServerLibrary
Library    REST    ${VNFM_SCHEMA}://${VNFM_HOST}:${VNFM_PORT}    ssl_verify=false
Library    OperatingSystem
Library    BuiltIn
Library    JSONLibrary
Library    Collections
Library    Process
Library    JSONSchemaLibrary    schemas/
Library    jwt

#Variables    descriptors/SOL001/VNFD/vnfd_SOL001.yaml
#Variables    descriptors/SOL006/VNFD/vnfd_SOL006.yaml

*** Keywords ***
Create Sessions   
    Start Process  java  -jar  ${MOCK_SERVER_JAR}    -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}

Get Vnf Instance 
    [Arguments]    ${vnfInstanceId}
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}
    ${body}=    Output    response body
    [Return]    ${body}

Check HTTP Response Status Code Is
    [Arguments]    ${expected_status}    
    Should Be Equal As Strings   ${response['status']}    ${expected_status}
    Log    Status code validated 

Check HTTP Response Header Contains ETag and Last-Modified
    Pass Execution If    ${HEADER_TEST_SUPPORTED} == 0    Headers testing not supported. Skipping the Test.
    Log    ${response['headers']}
    Should Contain    ${response['headers']}    Etag
    Should Contain    ${response['headers']}    Last-Modified
    Log    Etag and Last-Modified Headers are present

Check Operation Occurrence IdS
    ${vnfLcmOpOccId}=    Get Value From Json    ${response['headers']}    $..Location
    Should Not Be Empty    ${vnfLcmOpOccId}

Check Operation Occurrence Id existence 
    ${vnfLcmOpOccId}=    Get Value From Json    ${response['headers']}    $..Location
    Should Not Be Empty    ${vnfLcmOpOccId}
    
Check HTTP Response Body Json Schema Is
    [Arguments]    ${input}
    Run Keyword If    '${input}' == 'ProblemDetails'    Should Contain    ${response['headers']['Content-Type']}    application/problem+json
    ...    ELSE    Should Contain    ${response['headers']['Content-Type']}    application/json
    ${schema} =    Catenate    SEPARATOR=   ${input}    .schema.json
    Validate Json    ${schema}    ${response['body']}
    Log    Json Schema Validation OK

Check resource Instantiated
    Check VNF Instance    ${vnfInstanceId}
    Check HTTP Response Status Code Is    200
    Check VNF Status    ${response['body']['instantiationState']}    INSTANTIATED

Check resource not Instantiated
    Check VNF Instance    ${vnfInstanceId}
    Check HTTP Response Status Code Is    200
    Check VNF Status    ${response['body']['instantiationState']}     NOT_INSTANTIATED

Check VNF Instance
    [Arguments]    ${vnfId}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${response}=    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfId}

Check VNF Status
    [Arguments]    ${current}    ${expected}
    Should Be Equal As Strings    ${current}    ${expected}
    Log    VNF Status in the correct status

Check operation resource state is FAILED_TEMP
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId} 
    String    response body instantiationState    FAILED_TEMP     
    
Check operation resource state is FINALLY_FAILED
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId} 
    String    response body instantiationState    FINALLY_FAILED      
Get Vnf Scale Info
    [Arguments]    ${vnfInstanceId}
    ${vnfInstance}=    Get Vnf Instance    ${vnfInstanceId}
    ${scaleInfo}=    Get Value From Json    ${vnfInstance}    $..scaleStatus
    [Return]   ${scaleInfo} 

Get Vnf Flavour Info
    [Arguments]    ${vnfInstanceId}
    ${vnfInstance}=    Get Vnf Instance    ${vnfInstanceId}
    ${flavourInfo}=    Get Value From Json    ${vnfInstance}    $..flavourId
    [Return]    ${flavourInfo}

Get Vnf Operational State Info
    [Arguments]    ${vnfInstanceId}
    ${vnfInstance}=    Get Vnf Instance    ${vnfInstanceId}
    ${stateInfo}=    Get Value From Json    ${vnfInstance}    $..vnfState
    [Return]    ${stateInfo}

Get Vnf Ext Link Id
    [Arguments]    ${vnfInstanceId}
    ${vnfInstance}=    Get Vnf Instance    ${vnfInstanceId}
    [Return]    ${vnfInstance.instantiatedVnfInfo.extVirtualLinkInfo.id}

Check HTTP Response Header Contains
    [Arguments]    ${CONTENT_TYPE}
    Should Contain    ${response['headers']}    ${CONTENT_TYPE}
    Log    Header is present

Check Individual VNF LCM operation occurrence operationState is
    [Arguments]    ${status}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${opOccResponse}=    Get    ${response['headers']['Location']}
    Log    Validate operationState  
    Should Be Equal as Strings  ${opOccResponse['body']['operationState']}    ${status}
    Log    operationState validated

Send VNF Scale Out Request
    Log    Trying to scale a vnf Instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfOutRequest.json
    ${json}=    evaluate    json.loads('''${body}''')    json
    ${aspectId}=    Set Variable    ${json.aspectId}  
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale    ${body}
    
Send VNF Scale To Level Request
    [Documentation]    Instantiate VNF The POST method instantiates a VNF instance.
    Log    Trying to Instantiate a vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfToLevelRequest.json
    ${json}=    evaluate    json.loads('''${body}''')    json
    ${aspectId}=    Set Variable    ${json.aspectId}  
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level    ${body}
    
Send VNF Instance Resource Create Request
    Log    Create VNF instance by POST to ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/createVnfRequest.json
    ${body}=        Format String   ${template}     vnfdId=${Descriptor_ID}    vnfProvider=${Provider}    vnfProductName=${Product_Name}    vnfSoftwareVersion=${Software_Version}    vnfdVersion= ${Descriptor_Version}    flavourId=${Flavour_ID}
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances    ${body}

Send VNF Instance Resource Delete Request
    log    Delete an individual VNF instance
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${response}=    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}

Send Change VNF Flavour Request
    Log    Trying to change the deployment flavour of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/changeVnfFlavourRequest.json
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_flavour    ${body}

Send Change VNF Operational State Request
    Log    Trying to change the operational state of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/operateVnfRequest.json
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/operate    ${body}

Send Heal VNF Request
    Log    Trying to heal a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/healVnfRequest.json
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/heal    ${body}

Send Change Ext Connectivity Request
    Log    Trying to change the external connectivity of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/changeExtVnfConnectivityRequest.json
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_ext_conn    ${body}

Send Terminate VNF Request
    Log    Trying to terminate a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/terminateVnfRequest.json
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/terminate    ${body}

Send Info Modification Request
    Log    Trying to update information of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/patchBodyRequest.json
    ${response}=    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}    ${body}

Send Retry Operation Request
    Log    Retry a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/retry

Send Roll back Operation Request
    Log    Rollback a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/rollback

Send Fail Operation Request
    Log    Fail a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/fail

Send Cancel Operation Request
    Log    Cancel an ongoing VNF lifecycle operation
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${response}=    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/cancel     ${CancelMode}

POST Cancel operation task
    Log    Cancel an ongoing VNF lifecycle operation
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Cancel a VNF lifecycle operation
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/cancel    ${CancelMode}
    Log    Validate Status code
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Cancel operation task with conflict
    Log    Cancel an ongoing VNF lifecycle operation
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Cancel a VNF lifecycle operation
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccIdNotStartingProcessingRollingback}/cancel    ${CancelMode}
    Log    Validate Status code
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
POST Cancel operation task not existing
    Log    Cancel an ongoing VNF lifecycle operation
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Cancel a VNF lifecycle operation
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${notExistingVnfLcmOpOccId}/cancel    ${CancelMode}
    Log    Validate Status code
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Cancel operation task with permitted authorization scope
    Log    Cancel an ongoing VNF lifecycle operation with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Cancel a VNF lifecycle operation
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/cancel    ${CancelMode}
    Log    Validate Status code
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Cancel operation task with not permitted authorization scope
    Log    Cancel an ongoing VNF lifecycle operation with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${Authorization_Token}"}
    Log    Cancel a VNF lifecycle operation
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/cancel    ${CancelMode}
    Log    Validate Status code
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}   
    
GET Cancel operation task
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/cancel 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
PUT Cancel operation task
    log    Trying to perform a PUT. This method should not be implemented 
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/cancel  		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
PATCH Cancel operation task
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/cancel  	 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
DELETE Cancel operation task
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/cancel  	 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}    

POST Change External VNF Connectivity
    Log    Trying to change the external connectivity of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/changeExtVnfConnectivityRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_ext_conn    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Change External VNF Connectivity with permitted authorization scope
    Log    Trying to change the external connectivity of a VNF instance with permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${CHANGE_EXT_CONN_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${body}=    Get File    jsons/changeExtVnfConnectivityRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_ext_conn    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  

POST Change External VNF Connectivity with not permitted authorization scope
    Log    Trying to change the external connectivity of a VNF instance with not permitted authorization scope.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scopeValue=${CHANGE_EXT_CONN_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${body}=    Get File    jsons/changeExtVnfConnectivityRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_ext_conn    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  

GET Change External VNF Connectivity
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_ext_conn 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
PUT Change External VNF Connectivity
    log    Trying to perform a PUT. This method should not be implemented 
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_ext_conn  		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
PATCH Change External VNF Connectivity
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_ext_conn  	 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
DELETE Change External VNF Connectivity
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_ext_conn  	 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
POST Change VNF deployment flavour
    Log    Trying to change the deployment flavour of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/changeVnfFlavourRequest.json
    ${body}=        Format String   ${template}     newFlavourId=${Flavour_ID}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_flavour    ${body} 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Change VNF deployment flavour of NOT-INSTANTIATED VNF
    Log    Trying to change the deployment flavour of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/changeVnfFlavourRequest.json
    ${body}=        Format String   ${template}     newFlavourId=${newFlavourId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/change_flavour    ${body} 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Change VNF deployment flavour of not existing VNF
    Log    Trying to change the deployment flavour of a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/changeVnfFlavourRequest.json
    ${body}=        Format String   ${template}     newFlavourId=${newFlavourId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notExistingVnfInstanceId}/change_flavour    ${body} 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Change VNF deployment flavour with permitted authorization scope
    Log    Trying to change the deployment flavour of a VNF instance with permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${CHANGE_FLAVOUR_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/changeVnfFlavourRequest.json
    ${body}=        Format String   ${template}     newFlavourId=${Flavour_ID}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_flavour    ${body} 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Change VNF deployment flavour with not permitted authorization scope
    Log    Trying to change the deployment flavour of a VNF instance with not permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${CHANGE_FLAVOUR_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/changeVnfFlavourRequest.json
    ${body}=        Format String   ${template}     newFlavourId=${Flavour_ID}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_flavour    ${body} 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
    
GET Change VNF deployment flavour
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_flavour 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
PUT Change VNF deployment flavour
    log    Trying to perform a PUT. This method should not be implemented 
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_flavour 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
PATCH Change VNF deployment flavour
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_flavour   	 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
DELETE Change VNF deployment flavour
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_flavour   	 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

POST Create a new vnfInstance
    Log    Create VNF instance by POST to /vnf_instances
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/createVnfRequest.json
    ${body}=        Format String   ${template}     vnfdId=${vnfdId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

POST Create a new vnfInstance - Not ENABLED	
    Log    Create VNF instance by POST to /vnf_instances using vnfdId of a VNF Package which is not in ENABLED state or DNE.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/createVnfRequest.json
    ${body}=        Format String   ${template}     vnfdId=${vnfdId_notInEnabledState}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

GET multiple vnfInstances	
    Log    Query VNF The GET method queries information about multiple VNF instances.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances    
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 		

Get multiple vnfInstances with nextpage_opaque_marker parameter
    Log    Get next page of subscriptions
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query using the URI provided in the previous GET request and validate response
    Get    ${nextPageUri}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET multiple vnfInstances with bad attribute	
    Log    Query VNF The GET method queries information about multiple VNF instances.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances?attribute_not_exist=some_value  
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

GET multiple vnfInstances with bad filter	
    Log    Query VNF The GET method queries information about multiple VNF instances.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances?${NEG_FILTER_snapshot}  
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	
	
GET multiple vnfInstances using filter	
    Log    Query VNF The GET method queries information about multiple VNF instances using filter
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances?${POS_FILTER_snapshot}  
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

GET multiple vnfInstances with all_fields attribute selector
    Log    Query status information about multiple VNF instances, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances?all_fields
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
GET multiple vnfInstances with exclude_default attribute selector
    Log    Query status information about multiple VNF instances using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances?exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
GET multiple vnfInstances with fields attribute selector
    Log    Query status information about multiple VNF instances, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances?fields=${fields_vnfInstance}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}	
GET multiple vnfInstances with exclude_default and fields attribute selector
    Log    Query status information about multiple VNF instances, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances?fields=${fields_vnfInstance}&exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}	
GET multiple vnfInstances with exclude_fields attribute selector
    Log    Query status information about multiple VNF instances, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances?exclude_fields=${fields_vnfInstance}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 	

PUT VNFInstances
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances   
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 		     	

PATCH VNFInstances
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances   
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 		

DELETE VNFInstances
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances   
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

POST Create a new vnfInstance with permitted authorization scope	
    Log    Create VNF instance by POST to /vnf_instances with permitted authorization scope	
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_INSTANCE_INFO_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/createVnfRequest.json
    ${body}=        Format String   ${template}     vnfdId=${vnfdId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

POST Create a new vnfInstance with not permitted authorization scope	
    Log    Create VNF instance by POST to /vnf_instances with not permitted authorization scope	
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_INSTANCE_INFO_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/createVnfRequest.json
    ${body}=        Format String   ${template}     vnfdId=${vnfdId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	  
    
POST individual vnfInstance
    log    Trying to perform a POST. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}    
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}        

GET individual vnfInstance
    log    Trying to get information about an individual VNF instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET individual vnfInstance with permitted authorization scope
    log    Trying to get information about an individual VNF instance with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_INSTANCE_INFO_READONLY_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}      

GET individual vnfInstance with not permitted authorization scope
    log    Trying to get information about an individual VNF instance with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_INSTANCE_INFO_READONLY_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}   

PUT individual vnfInstance
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}	  	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
PATCH individual vnfInstance
    log    Trying to modify an individual VNF instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE_PATCH}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/patchBodyRequest.json
    ${body}=        Format String   ${template}     vnfdId=${Descriptor_ID}    vnfdVersion= ${Descriptor_Version}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
POST conflicting operation
    Log    Trying to create conflicting operation.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/terminateVnfRequest.json
    ${body}=        Format String   ${template}     terminationType=${terminationType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${anotherInstantiatedVnfInstanceId}/terminate    ${body}
    
PATCH individual vnfInstance with conflict
    POST conflicting operation
    log    Trying to modify an individual VNF instance with conflict
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE_PATCH}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/patchBodyRequest.json
    ${body}=        Format String   ${template}     vnfdId=${Descriptor_ID}    vnfdVersion= ${Descriptor_Version}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${anotherInstantiatedVnfInstanceId}    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

PATCH individual vnfInstance - Etag mismatch
    log    Trying to modify an individual VNF instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE_PATCH}"}
    Set Headers    {"If-Match": "${invalid_etag}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Run Keyword If    ${check_descriptors} == 1    PARSE the Descriptor File
    ${template}=    Get File    jsons/patchBodyRequest.json
    ${body}=        Format String   ${template}     vnfdId=${Descriptor_ID}    vnfdVersion= ${Descriptor_Version}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

DELETE individual vnfInstance
    log    Trying to delete an individual VNF instance
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
	
DELETE individual vnfInstance in INSTANTIATED state
    log    Trying to delete an individual VNF instance
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${instantiatedVnfInstanceId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
    
POST instantiate individual vnfInstance
    Log    Trying to Instantiate a vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/instantiateVnfRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/instantiate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

POST instantiate individual vnfInstance with conflict
    Log    Trying to Instantiate a vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/instantiateVnfRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${instantiatedVnfInstanceId}/instantiate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	
	
GET instantiate individual vnfInstance
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/instantiate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

PUT instantiate individual vnfInstance
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/instantiate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

PATCH instantiate individual vnfInstance
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/instantiate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

DELETE instantiate individual vnfInstance
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/instantiate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST instantiate individual vnfInstance with permitted authorization scope
    Log    Trying to Instantiate a vnf Instance with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${INSTANTIATE_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${body}=    Get File    jsons/instantiateVnfRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/instantiate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

POST instantiate individual vnfInstance with not permitted authorization scope
    Log    Trying to Instantiate a vnf Instance with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${INSTANTIATE_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${body}=    Get File    jsons/instantiateVnfRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/instantiate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

POST Scale vnfInstance
    Log    Trying to Instantiate a vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/scaleVnfRequest.json
    ${body}=        Format String   ${template}     scaletype=${scaletype}    scaleAspectId=${scaleAspectId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Scale vnfInstance with VNF NOT-INSTANTIATED
    Log    Trying to Instantiate a vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/scaleVnfRequest.json
    ${body}=        Format String   ${template}     scaletype=${scaletype}    scaleAspectId=${scaleAspectId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
POST Scale vnfInstance with not existing VNF
    Log    Trying to Instantiate a vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/scaleVnfRequest.json
    ${body}=        Format String   ${template}     scaletype=${scaletype}    scaleAspectId=${scaleAspectId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notExistingVnfInstanceId}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
POST Scale vnfInstance with permitted authorization scope
    Log    Trying to Instantiate a vnf Instance with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${SCALE_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/scaleVnfRequest.json
    ${body}=        Format String   ${template}     scaletype=${scaletype}    scaleAspectId=${scaleAspectId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
POST Scale vnfInstance with not permitted authorization scope
    Log    Trying to Instantiate a vnf Instance with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${SCALE_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/scaleVnfRequest.json
    ${body}=        Format String   ${template}     scaletype=${scaletype}    scaleAspectId=${scaleAspectId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
GET Scale vnfInstance				
    Log    Trying to get a scale a vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale    
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
PUT Scale vnfInstance				
    Log    Trying to modify a scale vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfRequest.json
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale   
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
PATCH Scale vnfInstance				
    Log    Trying to modify a scale vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfRequest.json
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale   
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	
DELETE Scale vnfInstance				
    Log    Trying to modify a scale vnf Instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfRequest.json
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale   
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Scale vnfInstance to level with InstantiationLevelId attribute
    Log    Trying to scale a vnf Instance to level
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/scaleVnfToLevelRequestInstantiationLevelId.json
    ${body}=        Format String   ${template}     instantiationLevelId=${instantiationLevelId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

POST Scale vnfInstance to level with scaleInfo attribute 
    Log    Trying to scale a vnf Instance to level
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/scaleVnfToLevelRequestScaleInfo.json
    ${body}=        Format String   ${template}    aspectId=${scaleAspectId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}     
POST Scale vnfInstance to level with VNF NOT-INSTANTIATED
    Log    Trying to scale a vnf Instance to level
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/scaleVnfToLevelRequest.json
    ${body}=        Format String   ${template}     instantiationLevelId=${instantiationLevelId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/scale_to_level    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 
POST Scale vnfInstance to level with not existing VNF
    Log    Trying to scale a vnf Instance to level
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/scaleVnfToLevelRequest.json
    ${body}=        Format String   ${template}     instantiationLevelId=${instantiationLevelId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notExistingVnfInstanceId}/scale_to_level    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
POST Scale vnfInstance to level with InstantiationLevelId attribute with permitted authorization scope
    Log    Trying to scale a vnf Instance to level with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${SCALE_TO_LEVEL_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/scaleVnfToLevelRequestInstantiationLevelId.json
    ${body}=        Format String   ${template}     instantiationLevelId=${instantiationLevelId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

POST Scale vnfInstance to level with InstantiationLevelId attribute with not permitted authorization scope
    Log    Trying to scale a vnf Instance to level with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${SCALE_TO_LEVEL_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/scaleVnfToLevelRequestInstantiationLevelId.json
    ${body}=        Format String   ${template}     instantiationLevelId=${instantiationLevelId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	
GET Scale vnfInstance to level
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		

PUT Scale vnfInstance to level
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		

PATCH Scale vnfInstance to level
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
	
DELETE Scale vnfInstance to level
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale_to_level 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
		
POST Terminate VNF
    Log    Trying to terminate a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/terminateVnfRequest.json
    ${body}=        Format String   ${template}     terminationType=${terminationType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/terminate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	

POST Terminate VNF in NOT-INSTANTIATED state
    Log    Trying to terminate a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/terminateVnfRequest.json
    ${body}=        Format String   ${template}     terminationType=${terminationType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/terminate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Terminate VNF with permitted authorization scope
    Log    Trying to terminate a VNF instance with permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${TERMINATE_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/terminateVnfRequest.json
    ${body}=        Format String   ${template}     terminationType=${terminationType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/terminate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Terminate VNF with not permitted authorization scope
    Log    Trying to terminate a VNF instance with not permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${TERMINATE_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/terminateVnfRequest.json
    ${body}=        Format String   ${template}     terminationType=${terminationType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/terminate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
    
GET Terminate VNF
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/terminate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		

PUT Terminate VNF
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/terminate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		

PATCH Terminate VNF
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/terminate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
	
DELETE Terminate VNF
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/terminate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Heal VNF
    Log    Trying to heal a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/healVnfRequest.json
    ${body}=        Format String   ${template}     healingCause=${healingCause}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/heal    ${body}		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Heal VNF Not Instantiated
    Log    Trying to heal a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/healVnfRequest.json
    ${body}=        Format String   ${template}     healingCause=${healingCause}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/heal    ${body}		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
	
POST Heal VNF Not Existing
    Log    Trying to heal a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/healVnfRequest.json
    ${body}=        Format String   ${template}     healingCause=${healingCause}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notExistingVnfInstanceId}/heal    ${body}		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Heal VNF with permitted authorization scope
    Log    Trying to heal a VNF instance with permitted authorization scope.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scopeValue=${HEAL_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/healVnfRequest.json
    ${body}=        Format String   ${template}     healingCause=${healingCause}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/heal    ${body}		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Heal VNF with not permitted authorization scope
    Log    Trying to heal a VNF instance with not permitted authorization scope.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scopeValue=${HEAL_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/healVnfRequest.json
    ${body}=        Format String   ${template}     healingCause=${healingCause}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/heal    ${body}		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET Heal VNF
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/heal  
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		

PUT Heal VNF
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/heal 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		

PATCH Heal VNF
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/heal 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
	
DELETE Heal VNF
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/heal 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Operate VNF
    Log    Trying to operate a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/operateVnfRequest.json
    ${body}=        Format String   ${template}     changeStateTo=${changeStateTo}    stopType=${stopType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/operate    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	

POST Operate VNF with conflict
    Log    Trying to operate a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/operateVnfRequest.json
    ${body}=        Format String   ${template}     changeStateTo=${changeStateTo}    stopType=${stopType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/operate    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
    
POST Operate VNF not existing
    Log    Trying to operate a VNF instance.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/operateVnfRequest.json
    ${body}=        Format String   ${template}     changeStateTo=${changeStateTo}    stopType=${stopType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notExistingVnfInstanceId}/operate    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET Operate VNF
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/operate  
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		

PUT Operate VNF
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/operate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		

PATCH Operate VNF
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/operate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
	
DELETE Operate VNF
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/operate 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Operate VNF with permitted authorization scope
    Log    Trying to operate a VNF instance with permitted authorization scope.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scopeValue=${OPERATE_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/operateVnfRequest.json
    ${body}=        Format String   ${template}     changeStateTo=${changeStateTo}    stopType=${stopType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/operate    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
	
POST Operate VNF with not permitted authorization scope
    Log    Trying to operate a VNF instance with not permitted authorization scope.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    ${scopeValue}=    Create Dictionary    scopeValue=${OPERATE_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/operateVnfRequest.json
    ${body}=        Format String   ${template}     changeStateTo=${changeStateTo}    stopType=${stopType}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/operate    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
	
Post VNF LCM Operation occurrences
    log    Trying to perform a POST. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs  	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
GET VNF LCM Operation occurrences
    Log    Query status information about multiple VNF lifecycle management operation occurrences.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Get VNF LCM Operation occurrences with nextpage_opaque_marker parameter
    Log    Get next page of subscriptions
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query using the URI provided in the previous GET request and validate response
    Get    ${nextPageUri}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
GET VNF LCM Operation occurrences invalid attribute
    Log    Query status information about multiple VNF lifecycle management operation occurrences.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"} 
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs?attribute_not_exist=some_value	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
GET VNF LCM Operation occurrences invalid filter
    Log    Query status information about multiple VNF lifecycle management operation occurrences.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"} 
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs?${NEG_FILTER_vnfLcmOpOcc}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
GET VNF LCM Operation occurrences using filter
    Log    Query status information about multiple VNF lifecycle management operation occurrences.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"} 
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs?${POS_FILTER_vnfLcmOpOcc}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Get VNF LCM Operation occurrences with all_fields attribute selector
    Log    Query status information about multiple VNF lifecycle management operation occurrences, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs?all_fields
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
Get VNF LCM Operation occurrences with exclude_default and fields attribute selector
    Log    Query status information about multiple VNF lifecycle management operation occurrences, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs?fields=${fields_vnfLcmOpOcc}&exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
Get VNF LCM Operation occurrences with exclude_default attribute selector
    Log    Query status information about multiple VNF lifecycle management operation occurrences using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs?exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
Get VNF LCM Operation occurrences with fields attribute selector
    Log    Query status information about multiple VNF lifecycle management operation occurrences, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs?fields=${fields_vnfLcmOpOcc}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}	
Get VNF LCM Operation occurrences with exclude_fields attribute selector
    Log    Query status information about multiple VNF lifecycle management operation occurrences, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs?exclude_fields=${fields_vnfLcmOpOcc}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
GET VNF LCM Operation occurrences with permitted authorization scope
    Log    Query status information about multiple VNF lifecycle management operation occurrences with permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
GET VNF LCM Operation occurrences with not permitted authorization scope
    Log    Query status information about multiple VNF lifecycle management operation occurrences with not permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
PUT VNF LCM Operation occurrences	
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
PATCH VNF LCM Operation occurrences	
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}			
DELETE VNF LCM Operation occurrences	
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Post Individual VNF LCM Operation occurrences 
    log    Trying to perform a POST. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Get Individual VNF LCM Operation occurrences
    Log    Query status information about multiple VNF lifecycle management operation occurrences.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Get Individual VNF LCM Operation occurrences with permitted authorization scope
    Log    Query status information about multiple VNF lifecycle management operation occurrences with permitted authorization scope.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Get Individual VNF LCM Operation occurrences with not permitted authorization scope
    Log    Query status information about multiple VNF lifecycle management operation occurrences with not permitted authorization scope.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Put multiple VNF instances	
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
Patch multiple VNF instances	
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Delete multiple VNF instances    
	log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Post Retry operation
    Log    Retry a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Post Retry operation with conflict
    Log    Retry a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccIdNotFailedTemp}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Post Retry operation not existing
    Log    Retry a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${notExistingVnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}   
Get Retry operation	
    Log    Trying to perform a GET. This method should not be implemented.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Put Retry operation	
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
Patch Retry operation	
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Delete Retry operation    
	log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Post Retry operation with permitted authorization scope
    Log    Retry a VNF lifecycle operation if that operation has experienced a temporary failure with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Post Retry operation with not permitted authorization scope
    Log    Retry a VNF lifecycle operation if that operation has experienced a temporary failure with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/retry	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
Post Rollback operation
    Log    Rollback a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/rollback
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Post Rollback operation with conflict
    Log    Rollback a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccIdNotFailedTemp}/rollback
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Post Rollback operation not existing
    Log    Rollback a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${notExistingVnfLcmOpOccId}/rollback
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	    
Get Rollback operation	
    Log    Trying to perform a GET. This method should not be implemented.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/rollback	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Put Rollback operation	
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/rollback	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
Patch Rollback operation	
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/rollback	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Delete Rollback operation    
	log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/rollback	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Post Rollback operation with permitted authorization scope
    Log    Rollback a VNF lifecycle operation if that operation has experienced a temporary failure with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/rollback
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Post Rollback operation with not permitted authorization scope
    Log    Rollback a VNF lifecycle operation if that operation has experienced a temporary failure with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/rollback
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Post Fail operation
    Log    mark as Failed a VNF lifecycle operation if that operation has experienced a temporary failure 
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/fail
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	    
Post Fail operation with conflict
    Log    mark as Failed a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccIdNotFailedTemp}/fail
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	    
	
Post Fail operation not existing
    Log    mark as Failed a VNF lifecycle operation if that operation has experienced a temporary failure
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${notExistingvnfLcmOpOccId}/fail
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Post Fail operation with permitted authorization scope
    Log    mark as Failed a VNF lifecycle operation if that operation has experienced a temporary failure with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/fail
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	    
Post Fail operation with not permitted authorization scope
    Log    mark as Failed a VNF lifecycle operation if that operation has experienced a temporary failure with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${OP_OCCS_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Log    Execute Query and validate response
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/fail
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	 
	
Get Fail operation	
    Log    Trying to perform a GET. This method should not be implemented.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/fail
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Put Fail operation	
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/fail	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
Patch Fail operation	
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/fail
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
Delete Fail operation    
	log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_lcm_op_occs/${vnfLcmOpOccId}/fail
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Post Create subscription
    Log    Create subscription instance by POST to ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/lccnSubscriptionRequest.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	Log    Verify notification endpoint is tested
	Verify Mock Expectation    ${notification_request}
	Clear Requests    ${callback_endpoint}
Post Create subscription - DUPLICATION
    Log    Trying to create a subscription with an already created content
    Pass Execution If    ${VNFM_DUPLICATION} == 0    VNFM is not permitting duplication. Skipping the test
    Set Headers    {"Accept": "${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/lccnSubscriptionRequest.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	Log    Verify notification endpoint is tested
	Verify Mock Expectation    ${notification_request}
	Clear Requests    ${callback_endpoint}

Post Create subscription - NO-DUPLICATION	
    Log    Trying to create a subscription with an already created content
    Pass Execution If    ${VNFM_DUPLICATION} == 1    VNFM permits duplication. Skipping the test
    Set Headers    {"Accept": "${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/lccnSubscriptionRequest.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	Log    Verify notification endpoint is tested
	Verify Mock Expectation    ${notification_request}
	Clear Requests    ${callback_endpoint}
	
Get subscriptions
    Log    Get the list of active subscriptions
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query and validate response
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
	
Get subscriptions with nextpage_opaque_marker parameter
    Log    Get next page of subscriptions
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query using the URI provided in the previous GET request and validate response
    Get    ${nextPageUri}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Get subscriptions - filter
    Log    Get the list of active subscriptions using a filter
    Set Headers    {"Accept": "${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${POS_filter_subscription}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
Get subscriptions - invalid filter  
    Log    Get the list of active subscriptions using an invalid filter
    Set Headers    {"Accept": "${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${NEG_filter_subscription}   
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
#Get subscriptions with all_fields attribute selector
#    Log    Get the list of active subscriptions, using fields
#    Set Headers    {"Accept": "${ACCEPT_JSON}"}
#    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
#    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?all_fields
#    ${output}=    Output    response
#    Set Suite Variable    ${response}    ${output}
#Get subscriptions with exclude_default attribute selector
#    Log    Get the list of active subscriptions, using fields
#    Set Headers    {"Accept": "${ACCEPT_JSON}"}
#    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
#    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?exclude_default
#    ${output}=    Output    response
#    Set Suite Variable    ${response}    ${output}
#Get subscriptions with fields attribute selector
#    Log    Get the list of active subscriptions, using fields
#    Set Headers    {"Accept": "${ACCEPT_JSON}"}
#    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
#    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?fields=${fields_subscription}
#    ${output}=    Output    response
#    Set Suite Variable    ${response}    ${output}	
#Get subscriptions with exclude_fields attribute selector
#    Log    Get the list of active subscriptions, using fields
#    Set Headers    {"Accept": "${ACCEPT_JSON}"}
#    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
#    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?exclude_fields=${fields_subscription}
#    ${output}=    Output    response
#    Set Suite Variable    ${response}    ${output}     		
Post Create subscription with permitted authorization scope
    Log    Create subscription instance by POST to ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${SUBSCRIPTIONS_PERMITTED_SCOPE}  
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/lccnSubscriptionRequest.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	Log    Verify notification endpoint is tested
	Verify Mock Expectation    ${notification_request}
	Clear Requests    ${callback_endpoint}
Post Create subscription with not permitted authorization scope
    Log    Create subscription instance by POST to ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${SUBSCRIPTIONS_NOT_PERMITTED_SCOPE}  
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/lccnSubscriptionRequest.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	Log    Verify notification endpoint is tested
	Verify Mock Expectation    ${notification_request}
	Clear Requests    ${callback_endpoint}
PUT subscriptions
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions  	   
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
PATCH subscriptions
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions  	   
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
DELETE subscriptions
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions  	   
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Post Create Individual subscription
    log    Trying to perform a POST. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}  	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
Get Individual subscription	
    log    Trying to get information about an individual subscription
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
    
Get Individual subscription with permitted authorization scope
    log    Trying to get information about an individual subscription with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${SUBSCRIPTIONS_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Get Individual subscription with not permitted authorization scope
    log    Trying to get information about an individual subscription with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${SUBSCRIPTIONS_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}				
Put Individual subscription	
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId} 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}		
Patch Individual subscription	
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId} 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	    
Delete Individual subscription
    log    Try to delete an individual subscription
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${Authorization_Token}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}  	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Put Notification subscription
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Put    ${callback_endpoint}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Patch Notification subscription
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Patch    ${callback_endpoint}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
Delete Notification subscription
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Delete    ${callback_endpoint}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
POST Operation occurrence
    log    The POST method delivers a notification from the server to the client.
    ${json}=	Get File	schemas/NsLcmOperationOccurrenceNotification.schema.json
    ${BODY}=	evaluate	json.loads('''${json}''')	json
    Log  Creating mock request and response to handle NSLcmOperationOccurrenceNotification
    &{req}=  Create Mock Request Matcher	POST  ${callback_endpoint}  body_type="JSON_SCHEMA"    body=${BODY}
    &{rsp}=  Create Mock Response	headers="Content-Type: application/json"  status_code=204
    Create Mock Expectation  ${req}  ${rsp}
    Sleep  ${sleep_interval}
    Log  Verifying results
    Verify Mock Expectation  ${req}
    Log  Cleaning the endpoint
    Clear Requests  ${callback_endpoint} 
POST Id creation
    log    The POST method delivers a notification from the server to the client.
    ${json}=	Get File	schemas/vnfIdentifierCreationNotification.schema.json
    ${BODY}=	evaluate	json.loads('''${json}''')	json
    Log  Creating mock request and response to handle vnfLcmOperationOccurrenceNotification
    &{req}=  Create Mock Request Matcher	POST  ${callback_endpoint}  body_type="JSON_SCHEMA"    body=${BODY}
    &{rsp}=  Create Mock Response	headers="Content-Type: application/json"  status_code=204
    Create Mock Expectation  ${req}  ${rsp}
    Sleep  ${sleep_interval}
    Log  Verifying results
    Verify Mock Expectation  ${req}
    Log  Cleaning the endpoint
    Clear Requests  ${callback_endpoint}
POST Id deletion
    log    The POST method delivers a notification from the server to the client.
    ${json}=	Get File	schemas/vnfIdentifierCreationNotification.schema.json
    ${BODY}=	evaluate	json.loads('''${json}''')	json
    Log  Creating mock request and response to handle vnfLcmOperationOccurrenceNotification
    &{req}=  Create Mock Request Matcher	POST  ${callback_endpoint}  body_type="JSON_SCHEMA"    body=${BODY}
    &{rsp}=  Create Mock Response	headers="Content-Type: application/json"  status_code=204
    Create Mock Expectation  ${req}  ${rsp}
    Sleep  ${sleep_interval}
    Log  Verifying results
    Verify Mock Expectation  ${req}
    Log  Cleaning the endpoint
    Clear Requests  ${callback_endpoint}    
GET test endpoint
    log    The GET method allows the server to test the notification endpoint
    &{req}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{rsp}=  Create Mock Response	headers="Content-Type: application/json"  status_code=204
    Create Mock Expectation  ${req}  ${rsp}
    Sleep  ${sleep_interval}
    Verify Mock Expectation  ${req}
    Clear Requests  ${callback_endpoint}    	    	
    
Check HTTP Response Header Contain Link
    ${linkURL}=    Get Value From Json    ${response['headers']}    $..Link
    Should Not Be Empty    ${linkURL}
    Log    Check that Link header contains rel set to next
    Should Contain    ${linkURL}    rel="next"
    Log    Check that Link header contains URI to next page between <>
    ${linkHeaderUri}=    Get Regexp Matches    ${linkURL}    (?<=\<)(.*?)(?=\>)
    ${length}=    Get Length    ${linkHeaderUri}	
    Should Be Equal As Integers	${length}	0
    Set Global Variable    ${nextPageUri}    ${linkHeaderUri[0]} 
    
Check HTTP Response Body vnfInstance content against VNF Descriptor
    Run Keyword If    ${check_descriptors} == 1    Check Individual VNF Instance Content
    
Check Individual VNF Instance Content    
    PARSE the Descriptor File
    Match the VNF Instance Response Attributes with Descriptors
    
PARSE the Descriptor File
    Run Keyword If  '${descriptorType}'=='SOL001'    Fetch Information from SOL001 descriptor file    ELSE    Fetch Information from SOL006 descriptor file
    
Fetch Information from SOL001 descriptor file
    Import Variables    ${CURDIR}/descriptors/SOL001/VNFD/vnfd_SOL001.yaml
    @{VDU_labels}=    Create List
    @{VNF_labels}=    Create List
    @{VirtualLink_labels}=    Create List
    @{CP_labels}=    Create List
    @{Storage_labels}=    Create List
    FOR    ${key}    IN    @{node_types.keys()}
        ${node_type}=    Get Variable Value    ${node_types['${key}']['derived_from']}
        ${is_VNF}=    Run Keyword And Return Status    Should Be Equal As Strings    ${node_type}    ${tosca_type_VNF}
        Run Keyword If    ${is_VNF}    Set Global Variable    ${tosca_type_derived_from_VNF}    ${key}
    END  
    ${derived_type_is_used}=    Run Keyword And Return Status    Should not be empty    ${tosca_type_derived_from_VNF}
    Run Keyword If    ${derived_type_is_used}    Set Global Variable    ${tosca_type_VNF}    ${tosca_type_derived_from_VNF}

    FOR    ${key}    IN    @{topology_template.node_templates.keys()} 
        ${key_type}=    Get Variable Value    ${topology_template.node_templates['${key}'].type}
        ${VDU_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}   ${tosca_type_VDU_compute}
        ${VNF_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_VNF}
        ${Link_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_virtual_link}
        ${VDUcp_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_VDU_cp}
        ${Storage_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_storage}
        Run Keyword If    ${VDU_check}    Append To List    ${VDU_labels}    ${key}
        Run Keyword If    ${VNF_check}    Append To List    ${VNF_labels}    ${key}
        Run Keyword If    ${VNF_check}    Get VNF Attributes from SOL001    ${key}            
        Run Keyword If    ${Link_check}    Append To List    ${VirtualLink_labels}    ${key}
        Run Keyword If    ${VDUcp_check}    Append To List    ${CP_labels}    ${key}
        Run Keyword If    ${Storage_check}    Append To List    ${Storage_labels}    ${key}
    END
    Set Global Variable    @{VDU_IDs}    @{VDU_labels}
    Set Global Variable    @{VNF_IDs}    @{VNF_labels}
    Set Global Variable    @{VirtualLink_IDs}    @{VirtualLink_labels}
    Set Global Variable    @{CP_IDs}    @{CP_labels}
    Set Global Variable    @{Storage_IDs}    @{Storage_labels}

Get VNF Attributes from SOL001
    [Arguments]    ${VNF_label}
    Import Variables    ${CURDIR}/descriptors/SOL001/VNFD/vnfd_SOL001.yaml    
    ${descriptor_id}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.descriptor_id}
    ${provider}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.provider}
    ${product_name}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.product_name}
    ${software_version}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.software_version}
    ${descriptor_version}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.descriptor_version}
    ${flavour_id}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.flavour_id}
    
    Set Global Variable    ${Descriptor_ID}    ${descriptor_id}
    Set Global Variable    ${Provider}    ${provider}
    Set Global Variable    ${Product_Name}    ${product_name}
    Set Global Variable    ${Software_Version}    ${software_version}
    Set Global Variable    ${Descriptor_Version}    ${descriptor_version}
    Set Global Variable    ${Flavour_ID}    ${flavour_id}


Fetch Information from SOL006 descriptor file
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    ${descriptor_id}=    Get Variable Value    ${nfv.vnfd[0].id}
    ${provider}=    Get Variable Value    ${nfv.vnfd[0].provider}
    ${product_name}=    Get Variable Value    ${nfv.vnfd[0]['product-name']}
    ${software_version}=    Get Variable Value    ${nfv.vnfd[0]['software-version']}
    ${descriptor_version}=    Get Variable Value    ${nfv.vnfd[0].version}    
    ${VDUcount}=    Get Length    ${nfv.vnfd[0].vdu}
    ${extCP_count}=    Get Length    ${nfv.vnfd[0]['ext-cpd']}
    ${virtualLink_count}=    Get length    ${nfv.vnfd[0]['int-virtual-link-desc']}
    ${instantiation_levels}=    Get Length    ${nfv.vnfd[0].df['instantiation-level']}    
    FOR    ${key}    IN    @{nfv.vnfd[0].keys()}
        ${VDU_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    vdu
        Run Keyword If    ${VDU_check}    Get VDU IDs    ${VDUcount}
        ${extCP_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    ext-cpd
        Run Keyword If    ${extCP_check}    Get External CP IDs    ${extCP_count}
        ${virtualLink_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    int-virtual-link-desc
        Run Keyword If    ${virtualLink_check}    Get Virtual Link IDs   ${virtualLink_count}    
        ${DF_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    df
        Run Keyword If    ${DF_check}    Get Instantiation Levels    ${instantiation_levels}    
    END
    Set Global Variable    ${Descriptor_ID}    ${descriptor_id}
    Set Global Variable    ${Provider}    ${provider}
    Set Global Variable    ${Product_Name}    ${product_name}
    Set Global Variable    ${Software_Version}    ${software_version}
    Set Global Variable    ${Descriptor_Version}    ${descriptor_version}
    
Get VDU IDs
    [Arguments]    ${count}    
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    @{VDU_labels}=    Create List
    ${Storage_labels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${VDU_labels}    ${nfv.vnfd[0].vdu[${i}]['id']}
        Append To List    ${Storage_labels}    ${nfv.vnfd[0].vdu[${i}]['virtual-storage-desc']} 
        Get Internal CPs for each VDU    ${i}   
    END
    Set Global Variable    ${VDU_IDs}    ${VDU_labels}
    Set Global Variable    ${Storage_IDs}    ${Storage_labels}
            
Get Internal CPs for each VDU
    [Arguments]    ${vdu}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml    
    ${count}=    Get Length    ${nfv.vnfd[0].vdu[${vdu}]['int-cpd']}    
    ${internal_CPs}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${internal_CPs}    ${nfv.vnfd[0].vdu[${vdu}]['int-cpd'][${i}]['id']} 
    END
    Set Global Variable    ${internalCP_IDs}    ${internal_CPs}
    
Get External CP IDs
    [Arguments]    ${count}    
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    @{external_CPs}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${external_CPs}    ${nfv.vnfd[0].ext_cpd[${i}].id} 
    END
    Set Global Variable    ${externalCP_IDs}    ${external_CPs}

Get Virtual Link IDs
    [Arguments]    ${count}    
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml    
    ${VirtualLink_labels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${VirtualLink_labels}    ${nfv.vnfd[0]['int-virtual-link-desc'][${i}]['id']} 
    END
    Set Global Variable    ${VirtualLink_IDs}    ${VirtualLink_labels}

Get Instantiation Levels
    [Arguments]    ${count}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    @{Instantiation_Levels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${Instantiation_Levels}    ${nfv.vnfd[0].df['instantiation-level'][${i}]['id']} 
    END
    Set Global Variable    ${InstantiationLevel_IDs}    ${Instantiation_Levels}
    
Match the VNF Instance Response Attributes with Descriptors
    #Checking Response attributes with VNF Descriptors
    Should Be Equal As Strings    ${response['body']['vnfdId']}    ${Descriptor_ID}
    Should Be Equal As Strings    ${response['body']['vnfProvider']}    ${Provider}
    Should Be Equal As Strings    ${response['body']['vnfProductName']}    ${Product_Name}
    Should Be Equal As Strings    ${response['body']['vnfSoftwareVersion']}    ${Software_Version}
    Should Be Equal As Strings    ${response['body']['vnfdVersion']}    ${Descriptor_Version}
    Should Be Equal As Strings    ${response['body']['instantiatedVnfInfo']['flavourId']}    ${Flavour_ID}
    List Should Contain Value    ${VDU_IDs}    ${response['body']['instantiatedVnfInfo']['vnfcResourceInfo']['vduId']}        
    Run Keyword If  '${descriptorType}'=='SOL006'    List Should Contain Value    ${externalCP_IDs}    ${response['body']['instantiatedVnfInfo']['extCpInfo'][0]['cpdId']}
    Run Keyword If  '${descriptorType}'=='SOL006'    List Should Contain Value    ${internalCP_IDs}    ${response['body']['instantiatedVnfInfo']['vnfcResourceInfo'][0]['vnfcCpInfo'][0]['cpdId']}
    Run Keyword If  '${descriptorType}'=='SOL001'    List Should Contain Value    @{CP_IDs}    ${response['body']['instantiatedVnfInfo']['extCpInfo'][0]['cpdId']}
    Run Keyword If  '${descriptorType}'=='SOL001'    List Should Contain Value    @{CP_IDs}    ${response['body']['instantiatedVnfInfo']['vnfcResourceInfo'][0]['vnfcCpInfo'][0]['cpdId']}
    List Should Contain value    ${Storage_IDs}    ${response['body']['instantiatedVnfInfo']['virtualStorageResourceInfo'][0]['virtualStorageDescId']}
    List Should Contain Value    ${VirtualLink_IDs}    ${response['body']['instantiatedVnfInfo']['extManagedVirtualLinkInfo'][0]['vnfVirtualLinkDescId']}

Check HTTP Response Body of Individual VNF LCM Operation occurrences content against VNF Descriptor
    Run Keyword If    ${check_descriptors} == 1    Check Individual VNF LCM Operation Occurence Content
    
Check Individual VNF LCM Operation Occurence Content
    PARSE the Descriptor File
    Match the VNF LCM Operation Occurence Response Attributes with Descriptors
    
Match the VNF LCM Operation Occurence Response Attributes with Descriptors
    #Checking Response attributes with VNF Descriptors
    Should Be Equal As Strings    ${response['body']['changedInfo']['vnfdId']}    ${Descriptor_ID}
    Should Be Equal As Strings    ${response['body']['changedInfo']['vnfdVersion']}    ${Descriptor_Version}
    List Should Contain Value    ${VDU_IDs}    ${response['body']['resourceChanges']['affectedVnfcs'][0]['vduId']}
    List Should Contain value    ${Storage_IDs}    ${response['body']['resourceChanges']['affectedVirtualStorages'][0]['virtualStorageDescId']}
    List Should Contain Value    ${VirtualLink_IDs}    ${response['body']['resourceChanges']['affectedVirtualLinks'][0]['virtualStorageDescId']}

POST Change Current VNF Package Task
    Log    Trying to change current VNF package task.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/changeCurrentVnfPkgRequest.json
    ${body}=        Format String   ${template}     vnfdId=${vnfdId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_vnfpkg    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

POST Change Current VNF Package Task with permitted authorization scope
    Log    Trying to change current VNF package task with permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${CHANGE_VNFPKG_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/changeCurrentVnfPkgRequest.json
    ${body}=        Format String   ${template}     vnfdId=${vnfdId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_vnfpkg    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  
	
POST Change Current VNF Package Task with not permitted authorization scope
    Log    Trying to change current VNF package task with not permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${CHANGE_VNFPKG_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/changeCurrentVnfPkgRequest.json
    ${body}=        Format String   ${template}     vnfdId=${vnfdId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_vnfpkg    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}      
	
GET Change Current VNF Package Task
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_vnfpkg 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

PUT Change Current VNF Package Task
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_vnfpkg 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

PATCH Change Current VNF Package Task
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_vnfpkg 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
DELETE Change Current VNF Package Task
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/change_vnfpkg 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

Check HTTP Location Header Contains URI of Created Resource
    ${URI}=    Get Value From Json    ${response['headers']}    $..Location
    Should Not Be Empty    ${URI}

POST Create VNF Snapshot Task
    Log    Trying to request a snapshot of a VNF instance
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/CreateVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/create_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Create VNF Snapshot Task NOT SUPPORTED
    Log    Trying to request a snapshot of a VNF instance for which task resource does not exist
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/CreateVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId_NoSnapshotTask}/create_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  

POST Create VNF Snapshot Task VNF_NOT_INSTANTIATED
    Log    Trying to request a snapshot of a VNF instance which is in NOT_INSTANTIATED state
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/CreateVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/create_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  
	
POST Create VNF Snapshot Task INVALID IDENTIFIER
    Log    Trying to request a snapshot of a VNF instance with invalid "Individual VNF Snapshot" resource identifier
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/CreateVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${invalidVnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/create_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Create VNF Snapshot Task with permitted authorization scope
    Log    Trying to request a snapshot of a VNF instance with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${CREATE_SNAPSHOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/CreateVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/create_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Create VNF Snapshot Task with not permitted authorization scope
    Log    Trying to request a snapshot of a VNF instance with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${CREATE_SNAPSHOT_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/CreateVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/create_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  
	
GET Create VNF Snapshot Task
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/create_snapshot 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

PUT Create VNF Snapshot Task
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/create_snapshot 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

PATCH Create VNF Snapshot Task
    log    Trying to perform a PATCH. This method should not be implemented
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/create_snapshot 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
DELETE Create VNF Snapshot Task
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/create_snapshot 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Revert to VNF Snapshot Task
    Log    Trying to initiate revert to VNF snapshot task.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/RevertToVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/revert_to_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
POST Revert to VNF Snapshot Task NOT SUPPORTED
    Log    Trying to revert to a snapshot of a VNF instance for which task resource does not exist
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/RevertToVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId_NoSnapshotTask}/revert_to_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  

POST Revert to VNF Snapshot Task VNF_NOT_INSTANTIATED
    Log    Trying to revert to a snapshot of a VNF instance which is in NOT_INSTANTIATED state
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/RevertToVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/revert_to_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  

GET Revert to VNF Snapshot Task
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/revert_to_snapshot 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

PUT Revert to VNF Snapshot Task
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/revert_to_snapshot 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

PATCH Revert to VNF Snapshot Task
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/revert_to_snapshot 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

DELETE Revert to VNF Snapshot Task
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/revert_to_snapshot 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Revert to VNF Snapshot Task with permitted authorization scope
    Log    Trying to initiate revert to VNF snapshot task with permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${REVERT_TO_SNAPSHOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/RevertToVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/revert_to_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Revert to VNF Snapshot Task with not permitted authorization scope
    Log    Trying to initiate revert to VNF snapshot task with not permitted authorization scope.
    ${scopeValue}=    Create Dictionary    scopeValue=${REVERT_TO_SNAPSHOT_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template}=    Get File    jsons/RevertToVnfSnapshotRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotInfoId=${vnfSnapshotInfoId} 
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/revert_to_snapshot    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  
	
POST Create a new VNF Snapshot
    Log    Create VNF snapshot by POST to ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template} =    Get File    jsons/CreateVnfSnapshotInfoRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotPkgId=${vnfSnapshotPkgId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	 	

GET multiple VNF Snapshots	
    Log    Query VNF The GET method queries information about multiple VNF snapshots.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots 
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 		

GET multiple VNF Snapshots with bad attribute	
    Log    Query VNF The GET method queries information about multiple VNF snapshots.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"} 
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots?attribute_not_exist=some_value
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	

GET multiple VNF Snapshots with bad filter	
    Log    Query VNF The GET method queries information about multiple VNF snapshots.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"} 
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots?${NEG_FILTER_snapshot}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	
	
GET multiple VNF Snapshots using filter	
    Log    Query VNF The GET method queries information about multiple VNF snapshots.
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"} 
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots?${POS_FILTER_snapshot}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 	
	
GET multiple VNF Snapshots with all_fields attribute selector
    Log    Query status information about multiple VNF snapshots, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots?all_fields
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET multiple VNF Snapshots with exclude_default attribute selector
    Log    Query status information about multiple VNF snapshots using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots?exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET multiple VNF Snapshots with fields attribute selector
    Log    Query status information about multiple VNF snapshots, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots?fields=${fields_snapshot}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}	    
    
GET multiple VNF Snapshots with exclude_default and fields attribute selector
    Log    Query status information about multiple VNF snapshots, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots?fields=${fields_snapshot}&exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET multiple VNF Snapshots with exclude_fields attribute selector
    Log    Query status information about multiple VNF snapshots, using fields
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots?exclude_fields=${fields_snapshot}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 	

GET VNF Snapshots without Paging support
    Run Keyword If    ${PAGING_SUPPORTED} == 0    GET multiple VNF Snapshots
    
GET VNF Snapshots with Paging support
    Run Keyword If    ${PAGING_SUPPORTED} == 1    GET multiple VNF Snapshots
    
Check HTTP Response Header Contains Link
    ${linkURL}=    Get Value From Json    ${response['headers']}    $..Link
    Should Not Be Empty    ${linkURL}

PUT VNF Snapshots
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots   	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
PATCH VNF Snapshots
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots	 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	
DELETE VNF Snapshots
    Log    Trying to perform a DELETE. This method should not be implemented
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots 	 	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Create a new VNF Snapshot with permitted authorization scope
    Log    Create VNF snapshot by POST to ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_SNAPSHOT_INFO_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template} =    Get File    jsons/CreateVnfSnapshotInfoRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotPkgId=${vnfSnapshotPkgId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Create a new VNF Snapshot with not permitted authorization scope
    Log    Create VNF snapshot by POST to ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_SNAPSHOT_INFO_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    ${template} =    Get File    jsons/CreateVnfSnapshotInfoRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotPkgId=${vnfSnapshotPkgId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

POST individual VNF Snapshot
    log    Trying to perform a POST. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}    
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}        

GET individual VNF Snapshot
    log    Trying to get information about an individual VNF snapshot
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET individual VNF Snapshot with invalid URI
    log    Trying to get information about an individual VNF snapshot
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${invalidVnfSnapshotInfoId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}      
	
PUT individual VNF Snapshot
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

PATCH individual VNF Snapshot
    log    Trying to perform a PATCH to modify individual VNF snapshot
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE_PATCH}"}
    Set Headers    {"If-Match": "${original_etag}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template} =    Get File    jsons/VnfSnapshotInfoModificationRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotPkgId=${vnfSnapshotPkgId}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}    ${body} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

PATCH individual VNF Snapshot - ETag mismatch
    log    Trying to perform a PATCH to modify individual VNF snapshot
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE_PATCH}"}
    Set Headers    {"If-Match": "${invalid_etag}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template} =    Get File    jsons/VnfSnapshotInfoModificationRequest.json
    ${body}=        Format String   ${template}     vnfSnapshotPkgId=${vnfSnapshotPkgId}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}    ${body} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

DELETE individual VNF Snapshot
    log    Trying to DELETE an individual VNF Snapshot
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

DELETE individual VNF Snapshot - CONFLICT
    log    Trying to DELETE an individual VNF Snapshot that is already in use by another operation
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId_InUse} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET individual VNF Snapshot with permitted authorization scope
    log    Trying to get information about an individual VNF snapshot with permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_SNAPSHOT_INFO_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET individual VNF Snapshot with not permitted authorization scope
    log    Trying to get information about an individual VNF snapshot with not permitted authorization scope
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_SNAPSHOT_INFO_NOT_PERMITTED_SCOPE}
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${authorizationToken}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId} 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  
	
Check Postcondition VNF Snapshot Resource Is Deleted
    GET individual VNF Snapshot
    Integer    response status    404
    Log    VNF Snapshot Resource deleted
    
Check Postcondition VNF Snapshot Resource Existence
    GET individual VNF Snapshot
    Integer    response status    200
    Log    VNF Snapshot Resource is not deleted

POST VNF State Snapshot
    log    Trying to perform a POST. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot   
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}        

GET VNF State Snapshot - Complete
    log    Trying to fetch the complete contents of an VNF State Snapshot
    Set Headers    {"Accept":"${ACCEPT_OCTET}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET VNF State Snapshot - Range Request
    Pass Execution If    ${VNFM_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    log    Trying to get information about an VNF State Snapshot using a range request
    Set Headers    {"Accept":"${ACCEPT_OCTET}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers    {"Range":"${range}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot 		
    Unset Headers  ["Range"]
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}  

GET VNF State Snapshot - Invalid Range
    Pass Execution If    ${VNFM_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    log    Trying to get information about a VNF State Snapshot with invalid range
    Set Headers    {"Accept":"${ACCEPT_OCTET}"}  
    Set Headers    {"Version": "${API_VERSION}"} 
    Set Headers    {"Range":"${erroneousRange}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot 		
    Unset Headers  ["Range"]
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}  
	
GET VNF State Snapshot - Conflict in State
    Log    Trying to get information about a VNF State Snapshot whose creation process is not completed
    Set Headers    {"Accept":"${ACCEPT_OCTET}"}  
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId_creationIncomplete}/vnf_state_snapshot 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET VNF State Snapshot - Complete with permitted authorization scope
    log    Trying to fetch the complete contents of an VNF State Snapshot with permitted authorization scope
    Set Headers    {"Accept":"${ACCEPT_OCTET}"}  
    Set Headers    {"Version": "${API_VERSION}"} 
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_STATE_SNAPSHOT_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

GET VNF State Snapshot - Complete with not permitted authorization scope
    log    Trying to fetch the complete contents of an VNF State Snapshot with not permitted authorization scope
    Set Headers    {"Accept":"${ACCEPT_OCTET}"}  
    Set Headers    {"Version": "${API_VERSION}"}
    ${scopeValue}=    Create Dictionary    scopeValue=${VNF_STATE_SNAPSHOT_NOT_PERMITTED_SCOPE} 
    ${authorizationToken}=    JWT Encode    payload=${scopeValue}    key=${OAUTH_KEY}    algorithm=${OAUTH_ENCRYPTION_ALGORITHM}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}  

PUT VNF State Snapshot
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

PATCH VNF State Snapshot
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

DELETE VNF State Snapshot
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_snapshots/${vnfSnapshotInfoId}/vnf_state_snapshot 		
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Select VNF Deployable Modules
    log    Trying to select deployable modules of VNF instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/SelectVnfDeployableModulesRequest.json
    ${body}=    Format String    ${template}
    Post    ${apiRoot}/vnflcm/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/select_depl_mods    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

POST Select VNF Deployable Modules Not Existing
    log    Trying to select deployable modules of VNF instance
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/SelectVnfDeployableModulesRequest.json
    ${body}=    Format String    ${template}
    Post    ${apiRoot}/vnflcm/${apiMajorVersion}/vnf_instances/${notExistingVnfInstanceId}/select_depl_mods    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

POST Select VNF Deployable Modules with conflict
    log    Trying to select deployable modules of VNF instance.
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template}=    Get File    jsons/SelectVnfDeployableModulesRequest.json
    ${body}=    Format String    ${template}
    Post    ${apiRoot}/vnflcm/${apiMajorVersion}/vnf_instances/${notInstantiatedVnfInstanceId}/select_depl_mods    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}


GET Select VNF Deployable Modules
    log    Trying to perform a GET. This method should not be implemented
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"} 
    Get    ${apiRoot}/vnflcm/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/select_depl_mods
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

PUT Select VNF Deployable Modules
    log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Put    ${apiRoot}/vnflcm/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/select_depl_mods
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

PATCH Select VNF Deployable Modules
    log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Patch    ${apiRoot}/vnflcm/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/select_depl_mods
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}

DELETE Select VNF Deployable Modules
    log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Delete    ${apiRoot}/vnflcm/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/select_depl_mods
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}


Check HTTP Content-Type Header Is Set
    Log    Check Content-Type HTTP Header
    Should Contain    ${response['headers']}    Content-Type
    Should Be Equal As Strings    ${response['headers']['Content-Type']}    ${CONTENT_TYPE_OCTET} 
    Log    Content type validated
    
Check HTTP Content-Range Header Is Set
    Log    Check Content-Range HTTP Header
    Should Contain    ${response['headers']}    Content-Range
    Should Be Equal As Strings    ${response['headers']['Content-Range']}    ${range}/${full_length}
    Log    Header Content-Range is present
    
Check HTTP Content-Length Header Is Set
    Log    Check Content-Range HTTP Header
    Should Contain    ${response['headers']}    Content-Length
    Should Be Equal As Strings    ${response['headers']['Content-Length']}    ${full_length}
    Log    Header Content-Length is present

JWT Encode
    [Arguments]    ${payload}    ${key}    ${algorithm}
    ${encoded}=    Evaluate    jwt.encode(${payload}, ${key}, ${algorithm})
    [Return]    ${encoded}
POST instantiate individual vnfInstance - Unauthorized
    Log    Trying to Instantiate a vnf Instance witout authorization
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    ${body}=    Get File    jsons/instantiateVnfRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/instantiate    ${body}	
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse} 

POST Create a new vnfInstance with Invalid vnfdId   
    Log    Create VNF instance by POST to ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances with invalid vnfdId
    Set Headers  {"Accept":"${ACCEPT}"}  
    Set Headers  {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${template} =    Get File    jsons/createVnfRequest.json
    ${body}=        Format String   ${template}     vnfdId=${invalidVnfdId}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances    ${body}
    ${outputResponse}=    Output    response
    Set Global Variable    ${response}    ${outputResponse}    

POST Scale vnfInstance Scale Vertical Compute
    Log    Trying to scale a vnf Instance vertically by scaling the compute
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfScaleVerticalComputeRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Scale vnfInstance Scale Vertical Storage
    Log    Trying to scale a vnf Instance vertically by scaling the storage
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfScaleVerticalStorageRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}

POST Scale vnfInstance Scale Vertical Oscontainer
    Log    Trying to scale a vnf Instance vertically by scaling the oscontainer
    Set Headers  {"Accept":"${ACCEPT}"}
    Set Headers    {"Version": "${API_VERSION}"}  
    Set Headers  {"Content-Type": "${CONTENT_TYPE}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/scaleVnfScaleVerticalOscontainerRequest.json
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_instances/${vnfInstanceId}/scale    ${body}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}