*** Settings ***
Library           String
Library           JSONSchemaLibrary    schemas/
Resource          environment/variables.txt    # Generic Parameters
Library           REST    ${NFVMANO_SCHEMA}://${NFVMANO_HOST}:${NFVMANO_PORT}    ssl_verify=false
Library           OperatingSystem
Library           JSONLibrary
Resource          environment/subscriptions.txt
Library           MockServerLibrary
Library           Process
Suite Setup       Create Sessions
Suite Teardown    Terminate All Processes    kill=true

*** Test Cases ***
GET all Performance Subscriptions
    [Documentation]    Test ID: 8.3.2.6.1
    ...    Test title: GET all Performance Subscriptions
    ...    Test objective: The objective is to test the retrieval of all performance subscriptions and perform a JSON schema validation of the returned subscriptions data structure.
    ...    Pre-conditions: An instance is instantiated. At least one performance subscription is available.
    ...    Reference: clause 6.5.8.3.2 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: none    
    Get all Performance Subscriptions
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    PmSubscriptions

GET Performance Subscriptions with attribute-based filter
    [Documentation]    Test ID: 8.3.2.6.2
    ...    Test title: GET Performance Subscriptions with attribute-based filter
    ...    Test objective: The objective is to test the retrieval of performance subscriptions using attribute-based filter, perform a JSON schema validation of the collected indicators data structure. 
    ...    Pre-conditions: An instance is instantiated. At least one performance subscription is available.
    ...    Reference: clause 6.5.8.3.2 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: none
    Get Performance Subscriptions with attribute-based filters
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    PmSubscriptions
    
GET Performance Management Subscriptions with Paged Response
    [Documentation]    Test ID: 8.3.2.6.3
    ...    Test title: GET Performance Management Subscriptions with Paged Response
    ...    Test objective: The objective is to query information about NFV-MANO Performance Subscriptions to get Paged Response.
    ...    Pre-conditions:  An instance is instantiated. At least one performance subscription is available.
    ...    Reference: clauseclause 6.5.8.3.2 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: none
    Get all Performance Subscriptions
    Check HTTP Response Status Code Is    200
    Check HTTP Response Header Contains Link

GET Performance Subscriptions with invalid attribute-based filter
    [Documentation]    Test ID: 8.3.2.6.4
    ...    Test title: GET Performance Subscriptions with invalid attribute-based filter
    ...    Test objective: The objective is to test that the retrieval of performance subscriptions fails when using invalid attribute-based filters, and perform the JSON schema validation of the failed operation HTTP response. 
    ...    Pre-conditions: An instance is instantiated. At least one performance subscription is available.
    ...    Reference: clause 6.5.8.3.2 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: none
    Get Performance Subscriptions with invalid attribute-based filters
    Check HTTP Response Status Code Is    400
    Check HTTP Response Body Json Schema Is   ProblemDetails 
    
GET Performance Subscriptions - Bad Request Response too Big
    [Documentation]    Test ID: 8.3.2.6.5
    ...    Test title: GET Performance Subscriptions - Bad Request Response too Big
    ...    Test objective: The objective is to test that the retrieval of NFV-MANO performance subscriptions fails when response is too big, and perform the JSON schema validation of the failed operation HTTP response.
    ...    Pre-conditions:  An instance is instantiated. At least one performance subscription is available.
    ...    Reference: clause 6.5.8.3.2 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: 
    ...    Post-Conditions: 
    Get all Performance Subscriptions
    Check HTTP Response Status Code Is    400
    Check HTTP Response Body Json Schema Is    ProblemDetails

GET Performance Subscriptions with invalid resource endpoint
    [Documentation]    Test ID: 8.3.2.6.6
    ...    Test title: GET Performance Subscriptions with invalid resource endpoint
    ...    Test objective: The objective is to test that the retrieval of all performance subscriptions fails when using invalid resource endpoint.
    ...    Pre-conditions: An instance is instantiated. At least one performance subscription is available.
    ...    Reference: clause 6.5.8.3.2 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: none    
    Get Performance Subscriptions with invalid resource endpoint
    Check HTTP Response Status Code Is    404
    
Create new Performance subscription
    [Documentation]    Test ID: 8.3.2.6.7
    ...    Test title: Create new Performance subscription
    ...    Test objective: The objective is to test the creation of a new performance subscription and perform a JSON schema and content validation of the returned subscription data structure.
    ...    Pre-conditions: An instance is instantiated.
    ...    Reference: clause 6.5.8.3.1 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: The performance subscription is successfully set and it matches the issued subscription. 
    Send Post Request for Performance Subscription
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is    PmSubscription
    Check HTTP Response Body PmSubscription Attributes Values Match the Issued Subscription
    Check Postcondition Performance Subscription Is Set 


Create duplicated Performance subscription with NFV-MANO not creating duplicated subscriptions
    [Tags]    no-duplicated-subs
    [Documentation]    Test ID: 8.3.2.6.8
    ...    Test title: Create duplicated Performance subscription with NFV-MANO not creating duplicated subscriptions
    ...    Test objective: The objective is to test the attempt of a creation of a duplicated performance subscription and check that no new subscription is created by the NFV-MANO and a link to the original subscription is returned
    ...    Pre-conditions: An instance is instantiated. At least one performance subscription is available in the NFV-MANO.
    ...    Reference: clause 6.5.8.3.1 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: The NFV-MANO does not support the creation of duplicated subscriptions
    ...    Post-Conditions: The existing performance subscription returned is available in the NFV-MANO
    Send Post Request for Duplicated Performance Subscription
    Check HTTP Response Status Code Is    303
    Check HTTP Response Body Is Empty
    Check HTTP Response Header Contains    Location
    Check Postcondition Subscription Resource Returned in Location Header Is Available

Create duplicated Performance subscription with NFV-MANO creating duplicated subscriptions
    [Tags]    duplicated-subs
    [Documentation]    Test ID: 8.3.2.6.9
    ...    Test title: Create duplicated Performance subscription with NFV-MANO creating duplicated subscriptions
    ...    Test objective: The objective is to test the creation of a duplicated performance subscription and perform a JSON schema and content validation of the returned duplicated subscription data structure.
    ...    Pre-conditions: An instance is instantiated. At least one performance subscription is available in the NFV-MANO.
    ...    Reference: clause 6.5.8.3.1 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: The NFV-MANO supports the creation of duplicated subscriptions
    ...    Post-Conditions: The duplicated performance subscription is successfully set and it matches the issued subscription
    Send Post Request for Duplicated Performance Subscription
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is    PmSubscription
    Check HTTP Response Body PmSubscription Attributes Values Match the Issued Subscription
    Check Postcondition Performance Subscription Is Set 

PUT Performance Subscriptions - Method not implemented
    [Documentation]    Test ID: 8.3.2.6.10
    ...    Test title: PUT Performance Subscriptions - Method not implemented
    ...    Test objective: The objective is to test that PUT method is not allowed to modify performance subscriptions
    ...    Pre-conditions: none
    ...    Reference: clause 6.5.8.3.3 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: none
    Send Put Request for Performance Subscriptions
    Check HTTP Response Status Code Is    405 
    
PATCH Performance Subscriptions - Method not implemented
    [Documentation]    Test ID: 8.3.2.6.11
    ...    Test title: PATCH Performance Subscriptions - Method not implemented
    ...    Test objective: The objective is to test that PATCH method is not allowed to update performance subscriptions
    ...    Pre-conditions: none
    ...    Reference: clause 6.5.8.3.4 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: none
    Send Patch Request for Performance Subscriptions
    Check HTTP Response Status Code Is    405
    
DELETE Performance Subscriptions - Method not implemented
    [Documentation]    Test ID: 8.3.2.6.12
    ...    Test title: DELETE Performance Subscriptions - Method not implemented
    ...    Test objective: The objective is to test that DELETE method is not allowed to delete performance subscriptions
    ...    Pre-conditions: none
    ...    Reference: clause 6.5.8.3.5 - ETSI GS NFV-SOL 009 [7] v3.3.1
    ...    Config ID: Config_prod_NFV-MANO
    ...    Applicability: none
    ...    Post-Conditions: none
    Send Delete Request for Performance Subscriptions
    Check HTTP Response Status Code Is    405
    
*** Keywords ***

Create Sessions
    Pass Execution If    ${NFVMANO_CHECKS_NOTIF_ENDPOINT} == 0   MockServer not necessary to run    
    Start Process  java  -jar  ${MOCK_SERVER_JAR}    -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}
    
Check Notification Endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}    
    &{notification_response}=  Create Mock Response	headers="Content-Type: application/json"  status_code=204
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Wait Until Keyword Succeeds    ${total_polling_time}   ${polling_interval}   Verify Mock Expectation    ${notification_request}
    Clear Requests  ${callback_endpoint}
    
Check HTTP Response Body Subscription Identifier matches the requested Subscription
    Log    Trying to check response ID
    Should Be Equal As Strings   ${response['body']['id']}    ${subscriptionId} 
    Log    Subscription identifier as expected
    
Check HTTP Response Status Code Is
    [Arguments]    ${expected_status}    
    Should Be Equal As Strings    ${response['status']}    ${expected_status}
    Log    Status code validated 
    
Check HTTP Response Header Contains Link
    ${linkURL}=    Get Value From Json    ${response['headers']}    $..Link
    Should Not Be Empty    ${linkURL}
    
Check HTTP Response Body Json Schema Is
    [Arguments]    ${input}
    Run Keyword If    '${input}' == 'ProblemDetails'    Should Contain    ${response['headers']['Content-Type']}    application/problem+json
    ...    ELSE    Should Contain    ${response['headers']['Content-Type']}    application/json
    ${schema} =    Catenate    SEPARATOR=    ${input}    .schema.json
    Validate Json    ${schema}    ${response['body']}
    Log    Json Schema Validation OK  

Check HTTP Response Body Is Empty
    Should Be Empty    ${response['body']}    
    Log    No json schema is provided. Validation OK  
    
Get all Performance Subscriptions
    Set headers    {"Accept": "application/json"}
    Set headers    {"Content-Type": "application/json"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
 Get Performance Subscriptions with attribute-based filters
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${filter_ok}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Performance Subscriptions with invalid attribute-based filters
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${filter_ko}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Performance Subscriptions with invalid resource endpoint
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscription
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Post Request for Performance Subscription
    Set headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}     callback_uri=${callback_uri}    callback_endpoint=${callback_endpoint}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Run Keyword If    ${NFVMANO_CHECKS_NOTIF_ENDPOINT} == 1
    ...       Check Notification Endpoint  

Check HTTP Response Body PmSubscription Attributes Values Match the Issued Subscription
    Log    Check Response matches subscription
    ${template}=    Get File    jsons/subscriptions.json
     ${body}=        Format String   ${template}     callback_uri=${callback_uri}    callback_endpoint=${callback_endpoint}
    ${subscription}=    evaluate    json.loads('''${body}''')    json
    Should Be Equal As Strings    ${response['body']['callbackUri']}    ${subscription['callbackUri']}
    
Check Postcondition Performance Subscription Is Set
    Log    Check Postcondition subscription exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": "${AUTHORIZATION}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${response['body']['id']}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Check HTTP Response Status Code Is    200
    
Send Post Request for Duplicated Performance Subscription
    Set headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Set headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    ${template}=    Get File    jsons/subscriptions.json
     ${body}=        Format String   ${template}     callback_uri=${callback_uri}    callback_endpoint=${callback_endpoint}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Check HTTP Response Header Contains
    [Arguments]    ${CONTENT_TYPE}
    Should Contain    ${response['headers']}    ${CONTENT_TYPE}
    Log    Header is present
    
Check Postcondition Subscription Resource Returned in Location Header Is Available
    Log    Going to check postcondition
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": "${AUTHORIZATION}"}
    GET    ${response['headers']['Location']}
    Integer    response status    200
    Log    Received a 200 OK as expected
    ${contentType}=    Output    response headers Content-Type
    Should Contain    ${contentType}    application/json
    ${result}=    Output    response body
    Validate Json    PmSubscription.schema.json    ${result}
    Log    Validated PmSubscription schema
    
Send Put Request for Performance Subscriptions
    [Documentation]    This method is not supported. When this method is requested on this resource, the NFV-MANO shall return a "405 Method
    ...    Not Allowed" response as defined in clause 4.3.5.4.
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Patch Request for Performance Subscriptions
    [Documentation]    This method is not supported. When this method is requested on this resource, the NFV-MANO shall return a "405 Method
    ...    Not Allowed" response as defined in clause 4.3.5.4.
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Delete Request for Performance Subscriptions
    [Documentation]    This method is not supported. When this method is requested on this resource, the NFV-MANO shall return a "405 Method
    ...    Not Allowed" response as defined in clause 4.3.5.4.
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"Authorization": ${AUTHORIZATION}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}   