*** Settings ***
Library           String
Library           JSONSchemaLibrary    schemas/
Resource          environment/variables.txt    # Generic Parameters
Library           JSONLibrary
Library           REST    ${VNFM_SCHEMA}://${VNFM_HOST}:${VNFM_PORT}    ssl_verify=false
Library           OperatingSystem
Library           Process
Library           MockServerLibrary

Suite Setup       Create Sessions
Suite Teardown    Terminate All Processes    kill=true

*** Test Cases ***
GET All Performance Thresholds
    [Documentation]    Test ID: 7.3.4.4.1
    ...    Test title: GET All Performance Thresholds
    ...    Test objective: The objective is to test the retrieval of all the available VNF performance thresholds and perform a JSON schema validation of the collected thresholds data structure
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNFM.
    ...    Reference: Clause 6.4.5.3.2 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    GET all Performance Thresholds
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is   Thresholds

GET Performance Thresholds with attribute-based filter
    [Documentation]    Test ID: 7.3.4.4.2
    ...    Test title: GET Performance Thresholds with attribute-based filter
    ...    Test objective: The objective is to test the retrieval of all the available VNF performance thresholds when using attribute-based filters, perform a JSON schema validation of the collected thresholds data structure, and verify that the retrieved information matches the issued attribute-based filter
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNFM.
    ...    Reference: Clause 6.4.5.3.2 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    GET Performance Thresholds with attribute-based filter
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is   Thresholds
    Check HTTP Response Body Thresholds match the requested attribute-based filter

GET Performance Thresholds with invalid attribute-based filter
    [Documentation]    Test ID: 7.3.4.4.3
    ...    Test title: GET Performance Thresholds with invalid attribute-based filter
    ...    Test objective: The objective is to test that the retrieval of VNF performance thresholds fails when using invalid attribute-based filter, and perform the JSON schema validation of the failed operation HTTP response
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNFM.
    ...    Reference: Clause 6.4.5.3.2 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    GET Performance Thresholds with invalid attribute-based filter
    Check HTTP Response Status Code Is    404

GET Performance Thresholds with invalid resource endpoint
    [Documentation]    Test ID: 7.3.4.4.4
    ...    Test title: GET Performance Thresholds with invalid resource endpoint
    ...    Test objective: The objective is to test that the retrieval of VNF performance thresholds fails when using invalid resource endpoint, and perform the JSON schema validation of the failed operation HTTP response
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNFM.
    ...    Reference: Clause 6.4.5.3.2 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    GET VNF Performance Thresholds with invalid resource endpoint
    Check HTTP Response Status Code Is    400
    Check HTTP Response Body Json Schema Is   ProblemDetails

Create new Performance Threshold
    [Documentation]    Test ID: 7.3.4.4.5
    ...    Test title:  Create new Performance Threshold
    ...    Test objective: The objective is to test the creation of a new VNF performance threshold and perform the JSON schema validation of the returned threshold data structure
    ...    Pre-conditions: A VNF instance is instantiated.
    ...    Reference: Clause 6.4.5.3.1 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: The VNF Performance Threshold is successfully created on the VNFM
    Send Post Request Create new Performance Threshold
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is   Threshold
    Check HTTP Response Header Contains    Location
    Check Postcondition Threshold Exists

PUT Performance Thresholds - Method not implemented
    [Documentation]    Test ID: 7.3.4.4.6
    ...    Test title: PUT Performance Thresholds - Method not implemented
    ...    Test objective: The objective is to test that PUT method is not allowed to modify VNF Performance Thresholds
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNF.
    ...    Reference: Clause 6.4.5.3.3 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    Send PUT Request for all Performance Thresholds
    Check HTTP Response Status Code Is    405

PATCH Performance Thresholds - Method not implemented
    [Documentation]    Test ID: 7.3.4.4.7
    ...    Test title: PATCH Performance Thresholds - Method not implemented
    ...    Test objective: The objective is to test that PATCH method is not allowed to modify VNF Performance Thresholds
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNFM.
    ...    Reference: Clause 6.4.5.3.4 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    Send PATCH Request for all Performance Thresholds
    Check HTTP Response Status Code Is    405

DELETE Performance Thresholds - Method not implemented
    [Documentation]    Test ID: 7.3.4.4.8
    ...    Test title: DELETE Performance Thresholds - Method not implemented
    ...    Test objective: The objective is to test that DELETE method is not allowed to update VNF Performance Thresholds
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNFM.
    ...    Reference: Clause 6.4.5.3.5 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: The VNF performance thresholds are not deleted by the failed operation
    Send DELETE Request for all Performance Thresholds
    Check HTTP Response Status Code Is    405
    Check Postcondition Thresholds Exist
    
GET All Performance Thresholds as Paged Response
    [Documentation]    Test ID: 7.3.4.4.9
    ...    Test title: GET All Performance Thresholds as Paged Response
    ...    Test objective: The objective is to test the retrieval of all the available VNF performance thresholds as a Paged Response.
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNFM.
    ...    Reference: Clause 6.4.5.3.2 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    GET all Performance Thresholds
    Check HTTP Response Status Code Is    200
    Check LINK in Header
    
GET Performance Thresholds - Bad Request Response too Big
    [Documentation]    Test ID: 7.3.4.4.10
    ...    Test title: GET Performance Thresholds - Bad Request Response too Big
    ...    Test objective: The objective is to test that the retrieval of VNF performance thresholds fails because response is too big, and perform the JSON schema validation of the failed operation HTTP response
    ...    Pre-conditions: A VNF instance is instantiated. One or more VNF performance thresholds are set in the VNFM.
    ...    Reference: Clause 6.4.5.3.2 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    GET all Performance Thresholds
    Check HTTP Response Status Code Is    400
    Check HTTP Response Body Json Schema Is   ProblemDetails
    
POST create new Performance Threshold - Unprocessable Entity
    [Documentation]    Test ID: 7.3.4.4.11
    ...    Test title:  POST create new Performance Threshold - Unprocessable Entity
    ...    Test objective: The objective is to test that creation of new VNF performance threshold fails when a request contains syntactically correct data but the data cannot be processed.
    ...    Pre-conditions: A VNF instance is instantiated.
    ...    Reference: Clause 6.4.5.3.1 - ETSI GS NFV-SOL 003 [1] v3.5.1
    ...    Config ID: Config_prod_VNFM
    ...    Applicability: none
    ...    Post-Conditions: none
    Send Post Request Create new Performance Threshold with unprocessable entity
    Check HTTP Response Status Code Is    422
    Check HTTP Response Body Json Schema Is   ProblemDetails
    
*** Keywords ***
Create Sessions
    Start Process  java  -jar  ${MOCK_SERVER_JAR}    -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}
    
GET all Performance Thresholds
    Log    Trying to get all thresholds present in the VNFM    
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Performance Thresholds with attribute-based filter
    Log    Trying to get thresholds present in the VNFM with filter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds?${FILTER_OK}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Performance Thresholds with invalid attribute-based filter
    Log    Trying to get thresholds present in the VNFM with invalid filter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds?${FILTER_KO}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET VNF Performance Thresholds with invalid resource endpoint
    Log    Trying to get thresholds present in the VNFM with invalid resource endpoint
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/threshold
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Post Request Create new Performance Threshold
    Log    Creating a new THreshold
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${template}=    Get File    jsons/CreateThresholdRequest.json
    ${request}=        Format String   ${template}     objectInstanceIds=${objectInstanceIds}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds    ${request}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Post Request Create new Performance Threshold with unprocessable entity
    Log    trying to create a new Threshold
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${template}=    Get File    jsons/CreateThresholdRequest.json
    ${request}=        Format String   ${template}     objectInstanceIds=${objectInstanceIds}    callback_uri=${callback_uri}    callback_port=${callback_port}    callback_uri=${callback_endpoint} 
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds    ${request}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Verify Mock Expectation    ${notification_request}
	Clear Requests    ${callback_endpoint}

Send PUT Request for all Performance Thresholds
    Log    PUT THresholds
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send PATCH Request for all Performance Thresholds
    Log    PUT THresholds
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for all Performance Thresholds
    Log    DELETE THresholds
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check Postcondition Thresholds Exist
    Log    Checking that Thresholds still exists
    GET all Performance Thresholds
    
Check Postcondition Threshold Exists
    Log    Checking that Threshold exists
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/thresholds/${response['body']['id']}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    Threshold
        
Check HTTP Response Body Thresholds match the requested attribute-based filter
    Log    Checking that attribute-based filter is matched
    #todo
    
Check HTTP Response Status Code Is
    [Arguments]    ${expected_status}
    ${status}=    Convert To Integer    ${expected_status}    
    Should Be Equal As Strings   ${response['status']}    ${status} 
    Log    Status code validated

Check HTTP Response Header Contains
    [Arguments]    ${CONTENT_TYPE}
    Should Contain    ${response['headers']}    ${CONTENT_TYPE}
    Log    Header is present
    
Check HTTP Response Body Json Schema Is
    [Arguments]    ${input}
    Run Keyword If    '${input}' == 'ProblemDetails'    Should Contain    ${response['headers']['Content-Type']}    application/problem+json
    ...    ELSE    Should Contain    ${response['headers']['Content-Type']}    application/json
    ${schema} =    Catenate    SEPARATOR=    ${input}    .schema.json
    Validate Json    ${schema}    ${response['body']}
    Log    Json Schema Validation OK

Check LINK in Header
    ${linkURL}=    Get Value From Json    ${response['headers']}    $..Link
    Should Not Be Empty    ${linkURL}

