*** Settings ***
Library           JSONSchemaLibrary    schemas/
Resource          environment/variables.txt    # Generic Parameters
Library           OperatingSystem
Library           JSONLibrary
Library           Process
Library           MockServerLibrary
Library           String 
Library           REST    ${EM-VNF_SCHEMA}://${EM-VNF_HOST}:${EM-VNF_PORT}    ssl_verify=false
Suite Setup       Create Sessions
Suite Teardown    Terminate All Processes    kill=true

*** Test Cases ***
GET VNF Indicators Subscriptions
    [Documentation]    Test ID: 6.3.2.4.1
    ...    Test title: GET VNF Indicators Subscriptions
    ...    Test objective: The objective is to test the retrieval of all VNF indicators subscriptions and perform a JSON schema validation of the returned subscriptions data structure
    ...    Pre-conditions: A VNF instance is instantiated. At least one VNF indicator subscription is available in the VNF.
    ...    Reference: Clause 8.4.5.3.2 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators
    ...    Post-Conditions: none
    Get VNF Indicators Subscriptions
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is   VnfIndicatorSubscriptions

GET VNF Indicators Subscriptions with attribute-based filter
    [Documentation]    Test ID: 6.3.2.4.2
    ...    Test title: GET VNF Indicators Subscriptions with attribute-based filter
    ...    Test objective: The objective is to test the retrieval of all VNF indicators subscriptions using attribute-based filter and perform a JSON schema and content validation of the returned subscriptions data structure
    ...    Pre-conditions: A VNF instance is instantiated. At least one VNF indicator subscription is available in the VNF.
    ...    Reference: Clause 8.4.5.3.2 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators
    ...    Post-Conditions: none
    Get VNF Indicators Subscriptions with filter
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is   VnfIndicatorSubscriptions
    Check HTTP Response Body Subscriptions Match the requested Attribute-Based Filter

GET VNF Indicators Subscriptions with invalid attribute-based filter
    [Documentation]    Test ID: 6.3.2.4.3
    ...    Test title: GET VNF Indicators Subscriptions with invalid attribute-based filter
    ...    Test objective: The objective is to test that the retrieval of all VNF indicators subscriptions fails when using invalid attribute-based filter. The test also checks the JSON schema of the unsuccessful operation HTTP response.
    ...    Pre-conditions: A VNF instance is instantiated. At least one VNF indicator subscription is available in the VNF.
    ...    Reference: Clause 8.4.5.3.2 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators
    ...    Post-Conditions: none
    Get VNF Indicators Subscriptions with invalid filter
    Check HTTP Response Status Code Is    400
    Check HTTP Response Body Json Schema Is   ProblemDetails    

GET VNF Indicators Subscriptions with invalid resource endpoint
    [Documentation]    Test ID: 6.3.2.4.4
    ...    Test title: GET VNF Indicators Subscriptions with invalid resource endpoint
    ...    Test objective: The objective is to test that the retrieval of all VNF indicators subscriptions fails when using invalid resource endpoint.
    ...    Pre-conditions: A VNF instance is instantiated. At least one VNF indicator subscription is available in the VNF.
    ...    Reference: Clause 8.4.5.3.2 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators.
    ...    Post-Conditions: none
    Get VNF Indicators Subscriptions with invalid resource endpoint
    Check HTTP Response Status Code Is    404
    Check HTTP Response Body Json Schema Is   ProblemDetails

Create new VNF indicator subscription
    [Documentation]    Test ID: 6.3.2.4.5
    ...    Test title: Create new VNF indicator subscription
    ...    Test objective: The objective is to test the creation of a new VNF indicator subscription perform a JSON schema and content validation of the returned subscriptions data structure
    ...    Pre-conditions: A VNF instance is instantiated.
    ...    Reference: Clause 8.4.5.3.1 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators.
    ...    Post-Conditions: The VNF indicator subscription is successfully set and it matches the issued subscription
    Send Post Request for VNF Indicator Subscription
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is    VnfIndicatorSubscription
    Check HTTP Response Body Matches the Subscription
    Check Postcondition VNF Indicator Subscription Is Set

PUT VNF Indicator Subscriptions - Method not implemented
    [Documentation]    Test ID: 6.3.2.4.6
    ...    Test title: PUT VNF Indicator Subscriptions - Method not implemented
    ...    Test objective: The objective is to test that PUT method is not allowed to modify VNF indicator subscriptions
    ...    Pre-conditions: none
    ...    Reference: Clause 8.4.5.3.3 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators.
    ...    Post-Conditions: none
    Send Put Request for VNF Indicator Subscriptions
    Check HTTP Response Status Code Is    405

PATCH VNF Indicator Subscriptions - Method not implemented
    [Documentation]    Test ID: 6.3.2.4.7
    ...    Test title: PATCH VNF Indicator Subscriptions - Method not implemented
    ...    Test objective: The objective is to test that PATCH method is not allowed to update VNF indicator subscriptions
    ...    Pre-conditions: none
    ...    Reference: Clause 8.4.5.3.4 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators.
    ...    Post-Conditions: none
    Send Patch Request for VNF Indicator Subscriptions
    Check HTTP Response Status Code Is    405

DELETE VNF Indicator Subscriptions - Method not implemented
    [Documentation]    Test ID: 6.3.2.4.8
    ...    Test title: DELETE VNF Indicator Subscriptions - Method not implemented
    ...    Test objective: The objective is to test that DELETE method is not allowed to delete VNF indicator subscriptions
    ...    Pre-conditions: none
    ...    Reference: Clause 8.4.5.3.5 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators.
    ...    Post-Conditions: none
    Send Delete Request for VNF Indicator Subscriptions
    Check HTTP Response Status Code Is    405
    
GET VNF Indicators Subscriptions to get Paged Response
    [Documentation]    Test ID: 6.3.2.4.9
    ...    Test title: GET VNF Indicators Subscriptions to get Paged Response
    ...    Test objective: The objective is to test the retrieval of all VNF indicators subscriptions to get Paged Response.
    ...    Pre-conditions: A VNF instance is instantiated. At least one VNF indicator subscription is available in the VNF.
    ...    Reference: Clause 8.4.5.3.2 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators
    ...    Post-Conditions: none
    Get VNF Indicators Subscriptions
    Check HTTP Response Status Code Is    200
    Check HTTP Response Header Contain Link
 
GET VNF Indicators Subscriptions - Bad Request Response too Big
    [Documentation]    Test ID: 6.3.2.4.10
    ...    Test title: GET VNF Indicators Subscriptions - Bad Request Response too Big
    ...    Test objective: The objective is to test that the retrieval of all VNF indicators subscriptions fails because response is too big. The test also checks the JSON schema of the unsuccessful operation HTTP response.
    ...    Pre-conditions: A VNF instance is instantiated. At least one VNF indicator subscription is available in the VNF.
    ...    Reference: Clause 8.4.5.3.2 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: The VNF supports the generation and maintenance of indicators
    ...    Post-Conditions: none
    Get VNF Indicators Subscriptions
    Check HTTP Response Status Code Is    400
    Check HTTP Response Body Json Schema Is   ProblemDetails   
    
POST Create a new Subscription - Unprocessable Entity
    [Documentation]    Test ID: 6.3.2.4.11
    ...    Test title: POST Create a new Subscription - Unprocessable Entity
    ...    Test objective: The objective is to test that content type of the payload body is supported and the payload body of a request contains syntactically correct data but the data cannot be processed.
    ...    Pre-conditions: none
    ...    Reference: Clause 8.4.5.3.1 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: none
    ...    Post-Conditions: none
    Send Post Request for VNF Indicator Subscription with unprocessable entity
    Check HTTP Response Status Code Is    422
    Check HTTP Response Body Json Schema Is  ProblemDetails
 
Create new VNF indicator Subscription - DUPLICATION
    [Documentation]    Test ID: 6.3.2.4.12
    ...    Test title: POST Create a new subscription - DUPLICATION
    ...    Test objective: The POST method creates a new subscription even if an existing subscription to same content exist
    ...    Pre-conditions: none
    ...    Reference: Clause 8.4.5.3.1 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: SUT should support duplication of subscription creation
    ...    Post-Conditions: In response header Location shall not be null
    Post Create subscription - DUPLICATION
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is    VnfIndicatorSubscription
    Check Operation Occurrence Id

Create new VNF indicator Subscription - NO-DUPLICATION
    [Documentation]    Test ID: 6.3.2.4.13
    ...    Test title: POST Create a new subscription - NO-DUPLICATION
    ...    Test objective: The POST method creates a new subscription even if an existing subscription to same content exist
    ...    Pre-conditions: none
    ...    Reference: Clause 8.4.5.3.1 - ETSI GS NFV-SOL 002 [2] v2.8.1
    ...    Config ID: Config_prod_VE
    ...    Applicability: SUT should not support duplication of subscription creation
    ...    Post-Conditions: In response header Location shall not be null
    Post Create subscription - NO-DUPLICATION
    Check HTTP Response Status Code Is    303
    Check Operation Occurrence Id
 
 *** Keywords ***
Get VNF Indicators Subscriptions
    Log    Trying to get the list of subscriptions
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get VNF Indicators Subscriptions with filter
    Log    Trying to get the list of subscriptions using filters
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${POS_FILTER}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get VNF Indicators Subscriptions with invalid filter   
    Log    Trying to get the list of subscriptions using filters with wrong attribute name
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${NEG_FILTER}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}   
    
Get VNF Indicators Subscriptions with invalid resource endpoint
    Log    Trying to perform a request on a Uri which doesn't exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscription
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}   

Send Post Request for VNF Indicator Subscription
    Log    Trying to create a new subscription
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String    ${template}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    
	${output}=    Output    response
	Set Global Variable    ${response}    ${output}
	Clear Requests    ${callback_endpoint}
    #Run Keyword If    ${NFVO_CHECKS_NOTIF_ENDPOINT} == 1    #removed as the check of notif endpoint is done by NFVO while processing subscription reques
    #...    Check Notification Endpoint

Send Post Request for VNF Indicator Subscription with unprocessable entity
    Log    Trying to create a new subscription
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String    ${template}     callback_uri=localhost    #not reachable endpoint
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    
	${output}=    Output    response
	Set Global Variable    ${response}    ${output}

Send Put Request for VNF Indicator Subscriptions
    Log    Trying to create a new subscription
    ${body}=    Get File    jsons/subscriptions.json
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Patch Request for VNF Indicator Subscriptions
    Log    Trying to create a new subscription
    ${body}=    Get File    jsons/subscriptions.json
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete Request for VNF Indicator Subscriptions
    Log    Trying to create a new subscription
    ${body}=    Get File    jsons/subscriptions.json
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Status Code Is
    [Arguments]    ${expected_status}    
    ${status}=    Convert To Integer    ${expected_status}    
    Should Be Equal    ${response['status']}    ${status}
    Log    Status code validated

Check HTTP Response Header Contains
    [Arguments]    ${CONTENT_TYPE}
    Log    ${response['headers']}
    Should Contain    ${response['headers']}    ${CONTENT_TYPE}
    Log    Header is present
    
Check HTTP Response Body Json Schema Is
    [Arguments]    ${input}
    Run Keyword If    '${input}' == 'ProblemDetails'    Should Contain    ${response['headers']['Content-Type']}    application/problem+json
    ...    ELSE    Should Contain    ${response['headers']['Content-Type']}    application/json
    ${schema} =    Catenate    SEPARATOR=    ${input}	.schema.json
    Validate Json    ${schema}    ${response['body']}
    Log    Json Schema Validation OK

Check HTTP Response Body Matches the Subscription
    Log    Check Response matches subscription
    ${body}=    Get File    jsons/subscriptions.json
    ${subscription}=    evaluate    json.loads('''${body}''')    json
    Should Be Equal    ${response['body']['callbackUri']}    ${subscription['callbackUri']}

Check Postcondition VNF Indicator Subscription Is Set
    Log    Check Postcondition subscription exist
    Log    Trying to get the subscription
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${response['body']['id']}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Check HTTP Response Status Code Is    200
    
Check HTTP Response Body Subscriptions Match the requested Attribute-Based Filter
    Log    Check Response includes VNF Indicators according to filter
    @{words} =  Split String    ${POS_FILTER}       ${SEPARATOR} 
    Should Be Equal As Strings    ${response['body'][0]['callbackUri']}    @{words}[1]
    
Create Sessions 
    Start Process  java  -jar  ${MOCK_SERVER_JAR}    -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}
    
Check Notification Endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}    
    &{notification_response}=  Create Mock Response	headers="Content-Type: application/json"  status_code=204
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Wait Until Keyword Succeeds    ${total_polling_time}   ${polling_interval}   Verify Mock Expectation    ${notification_request}
    Clear Requests  ${callback_endpoint}
    
Check HTTP Response Header Contain Link
    ${linkURL}=    Get Value From Json    ${response['headers']}    $..Link
    Should Not Be Empty    ${linkURL}
    
Post Create subscription - DUPLICATION
    Log    Trying to create a subscription with an already created content
    Pass Execution If    ${VNFM_CHECKS_NOTIF_ENDPOINT} == 0    VNFM is not permitting duplication. Skipping the test
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/subscriptions.json
    ${body}=        Format String    ${body}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	Clear Requests    ${callback_endpoint}
    #Run Keyword If    ${NFVO_CHECKS_NOTIF_ENDPOINT} == 1    #removed as the check of notif endpoint is done by NFVO while processing subscription reques
    #...    Check Notification Endpoint 
		
Post Create subscription - NO-DUPLICATION	
    Log    Trying to create a subscription with an already created content
    Pass Execution If    ${VNFM_CHECKS_NOTIF_ENDPOINT} == 1    VNFM permits duplication. Skipping the test
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    ${body}=    Get File    jsons/subscriptions.json
    ${body}=        Format String    ${body}     callback_uri=${callback_uri}:${callback_port}    callback_endpoint=${callback_endpoint}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}
	Clear Requests    ${callback_endpoint}
    #Run Keyword If    ${NFVO_CHECKS_NOTIF_ENDPOINT} == 1    #removed as the check of notif endpoint is done by NFVO while processing subscription reques
    #...    Check Notification Endpoint 
	
Check Operation Occurrence Id
    ${vnfLcmOpOccId}=    Get Value From Json    ${response['headers']}    $..Location
    Should Not Be Empty    ${vnfLcmOpOccId}
