*** Settings ***
Resource    environment/variables.txt
Library    REST    ${NFVO_SCHEMA}://${NFVO_HOST}:${NFVO_PORT}    ssl_verify=false
Library    MockServerLibrary 
Library    OperatingSystem
Library    BuiltIn
Library    JSONLibrary
Library    Collections
Library    JSONSchemaLibrary    schemas/
Library    Process
Library    String
#Variables    descriptors/SOL001/VNFD/vnfd_SOL001.yaml
#Variables    descriptors/SOL006/VNFD/vnfd_SOL006.yaml

*** Keywords ***
Get all VNF Packages
    Log    Trying to get all VNF Packages present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}   

Get all VNF Packages with nextpage_opaque_marker parameter
    Log    Get next page of packages
    Set Headers  {"Accept":"${ACCEPT_JSON}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query using the URI provided in the previous GET request and validate response
    Get    ${nextPageUri}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	

Check HTTP Response Body Does Not Contain softwareImages
    Log    Checking that field element is missing
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Should Be Empty    ${softwareImages}
    Log    Element is empty as expected
    
Check HTTP Response Body Does Not Contain additionalArtifacts
    Log    Checking that field element is missing
    ${additionalArtifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Should Be Empty    ${additionalArtifacts}
    Log    Element is empty as expected
    
Check HTTP Response Body Does Not Contain userDefinedData 
    Log    Checking that field element is missing
    ${userDefinedData}=    Get Value From Json    ${response['body']}    $..userDefinedData
    Should Be Empty    ${userDefinedData}
    Log    Element is empty as expected

GET VNF Packages with attribute-based filter
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using filter params
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?${POS_FILTER}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body VnfPkgsInfo Matches the requested Attribute-Based Filter
    Log    Checking that attribute-based filter is matched
    @{attr} =    Split String    ${POS_FILTER}       ,${VAR_SEPERATOR} 
    @{var_id} =    Split String    @{attr}[0]       ,${SEPERATOR}
    @{var_provider} =    Split String    @{attr}[1]       ,${SEPERATOR}
    Should Be True     "${response['body'][0]['vnfdId']}"=="@{var_id}[1]" and "${response['body'][0]['vnfProvider']}"=="@{var_provider}[1]"

GET VNF Packages with invalid attribute-based filter
    Log    Trying to perform a negative get, filtering by the inexistent filter 'nfvId'
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?${NEG_FILTER}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET VNF Packages with all_fields attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using filter params
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?all_fields
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
   
Check HTTP Response Body VnfPkgsInfo Matches the requested all_fields selector
    Log    Trying to validate softwareImages schema
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Validate Json    softwareImage.schema.json    ${softwareImages[0]}
    Log    Validation for softwareImage schema OK
    Log    Trying to validate additionalArtifacts schema
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Validate Json    additionalArtifacts.schema.json    ${additional_artifacts[0]}
    Log    Validation for additionalArtifacts schema OK
    ${links}=    Get Value From Json    ${response['body']}    $.._links
    Validate Json    links.schema.json    ${links[0]}
    Log    Validation for _links schema OK

GET VNF Packages with exclude_default attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using exclude_default filter.
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?exclude_default   
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body VnfPkgsInfo Matches the requested exclude_default selector
    Log    Checking missing information for softwareImages element
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Should Be Empty    ${softwareImages}
    Log    softwareImages element is missing as excepted
    Log    Checking missing information for additionalArtifact element
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Should Be Empty    ${additional_artifacts}
    Log    additionalArtifact element is missing as excepted

GET VNF Packages with fields attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using fields
    Pass Execution If    ${NFVO_FIELDS} == 0    The NFVO is not able to use fields parameter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body vnfPkgsInfo Matches the requested fields selector
    Log    Trying to validate softwareImages schema
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Validate Json    softwareImage.schema.json    ${softwareImages[0]}
    Log    Validation for softwareImage schema OK
    Log    Trying to validate additionalArtifacts schema
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Validate Json    additionalArtifacts.schema.json    ${additional_artifacts[0]}
    Log    Validation for additionalArtifacts schema OK
    
GET VNF Packages with fields and exclude_default attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using fields
    Pass Execution If    ${NFVO_FIELDS} == 0    The NFVO is not able to use fields parameter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?fields=${fields}&exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body vnfPkgsInfo Matches the requested fields and exclude_default selector
    Log    Trying to validate softwareImages schema
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Validate Json    softwareImage.schema.json    ${softwareImages[0]}
    Log    Validation for softwareImage schema OK
    Log    Trying to validate additionalArtifacts schema
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Validate Json    additionalArtifacts.schema.json    ${additional_artifacts[0]}
    Log    Validation for additionalArtifacts schema OK
    
GET VNF Packages with exclude_fields attribute selector
    Log    Trying to get all VNF Packages present in the NFVO Catalogue, using fields
    Pass Execution If    ${NFVO_FIELDS} == 0    The NFVO is not able to use fields parameter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages?exclude_fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Body vnfPkgsInfo Matches the requested exclude_fields selector
    Log    Checking missing information for softwareImages element
    ${softwareImages}=    Get Value From Json    ${response['body']}    $..softwareImages
    Should Be Empty    ${softwareImages}
    Log    softwareImages element is missing as excepted
    Log    Checking missing information for additionalArtifact element
    ${additional_artifacts}=    Get Value From Json    ${response['body']}    $..additionalArtifacts
    Should Be Empty    ${additional_artifacts}
    Log    additionalArtifact element is missing as excepted

GET all VNF Packages with invalid resource endpoint
    Log    Trying to perform a GET on a erroneous URI
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_package
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request for all VNF Packages
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for all VNF Packages
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PATCH Request for all VNF Packages
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for all VNF Packages
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package
    Log    Trying to get a VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check HTTP Response Body vnfPkgInfo Identifier matches the requested VNF Package
    Log    Going to validate pacakge info retrieved
    Should Be Equal    ${response['body']['id']}    ${vnfPackageId} 
    Log    Pacakge identifier as expected

GET Individual VNF Package with invalid resource identifier
    Log    Trying to perform a negative get, using wrong authorization bearer
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request for individual VNF Package
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual VNF Package
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for individual VNF Package
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual VNF Package
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual VNF Package Content  
    Log    Trying to get a VNF Package Content
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Header Content-Type Is
    [Arguments]   ${header}
    Should Contain    ${response['headers']['Content-Type']}    ${header}

GET Individual VNF Package Content with Range Request
    Log    Trying to get a VNF Package Content using RANGE using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual VNF Package Content with explicit Range Request
    Log    Trying to get a VNF Package Content using RANGE using an NFVO that can handle it
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}


Check HTTP Response Header Content-Range Is Present and Matches the requested range
    Log    Check Content-Range HTTP Header
    Should Contain    ${response['headers']}    Content-Range
    Should Be Equal As Strings    ${response['headers']['Content-Range']}    ${range}/${full_size}
    Log    Header Content-Range is present
    
Check HTTP Response Header Content-Length Is Present and Matches the requested range length
    Log    Check Content-Length HTTP Header
    Should Contain    ${response['headers']}    Content-Length
    Should Be Equal As Integers    ${response['headers']['Content-Length']}    ${length}
    Log    Header Content-Length is present

GET Individual VNF Package Content with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package Content with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPkgId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Content for VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a VNF Package content present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request for individual VNF Package Content
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual VNF Package Content
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for individual VNF Package Content
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual VNF Package Content
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Get all VNF Package Subscriptions
    Log    Trying to get the list of subscriptions
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET All VNF Package Subscriptions with nextpage_opaque_marker parameter
    Log    Get next page of subscriptions
    Set Headers  {"Accept":"${ACCEPT_JSON}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query using the URI provided in the previous GET request and validate response
    Get    ${nextPageUri}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	

Get VNF Package Subscriptions with attribute-based filters
    Log    Trying to get the list of subscriptions using filters
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${filter_ok}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}    

Get VNF Package Subscriptions with invalid attribute-based filters
    Log    Trying to get the list of subscriptions using filters with wrong attribute name
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions?${filter_ko}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Get VNF Package Subscriptions with invalid resource endpoint
    Log    Trying to perform a request on a Uri which doesn't exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscription
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send Post Request for VNF Package Subscription
    Log    Trying to create a new subscription
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}    callback_uri=${callback_uri}    callback_port=${callback_port}    callback_uri=${callback_endpoint} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${output}=    Output    response
	Set Global Variable    ${response}    ${output}
	Clear Requests    ${callback_endpoint}
    #Run Keyword If    ${NFVO_CHECKS_NOTIF_ENDPOINT} == 1    #removed as the check of notif endpoint is done by NFVO while processing subscription reques
    #...    Check Notification Endpoint 

Send Post Request for Duplicated VNF Package Subscription
    Log    Trying to create a subscription with an already created content
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Set Headers    {"Content-Type": "${CONTENT_TYPE_JSON}"}
    ${template}=    Get File    jsons/subscriptions.json
    ${body}=        Format String   ${template}    callback_uri=${callback_uri}    callback_port=${callback_port}    callback_uri=${callback_endpoint} 
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log  Creating mock request and response to handle GET operation on notification endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}
    &{notification_response}=  Create Mock Response	status_code=204
    Log    Issue the subscription request
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Post    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions    ${body}    allow_redirects=false
	${output}=    Output    response
	Set Global Variable    ${response}    ${output}
	Clear Requests    ${callback_endpoint}
    #Run Keyword If    ${NFVO_CHECKS_NOTIF_ENDPOINT} == 1    #removed as the check of notif endpoint is done by NFVO while processing subscription reques
    #...    Check Notification Endpoint 

Send Put Request for VNF Package Subscriptions
    Log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send Patch Request for VNF Package Subscriptions
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send Delete Request for VNF Package Subscriptions
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}    

Get single file VNFD in Individual VNF Package in Plain Format
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgPlainVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Get VNFD in Individual VNF Package in Zip Format
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Get single file VNFD in Individual VNF Package in Plain or Zip Format
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgPlainVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get multi file VNFD in Individual VNF Package in Plain or Zip Format
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Check HTTP Response Header Content-Type Is Any of
    [Arguments]   ${header1}    ${header2}
    Should Contain Any  ${response['headers']['Content-Type']}    ${header1}    ${header2}

Get multi file VNFD in Individual VNF Package in Plain Format
    Log    Trying to get a negative case performing a get on a VNFD from a given VNF Package present in the NFVO Catalogue. Accept will be text/plain but VNFD is composed my multiple files.
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Get VNFD in Individual VNF Package with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPkgId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Get VNFD in Individual VNF Package Content with conflict due to onboarding state 
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send POST Request for VNFD in individual VNF Package
    Log    Trying to perform a POST. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send PUT Request for VNFD in individual VNF Package
    Log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Send PATCH Request for VNFD in individual VNF Package
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send DELETE Request for VNFD in individual VNF Package
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

GET Individual VNF Package Artifact
    Log    Trying to get a VNF Package Artifact
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package Artifact in octet stream format  
    Log    Trying to get a VNF Package Artifact
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageOctetStreamId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual VNF Package Artifact with Range Request
    Log    Trying to get an Artifact using RANGE Header and using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual VNF Package Artifact with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
GET Individual VNF Package Artifact with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${erroneousVnfPkgId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Artifact for VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a VNF Package artifact present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request for individual VNF Package Artifact
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual VNF Package Artifact
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for individual VNF Package Artifact
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual VNF Package Artifact
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check HTTP Response Status Code Is
    [Arguments]    ${expected_status}    
    Should Be Equal As Strings   ${response['status']}    ${expected_status}
    Log    Status code validated 
    

Check HTTP Response Body Json Schema Is
    [Arguments]    ${input}
    Run Keyword If    '${input}' == 'ProblemDetails'    Should Contain    ${response['headers']['Content-Type']}    application/problem+json
    ...    ELSE    Should Contain    ${response['headers']['Content-Type']}    application/json
    ${schema} =    Catenate    SEPARATOR=    ${input}    .schema.json
    Validate Json    ${schema}    ${response['body']}
    Log    Json Schema Validation OK  


Check HTTP Response Body Is Empty
    Should Be Empty    ${response['body']}    
    Log    No json schema is provided. Validation OK  


Check HTTP Response Body Subscriptions Match the requested Attribute-Based Filter
    Log    Check Response includes VNF Package Management according to filter
    @{words} =  Split String    ${filter_ok}       ,${SEPERATOR} 
    Should Be Equal As Strings    ${response['body'][0]['callbackUri']}    @{words}[1]

Check HTTP Response Body PkgmSubscription Attributes Values Match the Issued Subscription
    Log    Check Response matches subscription
    ${body}=    Get File    jsons/subscriptions.json
    ${subscription}=    evaluate    json.loads('''${body}''')    json
    Should Be Equal    ${response['body']['callbackUri']}    ${subscription['callbackUri']}


Check Postcondition VNF Package Subscription Is Set
    [Arguments]    ${location}=""
    Log    Check Postcondition subscription exist
    Log    Trying to get the subscription
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Run Keyword If    ${location} == Location
    ...    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${response['body']['id']}
    Run Keyword If    ${location} == Location
    ...    GET    ${response['headers']['Location']}  
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Check HTTP Response Status Code Is    200

Check Postcondition Subscription Resource Returned in Location Header Is Available
    Log    Going to check postcondition
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${response['headers']['Location']}
    Integer    response status    200
    Log    Received a 200 OK as expected
    ${contentType}=    Output    response headers Content-Type
    Should Contain    ${contentType}    application/json
    ${result}=    Output    response body
    Validate Json    PkgmSubscription.schema.json    ${result}
    Log    Validated PkgmSubscription schema
    ${body}=    Get File    jsons/subscriptions.json
    ${subscription}=    evaluate    json.loads('''${body}''')    json
    Should Be Equal    ${result['callbackUri']}    ${subscription['callbackUri']}
    Log    Validated Issued subscription is same as original

Get Individual VNF Package Subscription
    Log    Trying to get a single subscription identified by subscriptionId
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET individual VNF Package Subscription with invalid resource identifier
    Log    Trying to perform a request on a subscriptionID which doesn't exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${erroneousSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Delete request for individual VNF Package Subscription
    Log    Trying to perform a DELETE on a subscriptionId
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send Delete request for individual VNF Package Subscription with invalid resource identifier
    Log    Trying to perform a DELETE on a subscriptionId which doesn't exist
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${erroneousSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Post request for individual VNF Package Subscription
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${newSubscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send Put request for individual VNF Package Subscription
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send Patch request for individual VNF Package Subscription
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${origOutput}=    Output    response
    Set Suite Variable    ${origResponse}    ${origOutput}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/subscriptions/${subscriptionId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Check HTTP Response Header Contains
    [Arguments]    ${CONTENT_TYPE}
    Should Contain    ${response['headers']}    ${CONTENT_TYPE}
    Log    Header is present

Check HTTP Response Body Subscription Identifier matches the requested Subscription
    Log    Trying to check response ID
    Should Be Equal    ${response['body']['id']}    ${subscriptionId} 
    Log    Subscription identifier as expected

Create Sessions
    Pass Execution If    ${NFVO_CHECKS_NOTIF_ENDPOINT} == 0    MockServer not started as NFVO is not checking the notification endpoint
    Start Process  java  -jar  ${MOCK_SERVER_JAR}    -serverPort  ${callback_port}  alias=mockInstance
    Wait For Process  handle=mockInstance  timeout=5s  on_timeout=continue
    Create Mock Session  ${callback_uri}:${callback_port}
        
Check Notification Endpoint
    &{notification_request}=  Create Mock Request Matcher	GET  ${callback_endpoint}    
    &{notification_response}=  Create Mock Response	headers="Content-Type: application/json"  status_code=204
    Create Mock Expectation  ${notification_request}  ${notification_response}
    Wait Until Keyword Succeeds    ${total_polling_time}   ${polling_interval}   Verify Mock Expectation    ${notification_request}
    Clear Requests  ${callback_endpoint}
    
Check HTTP Response Header Contain Link
    ${linkURL}=    Get Value From Json    ${response['headers']}    $..Link
    Should Not Be Empty    ${linkURL}
    Log    Check that Link header contains rel set to next
    Should Contain    ${linkURL}    rel="next"
    Log    Check that Link header contains URI to next page between <>
    ${linkHeaderUri}=    Get Regexp Matches    ${linkURL}    (?<=\<)(.*?)(?=\>)
    ${length}=    Get Length    ${linkHeaderUri}	
    Should Be Equal As Integers	${length}	0
    Set Global Variable    ${nextPageUri}    ${linkHeaderUri[0]}  

Get all OnBoarded VNF Packages
    Log    Trying to get all OnBoarded VNF Packages present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Get all OnBoarded VNF Packages with nextpage_opaque_marker parameter
    Log    Get next page of packages
    Set Headers  {"Accept":"${ACCEPT_JSON}"}  
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    Log    Execute Query using the URI provided in the previous GET request and validate response
    Get    ${nextPageUri}
    ${outputResponse}=    Output    response
	Set Global Variable    ${response}    ${outputResponse}	
   
GET OnBoarded VNF Packages with attribute-based filter
    Log    Trying to get all OnBoarded VNF Packages present in the NFVO Catalogue, using filter params
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages?${POS_FILTER}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET OnBoarded VNF Packages with invalid attribute-based filter
    Log    Trying to perform a negative get, filtering by the inexistent filter 'nfvId'
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages?${NEG_FILTER}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET OnBoarded VNF Packages with all_fields attribute selector
    Log    Trying to get all OnBoarded VNF Packages present in the NFVO Catalogue, using filter params
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages?all_fields
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET OnBoarded VNF Packages with exclude_default attribute selector
    Log    Trying to get all OnBoarded VNF Packages present in the NFVO Catalogue, using exclude_default filter.
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages?exclude_default   
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET OnBoarded VNF Packages with fields attribute selector
    Log    Trying to get all OnBoarded VNF Packages present in the NFVO Catalogue, using fields
    Pass Execution If    ${NFVO_FIELDS} == 0    The NFVO is not able to use fields parameter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages?fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET OnBoarded VNF Packages with exclude_fields attribute selector
    Log    Trying to get all OnBoarded VNF Packages present in the NFVO Catalogue, using fields
    Pass Execution If    ${NFVO_FIELDS} == 0    The NFVO is not able to use fields parameter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages?exclude_fields=${fields}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET OnBoarded all VNF Packages with invalid resource endpoint
    Log    Trying to perform a GET on a erroneous URI
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_package
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send OnBoarded POST Request for all VNF Packages
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send OnBoarded PUT Request for all VNF Packages
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send OnBoarded PATCH Request for all VNF Packages
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send OnBoarded DELETE Request for all VNF Packages
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual OnBoarded VNF Package
    Log    Trying to get a OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual OnBoarded VNF Package with invalid resource identifier
    Log    Trying to perform a negative get, using wrong authorization bearer
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${erroneousVnfdId} 
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send POST Request for individual OnBoarded VNF Package
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual OnBoarded VNF Package
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for individual OnBoarded VNF Package
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual OnBoarded VNF Package
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get single file VNFD in Individual OnBoarded VNF Package in Plain Format
    Log    Trying to get a VNFD from a given OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get VNFD in Individual OnBoarded VNF Package in Zip Format
    Log    Trying to get a VNFD from a given OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get single file VNFD in Individual OnBoarded VNF Package in Plain or Zip Format
    Log    Trying to get a VNFD from a given OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get multi file VNFD in Individual OnBoarded VNF Package in Plain or Zip Format
    Log    Trying to get a VNFD from a given OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get multi file VNFD in Individual OnBoarded VNF Package in Plain Format
    Log    Trying to get a negative case performing a get on a VNFD from a given VNF Package present in the NFVO Catalogue. Accept will be text/plain but VNFD is composed my multiple files.
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndIdZipVnfd}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Get VNFD in Individual OnBoarded VNF Package with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${erroneousVnfdId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Get VNFD in Individual OnBoarded VNF Package Content with conflict due to onboarding state 
    Log    Trying to get a VNFD from a given OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${onboardingStateVnfdId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send POST Request for VNFD in individual OnBoarded VNF Package
    Log    Trying to perform a POST. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send PUT Request for VNFD in individual OnBoarded VNF Package
    Log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Send PATCH Request for VNFD in individual OnBoarded VNF Package
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send DELETE Request for VNFD in individual OnBoarded VNF Package
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndIdZipVnfd}/vnfd
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
GET Individual OnBoarded VNF Package Content  
    Log    Trying to get a VNF Package Content
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual OnBoarded VNF Package Content with Range Request
    Log    Trying to get a VNF Package Content using RANGE using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/package_content
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual OnBoarded VNF Package Content with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/package_content
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual OnBoarded VNF Package Content with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${erroneousVnfdId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Content for OnBoarded VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a VNF Package content present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${onboardingStateVnfdId}/package_content
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual OnBoarded VNF Package Artifact
    Log    Trying to get a VNF Package Artifact
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual OnBoarded VNF Package Artifact in octet stream format  
    Log    Trying to get a VNF Package Artifact
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfdOctetStreamId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual OnBoarded VNF Package Artifact with Range Request
    Log    Trying to get an Artifact using RANGE Header and using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual OnBoarded VNF Package Artifact with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
GET Individual OnBoarded VNF Package Artifact with invalid resource identifier
    Log    Trying to perform a negative get, using an erroneous package ID
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${erroneousVnfdId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Artifact for OnBoarded VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a VNF Package artifact present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${onboardingStateVnfdId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send POST Request for individual OnBoarded VNF Package Artifact
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for individual OnBoarded VNF Package Artifact
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for individual OnBoarded VNF Package Artifact
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for individual OnBoarded VNF Package Artifact
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get multi file VNFD in Individual OnBoarded VNF Package with security information
    Log    Trying to get a VNFD from a given OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/vnfd/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get multi file VNFD in Individual VNF Package with security information
    Log    Trying to get a VNFD from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPkgZipVNFD}/vnfd/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual VNF Package Artifact with security information 
    Log    Trying to get a VNF Package Artifact
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vndId}/artifacts/${artifactPath}/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual OnBoarded VNF Package Artifact with security information  
    Log    Trying to get a VNF Package Artifact
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual VNF Package Artifact with encrypted artifact
    Log    Trying to get a VNF Package Artifact
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vndEncryptedArtifactID}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual OnBoarded VNF Package Artifact with encrypted artifact
    Log    Trying to get a VNF Package Artifact
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndEncryptedArtifactID}/artifacts/${artifactPath}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Individual VNF Package Artifact with incompatable header
    Log    Trying to get a VNF Package Artifact
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vndId}/artifacts/${artifactPath}/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

GET Individual OnBoarded VNF Package Artifact with incompatable header  
    Log    Trying to get a VNF Package Artifact
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/artifacts/${artifactPath}/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Manifest in Individual VNF Package
    Log    Trying to get content of manifest file from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Manifest in Individual VNF Package with security information
    Log    Trying to get a manifest from a given VNF Package present in the NFVO Catalogue with security certificates
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/manifest/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Manifest in Individual VNF Package with conflict due to onboarding state
    Log    Trying to get a manifest from a given VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send POST Request for Manifest in individual VNF Package
    Log    Trying to perform a POST. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send PUT Request for Manifest in individual VNF Package
    Log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Send PATCH Request for Manifest in individual VNF Package
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send DELETE Request for Manifest in individual VNF Package
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Get Manifest in Individual OnBoarded VNF Package
    Log    Trying to get content of manifest file from a given OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Manifest in Individual OnBoarded VNF Package with security information
    Log    Trying to get a manifest from a given OnBoarded VNF Package present in the NFVO Catalogue with security certificates
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndIdZipVnfd}/manifest/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Manifest in Individual OnBoarded VNF Package with conflict due to onboarding state
    Log    Trying to get a manifest from a given OnBoarded VNF Package present in the NFVO Catalogue
    Set Headers    {"Accept": "${ACCEPT_PLAIN}"}
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${onboardingStateVnfdId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send POST Request for Manifest in individual OnBoarded VNF Package
    Log    Trying to perform a POST. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send PUT Request for Manifest in individual OnBoarded VNF Package
    Log    Trying to perform a PUT. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Send PATCH Request for Manifest in individual OnBoarded VNF Package
    Log    Trying to perform a PATCH. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Send DELETE Request for Manifest in individual OnBoarded VNF Package
    Log    Trying to perform a DELETE. This method should not be implemented
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vndId}/manifest
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 
    
Get Artifact in VNF Package
    Log    Trying to get Artifact in VNF Package
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with exclude_all_mano_artifacts parameter
    Log    Trying to get Artifact in VNF Package with exclude_all_mano_artifacts
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/exclude_all_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with include_external_artifacts parameter
    Log    Trying to get Artifact in VNF Package with include_external_artifacts
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/include_external_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with exclude_all_non_mano_artifacts parameter
    Log    Trying to get Artifact in VNF Package with security certificates
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with select_non_mano_artifact_sets parameter
    Log    Trying to get Artifact in VNF Package with security certificates
    Pass Execution If    ${NFVO_non-MANO_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/select_non_mano_artifact_sets=${non_mano_artifact_sets}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with include_signatures parameter
    Log    Trying to get Artifact in VNF Package with security certificates
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with Range Request
    Log    Trying to get an Artifact using RANGE Header and using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with exclude_all_non_mano_artifacts and select_non_mano_artifact_sets
    Log    Trying to get Artifact in VNF Package 
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts&select_non_mano_artifact_sets=${non_mano_artifact_sets}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in VNF Package with exclude_all_non_mano_artifacts and exclude_all_mano_artifacts
    Log    Trying to get Artifact in VNF Package 
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts&exclude_all_non_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Artifact of VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a VNF Package artifact present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${onboardingStateVnfPkgId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET VNF Package Artifact with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send POST Request for VNF Package Artifact
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for VNF Package Artifact
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for VNF Package Artifact
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for VNF Package Artifact
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Get Artifact in OnBoarded VNF Package
    Log    Trying to get Artifact in OnBoarded VNF Package
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in OnBoarded VNF Package with exclude_all_mano_artifacts parameter
    Log    Trying to get Artifact in OnBoarded VNF Package with exclude_all_mano_artifacts
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts/exclude_all_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in OnBoarded VNF Package with exclude_all_non_mano_artifacts parameter
    Log    Trying to get Artifact in OnBoarded VNF Package with security certificates
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in OnBoarded VNF Package with include_external_artifacts parameter
    Log    Trying to get Artifact in OnBoarded VNF Package with include_external_artifacts
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts/include_external_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in OnBoarded VNF Package with select_non_mano_artifact_sets parameter
    Log    Trying to get Artifact in OnBoarded VNF Package with security certificates
    Pass Execution If    ${NFVO_non-MANO_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts/select_non_mano_artifact_sets=${non_mano_artifact_sets}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in OnBoarded VNF Package with include_signatures parameter
    Log    Trying to get Artifact in OnBoarded VNF Package with security certificates
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts/include_signatures
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in OnBoarded VNF Package with Range Request
    Log    Trying to get an Artifact using RANGE Header and using an NFVO that can handle it
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Range": "${range}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in OnBoarded VNF Package with exclude_all_non_mano_artifacts and select_non_mano_artifact_sets
    Log    Trying to get Artifact in OnBoarded VNF Package 
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts&select_non_mano_artifact_sets=${non_mano_artifact_sets}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Get Artifact in OnBoarded VNF Package with exclude_all_non_mano_artifacts and exclude_all_mano_artifacts
    Log    Trying to get Artifact in OnBoarded VNF Package 
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts/exclude_all_non_mano_artifacts&exclude_all_non_mano_artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET Artifact of OnBoarded VNF Package in onboarding state different from ONBOARDED
    Log    Trying to get a OnBoarded VNF Package artifact present in the NFVO Catalogue, but not in ONBOARDED operationalStatus
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${onboardingStateVnfPkgId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET OnBoarded VNF Package Artifact with invalid Range Request
    Log    Trying to get a range of bytes of the limit of the OnBoarded VNF Package
    Pass Execution If    ${NFVO_RANGE_OK} == 0    Skipping this test as NFVO is not able to handle partial Requests.
    Set Headers    {"Accept": "${ACCEPT_ZIP}"}
    Set Headers    {"Range": "${erroneousRange}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts
    Unset Headers  ["Range"]
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Send POST Request for OnBoarded VNF Package Artifact
    Log    Trying to perform a POST (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send PUT Request for OnBoarded VNF Package Artifact
    Log    Trying to perform a PUT (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PUT    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
 
Send PATCH Request for OnBoarded VNF Package Artifact
    Log    Trying to perform a PATCH (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    PATCH    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Send DELETE Request for OnBoarded VNF Package Artifact
    Log    Trying to perform a DELETE (method should not be implemented)
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages/${vnfPackageId}/artifacts
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
GET OnBoarded VNF Packages with fields and exclude_default attribute selector
    Log    Trying to get all OnBoarded VNF Packages present in the NFVO Catalogue, using fields
    Pass Execution If    ${NFVO_FIELDS} == 0    The NFVO is not able to use fields parameter
    Set Headers    {"Accept": "${ACCEPT_JSON}"}
    Run Keyword If    ${AUTH_USAGE} == 1    Set Headers    {"${AUTHORIZATION_HEADER}":"${AUTHORIZATION_TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiMajorVersion}/onboarded_vnf_packages?fields=${fields}&exclude_default
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    
Check Postcondition VNF Package Subscription is Deleted
    Log    Check Postcondition Subscription is deleted
    GET individual VNF Package Subscription
    Check HTTP Response Status Code Is    404 
    
Check HTTP Response Body of Individual VNF Package content against VNF Descriptor
    Run Keyword If    ${check_descriptors} == 1    Check Individual VNF Package Content
    
Check Individual VNF Package Content
    PARSE the Descriptor File
    Match the VNF Package Response Attributes with Descriptors
    
PARSE the Descriptor File
    Run Keyword If  '${descriptorType}'=='SOL001'    Fetch Information from SOL001 descriptor file    ELSE    Fetch Information from SOL006 descriptor file    
    
Fetch Information from SOL001 descriptor file
    Import Variables    ${CURDIR}/descriptors/SOL001/VNFD/vnfd_SOL001.yaml
    @{VDU_labels}=    Create List
    @{VNF_labels}=    Create List
    @{VirtualLink_labels}=    Create List
    @{CP_labels}=    Create List
    @{Storage_labels}=    Create List
    FOR    ${key}    IN    @{node_types.keys()}
        ${node_type}=    Get Variable Value    ${node_types['${key}']['derived_from']}
        ${is_VNF}=    Run Keyword And Return Status    Should Be Equal As Strings    ${node_type}    ${tosca_type_VNF}
        Run Keyword If    ${is_VNF}    Set Global Variable    ${tosca_type_derived_from_VNF}    ${key}
    END  
    ${derived_type_is_used}=    Run Keyword And Return Status    Should not be empty    ${tosca_type_derived_from_VNF}
    Run Keyword If    ${derived_type_is_used}    Set Global Variable    ${tosca_type_VNF}    ${tosca_type_derived_from_VNF}              
    FOR    ${key}    IN    @{topology_template.node_templates.keys()} 
        ${key_type}=    Get Variable Value    ${topology_template.node_templates['${key}'].type}
        ${VDU_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}   ${tosca_type_VDU_compute}
        ${VNF_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_VNF}
        ${Link_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_virtual_link}
        ${VDUcp_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_VDU_cp}
        ${Storage_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key_type}    ${tosca_type_storage}
        Run Keyword If    ${VDU_check}    Append To List    ${VDU_labels}    ${key}
        Run Keyword If    ${VNF_check}    Append To List    ${VNF_labels}    ${key}
        Run Keyword If    ${VNF_check}    Get VNF Attributes from SOL001    ${key}            
        Run Keyword If    ${Link_check}    Append To List    ${VirtualLink_labels}    ${key}
        Run Keyword If    ${VDUcp_check}    Append To List    ${CP_labels}    ${key}
        Run Keyword If    ${Storage_check}    Append To List    ${Storage_labels}    ${key}
    END
    Set Global Variable    @{VDU_IDs}    @{VDU_labels}
    Set Global Variable    @{VNF_IDs}    @{VNF_labels}
    Set Global Variable    @{VirtualLink_IDs}    @{VirtualLink_labels}
    Set Global Variable    @{CP_IDs}    @{CP_labels}
    Set Global Variable    @{Storage_IDs}    @{Storage_labels}

Get VNF Attributes from SOL001
    [Arguments]    ${VNF_label}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL001.yaml    
    ${descriptor_id}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.descriptor_id}
    ${provider}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.provider}
    ${product_name}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.product_name}
    ${software_version}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.software_version}
    ${descriptor_version}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.descriptor_version}
    ${flavour_id}=    Get Variable Value    ${topology_template.node_templates['${VNF_label}'].properties.flavour_id}
    
    Set Global Variable    ${Descriptor_ID}    ${descriptor_id}
    Set Global Variable    ${Provider}    ${provider}
    Set Global Variable    ${Product_Name}    ${product_name}
    Set Global Variable    ${Software_Version}    ${software_version}
    Set Global Variable    ${Descriptor_Version}    ${descriptor_version}
    Set Global Variable    ${Flavour_ID}    ${flavour_id}
    
Fetch Information from SOL006 descriptor file
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    ${descriptor_id}=    Get Variable Value    ${nfv.vnfd[0].id}
    ${provider}=    Get Variable Value    ${nfv.vnfd[0].provider}
    ${product_name}=    Get Variable Value    ${nfv.vnfd[0]['product-name']}
    ${software_version}=    Get Variable Value    ${nfv.vnfd[0]['software-version']}
    ${descriptor_version}=    Get Variable Value    ${nfv.vnfd[0].version}
    ${VDUcount}=    Get Length    ${nfv.vnfd[0].vdu}
    ${extCP_count}=    Get Length    ${nfv.vnfd[0]['ext-cpd']}
    ${virtualLink_count}=    Get length    ${nfv.vnfd[0]['int-virtual-link-desc']}
    ${instantiation_levels}=    Get Length    ${nfv.vnfd[0].df['instantiation-level']}    
    FOR    ${key}    IN    @{nfv.vnfd[0].keys()}
        ${VDU_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    vdu
        Run Keyword If    ${VDU_check}    Get VDU IDs    ${VDUcount}
        ${extCP_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    ext-cpd
        Run Keyword If    ${extCP_check}    Get External CP IDs    ${extCP_count}
        ${virtualLink_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    int-virtual-link-desc
        Run Keyword If    ${virtualLink_check}    Get Virtual Link IDs   ${virtualLink_count}    
        ${DF_check}=    Run Keyword And Return Status    Should Be Equal As Strings    ${key}    df
        Run Keyword If    ${DF_check}    Get Instantiation Levels    ${instantiation_levels}    
    END
    
    Set Global Variable    ${Descriptor_ID}    ${descriptor_id}
    Set Global Variable    ${Provider}    ${provider}
    Set Global Variable    ${Product_Name}    ${product_name}
    Set Global Variable    ${Software_Version}    ${software_version}
    Set Global Variable    ${Descriptor_Version}    ${descriptor_version}
    
Get VDU IDs
    [Arguments]    ${count}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    @{VDU_labels}=    Create List
    ${Storage_labels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${VDU_labels}    ${nfv.vnfd[0].vdu[${i}]['id']}
        Append To List    ${Storage_labels}    ${nfv.vnfd[0].vdu[${i}]['virtual-storage-desc']} 
        Get Internal CPs for each VDU    ${i}   
    END
    Set Global Variable    ${VDU_IDs}    ${VDU_labels}
    Set Global Variable    ${Storage_IDs}    ${Storage_labels}
            
Get Internal CPs for each VDU
    [Arguments]    ${vdu}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    ${count}=    Get Length    ${nfv.vnfd[0].vdu[${vdu}]['int-cpd']}    
    ${internal_CPs}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${internal_CPs}    ${nfv.vnfd[0].vdu[${vdu}]['int-cpd'][${i}]['id']} 
    END
    Set Global Variable    ${internalCP_IDs}    ${internal_CPs}
    
Get External CP IDs
    [Arguments]    ${count} 
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml   
    @{external_CPs}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${external_CPs}    ${nfv.vnfd[0].ext_cpd[${i}].id} 
    END
    Set Global Variable    ${externalCP_IDs}    ${external_CPs}

Get Virtual Link IDs
    [Arguments]    ${count} 
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml   
    ${VirtualLink_labels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${VirtualLink_labels}    ${nfv.vnfd[0]['int-virtual-link-desc'][${i}]['id']} 
    END
    Set Global Variable    ${VirtualLink_IDs}    ${VirtualLink_labels}

Get Instantiation Levels
    [Arguments]    ${count}
    Import Variables    ${CURDIR}/descriptors/SOL006/VNFD/vnfd_SOL006.yaml
    @{Instantiation_Levels}=    Create List
    FOR    ${i}    IN RANGE    ${count}
        Append To List    ${Instantiation_Levels}    ${nfv.vnfd[0].df['instantiation-level'][${i}]['id']} 
    END
    Set Global Variable    ${InstantiationLevel_IDs}    ${Instantiation_Levels}

Match the VNF Package Response Attributes with Descriptors
    #Checking Response attributes with VNF Descriptors
    Should Be Equal As Strings    ${response['body']['vnfdId']}    ${Descriptor_ID}
    Should Be Equal As Strings    ${response['body']['vnfProvider']}    ${Provider}
    Should Be Equal As Strings    ${response['body']['vnfProductName']}    ${Product_Name}
    Should Be Equal As Strings    ${response['body']['vnfSoftwareVersion']}    ${Software_Version}
    Should Be Equal As Strings    ${response['body']['vnfdVersion']}    ${Descriptor_Version}

    
Check HTTP Response Body of Individual Subscription content against VNF Descriptor
    Run Keyword If    ${check_descriptors} == 1    Check Individual VNF Subscription Content
    
Check Individual VNF Subscription Content
    PARSE the Descriptor File
    Match the VNF subscription Response Attributes with Descriptors
    
Match the VNF subscription Response Attributes with Descriptors
    #Checking Response attributes with VNF Descriptors
    Should Be Equal As Strings    ${response['body']['filter']['vnfdId']}    ${Descriptor_ID}
    Should Be Equal As Strings    ${response['body']['filter']['vnfProductsFromProviders'][0]['vnfProvider']}    ${Provider}
    Should Be Equal As Strings    ${response['body']['filter']['vnfProductsFromProviders'][0]['vnfProducts'][0]['vnfProductName']}    ${Product_Name}
    Should Be Equal As Strings    ${response['body']['filter']['vnfProductsFromProviders'][0]['vnfProducts']['versions'][0]['vnfSoftwareVersion']}    ${Software_Version}
    Should Be Equal As Strings    ${response['body']['filter']['vnfProductsFromProviders'][0]['vnfProducts']['versions'][0]['vnfdVersion']}    ${Descriptor_Version}
    
