#include <stdexcept>
#include <regex>
#include <string>

#include "json_codec_factory.hh"

#include "loggers.hh"

#include "LibItsHttp_JsonMessageBodyTypes.hh"

int json_codec::encode (const LibItsHttp__JsonMessageBodyTypes::JsonBody& msg, OCTETSTRING& data)
{
  loggers::get_instance().log_msg(">>> json_codec::encode: ", (const Base_Type&)msg);

  TTCN_EncDec::clear_error();
  TTCN_EncDec::set_error_behavior(TTCN_EncDec::ET_ALL, TTCN_EncDec::EB_DEFAULT);
  TTCN_Buffer encoding_buffer;

  if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_ueIdentityTagInfo)) {
    const UEidentityAPI__TypesAndValues::UeIdentityTagInfo& ue_identity_tag_info = msg.ueIdentityTagInfo();
    ue_identity_tag_info.encode(UEidentityAPI__TypesAndValues::UeIdentityTagInfo_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data());
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_userTrackingSubscription)) {
    const LocationAPI__TypesAndValues::UserTrackingSubscription& user_tracking_subscription = msg.userTrackingSubscription();
    user_tracking_subscription.encode(LocationAPI__TypesAndValues::UserTrackingSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = char2oct(CHARSTRING("{\"userTrackingSubscription\": ")) + OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data()) + char2oct(CHARSTRING("}"));
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_periodicNotificationSubscription)) {
    const LocationAPI__TypesAndValues::PeriodicNotificationSubscription& periodic_notification_subscription = msg.periodicNotificationSubscription();
    periodic_notification_subscription.encode(LocationAPI__TypesAndValues::PeriodicNotificationSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = char2oct(CHARSTRING("{\"periodicNotificationSubscription\": ")) + OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data()) + char2oct(CHARSTRING("}"));
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_zonalTrafficSubscription)) {
    const LocationAPI__TypesAndValues::ZonalTrafficSubscription& zonal_traffic_subscription = msg.zonalTrafficSubscription();
    zonal_traffic_subscription.encode(LocationAPI__TypesAndValues::ZonalTrafficSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = char2oct(CHARSTRING("{\"zonalTrafficSubscription\": ")) + OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data()) + char2oct(CHARSTRING("}"));
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_circleNotificationSubscription)) {
    const LocationAPI__TypesAndValues::CircleNotificationSubscription& notif = msg.circleNotificationSubscription();
    notif.encode(LocationAPI__TypesAndValues::CircleNotificationSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = char2oct(CHARSTRING("{\"circleNotificationSubscription\": ")) + OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data()) + char2oct(CHARSTRING("}"));
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_distanceNotificationSubscription)) {
    const LocationAPI__TypesAndValues::DistanceNotificationSubscription& notif = msg.distanceNotificationSubscription();
    notif.encode(LocationAPI__TypesAndValues::DistanceNotificationSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = char2oct(CHARSTRING("{\"distanceNotificationSubscription\": ")) + OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data()) + char2oct(CHARSTRING("}"));
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_cellChangeSubscription)) {
    const RnisAPI__TypesAndValues::CellChangeSubscription& cell_change_subscription = msg.cellChangeSubscription();
    cell_change_subscription.encode(RnisAPI__TypesAndValues::CellChangeSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"CellChangeSubscription\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_appContext)) {
    const UEAppInterfaceAPI__TypesAndValues::AppContext& app_context = msg.appContext();
    app_context.encode(UEAppInterfaceAPI__TypesAndValues::AppContext_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = char2oct(CHARSTRING("{\"AppContext\": ")) + OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data()) + char2oct(CHARSTRING("}"));
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_serviceInfo)) {
    const EdgePlatformApplicationEnablementAPI__TypesAndValues::ServiceInfo& service_info = msg.serviceInfo();
    service_info.encode(EdgePlatformApplicationEnablementAPI__TypesAndValues::ServiceInfo_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"ServiceInfo\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_appTerminationNotificationSubscription)) {
    const EdgePlatformApplicationEnablementAPI__TypesAndValues::AppTerminationNotificationSubscription& app = msg.appTerminationNotificationSubscription();
    app.encode(EdgePlatformApplicationEnablementAPI__TypesAndValues::AppTerminationNotificationSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"AppTerminationNotificationSubscription\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_dnsRule)) {
    const EdgePlatformApplicationEnablementAPI__TypesAndValues::DnsRule& dns_rule = msg.dnsRule();
    dns_rule.encode(EdgePlatformApplicationEnablementAPI__TypesAndValues::DnsRule_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"DnsRule\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_serAvailabilityNotificationSubscription)) {
    const EdgePlatformApplicationEnablementAPI__TypesAndValues::SerAvailabilityNotificationSubscription& ser = msg.serAvailabilityNotificationSubscription();
    ser.encode(EdgePlatformApplicationEnablementAPI__TypesAndValues::SerAvailabilityNotificationSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"SerAvailabilityNotificationSubscription\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_trafficRule)) {
    const EdgePlatformApplicationEnablementAPI__TypesAndValues::TrafficRule& traffic_rule = msg.trafficRule();
    traffic_rule.encode(EdgePlatformApplicationEnablementAPI__TypesAndValues::TrafficRule_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"TrafficRule\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_bwInfo)) {
    const TrafficManagementAPI__TypesAndValues::BwInfo& bw_info = msg.bwInfo();
    bw_info.encode(TrafficManagementAPI__TypesAndValues::BwInfo_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"TrafficRule\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_mtsSessionInfo)) {
    const TrafficManagementAPI__TypesAndValues::MtsSessionInfo& mts_session_info = msg.mtsSessionInfo();
    mts_session_info.encode(TrafficManagementAPI__TypesAndValues::MtsSessionInfo_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"TrafficRule\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_predictedQos)) {
    const V2XInformationServiceAPI__TypesAndValues::PredictedQos& predicted_qos = msg.predictedQos();
    predicted_qos.encode(V2XInformationServiceAPI__TypesAndValues::PredictedQos_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"TrafficRule\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_assocStaSubscription)) {
    const WlanInformationAPI__TypesAndValues::AssocStaSubscription& assoc_sta_subscription = msg.assocStaSubscription();
    assoc_sta_subscription.encode(WlanInformationAPI__TypesAndValues::AssocStaSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"TrafficRule\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_staDataRateSubscription)) {
    const WlanInformationAPI__TypesAndValues::StaDataRateSubscription& sta_data_rate_subscription = msg.staDataRateSubscription();
    sta_data_rate_subscription.encode(WlanInformationAPI__TypesAndValues::StaDataRateSubscription_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = /*char2oct(CHARSTRING("{\"TrafficRule\": ")) + */OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data())/* + char2oct(CHARSTRING("}"))*/;
  } else {
    loggers::get_instance().error("json_codec::encode: Not supported");
  }

  loggers::get_instance().log("<<< json_codec::encode");
  return 0;
}

int json_codec::decode (const OCTETSTRING& p_data, LibItsHttp__JsonMessageBodyTypes::JsonBody& msg, params* p_params)
{
  loggers::get_instance().log_msg(">>> json_codec::decode: p_data=", p_data);

  // Sanity checks
  params::const_iterator it;
  if (p_params == nullptr) {
    loggers::get_instance().warning("json_codec::decode: Failed to access p_params (null pointer)");
    return -1; // TODO Use p_data instead of return -1
  } else {
    it = p_params->find("decode_str");
    if (it == p_params->cend()) {
      loggers::get_instance().warning("json_codec::decode: Failed to access p_params item (decode_str)");
      return -1; // TODO Use p_data instead of return -1
    }
    loggers::get_instance().log("json_codec::decode: it->second='%c' / '%s'", it->second.c_str()[0], it->second.c_str());
  }

  // Remove data structure name (if present) ...
  std::string str;
  if ((it->second[0] != '[') && (it->second[0] != '{')) {
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
  } else {
    str = it->second;
  }
  // ..and create the decoding buffer
  TTCN_EncDec::set_error_behavior(TTCN_EncDec::ET_ALL, TTCN_EncDec::EB_DEFAULT);
  TTCN_EncDec::clear_error();
  loggers::get_instance().log("json_codec::decode: decoding_buffer='%c' / '%s'", str[0], str.c_str());
  TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));

  if (it->second.find("\"userList\"") != std::string::npos) { // Be careful to the order
    // TODO To be refined, find("\"userList\"") is not optimal
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::UserList user_list;
    user_list.decode(LocationAPI__TypesAndValues::UserList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.userList() = user_list;
  } else if (it->second.find("\"accessPointList\"") != std::string::npos) { // Be careful to the order
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::AccessPointList access_point_list;
    access_point_list.decode(LocationAPI__TypesAndValues::AccessPointList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.accessPointList() = access_point_list;
  } else if (it->second.find("\"userTrackingSubscription\"") != std::string::npos) {
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::UserTrackingSubscription user_tracking_subscription;
    user_tracking_subscription.decode(LocationAPI__TypesAndValues::UserTrackingSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.userTrackingSubscription() = user_tracking_subscription;






  } else if (it->second.find("\"periodicNotificationSubscription\"") != std::string::npos) {
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::PeriodicNotificationSubscription periodic_notification_subscription;
    periodic_notification_subscription.decode(LocationAPI__TypesAndValues::PeriodicNotificationSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.periodicNotificationSubscription() = periodic_notification_subscription;
  } else if (it->second.find("\"zonalTrafficSubscription\"") != std::string::npos) {
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::ZonalTrafficSubscription zonal_traffic_subscription;
    zonal_traffic_subscription.decode(LocationAPI__TypesAndValues::ZonalTrafficSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.zonalTrafficSubscription() = zonal_traffic_subscription;
  } else if (it->second.find("\"periodicTrackingSubscription\"") != std::string::npos) {
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::PeriodicTrackingSubscription periodic_tracking_subscription;
    periodic_tracking_subscription.decode(LocationAPI__TypesAndValues::PeriodicTrackingSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.periodicTrackingSubscription() = periodic_tracking_subscription;
  } else if (it->second.find("\"accessPointId\"") != std::string::npos) {
    LocationAPI__TypesAndValues::UserInfo user_info;
    user_info.decode(LocationAPI__TypesAndValues::UserInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.userInfo() = user_info;
  } else if (it->second.find("\"terminalDistance\"") != std::string::npos) {
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::TerminalDistance terminal_distance;
    terminal_distance.decode(LocationAPI__TypesAndValues::TerminalDistance_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.terminalDistance() = terminal_distance;
  } else if (it->second.find("\"enteringLeavingCriteria\"") != std::string::npos) {
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::CircleNotificationSubscription notif;
    notif.decode(LocationAPI__TypesAndValues::CircleNotificationSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.circleNotificationSubscription() = notif;
  } else if (it->second.find("\"distanceNotificationSubscription\"") != std::string::npos) {
    int idx_begin = it->second.find(":");
    int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
    str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
    TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    LocationAPI__TypesAndValues::DistanceNotificationSubscription notif;
    notif.decode(LocationAPI__TypesAndValues::DistanceNotificationSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.distanceNotificationSubscription() = notif;
  } else if (it->second.find("\"ueIdentityTagInfo\"") != std::string::npos) {
    UEidentityAPI__TypesAndValues::UeIdentityTagInfo ue_identity_tag_info;
    ue_identity_tag_info.decode(UEidentityAPI__TypesAndValues::UeIdentityTagInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.ueIdentityTagInfo() = ue_identity_tag_info;
  } else if (it->second.find("\"serName\"") != std::string::npos) { // Be careful to the order
                                                                    // TODO To be refined, find("\"accessPointList\"") is not optimal
    if (it->second[0] == '[') {
      std::size_t it_start;
      while ((it_start = str.find("null")) != std::string::npos) {
        loggers::get_instance().log("json_codec::decode: id_start=%d", it_start);
        loggers::get_instance().log("json_codec::decode: Before, str=%s", str.c_str());
        std::size_t it_stop = it_start + 4;
        loggers::get_instance().log("json_codec::decode: id_stop=%d", it_stop);
        loggers::get_instance().log("json_codec::decode: segment=%s", str.substr(it_start, it_stop - it_start).c_str());
        str = str.substr(0, it_start) + "{}" + str.substr(it_stop);
        loggers::get_instance().log("json_codec::decode: After, str=%s", str.c_str());
      } // End of 'while' statement
      decoding_buffer.clear();
      decoding_buffer.put_os(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
      EdgePlatformApplicationEnablementAPI__TypesAndValues::ServiceInfoList service_info_list;
      service_info_list.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::ServiceInfoList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
      msg.serviceInfoList() = service_info_list;
    } else {
      EdgePlatformApplicationEnablementAPI__TypesAndValues::ServiceInfo service_info;
      service_info.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::ServiceInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
      msg.serviceInfo() = service_info;
    }
  } else if (it->second.find("\"security\"") != std::string::npos) { // Be careful to the order
                                                                     // TODO To be refined, find("\"accessPointList\"") is not optimal
    std::size_t it_start;
    while ((it_start = str.find("null")) != std::string::npos) {
      loggers::get_instance().log("json_codec::decode: id_start=%d", it_start);
      loggers::get_instance().log("json_codec::decode: Before, str=%s", str.c_str());
      std::size_t it_stop = it_start + 4;
      loggers::get_instance().log("json_codec::decode: id_stop=%d", it_stop);
      loggers::get_instance().log("json_codec::decode: segment=%s", str.substr(it_start, it_stop - it_start).c_str());
      str = str.substr(0, it_start) + "{}" + str.substr(it_stop);
      loggers::get_instance().log("json_codec::decode: After, str=%s", str.c_str());
    } // End of 'while' statement
    decoding_buffer.clear();
    decoding_buffer.put_os(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    EdgePlatformApplicationEnablementAPI__TypesAndValues::TransportInfoList transport_info_list;
    transport_info_list.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::TransportInfoList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.transportInfoList() = transport_info_list;
  } else if (it->second.find("\"trafficRuleId\"") != std::string::npos) {
    if (it->second[0] == '[') {
      EdgePlatformApplicationEnablementAPI__TypesAndValues::TrafficRuleList traffic_rule_list;
      traffic_rule_list.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::TrafficRuleList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
      msg.trafficRuleList() = traffic_rule_list;
    } else {
      EdgePlatformApplicationEnablementAPI__TypesAndValues::TrafficRule traffic_rule;
      traffic_rule.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::TrafficRule_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
      msg.trafficRule() = traffic_rule;
    }
  } else if (it->second.find("\"dnsRuleId\"") != std::string::npos) {
    if (it->second[0] == '[') {
      EdgePlatformApplicationEnablementAPI__TypesAndValues::DnsRuleList dns_rule_list;
      dns_rule_list.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::DnsRuleList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
      msg.dnsRuleList() = dns_rule_list;
    } else {
      EdgePlatformApplicationEnablementAPI__TypesAndValues::DnsRule dns_rule;
      dns_rule.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::DnsRule_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
      msg.dnsRule() = dns_rule;
    }
  } else if ((it->second.find("\"_links\"") != std::string::npos) && (it->second.find("\"subscriptions\"") != std::string::npos)) {
    EdgePlatformApplicationEnablementAPI__TypesAndValues::SubscriptionLinkList sub_link_list;
    sub_link_list.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::SubscriptionLinkList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.subscriptionLinkList__app__ens() = sub_link_list;
  } else if (it->second.find("\"AppTerminationNotificationSubscription\"") != std::string::npos) {
    EdgePlatformApplicationEnablementAPI__TypesAndValues::AppTerminationNotificationSubscription app_term;
    app_term.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::AppTerminationNotificationSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.appTerminationNotificationSubscription() = app_term;
  } else if ((it->second.find("\"subscriptionType\"") != std::string::npos) && (it->second.find("\"SerAvailabilityNotificationSubscription\"") != std::string::npos)) {
    EdgePlatformApplicationEnablementAPI__TypesAndValues::SerAvailabilityNotificationSubscription ser_av;
    ser_av.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::SerAvailabilityNotificationSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.serAvailabilityNotificationSubscription() = ser_av;
  } else if ((it->second.find("\"ntpServers\"") != std::string::npos) || (it->second.find("{\"timeStamp\"") == 0)) {
    EdgePlatformApplicationEnablementAPI__TypesAndValues::TimingCaps timing_caps;
    timing_caps.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::TimingCaps_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.timingCaps() = timing_caps;
  } else if (it->second.find("\"timeSourceStatus\"") != std::string::npos) {
    EdgePlatformApplicationEnablementAPI__TypesAndValues::CurrentTime current_time;
    current_time.decode(EdgePlatformApplicationEnablementAPI__TypesAndValues::CurrentTime_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.currentTime() = current_time;
  } else if (it->second.find("\"CellChangeSubscription\"") != std::string::npos) {
    RnisAPI__TypesAndValues::CellChangeSubscription cell_change_subscription;
    cell_change_subscription.decode(RnisAPI__TypesAndValues::CellChangeSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.cellChangeSubscription() = cell_change_subscription;
    //TODO Continue with other ChangeSubscription
  } else if (it->second.find("\"cellInfo\"") != std::string::npos) {
    RnisAPI__TypesAndValues::L2Meas l2_meas;
    l2_meas.decode(RnisAPI__TypesAndValues::L2Meas_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.l2Meas() = l2_meas;
  } else if ((it->second.find("\"appInstanceId\"") != std::string::npos) && (it->second.find("\"requestId\"") != std::string::npos)) {
    RnisAPI__TypesAndValues::RabInfo rab_info;
    rab_info.decode(RnisAPI__TypesAndValues::RabInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.rabInfo() = rab_info;
  } else if (it->second.find("\"S1BearerInfo\"") != std::string::npos) {
    RnisAPI__TypesAndValues::S1BearerInfo s1_nearer_info;
    s1_nearer_info.decode(RnisAPI__TypesAndValues::S1BearerInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.s1BearerInfo() = s1_nearer_info;
  } else if ((it->second.find("\"appInstanceId\"") != std::string::npos) && (it->second.find("\"plmn\"") != std::string::npos)) {
    RnisAPI__TypesAndValues::PlmnInfos plmn_infos;
    plmn_infos.decode(RnisAPI__TypesAndValues::PlmnInfos_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.plmnInfos() = plmn_infos;
  } else if (it->second.find("\"RabEstSubscription\"") != std::string::npos) {
    RnisAPI__TypesAndValues::RabEstSubscription rab_est_subscription;
    rab_est_subscription.decode(RnisAPI__TypesAndValues::RabEstSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.rabEstSubscription() = rab_est_subscription;
  } else if (it->second.find("\"RabModSubscription\"") != std::string::npos) {
    RnisAPI__TypesAndValues::RabModSubscription rab_mod_subscription;
    rab_mod_subscription.decode(RnisAPI__TypesAndValues::RabModSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.rabModSubscription() = rab_mod_subscription;
  } else if (it->second.find("\"RabRelSubscription\"") != std::string::npos) {
    RnisAPI__TypesAndValues::RabRelSubscription rab_rel_subscription;
    rab_rel_subscription.decode(RnisAPI__TypesAndValues::RabRelSubscription_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.rabRelSubscription() = rab_rel_subscription;
  } else if ((it->second.find("\"_links\"") != std::string::npos) && (it->second.find("subscriptions") != std::string::npos)) { // SubscriptionLinkList
    // FIXME Conflict with MEC-028 SubscriptionLinkList data type
    RnisAPI__TypesAndValues::SubscriptionLinkList subscription_link_list;
    subscription_link_list.decode(RnisAPI__TypesAndValues::SubscriptionLinkList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.subscriptionLinkList() = subscription_link_list;
  } else if (it->second.find("\"AppContext\"") != std::string::npos) {
    UEAppInterfaceAPI__TypesAndValues::AppContext appContext;
    appContext.decode(UEAppInterfaceAPI__TypesAndValues::AppContext_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.appContext() = appContext;
  } else if (it->second.find("\"AppInfo\"") != std::string::npos) {
    UEAppInterfaceAPI__TypesAndValues::AppInfo appInfo;
    appInfo.decode(UEAppInterfaceAPI__TypesAndValues::AppInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.appInfo() = appInfo;
  } else if ((it->second.find("\"bwInfo\"") != std::string::npos) || (it->second.find("\"fixedBWPriority\"") != std::string::npos)) {
    // Need to change some enumerated string not supported by TTCN-3 language
    while (true) {
      if (str.find("\"00\"") != std::string::npos) {
        str = regex_replace(str, regex("\"00\""), "\"Downlink\"");
      } else if (str.find("\"01\"") != std::string::npos) {
        str = regex_replace(str, regex("\"01\""), "\"Uplink\"");
      } else if (str.find("\"10\"") != std::string::npos) {
        str = regex_replace(str, regex("\"10\""), "\"Symmetrical\"");
      } else {
        loggers::get_instance().log("json_codec::decode: BwInfo in break");
        break;
      }
    } // End of 'while' loop
    loggers::get_instance().log("json_codec::decode: BwInfo after converting enum: %s", str.c_str());
    if (it->second[0] == '[') { // BwInfoList
      TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
      TrafficManagementAPI__TypesAndValues::BwInfoList bw_info_list;
      bw_info_list.decode(TrafficManagementAPI__TypesAndValues::BwInfoList_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
      msg.bwInfoList() = bw_info_list;
    } else { // BwInfo
      TrafficManagementAPI__TypesAndValues::BwInfo bw_info;
      TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
      bw_info.decode(TrafficManagementAPI__TypesAndValues::BwInfo_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
      msg.bwInfo() = bw_info;
    }
  } else if ((it->second.find("\"mtsCapabilityInfo\"") != std::string::npos) || (it->second.find("\"mtsAccessInfo\"") != std::string::npos)) {
    TrafficManagementAPI__TypesAndValues::MtsCapabilityInfo mts_cap_info;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    mts_cap_info.decode(TrafficManagementAPI__TypesAndValues::MtsCapabilityInfo_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.mtsCapabilityInfo() = mts_cap_info;
  } else if ((it->second.find("\"mtsSessionInfo\"") != std::string::npos) || (it->second.find("\"mtsMode\"") != std::string::npos)) {
    TrafficManagementAPI__TypesAndValues::MtsSessionInfo mts_session_info;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    mts_session_info.decode(TrafficManagementAPI__TypesAndValues::MtsSessionInfo_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.mtsSessionInfo() = mts_session_info;
  } else if ((it->second.find("\"uuUnicastProvisioningInfo\"") != std::string::npos) || (it->second.find("\"proInfoUuUnicast\"") != std::string::npos)) {
    V2XInformationServiceAPI__TypesAndValues::UuUnicastProvisioningInfo uu_unicast_provisioning_info;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    uu_unicast_provisioning_info.decode(V2XInformationServiceAPI__TypesAndValues::UuUnicastProvisioningInfo_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.uuUnicastProvisioningInfo() = uu_unicast_provisioning_info;
  } else if ((it->second.find("\"uuMbmsProvisioningInfo\"") != std::string::npos) || (it->second.find("\"proInfoUuMbms\"") != std::string::npos)) {
    V2XInformationServiceAPI__TypesAndValues::UuMbmsProvisioningInfo uu_mbms_provisioning_info;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    uu_mbms_provisioning_info.decode(V2XInformationServiceAPI__TypesAndValues::UuMbmsProvisioningInfo_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.uuMbmsProvisioningInfo() = uu_mbms_provisioning_info;
  } else if ((it->second.find("\"pc5ProvisioningInfo\"") != std::string::npos) || (it->second.find("\"proInfoPc5\"") != std::string::npos)) {
    V2XInformationServiceAPI__TypesAndValues::Pc5ProvisioningInfo pc5_provisioning_info;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    pc5_provisioning_info.decode(V2XInformationServiceAPI__TypesAndValues::Pc5ProvisioningInfo_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.pc5ProvisioningInfo() = pc5_provisioning_info;
  } else if ((it->second.find("\"predictedQos\"") != std::string::npos) || (it->second.find("\"locationGranularity\"") != std::string::npos)) {
    V2XInformationServiceAPI__TypesAndValues::PredictedQos predicted_qos;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    predicted_qos.decode(V2XInformationServiceAPI__TypesAndValues::PredictedQos_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.predictedQos() = predicted_qos;
  } else if (it->second.find("\"ProvChgUuUniSubscription\"") != std::string::npos) {
    V2XInformationServiceAPI__TypesAndValues::ProvChgUuUniSubscription prov_chg_uu_uni_subscription;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    prov_chg_uu_uni_subscription.decode(V2XInformationServiceAPI__TypesAndValues::ProvChgUuUniSubscription_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.provChgUuUniSubscription() = prov_chg_uu_uni_subscription;
  } else if (it->second.find("\"ProvChgUuMbmsSubscription\"") != std::string::npos) {
    V2XInformationServiceAPI__TypesAndValues::ProvChgUuMbmsSubscription prov_chg_uu_mbms_subscription;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    prov_chg_uu_mbms_subscription.decode(V2XInformationServiceAPI__TypesAndValues::ProvChgUuMbmsSubscription_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.provChgUuMbmsSubscription() = prov_chg_uu_mbms_subscription;
  } else if (it->second.find("\"ProvChgPc5Subscription\"") != std::string::npos) {
    V2XInformationServiceAPI__TypesAndValues::ProvChgPc5Subscription prov_chg_pc5_subscription;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    prov_chg_pc5_subscription.decode(V2XInformationServiceAPI__TypesAndValues::ProvChgPc5Subscription_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.provChgPc5Subscription() = prov_chg_pc5_subscription;


  } else if (it->second.find("\"ProvChgUuUniNotification\"") != std::string::npos) {
    V2XInformationServiceAPI__TypesAndValues::ProvChgUuUniNotification prov_chg_uu_uni_notification;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    prov_chg_uu_uni_notification.decode(V2XInformationServiceAPI__TypesAndValues::ProvChgUuUniNotification_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.provChgUuUniNotification() = prov_chg_uu_uni_notification;
  } else if (it->second.find("\"ProvChgUuMbmsNotification\"") != std::string::npos) {
    V2XInformationServiceAPI__TypesAndValues::ProvChgUuMbmsNotification prov_chg_uu_mbms_notification;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    prov_chg_uu_mbms_notification.decode(V2XInformationServiceAPI__TypesAndValues::ProvChgUuMbmsNotification_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.provChgUuMbmsNotification() = prov_chg_uu_mbms_notification;
  } else if (it->second.find("\"provChgPc5Notification\"") != std::string::npos) {
    V2XInformationServiceAPI__TypesAndValues::ProvChgPc5Notification prov_chg_pc5_notification;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    prov_chg_pc5_notification.decode(V2XInformationServiceAPI__TypesAndValues::ProvChgPc5Notification_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.provChgPc5Notification() = prov_chg_pc5_notification;


  } else if (it->second.find("\"AssocStaSubscription\"") != std::string::npos) {
    if (it->second.find("\"subscription\"") != std::string::npos) { // SubscriptionLinkList
      WlanInformationAPI__TypesAndValues::SubscriptionLinkList subscription_link_list;
      TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
      subscription_link_list.decode(WlanInformationAPI__TypesAndValues::SubscriptionLinkList_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
      msg.subscriptionLinkList__wlan() = subscription_link_list;
    } else { // AssocStaSubscription
      WlanInformationAPI__TypesAndValues::AssocStaSubscription assoc_sta_subscription;
      TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
      assoc_sta_subscription.decode(WlanInformationAPI__TypesAndValues::AssocStaSubscription_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
      msg.assocStaSubscription() = assoc_sta_subscription;
    }
  } else if (it->second.find("\"StaDataRateSubscription\"") != std::string::npos) {
    WlanInformationAPI__TypesAndValues::StaDataRateSubscription sta_data_rate_subscription;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    sta_data_rate_subscription.decode(WlanInformationAPI__TypesAndValues::StaDataRateSubscription_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.staDataRateSubscription() = sta_data_rate_subscription;
  } else if ((it->second.find("\"_links\"") != std::string::npos) && (it->second.find("subscriptions") != std::string::npos)) { // SubscriptionLinkList
    // decode_str: {"_links":{"self":{"href":"https://try-mec.etsi.org/sbxpb4c4k2/wai/v2/subscriptions"}}}
    WlanInformationAPI__TypesAndValues::SubscriptionLinkList subscription_link_list;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    subscription_link_list.decode(WlanInformationAPI__TypesAndValues::SubscriptionLinkList_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.subscriptionLinkList__wlan() = subscription_link_list;
  } else if ((it->second.find("\"apInfo\"") != std::string::npos) || (it->second.find("\"apId\"") != std::string::npos)) {
    WlanInformationAPI__TypesAndValues::ApInfoList ap_info_list;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    ap_info_list.decode(WlanInformationAPI__TypesAndValues::ApInfoList_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.apInfoList() = ap_info_list;

  } else if ((it->second.find("\"staInfo\"") != std::string::npos) || (it->second.find("\"rssi\"") != std::string::npos)) {
    WlanInformationAPI__TypesAndValues::StaInfoList sta_info_list;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    sta_info_list.decode(WlanInformationAPI__TypesAndValues::StaInfoList_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.staInfoList() = sta_info_list;
  } else if (it->second.find("\"AssocStaNotification\"") != std::string::npos) {
    WlanInformationAPI__TypesAndValues::AssocStaNotification assoc_sta_notification;
    TTCN_Buffer decoding_buffer_(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));
    assoc_sta_notification.decode(WlanInformationAPI__TypesAndValues::AssocStaNotification_descr_, decoding_buffer_, TTCN_EncDec::CT_JSON);
    msg.assocStaNotification() = assoc_sta_notification;









  } else if (it->second.find("\"detail\"") != std::string::npos) {
    LibMec__TypesAndValues::ProblemDetails problem_details;
    problem_details.decode(LibMec__TypesAndValues::ProblemDetails_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.problemDetails() = problem_details;
  } else {
    loggers::get_instance().warning("json_codec::decode: Unsupported variant");
    return -1;
  }

  loggers::get_instance().log_msg("<<< json_codec::decode: ", (const Base_Type&)msg);
  return 0;
}

json_codec_factory json_codec_factory::_f;
