#include <stdexcept>
#include <regex>
#include <string>

#include "json_codec_factory.hh"

#include "loggers.hh"

#include "LibItsHttp_JsonMessageBodyTypes.hh"

int json_codec::encode (const LibItsHttp__JsonMessageBodyTypes::JsonBody& msg, OCTETSTRING& data)
{
  loggers::get_instance().log_msg(">>> json_codec::encode: ", (const Base_Type&)msg);

  TTCN_EncDec::clear_error();
  TTCN_EncDec::set_error_behavior(TTCN_EncDec::ET_ALL, TTCN_EncDec::EB_DEFAULT);
  TTCN_Buffer encoding_buffer;

  if (msg.ischosen(LibItsHttp__JsonMessageBodyTypes::JsonBody::ALT_ueIdentityTagInfo)) {
    const UEidentityAPI__TypesAndValues::UeIdentityTagInfo& ue_identity_tag_info = msg.ueIdentityTagInfo();
    ue_identity_tag_info.encode(UEidentityAPI__TypesAndValues::UeIdentityTagInfo_descr_, encoding_buffer, TTCN_EncDec::CT_JSON);
    data = OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data());
  } else {
    loggers::get_instance().error("json_codec::encode: Not supported");
  }
  
  loggers::get_instance().log("<<< json_codec::encode");
  return 0;
}

int json_codec::decode (const OCTETSTRING& p_data, LibItsHttp__JsonMessageBodyTypes::JsonBody& msg, params* p_params)
{
  loggers::get_instance().log_msg(">>> json_codec::decode: p_data=", p_data);

  // Sanity checks
  params::const_iterator it;
  if (p_params == nullptr) {
    loggers::get_instance().warning("json_codec::decode: Failed to access p_params (null pointer)");
    return -1;
  } else {
    it = p_params->find("decode_str");
    if (it == p_params->cend()) {
      loggers::get_instance().warning("json_codec::decode: Failed to access p_params item (decode_str)");
      return -1;
    }
  }
  
  TTCN_EncDec::set_error_behavior(TTCN_EncDec::ET_ALL, TTCN_EncDec::EB_DEFAULT);
  TTCN_EncDec::clear_error();
  // Remove data structure name...
  int idx_begin = it->second.find(":");
  int idx_end = it->second.rfind("}") - 1; // Remove the last '}'
  std::string str = it->second.substr(idx_begin + 1, idx_end - idx_begin);
  // ..and create the decoding buffer
  TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));

  if (it->second.find("\"userList\"") != std::string::npos) { // Be carefull to the order
                                                                // TODO To be refined, find("\"userList\"") is not optimal
      LocationAPI__TypesAndValues::UserList user_list;
      user_list.decode(LocationAPI__TypesAndValues::UserList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
      msg.userList() = user_list;
    } else if (it->second.find("\"accessPointList\"") != std::string::npos) { // Be carefull to the order
                                                                       // TODO To be refined, find("\"accessPointList\"") is not optimal
    LocationAPI__TypesAndValues::AccessPointList access_point_list;
    access_point_list.decode(LocationAPI__TypesAndValues::AccessPointList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.accessPointList() = access_point_list;
    } else if (it->second.find("\"SubscriptionLinkList\"") != std::string::npos) { // Be carefull to the order
                                                                                   // TODO To be refined, find("\"accessPointList\"") is not optimal
    RnisAPI__TypesAndValues::SubscriptionLinkList subscription_link_list;
    subscription_link_list.decode(RnisAPI__TypesAndValues::SubscriptionLinkList_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.subscriptionLinkList() = subscription_link_list;
  } else if (it->second.find("\"userInfo\"") != std::string::npos) {
    LocationAPI__TypesAndValues::UserInfo user_info;
    user_info.decode(LocationAPI__TypesAndValues::UserInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.userInfo() = user_info;
  } else if (it->second.find("\"ueIdentityTagInfo\"") != std::string::npos) {
    UEidentityAPI__TypesAndValues::UeIdentityTagInfo ue_identity_tag_info;
    ue_identity_tag_info.decode(UEidentityAPI__TypesAndValues::UeIdentityTagInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.ueIdentityTagInfo() = ue_identity_tag_info;
  } else if (it->second.find("\"bwInfo\"") != std::string::npos) {
    BwManagementAPI__TypesAndValues::BwInfo bw_info;
    bw_info.decode(BwManagementAPI__TypesAndValues::BwInfo_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.bwInfo() = bw_info;
  } else if (it->second.find("\"problemDetails\"") != std::string::npos) {
    UEidentityAPI__TypesAndValues::ProblemDetails problem_details;
    problem_details.decode(UEidentityAPI__TypesAndValues::ProblemDetails_descr_, decoding_buffer, TTCN_EncDec::CT_JSON);
    msg.problemDetails__ue__identity() = problem_details;
  } else {
    loggers::get_instance().warning("json_codec::decode: Unsupported variant");
    return -1;
  }
  
  loggers::get_instance().log_msg("<<< json_codec::decode: ", (const Base_Type&)msg);
  return 0;
}

json_codec_factory json_codec_factory::_f;
