Y''[Documentation]   robot --outputdir ../../../outputs ./CustomerSelfServiceEnablement.robot
...    Test Suite to validate CustomerSelfServiceEnablement Service API operations.

*** Settings ***
Resource    environment/variables.txt
Resource    ../../../pics.txt
Resource    ../../../GenericKeywords.robot
Library     REST    ${SCHEMA}://${HOST}:${PORT}    ssl_verify=false
Library     OperatingSystem
Library     libraries/Server.py
Library     libraries/StressGenerator.py
Library     JSONLibrary
Library     String    
Library     Collections
Library     DateTime 

*** Test Cases ***
TC_MEC_MEC048_MEO_CSE_001_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_001_OK
    ...    Check that the IUT responds with a list of all available tenantInfo when requested to a CSE - No query parameters
    ...    ETSI GS MEC 048 v3.1.1, clause 7.3.3.1
    ...    ETSI GS MEC 048 v3.1.1, clause 6.2.2 (TenantInfo)
    [Tags]    PIC_SERVICES
    [Setup]   Register mutiple tenant Info and get tenant Ids   TenantInfo    TenantInfo2    TenantInfo3
    Retrieve all tenant Info resources
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  TenantInfoList
    [Teardown]  Remove mutiple tenant Info    ${TENANT_ID_1}    ${TENANT_ID_2}    ${TENANT_ID_3}

TC_MEC_MEC048_MEO_CSE_001_OK_02
    [Documentation]    TP_MEC_MEC048_MEO_CSE_001_OK_02
    ...    Check that the IUT responds with a list of tenantInfo when queried with customerId as query parameter
    ...    ETSI GS MEC 048 v3.1.1, clause 7.3.3.1
    ...    ETSI GS MEC 048 v3.1.1, clause 6.2.2 (TenantInfo)
    [Tags]    PIC_SERVICES
    [Setup]   Register mutiple tenant Info and get tenant Ids   TenantInfo    TenantInfo2    TenantInfo3
    Retrieve all tenant info resources with query parameters    customerId=${CUSTOMER_ID_1}
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  TenantInfoList
    FOR  ${tenant}  IN  @{response['body']}
      Should Be Equal As Strings    ${tenant}[tenantId]      ${TENANT_ID_1}
    END
    [Teardown]  Remove mutiple tenant Info    ${TENANT_ID_1}    ${TENANT_ID_2}    ${TENANT_ID_3}

TC_MEC_MEC048_MEO_CSE_001_OK_03
    [Documentation]    TP_MEC_MEC048_MEO_CSE_001_OK_03
    ...    Check that the IUT responds with a list of tenantInfo when queried with customerName as query parameter
    ...    ETSI GS MEC 048 v3.1.1, clause 7.3.3.1
    ...    ETSI GS MEC 048 v3.1.1, clause 6.2.2 (TenantInfo)
    [Tags]    PIC_SERVICES
    [Setup]   Register mutiple tenant Info and get tenant Ids   TenantInfo    TenantInfo2    TenantInfo3
    Retrieve all tenant info resources with query parameters    customerName=${CUSTOMER_NAME_1}
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  TenantInfoList
    FOR  ${tenant}  IN  @{response['body']}
      Should Be Equal As Strings    ${tenant}[customerName]      ${CUSTOMER_NAME_1}
    END
    [Teardown]  Remove mutiple tenant Info    ${TENANT_ID_1}    ${TENANT_ID_2}    ${TENANT_ID_3}

TC_MEC_MEC048_MEO_CSE_002_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_002_OK
    ...    Check that the IUT creates a new tenantInfo when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.3.3.4
    ...    ETSI GS MEC 048 v3.1.1, clause 6.2.2 (TenantInfo)
    [Tags]    PIC_SERVICES
    ##No test setup needed because the tenantId is generated by IUT
    Register Tenant Info    TenantInfo
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is  TenantInfo  
    ${CUSTOMER_ID_VALUE}   Get value entry from JSON file    TenantInfo  customerId
    ${TENANT_NAME_VALUE}   Get value entry from JSON file    TenantInfo  tenantName

    Should Be Equal As Strings    ${response['body']['customerId']}       ${CUSTOMER_ID_VALUE}
    Should Be Equal As Strings    ${response['body']['tenantName']}       ${TENANT_NAME_VALUE}

    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}  

TC_MEC_MEC048_MEO_CSE_002_BR
    [Documentation]    TP_MEC_MEC048_MEO_CSE_002_BR
    ...    Check that the IUT responds with an error on creating a tenantInfo with both resourceUseInfo and siteList
    ...    ETSI GS MEC 048 v3.1.1, clause 7.3.3.4 and 6.2.2 (TenantInfo)
    [Tags]    PIC_SERVICES
    Register Tenant Info    TenantInfoBR
    Check HTTP Response Status Code Is    400

TC_MEC_MEC048_MEO_CSE_003_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_003_OK
    ...    Check that the IUT responds with a specific tenantInfo when requested to a MEC Consumer
    ...    ETSI GS MEC 048 v3.1.1, clause 7.3.3.1 and 6.2.2 (TenantInfo)
    [Tags]    PIC_SERVICES
    [Setup]   Register Tenant Info    TenantInfo
    Retrieve specific tenant info resource   ${TENANT_ID_1} 
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  TenantInfo
    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_003_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_003_NF
    ...    Check that the IUT returns an error when requesting a non-existing tenantInfo to a MEC Consumer
    ...    ETSI GS MEC 048 v3.1.1, clause 7.4.3.1 and 6.2.2
    [Tags]    PIC_SERVICES
    Retrieve specific tenant info resource    ${NOT_EXISTING_TENANT_ID} 
    Check HTTP Response Status Code Is    404

TC_MEC_MEC048_MEO_CSE_004_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_004_OK
    ...    Check that the IUT updates an existing tenantInfo when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.4.3.5 and 6.2.2
    [Tags]    PIC_SERVICES
    [Setup]   Register tenant Info    TenantInfo
    Update tenant Info    ${TENANT_ID_1}    TenantInfoUpdate
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  TenantInfo
    ${CUSTOMER_ID_VALUE}   Get value entry from JSON file    TenantInfoUpdate  customerId
    ${TENANT_NAME_VALUE}   Get value entry from JSON file    TenantInfoUpdate  tenantName
    ${CUSTOMER_NAME_VALUE}   Get value entry from JSON file    TenantInfoUpdate  customerName

    Should Be Equal As Strings    ${response['body']['customerId']}       ${CUSTOMER_ID_VALUE}
    Should Be Equal As Strings    ${response['body']['tenantName']}       ${TENANT_NAME_VALUE}
    Should Be Equal As Strings    ${response['body']['customerName']}       ${CUSTOMER_NAME_VALUE}

    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_004_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_004_NF
    ...    Check that the IUT returns an error on updating a not existing tenantInfo requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.4.3.5 and 6.2.2
    [Tags]    PIC_SERVICES
    Update tenant Info    ${NOT_EXISTING_TENANT_ID}    TenantInfoUpdate
    Check HTTP Response Status Code Is    404

TC_MEC_MEC048_MEO_CSE_004_BR
    [Documentation]    TP_MEC_MEC048_MEO_CSE_004_BR
    ...    Check that the IUT returns an error on updating with invalid values tenantInfo requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.4.3.2 and 6.2.2
    [Setup]   Register tenant Info    TenantInfo
    Update tenant Info    ${TENANT_ID_1}    TenantInfoBR
    Check HTTP Response Status Code Is    400
    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_005_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_005_OK
    ...    Check that the IUT removes a tenantInfo when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.4.3.5
    [Tags]    PIC_SERVICES
    [Setup]   Register tenant Info    TenantInfo
    Remove specific tenant info resource  ${TENANT_ID_1}  
    Check HTTP Response Status Code Is    204

TC_MEC_MEC048_MEO_CSE_005_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_005_NF
    ...    Check that the IUT returns an error on deleting a not existing tenantInfo when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.4.3.5
    [Tags]    PIC_SERVICES
    Remove specific tenant info resource  ${NOT_EXISTING_TENANT_ID}  
    Check HTTP Response Status Code Is    404 

TC_MEC_MEC048_MEO_CSE_006_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_006_OK
    ...    Check that the IUT returns a resourceQuotaInfo for a specific tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.5.3.1, clause 6.2.3
    [Tags]    PIC_SERVICES
    [Setup]   Register tenant Info    TenantInfo
    Register resource quota Info    ${TENANT_ID_1}    ResourceQuotaInfo
    Retrieve specific resource quota info resource   ${TENANT_ID_1} 
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  ResourceQuotaInfo
    Remove specific resource quota info resource    ${TENANT_ID_1}
    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_006_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_006_NF
    ...    Check that the IUT returns an error on retrieving resourceQuotaInfo for a not existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.5.3.1, clause 6.2.3
    [Tags]    PIC_SERVICES
    Retrieve specific resource quota info resource   ${NOT_EXISTING_TENANT_ID} 
    Check HTTP Response Status Code Is    404

# TO DO
TC_MEC_MEC048_MEO_CSE_007_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_007_OK
    ...    Check that the IUT updates the resourceQuotaInfo for an existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.5.3.2, clause 6.2.3
    [Tags]    PIC_SERVICES
    [Setup]   Register tenant Info    TenantInfo
    Register resource quota Info    ${TENANT_ID_1}    ResourceQuotaInfo
    Update resource quota Info   ${TENANT_ID_1}    ResourceQuotaInfoUpdate
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  ResourceQuotaInfo

    ${CPU_QUOTA_VALUE}   Get value entry from JSON file    ResourceQuotaInfoUpdate  cpuQuota
    ${MEMORY_QUOTA_VALUE}   Get value entry from JSON file    ResourceQuotaInfoUpdate  memoryQuota
    ${DISK_QUOTA_VALUE}   Get value entry from JSON file    ResourceQuotaInfoUpdate  diskQuota

    Should Be Equal As Strings    ${response['body']['cpuQuota']}       ${CPU_QUOTA_VALUE}
    Should Be Equal As Strings    ${response['body']['memoryQuota']}       ${MEMORY_QUOTA_VALUE}
    Should Be Equal As Strings    ${response['body']['diskQuota']}       ${DISK_QUOTA_VALUE}

    Remove specific resource quota info resource    ${TENANT_ID_1}
    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

# TO DO
TC_MEC_MEC048_MEO_CSE_007_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_007_NF
    ...    Check that the IUT returns an error on updating the resourceQuotaInfo for a not existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.5.3.2, clause 6.2.3
    [Tags]    PIC_SERVICES
    Update resource quota Info   ${NOT_EXISTING_TENANT_ID}    ResourceQuotaInfoUpdate
    Check HTTP Response Status Code Is    404
TC_MEC_MEC048_MEO_CSE_008_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_008_OK
    ...    Check that the IUT creates the resourceQuotaInfo for an existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.5.3.4, clause 6.2.3
    [Tags]    PIC_SERVICES
    [Setup]   Register tenant Info    TenantInfo
    Register resource quota Info    ${TENANT_ID_1}    ResourceQuotaInfo
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is  ResourceQuotaInfo
    Remove specific resource quota info resource    ${TENANT_ID_1}
    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_008_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_008_NF
    ...    Check that the IUT returns an error on creating the resourceQuotaInfo for a not existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.5.3.4, clause 6.2.3
    [Tags]    PIC_SERVICES
    Register resource quota Info   ${NOT_EXISTING_TENANT_ID}    ResourceQuotaInfo
    Check HTTP Response Status Code Is    404

# MISATKE IN URL (TEST PURPOSE) ,TO DO
TC_MEC_MEC048_MEO_CSE_009_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_009_OK
    ...    Check that the IUT returns the list of per site resource quota for a specific tenant when requested to a CSE - No filter
    ...    ETSI GS MEC 048 v3.1.1, clause 7.6.3.1, clause 6.2.4
    [Tags]    PIC_SERVICES
    [Setup]   Register tenant Info    TenantInfo
    Register mutiple site resource quota Info and get site Ids    ${TENANT_ID_1}    SiteResourceQuotaInfo    SiteResourceQuotaInfo2    SiteResourceQuotaInfo3
    Retrieve all site resource quota info resource   ${TENANT_ID_1} 
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  SiteResourceQuotaInfoList
    Remove mutiple site resource quota Info    ${TENANT_ID_1}    ${SITE_ID_1}    ${SITE_ID_2}    ${SITE_ID_3}
    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_009_OK_01
    [Documentation]    TP_MEC_MEC048_MEO_CSE_009_OK_01
    ...    Check that the IUT returns the list of per site resource quota for a specific tenant when requested to a CSE - siteId filter
    ...    ETSI GS MEC 048 v3.1.1, clause 7.6.3.1, clause 6.2.4
    [Tags]    PIC_SERVICES
    [Setup]   Register tenant Info    TenantInfo
    Register mutiple site resource quota Info and get site Ids    ${TENANT_ID_1}    SiteResourceQuotaInfo    SiteResourceQuotaInfo2    SKIP
    Retrieve all site resource quota info resource with query parameters   ${TENANT_ID_1}    siteId=${SITE_ID_1}
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is  SiteResourceQuotaInfoList
    FOR  ${element}  IN  @{response['body']}
      Should Be Equal As Strings    ${element}[siteId]      ${SITE_ID_1}
    END
    Remove mutiple site resource quota Info    ${TENANT_ID_1}    ${SITE_ID_1}    ${SITE_ID_2}    SKIP
    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_009_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_009_NF
    ...    Check that the IUT returns an error on requesting site resource quota info on a not existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.6.3.1, clause 6.2.4
    [Tags]    PIC_SERVICES
    Retrieve all site resource quota info resource    ${NOT_EXISTING_TENANT_ID}
    Check HTTP Response Status Code Is    404

TC_MEC_MEC048_MEO_CSE_010_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_010_OK
    ...    Check that the IUT creates a site resource quota info for an existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.6.3.4, clause 6.2.4
    [Tags]    PIC_SERVICES
    [Setup]   Register tenant Info    TenantInfo
    Register site resource quota Info    ${TENANT_ID_1}    SiteResourceQuotaInfo
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is  SiteResourceQuotaInfo
    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${response}[body][siteId]
    [Teardown]  Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_010_BR
    [Documentation]    TP_MEC_MEC048_MEO_CSE_010_BR
    ...    Check that the IUT returns an error on creating a site resource quota with wrong parameters
    ...    ETSI GS MEC 048 v3.1.1, clause 7.6.3.4, clause 6.2.4
    [Tags]    PIC_SERVICES
    [Setup]    Register tenant Info    TenantInfo
    Register site resource quota Info    ${TENANT_ID_1}    SiteResourceQuotaInfoBR
    Check HTTP Response Status Code Is    400
    [Teardown]    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_011_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_011_OK
    ...    Check that the IUT returns the site resource quota for a specific tenant and site identifier when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.7.3.1, clause 6.2.4
    [Tags]    PIC_SERVICES
    [Setup]    Register multiple site resource quotas for tenant
    Retrieve specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    SiteResourceQuotaInfo
    Should Be Equal As Strings    ${response['body']['siteId']}    ${SITE_ID_1}
    Should Be Equal As Integers    ${response['body']['cpuQuota']}    ${CPU_QUOTA_1}
    Should Be Equal As Integers    ${response['body']['memoryQuota']}    ${MEMORY_QUOTA_1}
    Should Be Equal As Integers    ${response['body']['diskQuota']}    ${DISK_QUOTA_1}
    [Teardown]    Remove tenant and site resources

TC_MEC_MEC048_MEO_CSE_011_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_011_NF
    ...    Check that the IUT returns an error on requesting the site resource quota for a specific tenant and a not existing site identifier when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.7.3.1, clause 6.2.4
    [Tags]    PIC_SERVICES
    [Setup]    Register tenant Info    TenantInfo
    Retrieve specific site resource quota info resource    ${TENANT_ID_1}    ${NOT_EXISTING_TENANT_ID}
    Check HTTP Response Status Code Is    404
    [Teardown]    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_012_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_012_OK
    ...    Check that the IUT updates a site resource quota info for an existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.7.3.2, clause 6.2.4
    [Tags]    PIC_SERVICES
    [Setup]    Register site resource quota for tenant
    # Update the site resource quota with new values
    Update site resource quota Info    ${TENANT_ID_1}    ${SITE_ID_1}    SiteResourceQuotaInfoUpdate
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    SiteResourceQuotaInfo
    
    # Verify the updated values match what we sent
    ${CPU_QUOTA_VALUE}    Get value entry from JSON file    SiteResourceQuotaInfoUpdate    cpuQuota
    ${MEMORY_QUOTA_VALUE}    Get value entry from JSON file    SiteResourceQuotaInfoUpdate    memoryQuota
    ${DISK_QUOTA_VALUE}    Get value entry from JSON file    SiteResourceQuotaInfoUpdate    diskQuota
    
    Should Be Equal As Strings    ${response['body']['siteId']}    ${SITE_ID_1}
    Should Be Equal As Integers    ${response['body']['cpuQuota']}    ${CPU_QUOTA_VALUE}
    Should Be Equal As Integers    ${response['body']['memoryQuota']}    ${MEMORY_QUOTA_VALUE}
    Should Be Equal As Integers    ${response['body']['diskQuota']}    ${DISK_QUOTA_VALUE}
    
    [Teardown]    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_012_BR
    [Documentation]    TP_MEC_MEC048_MEO_CSE_012_BR
    ...    Check that the IUT returns an error updating a site resource quota info for an existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.7.3.2, clause 6.2.4
    [Tags]    PIC_SERVICES
    [Setup]    Register site resource quota for tenant
    # Attempt to update the site resource quota with invalid data
    Update site resource quota Info    ${TENANT_ID_1}    ${SITE_ID_1}    SiteResourceQuotaInfoBR
    Check HTTP Response Status Code Is    400
    [Teardown]    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_012_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_012_NF
    ...    Check that the IUT returns an error updating a site resource quota info for a not existing tenant when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.7.3.2, clause 6.2.4
    [Tags]    PIC_SERVICES
    # No setup needed as we're testing with a non-existing tenant
    Update site resource quota Info    ${NOT_EXISTING_TENANT_ID}    ${NOT_EXISTING_SITE_ID}     SiteResourceQuotaInfoUpdate
    Check HTTP Response Status Code Is    404
    # No teardown needed as no resources were created

TC_MEC_MEC048_MEO_CSE_013_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_013_OK
    ...    Check that the IUT returns the list of subscriptions when requested to a CSE - no filter
    ...    ETSI GS MEC 048 v3.1.1, clause 7.8.3.1, clause 6.3.4
    [Tags]    PIC_SERVICES
    [Setup]    Create test subscriptions
    Retrieve all subscriptions
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    SubscriptionLinkList
    # Verify _links.self is present and correct
    Should Be Equal As Strings    ${response['body']['_links']['self']['href']}    ${apiRoot}/${apiName}/${apiVersion}/subscriptions
    # Verify subscription items exist
    ${subscriptions}=    Get From Dictionary    ${response['body']}    subscription
    ${subscription_count}=    Get Length    ${subscriptions}
    Should Be True    ${subscription_count} >= 3
    # Verify the subscription types are as expected - at least one of each type
    ${resource_type_count}=    Count Subscriptions Of Type    ${subscriptions}    ResourceUsageSubscription
    ${site_resource_type_count}=    Count Subscriptions Of Type    ${subscriptions}    SiteResourceUsageSubscription
    Should Be True    ${resource_type_count} >= 2    Found only ${resource_type_count} ResourceUsageSubscription, expected at least 2
    Should Be True    ${site_resource_type_count} >= 1    Found only ${site_resource_type_count} SiteResourceUsageSubscription, expected at least 1
    [Teardown]    Remove test subscriptions

TC_MEC_MEC048_MEO_CSE_013_OK_02
    [Documentation]    TP_MEC_MEC048_MEO_CSE_013_OK_02
    ...    Check that the IUT returns the list of subscriptions when requested to a CSE - subscription_type filter
    ...    ETSI GS MEC 048 v3.1.1, clause 7.8.3.1, clause 6.3.4
    [Tags]    PIC_SERVICES
    [Setup]    Create test subscriptions
    # Retrieve subscriptions with subscription_type filter
    Retrieve subscriptions with filter    subscription_type=ResourceUsageSubscription
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    SubscriptionLinkList
    # Verify _links.self is present and includes the query parameter
    Should Be Equal As Strings    ${response['body']['_links']['self']['href']}    ${apiRoot}/${apiName}/${apiVersion}/subscriptions?subscription_type=ResourceUsageSubscription
    # Verify subscription items exist
    ${subscriptions}=    Get From Dictionary    ${response['body']}    subscription
    ${subscription_count}=    Get Length    ${subscriptions}
    # Should have at least the ResourceUsageSubscription types we created
    Should Be True    ${subscription_count} >= 2
    # Verify ONLY ResourceUsageSubscription types are returned (no SiteResourceUsageSubscription)
    FOR    ${subscription}    IN    @{subscriptions}
        Should Be Equal As Strings    ${subscription['subscriptionType']}    ResourceUsageSubscription
    END
    [Teardown]    Remove test subscriptions

TC_MEC_MEC048_MEO_CSE_014_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_014_OK
    ...    Check that the IUT creates a new subscription when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.8.3.4, clause 6.3.2, clause 6.3.3
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant to use for the subscription
    [Setup]    Register tenant Info    TenantInfo
    Set Suite Variable    ${TENANT_ID_1}    ${response['body']['tenantId']}
    Set Suite Variable    ${CUSTOMER_ID_1}    ${response['body']['customerId']}
    
    # Create a new ResourceUsageSubscription
    Create ResourceUsageSubscription    ResourceUsageSubscription
    
    # Verify response status code
    Check HTTP Response Status Code Is    201
    
    # Verify response body schema
    Check HTTP Response Body Json Schema Is    ResourceUsageSubscription
    
    # Verify Location header (should contain the URI of the created subscription)
    ${location_header}=    Get From Dictionary    ${response['headers']}    Location
    Should Not Be Empty    ${location_header}
    Should Match Regexp    ${location_header}    ${apiRoot}/${apiName}/${apiVersion}/subscriptions/\\w+
    
    # Store subscription ID for cleanup
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    Set Suite Variable    ${SUBSCRIPTION_ID}    ${subscription_uri}
    
    # Verify response body content
    Should Be Equal As Strings    ${response['body']['subscriptionType']}    ResourceUsageSubscription
    Should Be Equal As Strings    ${response['body']['customerId']}    ${CUSTOMER_ID_1}
    Should Be Equal As Strings    ${response['body']['tenantId']}    ${TENANT_ID_1}
    Should Be Equal As Strings    ${response['body']['callbackReference']}    some/uri
    
    # Verify notification trigger details
    Should Be Equal As Integers    ${response['body']['notificationTrigger']['triggerType']}    10
    Should Be Equal As Integers    ${response['body']['notificationTrigger']['threshold']}    4
    # In the schema, greaterOrLess is a boolean, but in the specification it's 0/1
    # This could be handled either way depending on your implementation
    Should Be Equal    ${response['body']['notificationTrigger']['greaterOrLess']}    ${FALSE}
    
    # Verify self-link in response
    Should Not Be Empty    ${response['body']['_links']['self']['href']}
    
    # Teardown - clean up the created resources
    [Teardown]    Run Keywords
    ...    Delete Created Subscription    AND
    ...    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_014_OK_02
    [Documentation]    TP_MEC_MEC048_MEO_CSE_014_OK_02
    ...    Check that the IUT creates a new subscription when requested to a CSE - SiteResourceUsageSubscription
    ...    ETSI GS MEC 048 v3.1.1, clause 7.8.3.4, clause 6.3.2, clause 6.3.3
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant and site resources to use for the subscription
    [Setup]    Register site resource quota for tenant
    
    # Create a new SiteResourceUsageSubscription
    Register site Resource subscription   SiteResourceUsageSubscription1
    
    # Verify response status code
    Check HTTP Response Status Code Is    201
    
    # Verify response body schema
    Check HTTP Response Body Json Schema Is    SiteResourceUsageSubscription
    
    # Verify Location header (should contain the URI of the created subscription)
    ${location_header}=    Get From Dictionary    ${response['headers']}    Location
    Should Not Be Empty    ${location_header}
    Should Match Regexp    ${location_header}    ${apiRoot}/${apiName}/${apiVersion}/subscriptions/\\w+
    
    # Store subscription ID for cleanup
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    Set Suite Variable    ${SUBSCRIPTION_ID}    ${subscription_uri}
    
    # Verify response body content
    Should Be Equal As Strings    ${response['body']['subscriptionType']}    SiteResourceUsageSubscription
    Should Be Equal As Strings    ${response['body']['customerId']}    ${CUSTOMER_ID_1}
    Should Be Equal As Strings    ${response['body']['tenantId']}    ${TENANT_ID_1}
    Should Be Equal As Strings    ${response['body']['callbackReference']}    some/uri
    
    # Verify siteList contains our test site
    ${siteList}=    Get From Dictionary    ${response['body']}    siteList
    List Should Contain Value    ${siteList}    ${SITE_ID_1}
    
    # Verify notification trigger details
    Should Be Equal As Integers    ${response['body']['notificationTrigger']['triggerType']}    10
    Should Be Equal As Integers    ${response['body']['notificationTrigger']['threshold']}    4
    Should Be Equal    ${response['body']['notificationTrigger']['greaterOrLess']}    ${FALSE}
    
    # Verify self-link in response
    Should Not Be Empty    ${response['body']['_links']['self']['href']}
    
    # Teardown - clean up the created resources
    [Teardown]    Run Keywords
    ...    Delete Created Subscription    AND
    ...    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}    AND
    ...    Remove specific tenant info resource    ${TENANT_ID_1}
TC_MEC_MEC048_MEO_CSE_014_BR
    [Documentation]    TP_MEC_MEC048_MEO_CSE_014_BR
    ...    Check that the IUT returns an error on creating a new subscription when requested to a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.8.3.4, clause 6.3.2, clause 6.3.3
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant to use for the subscription
    [Setup]    Register tenant Info    TenantInfo
    Set Suite Variable    ${TENANT_ID_1}    ${response['body']['tenantId']}
    Set Suite Variable    ${CUSTOMER_ID_1}    ${response['body']['customerId']}
    
    # Attempt to create a subscription with missing required parameters
    Register subscription    ResourceUsageSubscriptionBR
    
    # Verify that a 400 Bad Request error is returned
    Check HTTP Response Status Code Is    400
    
    # Teardown - clean up the created tenant
    [Teardown]    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_015_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_015_OK
    ...    Check that the IUT returns a subscription when requested by a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.9.3.1, clause 6.3.2, clause 6.3.3
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant and site resources
    [Setup]    Run Keywords
    ...    Register site resource quota for tenant
    
    Register site Resource subscription    SiteResourceUsageSubscription
    
    # Verify subscription creation
    Check HTTP Response Status Code Is    201
    Check HTTP Response Body Json Schema Is    SiteResourceUsageSubscription
    
    # Save the subscription ID for retrieval and cleanup
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    ${subscription_id}=    Evaluate    "${subscription_uri}".split("/")[-1]
    Run Keyword If    '${subscription_id}' == ''    Fail    Failed to extract subscription ID from ${subscription_uri}
    Set Suite Variable    ${SUBSCRIPTION_ID}    ${subscription_id}
    
    # Retrieve the specific subscription
    Retrieve specific subscription    ${subscription_id}
    
    # Verify response status code
    Check HTTP Response Status Code Is    200
    
    # Verify response body schema
    Check HTTP Response Body Json Schema Is    SiteResourceUsageSubscription
    
    # Verify response body content
    Should Be Equal As Strings    ${response['body']['subscriptionType']}    SiteResourceUsageSubscription
    Should Be Equal As Strings    ${response['body']['customerId']}    ${CUSTOMER_ID_1}
    Should Be Equal As Strings    ${response['body']['tenantId']}    ${TENANT_ID_1}
    Should Be Equal As Strings    ${response['body']['callbackReference']}    some/uri
    
    # Verify siteList contains our test site
    ${siteList}=    Get From Dictionary    ${response['body']}    siteList
    List Should Contain Value    ${siteList}    ${SITE_ID_1}
    
    # Verify notification trigger details
    Should Be Equal As Integers    ${response['body']['notificationTrigger']['triggerType']}    10
    Should Be Equal As Integers    ${response['body']['notificationTrigger']['threshold']}    4
    Should Be Equal    ${response['body']['notificationTrigger']['greaterOrLess']}    ${FALSE}
    
    # Verify self-link in response
    Should Not Be Empty    ${response['body']['_links']['self']['href']}
    Should Be Equal As Strings    ${response['body']['_links']['self']['href']}    ${subscription_uri}
    
    # Teardown - clean up the created resources
    [Teardown]    Run Keywords
    ...    Run Keyword If    '${SUBSCRIPTION_ID}' != '' and '${SUBSCRIPTION_ID}' != 'None'    Delete Created Subscription
    ...    AND    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}
    ...    AND    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_015_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_015_NF
    ...    Check that the IUT returns an error on requesting a not existing subscription when requested by a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.9.3.1, clause 6.3.2, clause 6.3.3
    [Tags]    PIC_SERVICES
    
    # Attempt to retrieve a non-existent subscription
    Retrieve specific subscription    ${NOT_EXISTING_SUBSCRIPTION_ID}
    
    # Verify response status code is 404 Not Found
    Check HTTP Response Status Code Is    404

TC_MEC_MEC048_MEO_CSE_016_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_016_OK
    ...    Check that the IUT updates an existing subscription when requested by a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.9.3.2, clause 6.3.2, clause 6.3.3
    [Tags]    PIC_SERVICES
    [Setup]    Register site resource quota for tenant
    Register site Resource subscription    SiteResourceUsageSubscription
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    ${subscription_id}=    Evaluate    "${subscription_uri}".split("/")[-1]
    Run Keyword If    '${subscription_id}' == ''    Fail    Failed to extract subscription ID from ${subscription_uri}
    Set Suite Variable    ${SUBSCRIPTION_ID}    ${subscription_id}
    Set Suite Variable    ${OLD_CUSTOMER_ID}    ${response['body']['customerId']}
    Register tenant Info    TenantInfo2    ${True}    TENANT_ID_2
    Set Suite Variable    ${NEW_CUSTOMER_ID}    ${response['body']['customerId']}
    Set Suite Variable    ${TENANT_ID_2}    ${response['body']['tenantId']}
    Update subscription    ${subscription_id}    SiteResourceUsageSubscriptionUpdate
    Check HTTP Response Status Code Is    200
    Check HTTP Response Body Json Schema Is    SiteResourceUsageSubscription
    Should Be Equal As Strings    ${response['body']['subscriptionType']}    SiteResourceUsageSubscription
    Should Be Equal As Strings    ${response['body']['customerId']}    ${NEW_CUSTOMER_ID}
    Should Not Be Equal As Strings    ${response['body']['customerId']}    ${OLD_CUSTOMER_ID}
    Should Be Equal As Strings    ${response['body']['tenantId']}    ${TENANT_ID_1}
    Should Be Equal As Strings    ${response['body']['callbackReference']}    some/uri/updated
    ${siteList}=    Get From Dictionary    ${response['body']}    siteList
    List Should Contain Value    ${siteList}    ${SITE_ID_1}
    Should Be Equal As Integers    ${response['body']['notificationTrigger']['triggerType']}    10
    Should Be Equal As Integers    ${response['body']['notificationTrigger']['threshold']}    4
    Should Be Equal    ${response['body']['notificationTrigger']['greaterOrLess']}    ${FALSE}
    Should Not Be Empty    ${response['body']['_links']['self']['href']}
    Should Be Equal As Strings    ${response['body']['_links']['self']['href']}    ${subscription_uri}
    [Teardown]    Run Keywords
    ...    Delete Created Subscription    AND
    ...    Run Keyword If    '${SITE_ID_1}' != '' and '${SITE_ID_1}' != 'None'    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}    AND
    ...    Remove specific tenant info resource    ${TENANT_ID_1}    AND
    ...    Remove specific tenant info resource    ${TENANT_ID_2}

TC_MEC_MEC048_MEO_CSE_016_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_016_NF
    ...    Check that the IUT returns an error updating an not existing subscription when requested by a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.9.3.2, clause 6.3.2, clause 6.3.3
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant and site resources for the update request
    [Setup]    Register site resource quota for tenant
    
    # Create a JSON template for another tenant to get a customer ID
    Register tenant Info    TenantInfo2
    Set Suite Variable    ${NEW_CUSTOMER_ID}    ${response['body']['customerId']}
    Set Suite Variable    ${TENANT_ID_2}    ${response['body']['tenantId']}
    
    # Attempt to update a non-existent subscription
    Update subscription    ${NOT_EXISTING_SUBSCRIPTION_ID}    SiteResourceUsageSubscriptionUpdate
    
    # Verify response status code is 404 Not Found
    Check HTTP Response Status Code Is    404
    
    # Teardown - clean up the created resources
    [Teardown]    Run Keywords
    ...    Run Keyword If    '${SITE_ID_1}' != '' and '${SITE_ID_1}' != 'None'    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}    AND
    ...    Remove specific tenant info resource    ${TENANT_ID_1}    AND
    ...    Remove specific tenant info resource    ${TENANT_ID_2}

TC_MEC_MEC048_MEO_CSE_016_BR
    [Documentation]    TP_MEC_MEC048_MEO_CSE_016_BR
    ...    Check that the IUT returns an error updating an existing subscription with wrong parameters when requested by a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.9.3.2, clause 6.3.2, clause 6.3.3
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant, site resources, and a valid subscription
    [Setup]    Register site resource quota for tenant
    
    # Create a SiteResourceUsageSubscription to be updated later
    Register subscription    SiteResourceUsageSubscription
    
    # Save the subscription ID for update and cleanup
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    ${subscription_id}=    Fetch From Right    ${subscription_uri}    /subscriptions/
    Set Suite Variable    ${SUBSCRIPTION_ID}    ${subscription_id}
    
    # Create another tenant to get a new customer ID for the update
    Register tenant Info    TenantInfo2    ${True}    TENANT_ID_2
    Set Suite Variable    ${NEW_CUSTOMER_ID}    ${response['body']['customerId']}
    Set Suite Variable    ${TENANT_ID_2}    ${response['body']['tenantId']}
    
    # Attempt to update the subscription with invalid parameters (missing both callbackReference and websockNotifConfig)
    Update subscription    ${subscription_id}    SiteResourceUsageSubscriptionBR
    
    # Verify response status code is 400 Bad Request
    Check HTTP Response Status Code Is    400
    
    # Teardown - clean up the created resources
    [Teardown]    Run Keywords
    ...    Run Keyword If    '${SUBSCRIPTION_ID}' != '' and '${SUBSCRIPTION_ID}' != 'None'    Delete Created Subscription    AND
    ...    Run Keyword If    '${SITE_ID_1}' != '' and '${SITE_ID_1}' != 'None'    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}    AND
    ...    Run Keyword If    '${TENANT_ID_1}' != '' and '${TENANT_ID_1}' != 'None'    Remove specific tenant info resource    ${TENANT_ID_1}    AND
    ...    Run Keyword If    '${TENANT_ID_2}' != '' and '${TENANT_ID_2}' != 'None'    Remove specific tenant info resource    ${TENANT_ID_2}

TC_MEC_MEC048_MEO_CSE_017_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_017_OK
    ...    Check that the IUT deletes an existing subscription when requested by a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.9.3.2
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant, site resources, and subscription
    [Setup]    Register site resource quota for tenant
    
    # Create a SiteResourceUsageSubscription to be deleted
    Register subscription    SiteResourceUsageSubscription
    
    # Save the subscription ID for deletion
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    ${subscription_id}=    Fetch From Right    ${subscription_uri}    /subscriptions/
    
    # Delete the subscription
    Delete subscription    ${subscription_id}
    
    # Verify response status code is 204 No Content
    Check HTTP Response Status Code Is    204
    
    # Verify the subscription no longer exists (optional)
    Retrieve specific subscription    ${subscription_id}
    Check HTTP Response Status Code Is    404
    
    # Teardown - clean up the created resources (only tenant and site as subscription is deleted)
    [Teardown]    Run Keywords
    ...    Run Keyword If    '${SITE_ID_1}' != '' and '${SITE_ID_1}' != 'None'    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}    AND
    ...    Run Keyword If    '${TENANT_ID_1}' != '' and '${TENANT_ID_1}' != 'None'    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_017_NF
    [Documentation]    TP_MEC_MEC048_MEO_CSE_017_NF
    ...    Check that the IUT returns an error on a deletion of a not existing subscription when requested by a CSE
    ...    ETSI GS MEC 048 v3.1.1, clause 7.9.3.2
    [Tags]    PIC_SERVICES
    
    # Attempt to delete a non-existent subscription
    Delete subscription    ${NOT_EXISTING_SUBSCRIPTION_ID}
    
    # Verify response status code is 404 Not Found
    Check HTTP Response Status Code Is    404

TC_MEC_MEC048_MEO_CSE_018_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_018_OK
    ...    Check that the CSE sends a site resource usage notification if the CSE has an associated subscription and the event is generated
    ...    ETSI GS MEC 048 v3.1.1, clause 7.8.3.4, clause 6.4.3
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant and site resources
    [Setup]    Register site resource quota for tenant
    
    # Create a SiteResourceUsageSubscription with callback to our notification server
    ${subscription_json}=    Get File    jsons/SiteResourceUsageSubscription.json
    ${callback_url}=    Set Variable    http://${NOTIFICATION_SERVER_IP}:${NOTIFICATION_SERVER_PORT}${NOTIFICATION_SERVER_URI}
    ${subscription_json}=    Replace String    ${subscription_json}    "some/uri"    "${callback_url}"
    ${subscription_json}=    Replace String    ${subscription_json}    "placeholder-tenant"    "${TENANT_ID_1}"
    ${subscription_json}=    Replace String    ${subscription_json}    "placeholder-customer"    "${CUSTOMER_ID_1}"
    ${subscription_json}=    Replace String    ${subscription_json}    ["placeholder-site"]    ["${SITE_ID_1}"]
    
    # Register the subscription with our callback URL
    Post Raw Subscription    ${subscription_json}
    Check HTTP Response Status Code Is    201
    
    # Save the subscription ID for cleanup
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    ${subscription_id}=    Fetch From Right    ${subscription_uri}    /subscriptions/
    Set Suite Variable    ${SUBSCRIPTION_ID}    ${subscription_id}
    
    # Generate load to trigger the notification
    Log    Generating CPU load to trigger notification
    ${result}=    Generate CPU Load    60    5
    Log    ${result}
    
    # Start the notification server to wait for and capture the notification
    ${notification}=    Spawn Notification Server
    # Verify notification exists
    Should Not Be Empty    ${notification}
    
    # Verify notification content
    Should Be Equal As Strings    ${notification['notificationType']}    SiteResourceUsageNotification
    Should Be Equal As Strings    ${notification['customerId']}    ${CUSTOMER_ID_1}
    Should Be Equal As Strings    ${notification['tenantId']}    ${TENANT_ID_1}
    
    # Verify resource usage information
    ${site_info}=    Get From List    ${notification['resourceUseInfo']}    0
    Should Be Equal As Strings    ${site_info['siteId']}    ${SITE_ID_1}
    Should Be Equal As Integers    ${site_info['cpuUsed']}    5
    
    # Teardown - clean up resources
    [Teardown]    Run Keywords
    ...    Run Keyword If    '${SUBSCRIPTION_ID}' != '' and '${SUBSCRIPTION_ID}' != 'None'    Delete Created Subscription    AND
    ...    Run Keyword If    '${SITE_ID_1}' != '' and '${SITE_ID_1}' != 'None'    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}    AND
    ...    Run Keyword If    '${TENANT_ID_1}' != '' and '${TENANT_ID_1}' != 'None'    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_019_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_019_OK
    ...    Check that the CSE sends a resource usage notification if the CSE has an associated subscription and the event is generated
    ...    ETSI GS MEC 048 v3.1.1, clause 7.8.3.4, clause 6.4.2
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant for the subscription
    [Setup]    Register tenant Info    TenantInfo
    
    # Create a ResourceUsageSubscription with callback to notification server
    ${subscription_json}=    Get File    jsons/ResourceUsageSubscription.json
    ${callback_url}=    Set Variable    http://${NOTIFICATION_SERVER_IP}:${NOTIFICATION_SERVER_PORT}${NOTIFICATION_SERVER_URI}
    ${subscription_json}=    Replace String    ${subscription_json}    "some/uri"    "${callback_url}"
    ${subscription_json}=    Replace String    ${subscription_json}    "tenant-123"    "${TENANT_ID_1}"
    ${subscription_json}=    Replace String    ${subscription_json}    "customer-123"    "${CUSTOMER_ID_1}"
    
    # Register the subscription
    Post Raw Subscription    ${subscription_json}
    Check HTTP Response Status Code Is    201
    
    # Save the subscription ID for cleanup
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    ${subscription_id}=    Fetch From Right    ${subscription_uri}    /subscriptions/
    Set Suite Variable    ${SUBSCRIPTION_ID}    ${subscription_id}
    
    # Generate CPU load to trigger the notification
    Log    Generating CPU load to trigger notification (cpuUsed: 5)
    ${result}=    Generate CPU Load    60    5
    Log    ${result}
    
    # Start the notification server to capture the notification
    ${notification}=    Spawn Notification Server
    
    # Verify notification exists
    Should Not Be Empty    ${notification}
    
    # Verify notification content
    Should Be Equal As Strings    ${notification['notificationType']}    ResourceUsageNotification
    Should Be Equal As Strings    ${notification['customerId']}    ${CUSTOMER_ID_1}
    Should Be Equal As Strings    ${notification['tenantId']}    ${TENANT_ID_1}
    
    # Verify resource usage information
    Should Be Equal As Integers    ${notification['resourceUseInfo']['cpuUsed']}    5
    
    # Teardown - clean up resources
    [Teardown]    Run Keywords
    ...    Run Keyword If    '${SUBSCRIPTION_ID}' != '' and '${SUBSCRIPTION_ID}' != 'None'    Delete Created Subscription    AND
    ...    Run Keyword If    '${TENANT_ID_1}' != '' and '${TENANT_ID_1}' != 'None'    Remove specific tenant info resource    ${TENANT_ID_1}

TC_MEC_MEC048_MEO_CSE_020_OK
    [Documentation]    TP_MEC_MEC048_MEO_CSE_020_OK
    ...    Check that the CSE sends a notification on subscription expiration if the CSE has an associated subscription and the event is generated
    ...    ETSI GS MEC 048 v3.1.1, clause 7.8.3.4, clause 6.4.4
    [Tags]    PIC_SERVICES
    
    # Setup - Create a tenant for the subscription
    [Setup]    Register tenant Info    TenantInfo
    
    # Get the current time and add a short expiry deadline (10 seconds in the future)
    ${current_time}=    Get Current Date
    ${expiry_time}=    Add Time To Date    ${current_time}    10 seconds
    ${expiry_time_iso}=    Convert Date    ${expiry_time}    result_format=%Y-%m-%dT%H:%M:%SZ
    Set Suite Variable    ${EXPIRY_DEADLINE}    ${expiry_time_iso}
    
    # Create a ResourceUsageSubscription with expiry
    ${subscription_json}=    Get File    jsons/ResourceUsageSubscriptionWithExpiry.json
    ${callback_url}=    Set Variable    http://${NOTIFICATION_SERVER_IP}:${NOTIFICATION_SERVER_PORT}${NOTIFICATION_SERVER_URI}
    ${subscription_json}=    Replace String    ${subscription_json}    "some/uri"    "${callback_url}"
    ${subscription_json}=    Replace String    ${subscription_json}    "tenant-123"    "${TENANT_ID_1}"
    ${subscription_json}=    Replace String    ${subscription_json}    "customer-123"    "${CUSTOMER_ID_1}"
    ${subscription_json}=    Replace String    ${subscription_json}    "2099-12-31T23:59:59Z"    "${EXPIRY_DEADLINE}"
    
    # Register the subscription
    Post Raw Subscription    ${subscription_json}
    Check HTTP Response Status Code Is    201
    
    # Save the subscription ID and HREF
    ${subscription_uri}=    Set Variable    ${response['body']['_links']['self']['href']}
    ${subscription_id}=    Fetch From Right    ${subscription_uri}    /subscriptions/
    Set Suite Variable    ${SUBSCRIPTION_ID}    ${subscription_id}
    Set Suite Variable    ${SUBSCRIPTION_HREF}    ${subscription_uri}
    
    # Trigger subscription expiry
    Log    Waiting for subscription to expire
    Sleep    12s    # Wait past the 10-second expiry deadline
    
    # Start the notification server to capture the expiry notification
    ${notification}=    Spawn Notification Server
    
    # Verify notification exists
    Should Not Be Empty    ${notification}
    
    # Verify notification content
    Should Be Equal As Strings    ${notification['notificationType']}    ExpiryNotification
    Should Be Equal As Strings    ${notification['_links']['subscription']['href']}    ${SUBSCRIPTION_HREF}
    Should Be Equal As Strings    ${notification['expiryDeadline']}    ${EXPIRY_DEADLINE}
    
    # Verify the subscription no longer exists
    Retrieve specific subscription    ${SUBSCRIPTION_ID}
    Check HTTP Response Status Code Is    404
    
    # Teardown - clean up tenant (subscription auto-removed after expiry)
    [Teardown]    Run Keyword If    '${TENANT_ID_1}' != '' and '${TENANT_ID_1}' != 'None'    Remove specific tenant info resource    ${TENANT_ID_1}

*** Keywords ***

Spawn Expiry Notification Server
    Log    Starting notification server on ${NOTIFICATION_SERVER_IP}:${NOTIFICATION_SERVER_PORT} for expiry notification
    ${notification}=    Spawn Web Server  ${NOTIFICATION_SERVER_IP}  ${NOTIFICATION_SERVER_PORT}  ${NOTIFICATION_SERVER_TIMEOUT}  ${NOTIFICATION_SERVER_HTTP_METHOD}  ${NOTIFICATION_SERVER_URI}  ResourceUsageSubscription
    [Return]    ${notification}

Spawn Resource Notification Server
    Log    Starting notification server on ${NOTIFICATION_SERVER_IP}:${NOTIFICATION_SERVER_PORT}
    ${notification}=    Spawn Web Server  ${NOTIFICATION_SERVER_IP}  ${NOTIFICATION_SERVER_PORT}  ${NOTIFICATION_SERVER_TIMEOUT}  ${NOTIFICATION_SERVER_HTTP_METHOD}  ${NOTIFICATION_SERVER_URI}  ResourceUsageSubscription
    [Return]    ${notification}

Spawn Notification Server
    Log    Starting notification server on ${NOTIFICATION_SERVER_IP}:${NOTIFICATION_SERVER_PORT}
    ${notification}=    Spawn Web Server  ${NOTIFICATION_SERVER_IP}  ${NOTIFICATION_SERVER_PORT}  ${NOTIFICATION_SERVER_TIMEOUT}  ${NOTIFICATION_SERVER_HTTP_METHOD}  ${NOTIFICATION_SERVER_URI}  SiteResourceUsageSubscription
    [Return]    ${notification}

Post Raw Subscription
    [Arguments]    ${subscription_json}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    POST    ${apiRoot}/${apiName}/${apiVersion}/subscriptions    ${subscription_json}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Update subscription
    [Arguments]    ${subscription_id}    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    Log    Variables - TENANT_ID_1: ${TENANT_ID_1}, TENANT_ID_2: ${TENANT_ID_2}, NEW_CUSTOMER_ID: ${NEW_CUSTOMER_ID}, SITE_ID_1: ${SITE_ID_1}
    ${body}=    Replace String    ${body}    "placeholder-tenant"    "${TENANT_ID_1}"
    ${body}=    Replace String    ${body}    "customer-new"    "${NEW_CUSTOMER_ID}"
    ${body}=    Replace String    ${body}    "site-placeholder"    "${SITE_ID_1}"
    Log    Request body after replacements: ${body}
    PUT    ${apiRoot}/${apiName}/${apiVersion}/subscriptions/${subscription_id}    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Retrieve specific subscription
    [Arguments]    ${subscription_id}
    Log    Getting specific subscription: ${subscription_id}
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/subscriptions/${subscription_id}
    ${output}=    Output    response
    Log    GET Response: ${output}
    Set Suite Variable    ${response}    ${output}

Create ResourceUsageSubscription
    [Arguments]    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    # Replace placeholder values in the JSON with actual test values
    ${body}=    Replace String    ${body}    "placeholder"    "${TENANT_ID_1}"    count=1
    ${body}=    Replace String    ${body}    "placeholder"    "${CUSTOMER_ID_1}"    count=1
    POST    ${apiRoot}/${apiName}/${apiVersion}/subscriptions    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Delete Created Subscription
    ${sub_id}=    Fetch From Right    ${SUBSCRIPTION_ID}    /subscriptions/
    Delete subscription    ${sub_id}
Retrieve subscriptions with filter
    [Arguments]    ${query_params}
    Log    Getting subscriptions with filter: ${query_params}
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/subscriptions?${query_params}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Register site Resource subscription
    [Arguments]    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1    msg=PIC_MEC_PLAT must be 1
    Should Be True    ${PIC_SERVICES} == 1    msg=PIC_SERVICES must be 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    Log    Loading JSON file: ${file}
    ${file_exists}=    Run Keyword And Return Status    File Should Exist    ${file}
    Run Keyword If    not ${file_exists}    Fail    JSON file not found: ${file}
    ${body}=    Get File    ${file}
    Log    Request Body Before Replacement: ${body}
    # Validate variables
    Should Not Be Empty    ${TENANT_ID_1}    msg=TENANT_ID_1 is not defined
    Should Not Be Empty    ${CUSTOMER_ID_1}    msg=CUSTOMER_ID_1 is not defined
    Should Not Be Empty    ${SITE_ID_1}    msg=SITE_ID_1 is not defined
    # Replace placeholders
    ${body}=    Replace String    ${body}    "placeholder-tenant"    "${TENANT_ID_1}"
    ${body}=    Replace String    ${body}    "placeholder-customer"    "${CUSTOMER_ID_1}"
    ${body}=    Replace String    ${body}    "placeholder-site"    "${SITE_ID_1}"
    Log    Request Body After Replacement: ${body}
    # Validate JSON using Collections
    # ${json_dict}=    Evaluate    json.loads($body)    modules=json
    # Should Be True    isinstance($json_dict, dict)    msg=JSON is not a valid dictionary: ${body}
    POST    ${apiRoot}/${apiName}/${apiVersion}/subscriptions    ${body}
    ${output}=    Output    response
    Log    POST Response: ${output}
    Set Suite Variable    ${response}    ${output}
    Run Keyword If    ${response['status']} != 201    Fail    Subscription creation failed: ${response['body']['error']}    status=${response['status']}
Register subscription
    [Arguments]    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    Log    Variables - TENANT_ID_1: ${TENANT_ID_1}, CUSTOMER_ID_1: ${CUSTOMER_ID_1}, SITE_ID_1: ${SITE_ID_1}
    ${body}=    Replace String    ${body}    "placeholder-tenant"    "${TENANT_ID_1}"
    ${body}=    Replace String    ${body}    "placeholder-customer"    "${CUSTOMER_ID_1}"
    ${body}=    Replace String    ${body}    "placeholder-site"    "${SITE_ID_1}"
    Log    Request body after replacements: ${body}
    POST    ${apiRoot}/${apiName}/${apiVersion}/subscriptions    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Retrieve all subscriptions
    Log    Getting all subscriptions
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/subscriptions
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Remove test subscriptions
    # Extract subscription IDs from the hrefs
    ${sub_id_1}=    Fetch From Right    ${SUBSCRIPTION_ID_1}    /subscriptions/
    ${sub_id_2}=    Fetch From Right    ${SUBSCRIPTION_ID_2}    /subscriptions/
    ${sub_id_3}=    Fetch From Right    ${SUBSCRIPTION_ID_3}    /subscriptions/
    
    # Delete the created subscriptions
    Delete subscription    ${sub_id_1}
    Delete subscription    ${sub_id_2}
    Delete subscription    ${sub_id_3}
    
    # Remove the tenant
    Remove specific tenant info resource    ${TENANT_ID_1}

Delete subscription
    [Arguments]    ${subscription_id}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    DELETE    ${apiRoot}/${apiName}/${apiVersion}/subscriptions/${subscription_id}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Count Subscriptions Of Type
    [Arguments]    ${subscriptions}    ${subscription_type}
    ${count}=    Set Variable    ${0}
    FOR    ${subscription}    IN    @{subscriptions}
        ${type}=    Get From Dictionary    ${subscription}    subscriptionType
        ${count}=    Run Keyword If    '${type}' == '${subscription_type}'    Evaluate    ${count} + 1    ELSE    Set Variable    ${count}
    END
    [Return]    ${count}

Create test subscriptions
    # First create a tenant for the subscriptions
    Register tenant Info    TenantInfo
    Set Suite Variable    ${TENANT_ID_1}    ${response['body']['tenantId']}
    Set Suite Variable    ${CUSTOMER_ID_1}    ${response['body']['customerId']}
    
    # Create ResourceUsageSubscription subscriptions
    Register subscription    ResourceUsageSubscription1
    Set Suite Variable    ${SUBSCRIPTION_ID_1}    ${response['body']['_links']['self']['href']}
    Register subscription    ResourceUsageSubscription2
    Set Suite Variable    ${SUBSCRIPTION_ID_2}    ${response['body']['_links']['self']['href']}
    
    # Create SiteResourceUsageSubscription subscription
    Register subscription    SiteResourceUsageSubscription
    Set Suite Variable    ${SUBSCRIPTION_ID_3}    ${response['body']['_links']['self']['href']}


Register site resource quota for tenant
    # Create tenant
    Register tenant Info    TenantInfo
    Set Suite Variable    ${TENANT_ID_1}    ${response['body']['tenantId']}
    Set Suite Variable    ${CUSTOMER_ID_1}    ${response['body']['customerId']}
    # Create site resource quota
    Register site resource quota Info    ${TENANT_ID_1}    SiteResourceQuotaInfo
    Set Suite Variable    ${SITE_ID_1}    ${response['body']['siteId']}

Update site resource quota Info
    [Arguments]    ${tenantId}    ${siteId}    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    PUT    ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}/resources/quota_in_sites/${siteId}    ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Register multiple site resource quotas for tenant
    # Create tenant
    Register tenant Info    TenantInfo
    Set Suite Variable    ${TENANT_ID_1}    ${response['body']['tenantId']}
    # Create three site resource quotas
    Register site resource quota Info    ${TENANT_ID_1}    SiteResourceQuotaInfo
    Set Suite Variable    ${SITE_ID_1}    ${response['body']['siteId']}
    ${CPU_QUOTA}=    Convert To Integer    ${response['body']['cpuQuota']}
    ${MEMORY_QUOTA}=    Convert To Integer    ${response['body']['memoryQuota']}
    ${DISK_QUOTA}=    Convert To Integer    ${response['body']['diskQuota']}
    Set Suite Variable    ${CPU_QUOTA_1}    ${CPU_QUOTA}
    Set Suite Variable    ${MEMORY_QUOTA_1}    ${MEMORY_QUOTA}
    Set Suite Variable    ${DISK_QUOTA_1}    ${DISK_QUOTA}
    Register site resource quota Info    ${TENANT_ID_1}    SiteResourceQuotaInfo2
    Set Suite Variable    ${SITE_ID_2}    ${response['body']['siteId']}
    Register site resource quota Info    ${TENANT_ID_1}    SiteResourceQuotaInfo3
    Set Suite Variable    ${SITE_ID_3}    ${response['body']['siteId']}

Retrieve specific site resource quota info resource
    [Arguments]    ${tenantId}    ${siteId}
    Log    Getting specific site resource quota for tenant ${tenantId} and site ${siteId}
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}/resources/quota_in_sites/${siteId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Remove tenant and site resources
    # Remove site resource quotas
    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_1}
    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_2}
    Remove specific site resource quota info resource    ${TENANT_ID_1}    ${SITE_ID_3}
    # Remove tenant
    Remove specific tenant info resource    ${TENANT_ID_1}

Retrieve all tenant info resources
    Log    Getting all Tenant Info
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    Get    ${apiRoot}/${apiName}/${apiVersion}/tenants    
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Retrieve specific tenant info resource
    [Arguments]     ${TEANNT_ID}
    Log    Getting Specific Tenant Info Resource
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/tenants/${TEANNT_ID}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Retrieve all tenant info resources with query parameters
    [Arguments]     ${query_params}
    Log    Getting all Tenant Info with query parameter
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/tenants?${query_params}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Register tenant Info
    [Arguments]     ${content}    ${is_individual_call}=${True}    ${tenant_var}=TENANT_ID_1
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    POST   ${apiRoot}/${apiName}/${apiVersion}/tenants   ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Run Keyword If    ${is_individual_call} and ${response}[status] != 400    Set Suite Variable    \${${tenant_var}}    ${response}[body][tenantId]

Update tenant Info
    [Arguments]     ${tenantId}    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    PUT   ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}   ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}
    Run Keyword If    ${response['status']} != 400 and ${response['status']} != 404    Set Suite Variable    ${TENANT_ID_1}    ${response['body']['tenantId']}

Update resource quota Info
    [Arguments]     ${tenantId}    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    PUT   ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}/resources/quota_in_system   ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Register mutiple tenant Info and get tenant Ids
    [Arguments]     ${content_01}     ${content_02}    ${content_03}
    Register tenant Info    ${content_01}    ${False}
    Set Suite Variable      ${TENANT_ID_1}    ${response}[body][tenantId]
    Set Suite Variable      ${CUSTOMER_ID_1}    ${response}[body][customerId]
    Set Suite Variable      ${CUSTOMER_NAME_1}    ${response}[body][customerName]
    Register tenant Info    ${content_02}    ${False}
    Set Suite Variable      ${TENANT_ID_2}    ${response}[body][tenantId]
    Set Suite Variable      ${CUSTOMER_ID_2}    ${response}[body][customerId]
    Set Suite Variable      ${CUSTOMER_NAME_2}    ${response}[body][customerName]
    Register tenant Info    ${content_03}    ${False}
    Set Suite Variable      ${TENANT_ID_3}    ${response}[body][tenantId]
    Set Suite Variable      ${CUSTOMER_ID_3}    ${response}[body][customerId]
    Set Suite Variable      ${CUSTOMER_NAME_3}    ${response}[body][customerName]

Remove mutiple tenant Info
    [Arguments]     ${tenant_id_1}   ${tenant_id_2}    ${tenant_id_3}    
    Run Keyword If    '${tenant_id_1}' != 'SKIP'    Remove specific tenant info resource   ${tenant_id_1} 
    Remove specific tenant info resource   ${tenant_id_2} 
    Remove specific tenant info resource   ${tenant_id_3}

Remove specific tenant info resource 
    [Arguments]     ${tenantId} 
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    DELETE  ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output} 

Retrieve specific resource quota info resource
    [Arguments]     ${TEANNT_ID}
    Log    Getting Specific Resource Quota Info
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/tenants/${TEANNT_ID}/resources/quota_in_system 
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Retrieve all resource quota info resource
    [Arguments]     ${TEANNT_ID}
    Log    Getting All Resource Quota Info
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/tenants/${TEANNT_ID}/resources/quota_in_system 
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Retrieve all site resource quota info resource
    [Arguments]     ${TEANNT_ID}
    Log    Getting All Site Resource Quota Info
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/tenants/${TEANNT_ID}/resources/quota_in_sites 
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Retrieve all site resource quota info resource with query parameters
    [Arguments]     ${TEANNT_ID}    ${query_params}
    Log    Getting All Site Resource Quota Info with query parameters
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"*/*"}
    Set Headers    {"Authorization":"${TOKEN}"}
    GET    ${apiRoot}/${apiName}/${apiVersion}/tenants/${TEANNT_ID}/resources/quota_in_sites?${query_params}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Register mutiple site resource quota Info and get site Ids
    [Arguments]     ${tenantId}    ${content_01}     ${content_02}    ${content_03}
    Register site resource quota Info    ${tenantId}    ${content_01}
    Set Suite Variable      ${SITE_ID_1}    ${response}[body][siteId]
    Register site resource quota Info    ${tenantId}    ${content_02}
    Set Suite Variable      ${SITE_ID_2}    ${response}[body][siteId]
    Run Keyword If    '${content_03}' != 'SKIP'    Register site resource quota Info    ${tenantId}    ${content_03}
    Run Keyword If    '${content_03}' != 'SKIP'    Set Suite Variable      ${SITE_ID_3}    ${response}[body][siteId]

Register resource quota Info
    [Arguments]     ${tenantId}    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    POST   ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}/resources/quota_in_system   ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Register site resource quota Info
    [Arguments]     ${tenantId}    ${content}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    Set Headers    {"Authorization":"${TOKEN}"}
    ${file}=    Catenate    SEPARATOR=    jsons/    ${content}    .json
    ${body}=    Get File    ${file}
    POST   ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}/resources/quota_in_sites   ${body}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Remove mutiple site resource quota Info
    [Arguments]     ${tenant_id_1}   ${site_id_1}    ${site_id_2}    ${site_id_3}    
    Run Keyword If    '${tenant_id_1}' != 'SKIP'    Remove specific site resource quota info resource   ${tenant_id_1}    ${site_id_1}
    Remove specific site resource quota info resource   ${tenant_id_1}    ${site_id_2}
    Run Keyword If    '${site_id_3}' != 'SKIP'    Remove specific site resource quota info resource   ${tenant_id_1}    ${site_id_3}

Remove specific resource quota info resource 
    [Arguments]     ${tenantId} 
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    DELETE  ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}/resources/quota_in_system
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}

Remove specific site resource quota info resource 
    [Arguments]     ${tenantId}    ${siteId}
    Should Be True    ${PIC_MEC_PLAT} == 1
    Should Be True    ${PIC_SERVICES} == 1
    Set Headers    {"Accept":"application/json"}
    Set Headers    {"Content-Type":"application/json"}
    DELETE  ${apiRoot}/${apiName}/${apiVersion}/tenants/${tenantId}/resources/quota_in_sites/${siteId}
    ${output}=    Output    response
    Set Suite Variable    ${response}    ${output}