import logging

from asn1tools import parse_files, compile_dict, ParseError, CompileError
from glob import glob
from pathlib import Path

from pprint import pprint


def parseASN1File (asnFile):
    try:
        parse_files(asnFile)
    except ParseError as ex:
        return [ex]
    return []


def parseASN1Files (fileList):
    if len(fileList) == 0:
        logging.warning ("No files specified")
        return {}
    errors = {}
    logging.info("Parsing files...")
    for f in fileList:
        ex = parseASN1File(f)
        if ex: 
            logging.info (f"  {f}: Failed - {ex!r}")
        else: 
            logging.info (f"  {f}: OK")            
        errors[f] = ex
    return errors


def compileASN1Files (fileList):
    logging.info("Compiling files...")
    errors = []
    try:
        d = parse_files(fileList)
        for modulename, module in d.items():
            # Weird fix because the compiler doesn't like RELATIVE-OID as a type
            # Not sure if the on-the-wire encoding would be affected or not
            # but for most checking purposes this doesn't matter
            module['types']["RELATIVE-OID"] = {'type' : 'OBJECT IDENTIFIER'}
        c = compile_dict(d)
    except CompileError as ex:
        logging.info (f"Compiler  error: {ex}")
        errors.append(ex)
    except ParseError as ex:
        logging.info (f"Parse error: {ex}")
        errors.append(ex)
    logging.info ("Compiled OK")
    return errors


def validateASN1Files (fileList):
    parseErrors = parseASN1Files(fileList)
#    if len(parseErrors > 0):
#        logging.info ("Abandonding compile due to parse errors")
#    compileErrors = compileASN1Files(fileList)
#   leave this for now - TBD
    compileErrors = []
    return parseErrors, compileErrors


def validateAllASN1FilesInPath (path):
    p = Path(path)
    fileGlob = [str(f) for f in  p.rglob('*.asn')]
    fileGlob += [str(f) for f in p.rglob('*.asn1')]
    return validateASN1Files(fileGlob)    


if __name__ == '__main__':
    parseErrors, compileErrors = validateAllASN1FilesInPath("./")
    parseErrorCount = 0
    print ("ASN.1 Parser checks:")
    print ("-----------------------------")
    for filename, errors in parseErrors.items():
        if len(errors) > 0:
            parseErrorCount += len(errors)
            print (f"{filename}: {len(errors)} errors")
            for error in errors:
                print ("  " + str(error))
        else:
            print (f"{filename}: OK")
    print ("-----------------------------")
    print ("ASN.1 Compilation:")
    print ("-----------------------------")
    if len(compileErrors) > 0:
        for error in compileErrors:            
            print ("  " + str(error))
    else:
        print ("Compilation OK")
    print ("-----------------------------")
    print (f"{parseErrorCount} parse errors, {len(compileErrors)} compile errors")
    exit (parseErrorCount + len(compileErrors))

