#pragma once

#include <vector>
#include <memory>

#include "codec_gen.hh"
#include "params.hh"

#include "avp.hh" 

class TTCN_Typedescriptor_t;
class TTCN_Buffer;

namespace LibDiameter__TypesAndValues {
  class DiameterHeader; //! Forward declaration of TITAN class
}

class diameter_codec  {
  std::map<const int, std::unique_ptr<avp> > _avps;
  std::map<const int, std::unique_ptr<avp> >::const_iterator _it;
  std::unique_ptr<avp> _nullptr;
  const unsigned char* _start;
  const unsigned char* _end;

public:
  explicit diameter_codec() : _avps(), _it(), _nullptr(nullptr), _start(nullptr) { };
  virtual ~diameter_codec();

  int encode_header(const LibDiameter__TypesAndValues::DiameterHeader& p_header, TTCN_Buffer& p_encoding_buffer);
  int encode_session_id(const LibDiameter__Types__Base__AVPs::Session__Id__AVP& p_avp, TTCN_Buffer& p_encoding_buffer);
  int encode_auth_session_state(const LibDiameter__Types__Base__AVPs::Auth__Session__State__AVP& p_avp, TTCN_Buffer& p_encoding_buffer);
  int encode_origin_host(const LibDiameter__Types__Base__AVPs::Origin__Host__AVP& p_avp, TTCN_Buffer& p_encoding_buffer);
  int encode_origin_realm(const LibDiameter__Types__Base__AVPs::Origin__Realm__AVP& p_avp, TTCN_Buffer& p_encoding_buffer);

  int decode_header(TTCN_Buffer& p_decoding_buffer, LibDiameter__TypesAndValues::DiameterHeader& p_header, const params& p_params);
  int decode_avps(TTCN_Buffer& p_decoding_buffer, const LibDiameter__TypesAndValues::DiameterHeader& p_header, const params& p_params);

  const std::unique_ptr<avp>& get_avp(const int p_code);

public:
  int encode_avp_header(const LibDiameter__Types__Base__AVPs::AVP__Header& p_header, int* p_padding, TTCN_Buffer& p_encoding_buffer);

  int decode_avp(TTCN_Buffer& p_decoding_buffer, const LibDiameter__TypesAndValues::DiameterHeader& p_header, const params& p_params, avp** avp_ptr);

  std::string command_code_2_enumerated(const int p_code, const boolean p_request_flag);

  /** Command code of Request/Answer */
  static const int Code_CE=257;// Capabilities-Exchange
  static const int Code_RA=258;// Re-Auth
  static const int Code_AC=271;// Accounting
  static const int Code_AS=274;// Abort-Session
  static const int Code_ST=275;// Session-Termination
  static const int Code_DW=280;// Device-Watchdog
  static const int Code_DP=282;// Disconnect-Peer

  /* Codes on Cx */
  static const int Code_UA=300;// User-Authorization
  static const int Code_SA=301;// Server-Assignment-Request        SAR    (284 in RFC4740 or 301 in TS 129 229)
  static const int Code_LI=302;// Location-Info-Request            LIR    (285 in RFC4740 or 302 in TS 129 229)
  static const int Code_MA=303;// Multimedia-Auth-Request          MAR    (286 in RFC4740 or 303 in TS 129 229)
  static const int Code_RT=304;// Registration-Termination-Request RTR    (287 in RFC4740 or 304 in TS 129 229)
  static const int Code_PP=305;// Push-Profile-Request             PPR    (288 in RFC4740 or 305 in TS 129 229)

  /* Codes on Sh/Dh interface  */
  static const int Code_UD=306;// UDR User-Data-Request AS->HSS                UDR    (306)
  static const int Code_PU=307;// PUR Profile-Update-Request AS->HSS           PUR    (307)
  static const int Code_SN=308;// SNR Subscribe-Notifications-Request AS->($1) ($2)    (308)
  static const int Code_PN=309;// PNR Profile-Notification-Request HSS->AS     PNR    (309)

  /* Codes on Rx interface  */
  static const int Code_AA=265;// Auth-Accounting

  /* Codes on Gx interface - RFC4006 and TS 129 212  */
  static const int Code_CC=272;// Credit-Control 

  /* Codes on S6a interface - TS 129 272  */
  static const int Code_UL=316;  // Update-Location-Request                   ULR    (316)- Ref: TS 129 272 7.2.3-4
  static const int Code_CL=317;  // Cancel-Location-Request                   CLR    (317)- Ref: TS 129 272 7.2.7-8
  static const int Code_AI=318;  // Authentication-Information-Request        AIR    (318)- Ref: TS 129 272 7.2.5-6
  static const int Code_ID=319;  // Insert-Subscriber-Data-Request            IDR    (319)- Ref: TS 129 272 7.2.9-10
  static const int Code_DS=320;  // Delete-Subscriber-Data-Request            DSR    (320)- Ref: TS 129 272 7.2.11-12
  static const int Code_PUE=321; // Purge-UE-Request                          PUR    (321)- Ref: TS 129 272 7.2.13-14
  static const int Code_RS=322;  // Reset-Request                             DSR    (322)- Ref: TS 129 272 7.2.15-16
  static const int Code_NO=323;  // Notify-Request                            NOR    (323)- Ref: TS 129 272 7.2.17-18


  /*RFC3588 AVP codes*/
  const int AVP_Accounting_Realtime_Required_AVP = 483; 			//Ref: RFC3588 9.8.7.  Accounting-Realtime-Required AVP
  const int AVP_Accounting_Record_Number_AVP = 485;				    //Ref: RFC3588 9.8.3.  Accounting-Record-Number AVP
  const int AVP_Accounting_Record_Type_AVP = 480; 					//Ref: RFC3588 9.8.1.  Accounting-Record-Type AVP
  const int AVP_Accounting_Sub_Session_Id_AVP = 287;				//Ref: RFC3588 9.8.6.  Accounting-Sub-Session-Id AVP
  const int AVP_Acct_Application_Id_AVP = 259; 					    //Ref: RFC3588 6.9.  Acct-Application-Id AVP
  const int AVP_Acct_Interim_Interval_AVP = 85; 					//Ref: RFC3588 9.8.2.  Acct-Interim-Interval
  const int AVP_Acct_Multi_Session_Id_AVP = 50; 					//Ref: RFC3588 9.8.5.  Acct-Multi-Session-Id AVP
  const int AVP_Acct_Session_Id_AVP = 44; 							//Ref: RFC3588 9.8.4.  Acct-Session-Id AVP 
  const int AVP_Auth_Application_Id_AVP = 258; 					    //Ref: RFC3588 6.8.  Auth-Application-Id AVP
  const int AVP_Class_AVP = 25; 									//Ref: RFC3588 8.20.  Class AVP
  const int AVP_Destination_Host_AVP = 293; 						//Ref: RFC3588 6.5.  Destination-Host AVP
  const int AVP_Destination_Realm_AVP = 283; 						//Ref: RFC3588 6.6.  Destination-Realm AVP
  const int AVP_Disconnect_Cause_AVP = 273; 						//Ref: RFC3588 5.4.3.  Disconnect-Cause AVP
  const int AVP_Error_Message_AVP = 281; 							//Ref: RFC3588 7.3.  Error-Message AVP
  const int AVP_Error_Reporting_Host_AVP = 294; 					//Ref: RFC3588 7.4.  Error-Reporting-Host AVP
  const int AVP_Event_Timestamp_AVP = 55; 							//Ref: RFC3588 8.21.  Event-Timestamp AVP
  const int AVP_Failed_AVP = 279; 									//Ref: RFC3588 7.5.  Failed-AVP AVP
  const int AVP_Firmware_Revision_AVP = 267; 						//Ref: RFC3588 5.3.4.  Firmware-Revision AVP
  const int AVP_Host_IP_Address_AVP = 257; 						    //Ref: RFC3588 5.3.5.  Host-IP-Address AVP
  const int AVP_Inband_Security_Id_AVP = 299; 						//Ref: RFC3588 6.10.  Inband-Security-Id AVP
  const int AVP_Origin_Host_AVP = 264; 							    //Ref: RFC3588 6.3.  Origin-Host AVP
  const int AVP_Origin_Realm_AVP = 296; 							//Ref: RFC3588 6.4.  Origin-Realm AVP
  const int AVP_Origin_State_Id_AVP = 278; 						    //Ref: RFC3588 8.16.  Origin-State-Id AVP
  const int AVP_Product_Name_AVP = 269; 							//Ref: RFC3588 5.3.7.  Product-Name AVP
  const int AVP_Proxy_Host_AVP = 280; 								//Ref: RFC3588 6.7.3.  Proxy-Host AVP
  const int AVP_Proxy_Info_AVP = 284; 								//Ref: RFC3588 6.7.2.  Proxy-Info AVP
  const int AVP_Proxy_State_AVP = 33; 								//Ref: RFC3588 6.7.4.  Proxy-State AVP
  const int AVP_Re_Auth_Request_Type_AVP = 285; 					//Ref: RFC3588 8.12.  Re-Auth-Request-Type AVP
  const int AVP_Redirect_Host_AVP = 292; 							//Ref: RFC3588 6.12.  Redirect-Host AVP
  const int AVP_redirect_Max_Cache_Time_AVP = 262; 				//Ref: RFC3588 redirect_Max_Cache_Time_AVP is not defined. 
  const int AVP_Redirect_Host_Usage_AVP = 261; 					    //Ref: RFC3588 6.13.  Redirect-Host-Usage AVP
  const int AVP_Result_Code_AVP = 268; 							    //Ref: RFC3588 7.1.  Result-Code AVP
  const int AVP_Route_Record_AVP = 282; 							//Ref: RFC3588 6.7.1.  Route-Record AVP
  const int AVP_Session_Id_AVP = 263; 								//Ref: RFC3588 8.8.  Session-Id AVP
  const int AVP_Supported_Vendor_Id_AVP = 265; 					    //Ref; RFC3588 5.3.6.  Supported-Vendor-Id AVP
  const int AVP_Termination_Cause_AVP = 295; 						//Ref: RFC3588 8.15.  Termination-Cause AVP
  const int AVP_User_Name_AVP = 1; 								    //Ref: RFC3588 8.14.  User-Name AVP
  const int AVP_Vendor_Id_AVP = 266; 								//Ref: RFC3588 5.3.3.  Vendor-Id AVP
  const int AVP_Vendor_Specific_Appl_Id_AVP = 260; 				    //Ref: RFC3588 6.11.  Vendor-Specific-Application-Id AVP
  /*end RFC3588 AVP codes*/

  //RFC4740 AVP codes{
//  const int AVP_Auth_Grace_Period_AVP = 276; 						//Ref: RFC3588 8.10.  Auth-Grace-Period AVP								
  const int AVP_Auth_Session_State_AVP = 277; 						//Ref: RFC3588 8.11.  Auth-Session-State AVP							
//  const int AVP_Authorization_Lifetime_AVP = 291; 					//Ref: RFC3588 8.9.  Authorization-Lifetime AVP								
//  const int AVP_Redirect_Max_Cache_Time_AVP = 262;					//Ref: RFC3588 6.14.  Redirect-Max-Cache-Time AVP							
//  const int AVP_SIP_Accounting_Information_AVP = 368; 				//Ref: RFC4740 9.1.  SIP-Accounting-Information AVP								
//  const int AVP_SIP_Accounting_Server_URI_AVP = 369; 				//Ref: RFC4740 9.1.1.  SIP-Accounting-Server-URI AVP								
//  const int AVP_SIP_AOR_AVP = 122; 								//Ref: RFC4740 9.8.  SIP-AOR AVP							
  const int AVP_SIP_Auth_Data_Item_RFC_AVP = 376; 						//Ref: RFC4740 9.5.  SIP-Auth-Data-Item AVP
  const int AVP_SIP_Item_Number_AVP = 378;
  const int AVP_SIP_Authenticate_RFC_AVP = 379; 						//Ref: RFC4740 9.5.3.  SIP-Authenticate AVP								
//  const int AVP_SIP_Authentication_Info_AVP = 381;  				//Ref: RFC4740 9.5.5.  SIP-Authentication-Info AVP							
  const int AVP_SIP_Authentication_Scheme_RFC_AVP = 377; 				//Ref: RFC4740 9.5.1.  SIP-Authentication-Scheme AVP							
  const int AVP_SIP_Authorization_RFC_AVP = 380;  						//Ref: RFC4740 9.5.4.  SIP-Authorization AVP							
//  const int AVP_SIP_Credit_Control_Server_URI_AVP = 370; 			//Ref: RFC4740 9.1.2.  SIP-Credit-Control-Server-URI AVP								
//  const int AVP_SIP_Deregistration_Reason_AVP = 383; 				//Ref: RFC4740 9.5.5.  SIP-Authentication-Info AVP							
//  const int AVP_SIP_Item_Number_AVP = 378; 						//Ref: RFC4740 9.5.2 SIP-Item-Number AVP							
//  const int AVP_SIP_MandatoryCapability_AVP = 373; 				//Ref: RFC4740 9.3.1.  SIP-Mandatory-Capability AVP								
  const int AVP_SIP_Method_AVP = 393; 								//Ref: RFC4740 9.14.  SIP-Method AVP								
  const int AVP_SIP_Number_Auth_Items_RFC_AVP = 382; 					//Ref: RFC4740 9.6.  SIP-Number-Auth-Items AVP								
//  const int AVP_SIP_OptionalCapability_AVP = 374; 					//Ref: RFC4740 9.3.2.  SIP-Optional-Capability AVP								
//  const int AVP_SIP_Reason_Code_AVP = 384; 						//Ref: RFC4740 9.7.1.  SIP-Reason-Code AVP							
//  const int AVP_SIP_Reason_Info_AVP = 385;  						//Ref: RFC4740 9.7.2.  SIP-Reason-Info AVP							
//  const int AVP_SIP_Server_Assignment_Type_AVP = 375; 				//Ref: RFC4740 9.4.  SIP-Server-Assignment-Type AVP								
//  const int AVP_SIP_Server_Capabilities_AVP = 372; 				//Ref: RFC4740 9.3.  SIP-Server-Capabilities AVP								
//  const int AVP_SIP_Server_URI_AVP = 371; 							//Ref: RFC4740 9.2 SIP-Server-URI AVP							
//  const int AVP_SIP_Supported_User_Data_Type_AVP = 388; 			//Ref: RFC4740 9.12.1.  SIP-User-Data-Type AVP								
//  const int AVP_SIP_User_Authorization_Type_AVP = 387; 			//Ref: RFC4740 9.10.  SIP-User-Authorization-Type AVP						
//  const int AVP_SIP_User_Data_Already_Available_AVP = 392; 		//Ref: RFC4740 9.13.  SIP-User-Data-Already-Available AVP								
//  const int AVP_SIP_User_Data_AVP = 389; 							//Ref: RFC4740 9.12.  SIP-User-Data AVP								
//  const int AVP_SIP_User_Data_Contents_AVP = 391; 					//Ref: RFC4740 9.12.2.  SIP-User-Data-Contents AVP								
//  const int AVP_SIP_User_Data_Type_AVP = 390; 						//Ref: RFC4740 9.12.1.  SIP-User-Data-Type AVP								
//  const int AVP_SIP_Visited_Network_Id_AVP = 386; 					//Ref: RFC4740 9.9.  SIP-Visited-Network-Id AVP							
  //end of RFC4740 AVP codes

  const int AVP_Digest_Realm_AVP = 104;
  const int AVP_Digest_Nonce_AVP = 105;
  const int AVP_Digest_Domain_AVP = 119;
  const int AVP_Digest_Opaque_AVP = 116;
  const int AVP_Digest_Stale_AVP = 120;
  const int AVP_Digest_Algorithm_AVP = 111;
  const int AVP_Digest_QoP_AVP = 110;
  const int AVP_Digest_HA1_AVP = 121;
  const int AVP_Digest_Auth_Param_AVP = 117;
  const int AVP_Digest_Username_AVP = 115;
  const int AVP_Digest_URI_AVP = 109;
  const int AVP_Digest_Response_AVP = 103;
  const int AVP_Digest_CNonce_AVP = 113;
  const int AVP_Digest_Nonce_Count_AVP = 114;
  const int AVP_Digest_Method_AVP = 108;
  const int AVP_Digest_Entity_Body_Hash_AVP = 112;
  const int AVP_Digest_Nextnonce_AVP = 107;
  const int AVP_Digest_Response_Auth_AVP = 106;


  //RFC4005 AVP codes
  const int AVP_Called_Station_Id_AVP = 30;
  const int AVP_Framed_IP_Address_AVP  = 8;
  const int AVP_Framed_IPv6_Prefix_AVP = 97;
  const int AVP_Framed_Interface_Id_AVP = 96;
  //end of RFC4005 AVP codes

  //RFC4006 AVP codes
//  const int AVP_Rating_Group_AVP = 433; 							//Ref: RFC4006 8.29. Rating-Group AVP								
const int AVP_Tariff_Time_Change_AVP = 451; 						//Ref: RFC4006 8.20. Tariff-Time-Change AVP								
//  const int AVP_CC_Correlation_Id_AVP = 411; 						//Ref: RFC4006 8.1.  CC-Correlation-Id AVP								
const int AVP_CC_Input_Octets_AVP = 412; 						//Ref: RFC4006 8.24. CC-Input-Octets AVP
const int AVP_CC_Money_AVP = 413; 								//Ref: RFC4006 8.22. CC-Money AVP								
const int AVP_CC_Output_Octets_AVP = 414; 						//Ref: RFC4006 8.25. CC-Output-Octets AVP								
  const int AVP_CC_Request_Number_AVP = 415; 						//Ref: RFC4006 8.2.  CC-Request-Number AVP								
  const int AVP_CC_Request_Type_AVP = 416; 						//Ref: RFC4006 8.3.  CC-Request-Type AVP								
const int AVP_CC_Service_Specific_Units_AVP = 417; 				//Ref: RFC4006 8.26. CC-Service-Specific-Units AVP								
const int AVP_CC_Session_Failover_AVP = 418; 					//Ref: RFC4006 8.4.  CC-Session-Failover AVP								
//  const int AVP_vendIdCC_Sub_Session_Id_AVP = 419; 						//Ref: RFC4006 8.5.  CC-Sub-Session-Id AVP								
const int AVP_CC_Time_AVP = 420; 								//Ref: RFC4006 8.21. CC-Time AVP								
const int AVP_CC_Total_Octets_AVP = 421; 						//Ref: RFC4006 8.23. CC-Total-Octets AVP								
const int AVP_CC_Unit_Type_AVP = 454; 							//Ref: RFC4006 8.32. CC-Unit-Type AVP								
//  const int AVP_vendIdCheck_Balance_Result_AVP = 422; 					//Ref: RFC4006 8.6.  Check-Balance-Result AVP 12.8.  Check-Balance-Result AVP								
const int AVP_Cost_Information_AVP = 423; 						//Ref: RFC4006 8.7.  Cost-Information AVP								
const int AVP_Cost_Unit_AVP = 424; 								//Ref: RFC4006 8.12. Cost-Unit AVP								
const int AVP_Credit_Control_Failure_Handling_AVP = 427; 		//Ref: RFC4006 8.14. Credit-Control-Failure-Handling AVP								
const int AVP_Currency_Code_AVP = 425; 							//Ref: RFC4006 8.11. Currency-Code AVP								
const int AVP_Direct_Debiting_Failure_Handling_AVP = 428; 		//Ref: RFC4006 8.15. Direct-Debiting-Failure-Handling AVP								
const int AVP_Exponent_AVP = 429; 								//Ref: RFC4006 8.9.  Exponent AVP								
const int AVP_Filter_Id_AVP = 11; 								//Ref: RFC4005 6.7.  Filter-Id AVP								
const int AVP_Final_Unit_Action_AVP = 449; 						//Ref: RFC4006 8.35. Final-Unit-Action AVP								
const int AVP_Final_Unit_Indication_AVP = 430; 					//Ref: RFC4006 8.34. Final-Unit-Indication AVP								
const int AVP_G_S_U_Pool_Identifier_AVP = 453; 					//Ref: RFC4006 8.31. G-S-U-Pool-Identifier AVP								
const int AVP_G_S_U_Pool_Reference_AVP = 457; 					//Ref: RFC4006 8.30. G-S-U-Pool-Reference AVP								
const int AVP_Granted_Service_Unit_AVP = 431; 					//Ref: RFC4006 8.17. Granted-Service-Unit AVP								
//  const int AVP_vendIdMultiple_Services_Credit_Control_AVP = 456; 		//Ref: RFC4006 8.16. Multiple-Services-Credit-Control AVP								
const int AVP_Multiple_Services_Indicator_AVP = 455; 			//Ref: RFC4006 8.40. Multiple-Services-Indicator AVP								
  const int AVP_Redirect_Address_Type_AVP = 433; 					//Ref: RFC4006 8.38. Redirect-Address-Type AVP								
  const int AVP_Redirect_Server_Address_AVP = 435; 				//Ref: RFC4006 8.39. Redirect-Server-Address AVP								
  const int AVP_Redirect_Server_AVP = 434; 						//Ref: RFC4006 8.37. Redirect-Server AVP								
//  const int AVP_vendIdRequested_Action_AVP = 436; 						//Ref: RFC4006 8.41. Requested-Action AVP								
const int AVP_Requested_Service_Unit_AVP = 437; 					//Ref: RFC4006 8.18. Requested-Service-Unit AVP								
const int AVP_Restriction_Filter_Rule_AVP = 438; 				//Ref: RFC4006 8.36. Restriction-Filter-Rule AVP								
  const int AVP_Service_Context_Id_AVP = 461; 						//Ref: RFC4006 8.42. Service-Context-Id AVP								
  const int AVP_Service_Identifier_AVP = 439; 						//Ref: RFC4006 8.28. Service-Identifier AVP								
  const int AVP_Requested_Action_AVP = 436;
  //  const int AVP_vendIdService_Parameter_Info_AVP = 440; 					//Ref: RFC4006 8.43. Service-Parameter-Info AVP								
//  const int AVP_vendIdService_Parameter_Type_AVP = 441; 					//Ref: RFC4006 8.44. Service-Parameter-Type AVP								
//  const int AVP_vendIdService_Parameter_Value_AVP = 442; 				//Ref: RFC4006 8.45. Service-Parameter-Value AVP								
  const int AVP_Subscription_Id_AVP = 443; 						//Ref: RFC4006 8.46. Subscription-Id AVP								
  const int AVP_Subscription_Id_Data_AVP = 444; 					//Ref: RFC4006 								
  const int AVP_Subscription_Id_Type_AVP = 450; 					//Ref: RFC4006 8.47. Subscription-Id-Type AVP								
  const int AVP_Tariff_Change_Usage_AVP = 452;//Ref: RFC4006 8.27. Tariff-Change-Usage AVP 
  const int AVP_Rating_Group_AVP = 433;								
const int AVP_Unit_Value_AVP = 445; 								//Ref: RFC4006 8.8.  Unit-Value AVP								
const int AVP_Used_Service_Unit_AVP = 446; 						//Ref: RFC4006 8.19. Used-Service-Unit AVP								
//  const int AVP_vendIdUser_Equipment_Info_AVP = 458; 					//Ref: RFC4006 8.49. User-Equipment-Info AVP								
//  const int AVP_vendIdUser_Equipment_Info_Type_AVP = 459; 				//Ref: RFC4006 8.50. User-Equipment-Info-Type AVP								
//  const int AVP_vendIdUser_Equipment_Info_Value_AVP = 460; 				//Ref: RFC4006 8.51. User-Equipment-Info-Value AVP								
const int AVP_Validity_Time_AVP = 448; 							//Ref: RFC4006 8.33. Validity-Time AVP								
const int AVP_Value_Digits_AVP = 447; 							//Ref: RFC4006 8.10. Value-Digits AVP								
  /*end of RFC4006 AVP codes*/

  //TS129_229 AVP codes
//  const int AVP_vendIdCharging_Information_AVP = 61; 							//Ref: ETSI TS 129 229 Charging_Information_AVP
//  const int AVP_vendIdPrimary_Event_Charging_Function_Name_AVP = 619; 			//Ref: ETSI TS 129 229 Primary_Event_Charging_Function_Name_AVP
//  const int AVP_vendIdSecondary_Event_Charging_Function_Name_AVP = 620; 		//Ref: ETSI TS 129 229 Secondary_Event_Charging_Function_Name_AVP
//  const int AVP_vendIdPrimary_Charging_Collection_Function_Name_AVP = 621; 	//Ref: ETSI TS 129 229 Primary_Charging_Collection_Function_Name_AVP
//  const int AVP_vendIdSecondary_Charging_Collection_Function_Name_AVP = 622; 	//Ref: ETSI TS 129 229 Secondary_Charging_Collection_Function_Name_AVP
  const int AVP_Charging_Rule_Remove_AVP = 1002;
  const int AVP_Charging_Rule_Install_AVP = 1001;
  const int AVP_Charging_Rule_Report_AVP = 1018;
  const int AVP_Routing_Rule_Install_AVP = 1081;
  const int AVP_Routing_Rule_Remove_AVP = 1075;
  const int AVP_QoS_Information_AVP = 1016;
  const int AVP_Default_EPS_Bearer_QoS_AVP = 1049;
  const int AVP_Server_Capabilities_AVP = 603;							//Ref: ETSI TS 129 229 Server_Capabilities_AVP
//  const int AVP_vendIdMandatory_Capability_AVP = 604;							//Ref: ETSI TS 129 229 Mandatory_Capability_AVP
//  const int AVP_vendIdOptional_Capability_AVP = 605;							//Ref: ETSI TS 129 229 Optional_Capability_AVP
  const int AVP_IP_CAN_Type_AVP = 1027;
  const int AVP_RAT_Type_AVP = 1032;                                 //Ref: ETSI TS 129 212 RAT_Type_AVP
  const int AVP_Event_Trigger_AVP = 1006;
  const int AVP_Session_Release_Cause_AVP = 1045;
  const int AVP_Charging_Rule_Name_AVP = 1005;
  const int AVP_AN_GW_Address_AVP = 1050;

  const int AVP_Experimental_Result_AVP = 297;
  const int AVP_Experimental_Result_Code_AVP = 298;
  const int AVP_User_Data_AVP = 606;

  /*end of TS129_229 AVP codes*/

  //tS129_329
  const int AVP_Public_Identity_AVP = 601;
  const int AVP_Server_Name_AVP = 602;
  const int AVP_Feature_List_ID_AVP = 629; //    Ref: 3GPP TS 29.229, clause 6.3.30
  const int AVP_Feature_List_AVP = 630;             //    Ref: 3GPP TS 29.229, clause 6.3.31
  const int AVP_Supported_Features_AVP = 628;                           //    Ref: 3GPP TS 29.329    6.3.11 Supported-Features AVP (See 3GPP TS 29.229, clause 6.3.29)
  /*End TS129_329*/

  //Sh AVP codes

  ////const int AVP_Public_Identity_AVP = 601;
  const int AVP_User_Identity_AVP = 700;
  const int AVP_MSISDN_AVP= 701;
  const int AVP_User_Data_Sh_AVP= 702;
  const int AVP_Data_Reference_AVP = 703;
  const int AVP_Service_Indication_AVP = 704;
  const int AVP_Subs_Req_Type_AVP =705; 
  const int AVP_Requested_Domain_AVP = 706;
  const int AVP_Current_Location_AVP = 707;
  const int AVP_Identity_Set_AVP = 708;
  const int AVP_Expiry_Time_AVP= 709;
  const int AVP_Send_Data_Indication_AVP = 710;
  const int AVP_Repository_Data_ID_AVP = 715;									
  const int AVP_Sequence_Number_AVP = 716;
  ////const int AVP_Server_Name_AVP = 602;
  ////const int AVP_Supported_Features_AVP = 628; 
  ////const int AVP_Feature_List_ID_AVP = 629;
  ////const int AVP_Feature_List_AVP = 630;
  const int AVP_Wildcarded_PSI_AVP = 634;
  const int AVP_Wildcarded_IMPU_AVP = 636;
  const int AVP_DSAI_Tag_AVP = 711;
  ////const int AVP_Experimental_Result_Code_AVP = 298;
  ////const int AVP_Experimental_Result_AVP = 297; 
  ////const int AVP_User_Data_AVP = 606;
  //TS129_214 AVP codes
  const int AVP_Media_Component_Description_AVP = 517;
  const int AVP_Media_Component_Nr_AVP = 518;
  const int AVP_Media_Sub_Component_AVP = 519;
  const int AVP_AF_Application_Identifier_AVP = 504;
  const int AVP_Media_Type_AVP = 520;
  const int AVP_AF_Signalling_Protocol_AVP = 529;
  const int AVP_Reservation_Priority_AVP = 458;
  const int AVP_RS_Bandwidth_AVP = 522;
  const int AVP_RR_Bandwidth_AVP = 521;
  const int AVP_Codec_Data_AVP = 524;
  const int AVP_Flow_Description_AVP = 507;
  const int AVP_Flow_Number_AVP = 509;
  const int AVP_Flow_Status_AVP = 511;
  const int AVP_Flow_Usage_AVP = 512;
  const int AVP_Specific_Action_AVP = 513;
  const int AVP_Service_Info_Status_AVP = 527;
  const int AVP_Rx_Request_Type_AVP = 533;

  //TS129_215 AVP codes
  const int AVP_QoS_Rule_Name_AVP = 1054;
  const int AVP_Rule_Failure_Code_AVP = 1031;
  const int AVP_PDN_Connection_ID_AVP = 1065;
  const int AVP_QoS_Rule_Report_AVP = 1055;
  const int AVP_QoS_Rule_Install_AVP = 1051;
  const int AVP_Multiple_BBERF_Action_AVP = 2204;
  const int AVP_Session_Linking_Indicator_AVP = 1064;
  const int AVP_Subses_Decision_Info_AVP = 2200;
  const int AVP_Subses_Enforcement_Info_AVP = 2201;
  const int AVP_Subsession_Id_AVP = 2202;
  const int AVP_Subsession_Operation_AVP = 2203;
  //End TS129_215 AVP codes

   //TS129_272 AVP codes
   const int AVP_ULR_Flags_AVP = 1405;                                //ref: 7.3.7 ETSI TS 129 272
   const int AVP_ULA_Flags_AVP = 1406;                                //ref: 7.3.8
   const int AVP_Visited_PLMN_Id_AVP = 1407;                          //ref: 7.3.9
   const int AVP_UE_SRVCC_Capability_AVP = 1615;                      //ref: 7.3.130
   const int AVP_SGSN_Number_AVP = 1489;  							//ref: 7.3.102
   const int AVP_Homogeneous_Support_AVP = 1493;  					//ref: 7.3.107

   const int AVP_Context_Identifier_AVP = 1423;                      //ref: 7.3.27
   const int AVP_Service_Selection_AVP  = 493;                       //ref: 7.3.36
   const int AVP_MIP_Home_Agent_Address_AVP = 334;                   //ref: 7.3.42 - IETF RFC 4004 7.4
   const int AVP_MIP_Home_Agent_Host_AVP = 348;                      //ref: 7.3.43 - IETF RFC 4004 7.11
   const int AVP_MIP6_Home_Link_Prefix_AVP = 125;                    //ref:IETF RFC 5447 4.2.4
   const int AVP_MIP6_Agent_Info_AVP    = 486;                       //ref: 7.3.45
   const int AVP_Visited_Network_Identifier_AVP = 600;               //ref: 7.3.105 - TS 129 229 6.3.1
   const int AVP_SIP_Authentication_Context_AVP = 611;
   const int AVP_UAR_Flags_AVP = 637;
   const int AVP_User_Authorization_Type_AVP = 623;
   const int AVP_Server_Assignment_Type_AVP = 614;
   const int AVP_User_Data_Already_Available_AVP = 624;
   const int AVP_SIP_Digest_Authenticate_AVP = 635;
   const int AVP_SCSCF_Restoration_Info_AVP = 639;
   const int AVP_Restoration_Info_AVP = 649;
   const int AVP_Path_AVP = 640;
   const int AVP_Contact_AVP = 641;
   const int AVP_Subscription_Info_AVP = 642;
   const int AVP_Originating_Request_AVP = 652;
   const int AVP_Deregistration_Reason_AVP = 615;
   const int AVP_Reason_Code_AVP = 616;
   const int AVP_Reason_Info_AVP = 617;
   const int AVP_Specific_APN_Info_AVP  = 1472;                      //ref: 7.3.82
   const int AVP_Active_APN_AVP = 1612;                              //ref: 7.3.127
   const int AVP_Error_Diagnostic_AVP = 1614;						//ref: 7.3.128;
   const int AVP_Subscriber_Status_AVP = 1424;
   const int AVP_Operator_Determined_Barring_AVP = 1425;
   const int AVP_HPLMN_ODB_AVP = 1418;
   const int AVP_Access_Restriction_Data_AVP = 1426;
   const int AVP_APN_OI_Replacement_AVP = 1427;
   const int AVP_Max_Requested_Bandwidth_UL_AVP = 516;
   const int AVP_Max_Requested_Bandwidth_DL_AVP = 515;
   const int AVP_APN_Aggregate_Max_Bitrate_UL_AVP = 1041;
   const int AVP_APN_Aggregate_Max_Bitrate_DL_AVP = 1040;
   const int AVP_AMBR_AVP = 1435;

   const int AVP_All_APN_Conf_Included_Id_AVP = 1428;
   const int AVP_APN_Configuration_Profile_AVP = 1429;
   const int AVP_PDN_Type_AVP = 1456;
   const int AVP_QoS_Class_Identifier_AVP=1028;
   const int AVP_Priority_Level_AVP = 1046;
   const int AVP_Pre_emption_Capability_AVP = 1047;
   const int AVP_Pre_emption_Vulnerability_AVP = 1048;

   const int AVP_Allocation_Retention_Priority_AVP =1034;
   const int AVP_EPS_Subscribed_QoS_Profile_AVP = 1431;
   const int AVP_APN_Configuration_AVP = 1430;
   const int AVP_GMLC_Address_AVP = 2405;                            //ref: 7.3.109 - TS 129 173 6.4.7
   //....
   const int AVP_Complete_DL_Included_Indicator_AVP = 1468;
   const int AVP_QoS_Subscribed_AVP = 1404;
   const int AVP_PDP_Type_AVP = 1470;
   const int AVP_PDP_Context_AVP = 1469; 
   const int AVP_GPRS_Subscription_Data_AVP = 1467;
   const int AVP_Subscription_Data_AVP = 1400;                       //ref: 7.3.2
   const int AVP_Immediate_Response_Preferred_AVP = 1412;            //ref: 7.3.16
   const int AVP_Re_synchronization_Info_AVP = 1411;                 //ref: 7.3.15
   const int AVP_Req_EUTRAN_Auth_Info_AVP = 1408;                    //ref: 7.3.11
   const int AVP_Req_UTRAN_GERAN_Auth_Info_AVP = 1409;               //ref: 7.3.12

   const int AVP_Item_Number_AVP = 1419;
   const int AVP_RAND_AVP = 1447;
   const int AVP_XRES_AVP = 1448;
   const int AVP_AUTN_AVP = 1449;
   const int AVP_KASME_AVP = 1450;
   const int AVP_E_UTRAN_Vector_AVP = 1414;
   const int AVP_SIP_Authentication_Scheme_3GPP_AVP = 608;
   const int AVP_SIP_Authenticate_3GPP_AVP = 609;
   const int AVP_SIP_Authorization_3GPP_AVP = 610;
   const int AVP_SIP_Number_Auth_Items_3GPP_AVP = 607;
   const int AVP_SIP_Auth_Data_Item_3GPP_AVP = 612;//ETSI TS 129 229 V10.5.0 (2013-04) Clause 6.3.13 SIP-Auth-Data-Item AVP
   const int AVP_Identity_with_Emergency_Registration_AVP = 651;
   const int AVP_Confidentiality_Key_AVP = 625;
   const int AVP_Integrity_Key_AVP = 626;
   const int AVP_Line_Identifier_AVP = 500;
   const int AVP_UTRAN_Vector_AVP = 1415; 
   const int AVP_SRES_AVP = 1454;
   const int AVP_Kc_AVP = 1453;
   const int AVP_GERAN_Vector_AVP = 1416;
   const int AVP_Authentication_Info_AVP = 1413;

   const int AVP_Cancellation_Type_AVP = 1420;
   const int AVP_CLR_Flags_AVP = 1638;                               //ref: 7.3.152
   const int AVP_IDR_Flags_AVP = 1490;                               //ref: 7.3.103
   const int AVP_DSR_Flags_AVP = 1421;                               //ref: 7.3.25
   const int AVP_DSA_Flags_AVP = 1422;                               //ref: 7.3.26
   const int AVP_PUER_Flags_AVP= 1635;                               //ref: 7.3.149
   const int AVP_PUEA_Flags_AVP= 1442;                               //ref: 7.3.48
   const int AVP_IDA_Flags_AVP = 1441;                               //ref: 7.3.47
   const int AVP_NOR_Flags_AVP = 1443;                               //ref: 7.3.49
   const int AVP_IMS_Voice_PS_Sessions_Support_AVP = 1492;
   const int AVP_Last_UE_Activity_Time_AVP = 1494;
   const int AVP_EPS_User_State_AVP = 1495;
   const int AVP_EPS_Location_Information_AVP = 1496;
   //end TS129_272 AVP codes


  //TS132_299 AVP codes
   const int AVP_Charging_Information_AVP = 618;
   const int AVP_Primary_Event_Charging_Function_Name_AVP = 619;
   const int AVP_Secondary_Event_Charging_Function_Name_AVP = 620;
   const int AVP_Primary_Charging_Collection_Function_Name_AVP = 621;
   const int AVP_Secondary_Charging_Collection_Function_Name_AVP = 622;
//  const int AVP_User_CSG_Information_AVP = 2319; 		//Ref: ETSI TS 132 299 User_CSG_Information_AVP
//  const int AVP_CSG_Id_AVP = 1437; 						//Ref: ETSI TS 132 299 CSG_Id_AVP
//  const int AVP_CSG_Access_Mode_AVP = 2317; 				//Ref: ETSI TS 132 299 CSG_Access_Mode_AVP
//  const int AVP_CSG_Membership_Indication_AVP = 2318;	//Ref: ETSI TS 132 299 CSG_Membership_Indication_AVP
  const int AVP_Service_Information_AVP = 873;			//Ref: ETSI TS 132 299 Service_Information_AVP
  const int AVP_PS_Information_AVP = 874;				//Ref: ETSI TS 132 299 PS_Information_AVP
  const int AVP_WLAN_Information_AVP = 875;				//Ref: ETSI TS 132 299 WLAN_Information_AVP
  const int AVP_IMS_Information_AVP = 876;				//Ref: ETSI TS 132 299 IMS_Information_AVP
//  const int AVP_MMS_Information_AVP = 877;				//Ref: ETSI TS 132 299 MMS_Information_AVP
//  const int AVP_LCS_Information_AVP = 878;				//Ref: ETSI TS 132 299 LCS_Information_AVP
//  const int AVP_PoC_Information_AVP = 879;				//Ref: ETSI TS 132 299 PoC_Information_AVP
//  const int AVP_MBMS_Information_AVP = 880;				//Ref: ETSI TS 132 299 MBMS_Information_AVP
  const int AVP_Service_Generic_Information_AVP = 1256;	//Ref: ETSI TS 132 299 Service_Generic_Information_AVP
  const int AVP_Event_Type_AVP = 825;					//Ref: ETSI TS 132 299 Event_Type_AVP
  const int AVP_Event_AVP = 823;							//Ref: ETSI TS 132 299 Event_AVP
  const int AVP_Expires_AVP = 888;						//Ref: ETSI TS 132 299 Expires_AVP
  const int AVP_Role_Of_Node_AVP = 829;					//Ref: ETSI TS 132 299 Role_Of_Node_AVP
  const int AVP_Node_Functionality_AVP = 862;			//Ref: ETSI TS 132 299 Node_Functionality_AVP
  const int AVP_User_Session_ID_AVP = 830;				//Ref: ETSI TS 132 299 User_Session_ID_AVP
  const int AVP_Calling_Party_Address_AVP = 831;			//Ref: ETSI TS 132 299 Calling_Party_Address_AVP
  const int AVP_Called_Party_Address_AVP = 832;			//Ref: ETSI TS 132 299 Called_Party_Address_AVP
  const int AVP_Called_Asserted_Identity_AVP = 1250;		//Ref: ETSI TS 132 299 Called_Asserted_Identity_AVP
  const int AVP_Alternate_Charged_Party_Address_AVP = 1280;//Ref: ETSI TS 132 299 Alternate_Charged_Party_Address_AVP
  const int AVP_Requested_Party_Address_AVP = 1251;		//Ref: ETSI TS 132 299 Requested_Party_Address_AVP
  const int AVP_Associated_URI_AVP = 856;				//Ref: ETSI TS 132 299 Associated_URI_AVP
  const int AVP_Time_Stamps_AVP = 833;					//Ref: ETSI TS 132 299 Time_Stamps_AVP
  const int AVP_SIP_Request_Timestamp_AVP = 834;			//Ref: ETSI TS 132 299 SIP_Request_Timestamp_AVP
  const int AVP_SIP_Response_Timestamp_AVP = 835;		//Ref: ETSI TS 132 299 SIP_Response_Timestamp_AVP
  const int AVP_Application_Server_Information_AVP = 850;//Ref: ETSI TS 132 299 Application_Server_Information_AVP
  const int AVP_Application_Server_AVP = 836;			//Ref: ETSI TS 132 299 Application_Server_AVP
  const int AVP_Application_Provided_Called_Party_Address_AVP = 837;		//Ref: ETSI TS 132 299 Application_Provided_Called_Party_Address_AVP
  const int AVP_Inter_Operator_Identifier_AVP = 838;		//Ref: ETSI TS 132 299 Inter_Operator_Identifier_AVP
  const int AVP_Originating_IOI_AVP = 839;				//Ref: ETSI TS 132 299 Originating_IOI_AVP
  const int AVP_Terminating_IOI_AVP = 840;				//Ref: ETSI TS 132 299 Terminating_IOI_AVP
  const int AVP_IMS_Charging_Identifier_AVP = 841;		//Ref: ETSI TS 132 299 IMS_Charging_Identifier_AVP
  const int AVP_SDP_Session_Description_AVP = 842;		//Ref: ETSI TS 132 299 SDP_Session_Description_AVP
  const int AVP_SDP_Media_Component_AVP = 844;				//Ref: ETSI TS 132 299 SDP_Media_Name_AVP
//  const int AVP_SDP_Media_Name_AVP = 844;				//Ref: ETSI TS 132 299 SDP_Media_Name_AVP
//  const int AVP_SDP_Media_Description_AVP = 845;			//Ref: ETSI TS 132 299 SDP_Media_Description_AVP
//  const int AVP_Media_Initiator_Flag_AVP = 882;			//Ref: ETSI TS 132 299 Media_Initiator_Flag_AVP
//  const int AVP_Media_Initiator_Party_AVP = 1288;		//Ref: ETSI TS 132 299 Media_Initiator_Party_AVP
//  const int AVP_Authorized_QoS_AVP = 849;				//Ref: ETSI TS 132 299 Authorized_QoS_AVP
  const int AVP_Served_Party_IP_Address_AVP = 848;		//Ref: ETSI TS 132 299 Served_Party_IP_Address_AVP
  const int AVP_Trunk_Group_ID_AVP = 851;				//Ref: ETSI TS 132 299 Trunk_Group_ID_AVP
  const int AVP_Incoming_Trunk_Group_ID_AVP = 852;		//Ref: ETSI TS 132 299 Incoming_Trunk_Group_ID_AVP
  const int AVP_Outgoing_Trunk_Group_ID_AVP = 853;		//Ref: ETSI TS 132 299 Outgoing_Trunk_Group_ID_AVP

  const int AVP_Delivery_Status_AVP =2104;
  const int AVP_Application_Session_ID_AVP=2103;
  const int AVP_Application_Service_Type_AVP=2102;
  const int AVP_Application_Server_ID_AVP=2101;

  const int AVP_Reporting_Reason_AVP=872;
  const int AVP_Time_Quota_Threshold_AVP=868;
  const int AVP_Volume_Quota_Threshold_AVP=869;
  const int AVP_Unit_Quota_Threshold_AVP=1226;
  const int AVP_Quota_Holding_Time_AVP=871;
  const int AVP_Quota_Consumption_Time_AVP=881;
  const int AVP_Time_Quota_Type_AVP=1271;
  const int AVP_Base_Time_Interval_AVP=1265;
  const int AVP_Time_Quota_Mechanism_AVP=1270;
  const int AVP_Envelope_Start_Time_AVP=1269;
  const int AVP_Envelope_End_Time_AVP=1267;
  const int AVP_Envelope_AVP=1266;
  const int AVP_Envelope_Reporting_AVP=1268;
  const int AVP_Trigger_AVP=1264;
  const int AVP_Trigger_Type_AVP=870;
  const int AVP_Refund_Information_AVP=2022;
  const int AVP_Multiple_Services_Credit_Control_AVP = 456;

  const int AVP_Offline_Charging_AVP = 1278;
  const int AVP_Low_Balance_Indication_AVP=2020;
  const int AVP_Remaining_Balance_AVP = 2021;
  //end of TS132_299 AVP codes

private:
  //NOTE: The order of constants in the following array is very important.
  //It must exactly match order specified in TTCN-3 module.
  static const int _command_codes[];

}; // End of class diameter_codec
