/*!
 * \file      NGAPPort.cc
 * \brief     CC file for NGAPPort.
 * \author    ETSI TTF041
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */

#include "loggers.hh"

#include "LibNGAP_Interface.hh"
#include "NGAP_PDU_Descriptions.hh"
#include "ngap_layer_factory.hh"
#include "layer.hh"
#include "params.hh"

namespace NGAP__PDU__Descriptions{
  class NGAP__PDU;
}
namespace LibNGAP__Interface {

  NGAPPort::NGAPPort(const char *par_port_name): NGAPPort_BASE(par_port_name), _cfg_params(), _layer_params(), _layer(nullptr), _time_key("NGAPPort::") {
    // Nothing to do
  } // End of constructor

  NGAPPort::~NGAPPort() {
    loggers::get_instance().log(">>> NGAPPort::~NGAPPort");

    if (_layer != nullptr) {
      delete _layer;
    }
  } // End of destructor

  void NGAPPort::set_parameter(const char * parameter_name, const char * parameter_value)
  {
    loggers::get_instance().log("NGAPPort::set_parameter: %s=%s", parameter_name, parameter_value);
    _cfg_params.insert(std::pair<std::string, std::string>(std::string(parameter_name), std::string(parameter_value)));
  }

  /*void HttpPort::Handle_Fd_Event(int fd, boolean is_readable,
    boolean is_writable, boolean is_error) {}*/

  void NGAPPort::Handle_Fd_Event_Error(int /*fd*/)
  {

  }

  void NGAPPort::Handle_Fd_Event_Writable(int /*fd*/)
  {

  }

  void NGAPPort::Handle_Fd_Event_Readable(int /*fd*/)
  {

  }

  /*void HttpPort::Handle_Timeout(double time_since_last_call) {}*/

  void NGAPPort::user_map(const char * system_port)
  {
    loggers::get_instance().log(">>> NGAPPort::user_map: %s", system_port);
    // Build layer stack
    params::iterator it = _cfg_params.find(std::string("params"));
    if (it != _cfg_params.end()) {
      loggers::get_instance().log("NGAPPort::user_map: %s", it->second.c_str());
      // Setup parameters
      params::convert(_layer_params, it->second); // TODO This _layer_params seems to be useless
      // Create layer
      _layer = layer_stack_builder::get_instance()->create_layer_stack(it->second.c_str());

      if (static_cast<ngap_layer*>(_layer) == nullptr) {
        loggers::get_instance().error("NGAPPort::user_map: Invalid stack configurationi - NOT SCTP: %s", it->second.c_str());
      }
      static_cast<ngap_layer*>(_layer)->add_upper_port(this);

      // Create layer
      _layer = layer_stack_builder::get_instance()->create_layer_stack(it->second.c_str());
      if (static_cast<ngap_layer*>(_layer) == NULL) {
        loggers::get_instance().error("NGAPPort::user_map: Invalid stack configuration: %s", it->second.c_str());
      }
      static_cast<ngap_layer*>(_layer)->add_upper_port(this);
    } else {
      loggers::get_instance().error("NGAPPort::user_map: No layers defined in configuration file");
    }
  } // End of user_map method

  void NGAPPort::user_unmap(const char * system_port)
  {
    loggers::get_instance().log(">>> NGAPPort:user_unmap: %s", system_port);

    // Reset layers
    if (_layer != nullptr) {
      delete _layer;
      _layer = nullptr;
    }
  } // End of user_unmap method

  void NGAPPort::user_start()
  {
    loggers::get_instance().log(">>> NGAPPort::user_start");

  } // End of user_start method

  void NGAPPort::user_stop()
  {
    loggers::get_instance().log(">>> NGAPPort::user_stop");

  } // End of user_stop method

  void NGAPPort::outgoing_send(const NGAP__PDU__Descriptions::NGAP__PDU& send_par)
  {
    loggers::get_instance().log_msg(">>> NGAPPort::outgoing_send: payload=", send_par);
    
    float duration;
    loggers::get_instance().set_start_time(_time_key);
    params params;
    static_cast<ngap_layer*>(_layer)->sendMsg(send_par, params);
    loggers::get_instance().set_stop_time(_time_key, duration);
  }

  void NGAPPort::receiveMsg (const  NGAP__PDU__Descriptions::NGAP__PDU& p_ind, const params& p_params) {
    loggers::get_instance().log_msg(">>> NGAPPort::receive_msg: ", p_ind);
    float duration;
    loggers::get_instance().set_start_time(_time_key);
    // Sanity check
    if (!p_ind.is_bound()) {
      return;
    }

    incoming_message(p_ind);
    loggers::get_instance().set_stop_time(_time_key, duration);
  }
  
} // End of namespace LibNGAP__Interface

