-- © 2025 Nokia
-- Licensed under the BSD 3-Clause License
-- SPDX-License-Identifier: BSD-3-Clause

-- Store headerIDs with a counter for duplicates
local headerIDs = {}
local tocHTML = {}

function getHeaderLabelBase(headerText)
    -- Convert header name to a suitable link text
    
    -- Strip leading non-alphabet characters
    local startPos = headerText:find("[A-Za-z]")
    local headerLabelBase_1 = headerText:sub(startPos)
    
    -- Convert string to lowercase
    local headerLabelBase_2 = string.lower(headerLabelBase_1)

    -- Convert spaces to hyphens
    local headerLabelBase_3 = headerLabelBase_2:gsub(" ", "-")
    
    -- Remove ampersands
    local headerLabelBase_4 = headerLabelBase_3:gsub("%-%&%-", "-")
    
    -- Remove other non-alpha characters
    local headerLabelBase = headerLabelBase_4:gsub("[^%w-_.]", "")
        
    return headerLabelBase

end

function getHeaderLabel(headerText)
    -- Check if the trimmed header label already exists.
    -- If it does, then increment the label counter and
    -- append -<counter value> to the end of the label
    -- and return it.

    local headerLabelBase = getHeaderLabelBase(headerText)

    if headerIDs[headerLabelBase] == nil then
        headerIDs[headerLabelBase] = 0
        return headerLabelBase
    else
        headerIDs[headerLabelBase] = headerIDs[headerLabelBase] + 1
        return headerLabelBase .. "-" .. tostring(headerIDs[headerLabelBase])
    end
end

function getTOCString(headerText)
    -- Create HTML for the table of contents line

    local tocEntryHTMLTemplate = '<p class="nospace">%s <span class="page-number"><a href="#%s"></a></span></p>'

    local headerLabel = getHeaderLabel(headerText)
    
    return string.format(tocEntryHTMLTemplate, headerText, headerLabel)

end

function Header(header)
    local headerText = pandoc.utils.stringify(header.content)

    if string.find(headerText, "%a") ~= nil then
        -- print(getTOCString(headerText))
        table.insert(tocHTML, getTOCString(headerText))
    else
        print("Warning: Invalid header text found.")
    end
end

function Pandoc(doc)
    
    table.insert(tocHTML, 1, "<h1 class=\"TT\" id=\"contents\">Contents</h1>")
    table.insert(tocHTML, "<div class=\"pagebreak\"> </div>")
    
    tocHTMLBlock = pandoc.RawBlock("html", table.concat(tocHTML, "\r\n"))
    table.insert(doc.blocks, 1, tocHTMLBlock)

    return doc

end