-- © 2025 Nokia
-- Licensed under the BSD 3-Clause License
-- SPDX-License-Identifier: BSD-3-Clause

-- This pandoc lua script colorizes and indents ASN.1 inline code with HTML.

local keywords = {
    "BIT STRING",
	"BOOLEAN",
	"CHOICE",
	"ENUMERATED",
	"INTEGER",
	"OCTET STRING",
	"OF",
	"OPTIONAL", 
	"SEQUENCE",
	"SIZE", 
}

local function replaceInsideString(strIn, idxF, idxL, strRepl)

	-- strIn: original string
	-- idxF: first index of the text to replace
	-- idxL: last index of the text to replace
	-- strRepl: replacement text

	local firstPart = strIn:sub(1, idxF -1)
	local lastPart = strIn:sub(idxL + 1, string.len(strIn))
	return firstPart..strRepl..lastPart

end

local function colorizeString(str, colorClass)
	colorizedString = string.format('<span class="%s">%s</span>', colorClass, str)
	return colorizedString
end

local function colorizeKeywords(str)

	local colorClass = 'asn1-keyword'

	local processingString = str
	local processingIndex = 1

	for _, keyword in pairs(keywords) do
		processingIndex = 1
		while true do
			firstIndex, lastIndex = processingString:find(keyword, processingIndex)
			if not firstIndex then
				break
			end
			
			colorizedString = colorizeString(keyword, colorClass)
			processingString = replaceInsideString(processingString, firstIndex, lastIndex, colorizedString)
			processingIndex = firstIndex + string.len(colorizedString)
		end
	end

	return processingString
end

local function colorizeComments(str)

	local colorClass = 'asn1-comment'
	local commentPattern = "%-%-[^\r\n]+"
	
	local processingString = str
	local processingIndex = 1
	
	while true do
		firstIndex, lastIndex = processingString:find(commentPattern, processingIndex)
		if not firstIndex then
			break
		end
		
		colorizedString = colorizeString(processingString:sub(firstIndex, lastIndex), colorClass)
		processingString = replaceInsideString(processingString, firstIndex, lastIndex, colorizedString)
		processingIndex = firstIndex + string.len(colorizedString)
	end

	return processingString
end

local function indentASN1Source(str)

	local indentationLevel = 0
	local processingString = ""
	local openIndentDivTemplate = '<div class="asn1-indent-%d">'
	local closeDiv = '</div>'
	
	-- match all characters except newlines.
	for line in string.gmatch(str, "([^\n\r]+)") do
		--local has_open_brace = line:find("{")
		--local has_close_brace = line:find("}")
		
		-- usually braces that would require indenting the next line
		-- are at the very end of the line, before the new line character.
		local has_open_brace = (line:sub(-1) == "{")
		
		-- usually braces that would end an indentation level are
		-- at the very beginning of the line.
		local has_close_brace = (line:sub(1,1) == "}")
		
		if has_close_brace and not has_open_brace then
			-- Temporary fix in case there isn't a new line after a braces
			-- that should be opening a new level of hierarchy.
			indentationLevel = math.max(0, indentationLevel -1)
		end
		
		openIndentDiv = string.format(openIndentDivTemplate, indentationLevel)
		processingString = processingString..openIndentDiv..line..closeDiv.."\n"
		
		if has_open_brace and not has_close_brace then
			indentationLevel = indentationLevel + 1
		end
	end
	
	return processingString

end

local function wrapDivASN1Block(asn1Text)

	return '<div class="asn1">'..asn1Text..'</div>'

end



function CodeBlock(block)
	if block.classes[1] == "asn1" then
		asn1Text = colorizeKeywords(block.text)
		asn1Text = colorizeComments(asn1Text)
		asn1Text = indentASN1Source(asn1Text)
		asn1Text = wrapDivASN1Block(asn1Text)
		return pandoc.Para(pandoc.RawInline('html', asn1Text))
	end
	return block
end
