# Introduction
This Markdown Specification project implements an example of a structure to store and collaborate on a 3GPP specification in a git repository. The base version of the specification is v18.0.0 of TS38.331. The example recreates the RILs for the UAV work item starting in RAN2#125. The RILs are named as they were in that review, the review file for which can be found linked below in the resources section. The RILs were recreated to closely match their corresponding edits to the frozen specification, as if each RIL were submitted with text proposals for changes. The RILs were then implemented through merges and merge conflict resolution.

# Git
To get started with the basics of Git, it is recommended to read the [documentation](https://git-scm.com/book/en/v2). The most important sections to check are "Getting Started" and "Git Basics". For most, the following Git commands will be the most useful.

| Command | Function |
|----|----|
| git status | Print the status of the files in the git repository. Staged (added for commit) and unstaged files will be listed. This view is a good way to verify that all the intended files have been added prior to committing changes to ETSI Forge. |
| git branch | Print all of the branch names associated with the repository. Add -v, i.e., git branch -v, to show the latest commit message for each branch. |
| git checkout \<branch\> | Switch to branch `<branch>` |
| git checkout -b \<branch\> | Create a branch `<branch>` and swich to it. |
| git add <filename> | Add a file to the staging area for committing to the repository. It is best practice to specify each file separately. Never try to use the -a flag to add everything at once. |
| git commit | Commit staged (added) files to the repository. Without arguments, a text editor will launch where a commit message can be written. |
| git commit -t <template> | Commit staged (added) files to the repository. A text editor will open with the text of the specified template `<template>`, to be filled in. |
| git commit -m <message> | Commit staged (added) files to the repository with the commit message `<message>`. |
| git fetch | Retrieve the latest changes from the remote repository, including all new branches and modifications to existing branches. |
| git push origin \<branch\> | Push the committed changes of branch `<branch>` to the remote repository. |

It is important to ensure that the changes, which are first committed locally, are as intended prior to pushing changes to the repository. The following example shows a typical procedure for proposing changes to a file. The company name, which could eventually be a simpler identifier, is included as part of the branch name to avoid conflicts. The example is using Git BASH. Lines starting with \$ identify commands issued by the user. Lines starting with \> identify command outputs, and all other text is explanatory.


The Git repository is only cloned the first time.

`$ git clone ssh://git@forge.etsi.org:29419/fevold/markdown-specification.git`

Changes to the Git repository are only fetched in subsequent interactions with the repository.

`$ git fetch origin`

Create and checkout a new branch, identified by the WI code, company, and a short description.

```
$ git checkout -b NR_AIML_Air-core_Nokia_applicability_report
> Switched to a new branch 'General_Nokia_spec_version'
```

Edit 38331-i00.md

Verify changes implemented in the new branch.

```
$ git diff --word-diff main 38331-i00.md
> diff --git a/38331-i00.md b/38331-i00.md
> index 7528b2c..499968c 100644
> --- a/38331-i00.md
> +++ b/38331-i00.md
> @@ -1,4 +1,4 @@
> 3GPP TS 38.331 [-V18.0.0 (2023-12)-]{+V19.0.0 (2026-03)+}
> 
> ---
> 
> @@ -12,7 +12,7 @@ NR;
> 
> Radio Resource Control (RRC) protocol specification
> 
> (Release [-18)-]{+19)+}
> 
> ---
```

Check the status after modifying the file. Note that it is modified, but not staged for commit.

```
$ git status
> On branch General_Nokia_spec_version
> Changes not staged for commit:
>   (use "git add <file>..." to update what will be committed)
>   (use "git restore <file>..." to discard changes in working directory)
>         modified:   38331-i00.md
```

Add the file to be committed.

`$ git add 38331-i00.md`

Check the status after modifying the file. Note that it is shown as "to be committed".

```
$ git status
On branch NR_AIML_Air-core_Nokia_applicability_report
Changes to be committed:
  (use "git restore --staged <file>..." to unstage)
        modified:   38331-i00.md
```

Once satisfied with the changes, commit the changes with a message.

`$ git commit -m "Updated specification version to 19, v19.0.0."`

Push the branch to ETSI Forge.

`$ git push origin General_Nokia_spec_version`

# RILs
RILs are comments or direct suggestions for changes to a frozen release under review. Historically, RILs were written such that they would point to a specific example in the specification, with discussion that changes should apply to each relevant section of the specification. However, the RILs for this project have been created differently. Here, the RILs are able to modify related text throughout the specification as to propose a full set of common changes at the same time. For example, procedural text related to releasing configurations might need to be updated in several parts of the specification to address the same issue. One RIL could implement the chagnes all at once.

# Demonstration
In this demonstration, 27 RILs, which were agreed to be implemented in the specification, were transcribed from the initial review of 38.331 v18.0.0. Each RIL modifies the specification text directly and is stored in its own branch.

## Structure
The following structure has been developed for the demonstration. It is a first attempt, and is up for discussion.

* **main** - Contains the clean specification v18.0.0, and would eventually contain final versions of the modified specifications, e.g., v18.1.0.
  * **NR_UAV-core_R2_125** - Contains all changes related to a WI, which can eventually be merged as a running CR, either into the specification or into another running CR capturing other WIs.
    * **NR_UAV-core_R2_125_RIL** - Contains all changes related to the RIL for this WI. The agreed RILs would eventually be merged here.
      * **NR_UAV-core_R2_125_RIL_** - The RIL branches contain changes that make focused modifications ot the specification and can include changes in more than one part of the specification, but the changes should be related to one another.
      * **NR_UAV-core_R2_125_impl_v0** - Contains the first set of implemented changes as shown in the the running CR for the WI, which can be found [here](https://www.3gpp.org/ftp/tsg_ran/WG2_RL2/TSGR2_125/Docs/R2-2401605.zip)
        * **NR_UAV-core_R2_125_impl_v1** - Adds changes from a second set of changes to demonstrate how further changes could come during a meeting or in a revised CR.
          * **NR_UAV-core_R2_125_impl_v2** - Adds editorial changes and any inconsistently or uncompletely implemented changes.

As is demonstrated by the structure, RILs are submitted independently from one another and are merged in as they are agreed. Branches can derive from other branches, creating a hierarchy. Because the branches were based on the clean specification, the final set of changes are fully compatible with the clean specification and can be merged in as a set.

Check the [repository graph](https://forge.etsi.org/rep/fevold/markdown-specification/-/network/main?ref_type=heads) for a visualization of the branches and how they were merged into the implementation branches, impl_v0, impl_v1, and impl_v2.

# RIL Report

## Simple Report
RIL reports could come in different forms. On the most basic level, the branches and an excerpt from their comment can be easily produced directly by the git tool as shown or in the [web interface](https://forge.etsi.org/rep/fevold/markdown-specification/-/branches/active).

Note that this simple example can be improved upon with filtering and a full listing of the commoent associated with the latest commit to the branch. The web tool or the command-line tool can be used to determine the difference proposed by the change.

```git branch -v
NR_UAV-core_R2_125                    d256c74 Initial, clean commit of 38331 v18.0.0
NR_UAV-core_R2_125_RIL                d256c74 Initial, clean commit of 38331 v18.0.0
NR_UAV-core_R2_125_RIL_C003           6e9180e [RIL]: C003 [Delegate]: CATT (Tangxun) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_C004           da03d55 [RIL]: C004 [Delegate]: CATT (Tangxun) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_C008           098ce10 [RIL]: C008 [Delegate]: <CATT (Name)> [Class]: 1 [Status]: PropAgree [TDoc]: None [Proposed Conclusion]:
NR_UAV-core_R2_125_RIL_C009           21da34c [RIL]: C009 [Delegate]: CATT (Tangxun) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_C018           783620d [RIL]: C018 [Delegate]: <CATT (Tangxun)> [Class]: UAV [Status]: ToDo [TDoc]: None
NR_UAV-core_R2_125_RIL_E081           3ffa9c2 [RIL]: E081 [Delegate]: Ericsson (Helka-Liina) [Class]: 1 [Status]: PropAgree [TDoc]: None
NR_UAV-core_R2_125_RIL_E085           c9e885a [RIL]: E085 [Delegate]: Ericsson (Helka-Liina) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_E086           e745901 [RIL]: E086 [Delegate]: Ericsson (Helka-Liina) [Class]: 1 [Status]: PropAgree [TDoc]:
NR_UAV-core_R2_125_RIL_E094           42ae3be [RIL]: E094 [Delegate]: Ericsson (Helka-Liina) [Class]: 1 [Status]: PropAgree [TDoc]:
NR_UAV-core_R2_125_RIL_E122           a8b66ef [RIL]: E122 [Delegate]: Ericsson (Nithin) [Class]: 1 [Status]: PropAgree [TDoc]:
NR_UAV-core_R2_125_RIL_E125           488899f [RIL]: E125 [Delegate]: Ericsson (Nithin) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_E127           c5db256 [RIL]: E127 [Delegate]: Ericsson (Nithin) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_E129           bb9e70f [RIL]: E129 [Delegate]: Ericsson (Nithin) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_H059           2225895 [RIL]: H059 [Delegate]: Huawei (Lili) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_H743           bb5c0d8 [RIL]: H743 [Delegate]: Huawei (Simone) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_H745           7f695ab [RIL]: H745 [Delegate]: Huawei (Simone) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_I115           6e88e40 [RIL]: I115 [Delegate]: Intel [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_I116           62a7e6e [RIL]: I116 [Delegate]: Intel (Sudeep) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_J061           0ded1f2 [RIL]: J061 [Delegate]: Sharp (LIU Lei) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_J065           dadf1f5 [RIL]:  Sharp [Delegate]: LIU Lei [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_J074           e40c7b5 [RIL]: J074 [Delegate]: Sharp (LIU Lei) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_L004           52c4685 [RIL]:  L004 [Delegate]: LGE (Soo Kim) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_Rapp_C024_L003 3a292ef [RIL]: Rapp C024 L003 [Delegate]: < ()> [Class]: [Status]: [TDoc]:
NR_UAV-core_R2_125_RIL_W012           4a3daf2 [RIL]: W012 [Delegate]: NEC (Zonghui XIE) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_W014           7199ca4 [RIL]: W014 [Delegate]: NEC (Zonghui XIE) [Class]: 1 [Status]: ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_Z074           ddd53f0 [RIL]: Z074 [Delegate]: ZTE (ZhangMengjie) [Class]: 1 [Status]:ToDo [TDoc]:
NR_UAV-core_R2_125_RIL_Z075           59ce45f [RIL]: Z075 [Delegate]: ZTE (ZhangMengjie) [Class]: 1 [Status]: ToDo [TDoc]:
```

# Resources
[RIL File for 38.331, Review v229](https://www.3gpp.org/ftp/Email_Discussions/RAN2/%5BMisc%5D/ASN1%20review/Rel-18%202024-03/38331/ASN1%20review%20file/38331%20Rel18%20ASN1%20review%20v229.docx)

[Example PDF of 38.331-i00](https://forge.etsi.org/rep/fevold/markdown-specification/-/raw/main/examples/38331-i00.pdf?ref_type=heads&inline=false)

[Example PDF of 38.331-i00 after RIL](https://forge.etsi.org/rep/fevold/markdown-specification/-/raw/main/examples/38331-i00_impl_v2.pdf?ref_type=heads&inline=false)

[Example RIL Report](https://forge.etsi.org/rep/fevold/markdown-specification/-/raw/main/examples/38331-i00_ril_report.html?ref_type=heads&inline=false)

[Example Running CR](https://forge.etsi.org/rep/fevold/markdown-specification/-/raw/main/examples/impl_v2_cr.docx?ref_type=heads&inline=false)