/*
    This file implements ETSI TC CYBER QSC Quantum-safe Hybrid Key Exchanges
    (Version 1.1.1)

    This is not intended for production use.  It is intended to be a reference
    implementation for test vectors for the specification.

    It uses OpenSSL version 1.1.1d libcrypto.

    gcc -Wall -o etsi-hkex-test main.c qshkex.c -lcrypto
    ./etsi-hkex-test

    Copyright 2020 ETSI. All rights reserved
    SPDX-License-Identifier: BSD-3-Clause
*/

#include "qshkex.h"

#define TEST_VECTOR_CNT             7
#define HASH_FUNCTIONS_CNT          2
#define MAX_LABEL_BYTE_LEN          64
#define MAX_KEY_BYTE_LEN            128
#define MAX_LABEL_LEN               32
#define MAX_MSG_BYTE_LEN            4096
#define CONCAT_CONTEXT_TEST_VECTOR  "CONCATENATION TEST VECTOR 1"
#define CASCADE_CONTEXT_TEST_VECTOR "CASCADE TEST VECTOR 1"

const EVP_MD *(*evp_hash[TEST_VECTOR_CNT][HASH_FUNCTIONS_CNT])(void) = {
    {EVP_sha256, EVP_sha3_256},
    {EVP_sha384, EVP_sha3_384},
    {EVP_sha384, EVP_sha3_384},
    {EVP_sha512, EVP_sha3_512},
    {EVP_sha256, EVP_sha3_256},
    {EVP_sha384, EVP_sha3_384},
    {EVP_sha512, EVP_sha3_512},
};

const uint32_t key_length[TEST_VECTOR_CNT] = {16, 24, 24, 32, 16, 24, 32};
/* Label contribution from A */
const char *strLA1[] = {
    "0102030405060708090A0B0C0D0E0F10",
    "1102030405060708090A0B0C0D0E0F100102030405060708",
    "2102030405060708090A0B0C0D0E0F100102030405060709",
    "3102030405060708090A0B0C0D0E0F100102030405060708090A0B0C0D0E0F10",
    "10102030405060708090A0B0C0D0E0F1",
    "21102030405060708090A0B0C0D0E0F10010203040506070",
    "33102030405060708090A0B0C0D0E0F100102030405060708090A0B0C0D0E0F1"};
const char *strLA2[] = {
    "4102030405060708090A0B0C0D0E0F11",
    "5102030405060708090A0B0C0D0E0F110102030405060708",
    "6102030405060708090A0B0C0D0E0F110102030405060709",
    "7102030405060708090A0B0C0D0E0F110102030405060708090A0B0C0D0E0F11",
    "14102030405060708090A0B0C0D0E0F1",
    "25102030405060708090A0B0C0D0E0F11010203040506070",
    "3102030405060708090A0B0C0D0E0F110102030405060708090A0B0C0D0E0F11"};

const char *strPA1[] = {
    /* ECDH (NIST P-256, NIST P-384, NIST P-521) public keys from NIST CAVP SP
     * 800-56A ECCCDH Primitive Test Vectors.
     * http://csrc.nist.gov/groups/STM/cavp/documents/components
     * /ecccdhtestvectors.zip
     */
    "119F2F047902782AB0C9E27A54AFF5EB9B964829CA99C06B02DDBA95B0A3F6D08F52B72666\
4CAC366FC98AC7A012B2682CBD962E5ACB544671D41B9445704D1D",
    "9803807F2F6D2FD966CDD0290BD410C0190352FBEC7FF6247DE1302DF86F25D34FE4A97BEF\
60CFF548355C015DBB3E5FBA26CA69EC2F5B5D9DAD20CC9DA711383A9DBE34EA3FA5A2AF75B4650\
2629AD54DD8B7D73A8ABB06A3A3BE47D650CC99",
    "EA4018F5A307C379180BF6A62FD2CECEEBEEB7D4DF063A66FB838AA35243419791F7E2C9D4\
803C9319AA0EB03C416B6668835A91484F05EF028284DF6436FB88FFEBABCDD69AB0133E6735A1B\
CFB37203D10D340A8328A7B68770CA75878A1A6",
    "00602F9D0CF9E526B29E22381C203C48A886C2B0673033366314F1FFBCBA240BA42F4EF38A\
76174635F91E6B4ED34275EB01C8467D05CA80315BF1A7BBD945F550A501B7C85F26F5D4B2D7355\
CF6B02117659943762B6D1DB5AB4F1DBC44CE7B2946EB6C7DE342962893FD387D1B73D7A8672D1F\
236961170B7EB3579953EE5CDC88CD2D",
    "EAD218590119E8876B29146FF89CA61770C4EDBBF97D38CE385ED281D8A6B23028AF61281F\
D35E2FA7002523ACC85A429CB06EE6648325389F59EDFCE1405141",
    "FCFCEA085E8CF74D0DCED1620BA8423694F903A219BBF901B0B59D6AC81BAAD316A242BA32\
BDE85CB248119B852FAB66972E3C68C7AB402C5836F2A16ED451A33120A7750A6039F3FF15388EE\
622B7065F7122BF6D51AEFBC29B37B03404581B",
    "00D45615ED5D37FDE699610A62CD43BA76BEDD8F85ED31005FE00D6450FBBD101291ABD96D\
4945A8B57BC73B3FE9F4671105309EC9B6879D0551D930DAC8BA45D25501425332844E592B440C0\
027972AD1526431C06732DF19CD46A242172D4DD67C2C8C99DFC22E49949A56CF90C6473635CE82\
F25B33682FB19BC33BD910ED8CE3A7FA"};

const char *strPA2[] = {
    /* SIKE (sikep434, sikep503, sikep610, sikep751) public keys from Round 3 KATs,
     * https://csrc.nist.gov/CSRC/media/Projects/post-quantum-cryptography
     * /documents/round-3/submissions/SIKE-Round3.zip
     */
    "4484D7AADB44B40CC180DC568B2C142A60E6E2863F5988614A6215254B2F5F6F79B48F329A\
D1A2DED20B7ABAB10F7DBF59C3E20B59A700093060D2A44ACDC0083A53CF0808E0B3A827C45176B\
EE0DC6EC7CC16461E38461C12451BB95191407C1E942BB50D4C7B25A49C644B630159E6C4036538\
38E689FBF4A7ADEA693ED0657BA4A724786AF7953F7BA6E15F9BBF9F5007FB711569E72ACAB05D3\
463A458536CAB647F00C205D27D5311B2A5113D4B26548000DB237515931A040804E769361F94FF\
0167C78353D2630A1E6F595A1F80E87F6A5BCD679D7A64C5006F6191D4ADEFA1EA67F6388B7017D\
453F4FE2DFE80CCC709000B52175BFC3ADE52ECCB0CEBE1654F89D39131C357EACB61E5F13C80AB\
0165B7714D6BE6DF65F8DE73FF47B7F3304639F0903653ECCFA252F6E2104C4ABAD3C33AF24FD0E\
56F58DB92CC66859766035419AB2DF600",
    "05279D27FF7E3A38ABB05DCFE23B5831C030D832D3EAE35FE06A6538597532D22A0F4012FB\
2263E160495F8291B58D9DF8A8947C7CF3E6735520BB2D094912408829851AC4B85AA922069F2AA\
A0A4827DFA4730E9CF05485CBEE411C3D5169DD4953746B6A2E6574957EF920596B1612BE62A883\
740B5A0C157117AE1C3A07E4CE8CCCE7E9E88FE7C20A507FF019AE0893F34303E173D291F6CB7EC\
B4C3901FF34A48DE40771F5BAD72DA2B4C1CFD0A61F33E39327A8DA60F5640D4C2E71EF9C7297A4\
E9BC50493E3BA65D3664610A6D61035CB6600378D017D1E1810ACD113252D60F5915749C2B5CFB4\
452C40C86F1F40C63297DCCA900686F2D2266F9444539D9BA13B1F52FB2FC3BD4F3EDAA6EB707AA\
FCA5261EA271ED961B2ED195D5E3B0299179251866CE0EAA31C5C90B7999A8D63BA2DE84A8AFA19\
F11F2DC0CACA39B982CE053F71D269931D9EE26BCE592A8EA818553BC8F8D244F62FB4F5E5386E3\
EFF5CD231401C9EC2BA57FF42DC3B3791357A53E1E31394008",
    "671B24769304DD18C97AF0C5DE741C53E0B45A9E18C7A13A15C1758125E41605587E450F84\
52A2BF98B51C2AF6B0503CB8E01F8553C36079EBFADF4948FFA063ABF4866E7AB9B9D4C9A07CA40\
0C613607E6DB9BB6E7EB8ECA78894C7C8CE9E231B33179B2946C5C5BE1C783FA6AEA218F5EC4B4E\
6F914E5ED3724C5D7B79403F68438A40775E964C1B2C7D22E11A6C07474EB5D4CFF75965B400167\
E069FA9908A562DBABF5E30FED46BBA0A208ED4E50764CF320FB8556F07C7F6268084476A47D83B\
085DC77EB3CD30A2B5EE1E5829738077D52A0D7A4149EE9C1A70269BC047B4BE7E5B28007DEF74A\
4D813853396708A3A8498CC862F54015B79047014639EB8CA3BB786B27A2CFAF31E6BB9CCB152BE\
B3232465206973668597AA35EE1940A316F71241FA40D1AC233931E1967E79AAA600AA6D83FEC62\
80A63924E7375F22F7A47E1DE483FEA17E0DACBAEDBB13D58C0DC9BC21F2DC9525D46E4210AC5D8\
8567E4F23304EA5BE08D89D57A0246EA21C0CD28C096366D7F3C8D98F5A1FB00FE2F3A183E53A7E\
8B6C19E9BF979E8D20C703C957D6F06A142BE86A0A09B05ED40953BBD7A15E92098633941730DEB\
5BC1C5F5154E8BCA38E035580E101E6EE858D91BD8462B906EB2004C6E01",
    "E1A758EC0D418BFE86D8077B5BB169133C06C1F2A067D8B202D9D058FFC51F63FD26155A65\
77C74BA7F1A27E7BA51982517B923615DEB00BE408920A07831DF5978CFDDD0BF690A264353A4A1\
6B666F90586D7F89A193CE09375D389C1379A7A528581C3ACB002CD2DC4F0FD672568FF9050BA83\
65C7FEFC5E6ED089B921DE6804091A0744DE3EB14D426A3F7DA215C50312617C1C2697243980D06\
056F2CCE88AE7AE73C7343C0B7104C9F2870A94FED744CF6E94630514B6CEAB0E64733BB6FA67B9\
31E5D8206010475CBE8BC587248D65D89D8CD9C8BBFA93E8B5F9EB9130773DED665D52ABBD91C4C\
8C255F73C0FC82501AE33330E9F308DE7177CBF83E4E26E334D7CB09019E638147FC58ED372AF66\
0F14C194BC80E9666325C98E0F80877271D4A6BF514F603703D8A697874CD50A34D92F5AAEA8463\
3CCF96801BD517BF425DEE4A32AAF06684052473EA14643C3D535440FB2240A988D09F297C5A388\
CB3DE60ED943F124034B90EFF611221F80F78EC124956338A105F6636B063D7E48BFBD5D614310F\
B97D86F122E4AE6F9DDF4977A93ED7D0CE2A94E346A1A03D3219CF21907B85A5BCDC713F93A4406\
A22E03B1655A66E1F6741A2F953E6FE0868B2614BABEF1943BBBCB1B66D3E7017E533EA84F29124\
0B56AB33EF1DC3F3DE99DBF9E8BE51A0076E462BCDD825EA96D7F63C99177C305C257B31461F4C2\
3D43115F0220409E8880BBB2468586D03461E807BE824B693874911B2B52AF06FDBDC47F5A01597\
29641A7C950AB9E03F2DC045135",
    /* KYBER (kyber512, kyber768, kyber1024) public from Round 3 KATs,
     * https://csrc.nist.gov/CSRC/media/Projects/post-quantum-cryptography
     * /documents/round-3/submissions/Kyber-Round3.zip
     */
    "115ACE0E64677CBB7DCFC93C16D3A305F67615A488D711AA56698C5663AB7AC9CE66D547C0\
595F98A43F4650BBE08C364D976789117D34F6AE51AC063CB55C6CA32558227DFEF807D19C30DE4\
14424097F6AA236A1053B4A07A76BE372A5C6B6002791EBE0AFDAF54E1CA237FF545BA68343E745\
C04AD1639DBC590346B6B9569B56DBBFE53151913066E5C85527DC9468110A136A411497C227DCB\
8C9B25570B7A0E42AADA6709F23208F5D496EBAB7843F6483BF0C0C73A40296EC2C6440001394C9\
9CA173D5C775B7F415D02A5A26A07407918587C41169F2B7178755ACC27FC8B19C4C4B3FCD41053\
F2C74C8A10A8321241B2802432875AE808B9EF1365C7B8A52902F1317BA2FB0269F47930672107B\
4726FEF64547394D3320C8F120B3C2F4725B0305FAB88CC7981FCB09A76A1CBF7F179F43BB0A4C8\
B0590857F1E69708466C7F8607391E7BC5268BFD3D7A1DFFCB4ECA2A1C9B597593013D5FC4202EC\
2B74E57AB76BBCF3632BBAF97CDC418A6F16392838CA9BF45DDF023777B7561833C105190F94F30\
2C59B531900BBC816361FAA5B3380CA3A893104CA7388B185671B3E5FE3790E9A626EC46D9B0B33\
C7A419AF7B32B6859894F575D82AC5456B5490A7AF8FE61046360589ECBA7244236F4123116B617\
4AA179249A49195B356C72FC6641F0251812EAA98570B046699070E0819DC2713F469137DFC6A3D\
7B92B298995EE780369153AC366B06D7249CD09E1B3378FB04399CECB8650581D637C79AE67D6F2\
CAF6ABACF598159A7792CB3C971D1499D2373AD20F63F03BB59ED137384AC61A7155143B8CA4932\
612EC915E4CA346A9BCE5DD60417C6B2A89B1CC435643F875BDC5A7E5B3481CF919EA09172FEBC4\
6D4FC3FB0CB9591704EE2DBB61844B2F3314A06BB6C6D34005E485CE667BDC7D098586928D2D913\
40F00419EA401351A240A0B041058BEFB0C2FD32645B7A2DF8F5CBFD873327C978D7B351A280884\
38837024C52B9C295CD713646FB5D6C0CCFB470734AC2B2BC8123C2C13DF6938E92455A862639FE\
B8A64B85163E32707E037B38D8AC3922B45187BB65EAFD465FC64A0C5F8F3F9003489415899D59A\
543D8208C54A3166529B53922",
    "A72C2D9C843EE9F8313ECC7F86D6294D59159D9A879A542E260922ADF999051CC45200C9FF\
DB60449C49465979272367C083A7D6267A3ED7A7FD47957C219327F7CA73A4007E1627F00B11CC8\
0573C15AEE6640FB8562DFA6B240CA0AD351AC4AC155B96C14C8AB13DD262CDFD51C4BB5572FD61\
6553D17BDD430ACBEA3E95F0B698D66990AB51E5D03783A8B3D278A5720454CF9695CFDCA08485B\
A099C51CD92A7EA7587C1D15C28E609A81852601B0604010679AA482D51261EC36E36B871967621\
7FD74C54786488F4B4969C05A8BA27CA3A77CCE73B965923CA554E422B9B61F4754641608AC16C9\
B8587A32C1C5DD788F88B36B717A46965635DEB67F45B129B99070909C93EB80B42C2B3F3F70343\
A7CF37E8520E7BCFC416ACA4F18C7981262BA2BFC756AE03278F0EC66DC2057696824BA6769865A\
601D7148EF6F54E5AF5686AA2906F994CE38A5E0B938F239007003022C03392DF3401B1E4A3A7EB\
C6161449F73374C8B0140369343D9295FDF511845C4A46EBAAB6CA5492F6800B98C0CC803653A4B\
1D6E6AAED1932BACC5FEFAA818BA502859BA5494C5F5402C8536A9C4C1888150617F80098F6B2A9\
9C39BC5DC7CF3B5900A21329AB59053ABAA64ED163E859A8B3B3CA3359B750CCC3E710C7AC43C81\
91CB5D68870C06391C0CB8AEC72B897AC6BE7FBAACC676ED66314C83630E89448C88A1DF04ACEB2\
3ABF2E409EF333C622289C18A2134E650C45257E47475FA33AA537A5A8F7680214716C50D470E32\
84963CA64F54677AEC54B5272162BF52BC8142E1D4183FC017454A6B5A496831759064024745978\
CBD51A6CEDC8955DE4CC6D363670A47466E82BE5C23603A17BF22ACDB7CC984AF08C87E14E27753\
CF587A8EC3447E62C649E887A67C36C9CE98721B697213275646B194F36758673A8ED11284455AF\
C7A8529F69C97A3C2D7B8C636C0BA55614B768E624E712930F776169B01715725351BC74B47395E\
D52B25A1313C95164814C34C979CBDFAB85954662CAB485E75087A98CC74BB82CA2D1B5BF280323\
8480638C40E90B43C7460E7AA917F010151FAB1169987B372ABB59271F7006C24E60236B84B9DDD\
600623704254617FB498D89E58B0368BCB2103E79353EB587860C1422E476162E425BC2381DB82C\
6592737E1DD602864B0167A71EC1F223305C02FE25052AF2B3B5A55A0D7A2022D9A798DC0C5874A\
98702AAF4054C5D80338A5248B5B7BD09C53B5E2A084B047D277A861B1A73BB51488DE04EF573C8\
5230A0470B73175C9FA50594F66A5F50B4150054C93B68186F8B5CBC49316C8548A642B2B36A1D4\
54C7489AC33B2D2CE6668096782A2C1E0866D21A65E16B585E7AF8618BDF3184C19868785089172\
77B93E10706B1614972B2A94C7310FE9C708C231A1A8AC8D9314A529A97F469BF64962D82064844\
3099A076D55D4CEA824A58304844F99497C10A25148618A315D72CA857D1B04D575B94F85C01D19\
BEF211BF0AA3362E7041FD16596D808E867B44C4C00D1CDA3418967717F147D0EB21B42AAEE74AC\
35D0B92414B958531AADF463EC6305AE5ECAF79174002F26DDECC813BF32672E8529D95A4E730A7\
AB4A3E8F8A8AF979A665EAFD465FC64A0C5F8F3F9003489415899D59A543D8208C54A3166529B53\
922",
    "D22302CBD3399FACC630991FC8F28BDB4354762541527678BCF61F65C241146C426D23B9BF\
AA6B7DF18C97F20C1B6125BF874B1D89475852C448215DB0EB7737F91480E8CEBD9A0871574F5AB\
62D9020175EC6927CA0B54C09818E42CF92A383172422C7DC1831D63B0C295DE75159DB8034E9E0\
7F7B0B910C3C1E5FB66B3DC523F1FA6EB4910CB89A6C17562C83AB4C18D0CD7E0796592A372AA40\
9B1C557347CCACDC4644A119064D06DD474929D1C6FB4D686E5491CE4BC89A30BB4B8C41BCE5157\
DFC1360823B1AB618C14B10F98C25067398EA7018C278A4B3DF31334D603B2044EF187CD9BC6CE4\
2725BD962C264983E9E18155A8B9C47143D70460A26A56FE7658C1F150348C6087EF758AD167887\
860A007A5FC37358D43B5EBEE820ACEA474F0AC07B76802866199C61231D5C747C93774D2C1E0C1\
C67E6C81B82752173E125BAF39B4FD19A4F453DC57976B1D97FE6996992BBB65B7CB25D077BBAA6\
A13322899AF659CF1B3558C1B5001154B625809ED89AEEBB89E6EA7D67F723D045AB05715C42355\
DA6A5C8DD39C8ABE3037751A01ED1C7374919F3121B5A52C53D1487316769F80721DEEAAAD3C90F\
76E7AE9E12BA92B32B5FD457E3C752C2650DFB885771CB77AC3C785A8C562E6A1C63C2A55EA47CF\
8B90EB8225C123C346452566235B2F31823A33521E087937A345D8D663EEAA05658917BBAA008C2\
E335F8850A90A326D0E66432F44CEB8289E4ECB2D12958E984072ECACB88E1348FF0B55654ACBA5\
B54971CBAEBA88EC4B91A94C37192FA982BECB9F3DA421603B61A51BC8E36CBD053851C77B1B926\
B17A272AA9023246B02B3ED47F66A00BD5684823634E7CE58CF8F306E35B1E5322824D904801F0A\
2FA7C2BC9C252B0A56B7BA2AB0F636021745A70A9A43E2B0A8D615970B65309624B5184BCC30B91\
1679AEDD76025FE3908FD67897B0CF4BE5A6F5413D7DD98564B23E42A93E4AA8821CD45054C643E\
DC1158DB6B3DEB13FB5A51EBD1A8A78B87225A7338E101104C4A220D9BDEDD48C85A1C2DAE781A8\
0C40E13B87EAC73A764201C9B760CCFB1AE392699C7039D27C39362B27B8FC6F07A8A3D4410F154\
7C48A9997F62C61074452EF1515F8A649EBCA9437205A4E8A61606B41DAF6834D671F4D852C0C9C\
4096611648C6A3170678B1537CC1828D93580C9E5849A9653175ACB753F2BE7437BE45F6C603E48\
5F2EC301BB42B6C37C225D7495A584AE231890AB5C8C35C268CF4BBB0213C096019319561A8A694\
7637AA40D006B415BB2CFA2237E0890B6A3BC134ABF8F6585E108D15940F91F4BF5B0C818055B21\
DEA6E63B553988C47F4B94E7CF800A493B4734705EDC56A4B6021C629500675876804CF0B951F03\
8A5C7FE58E89774EF2992FD7C63099D352A7D21560B788B405709861817E59A96B3A3A83CBA803B\
16934331071905BBEC6532900155D8AC88CB32E4E21A3BD3A03FDEC325A51CD2773964E6784FCF1\
853737AA64EB67564727272661ABF84313A57A44B123C65509CFB7A6F6641CDCC3B57FE628C7B81\
92DB44FFBF5796A8613B1FA126F6076883C783DC24E2A4464C40B3A41CA70AE87620866CF4FCB2B\
D204BF5C283812BA056AC0C345E379C4BA24D750901279BB2F3A16F612BFADB35703332C7C136F6\
8EAB6755C66B6A4AD1AABA7B768A58ACAACC10A459A1CC8EF29377BC200E4D315A30A6BCC3256F9\
734D06E9779CAA5442A9A16069081377C76E75154368072DC446ED6C8B8E622A21E383CF9BA1FB4\
34E2ECC81E7B78CEE986B8FF798AB18CF9634543546284EDA2A26B47F05B735BCDB1202220076DC\
8B4E4B9F853533C8F6C7FF38817BA49712835785F17F14CA01D0C1C1E98810FE0B36E5B427157B9\
418449CEDD641A4293C85C32700102ACEC22EBAD98ED160A5F027BD4CDA57F1F3720A12C134654D\
D5E73F829676495390D0E7929D6034E9C55F7D55BA658BC587988E8AF94960F6CFB8D5AF7A00215\
35A6E25E437D49A780698BE22AC9953949F571B85A685725F8207A2B0AE849B601AB91B159B3DF4\
A154C2041E776070AFC42969322380917C97510799F3149131477E16663D3174C7C1CAEA788535C\
6C005A64F2868631B31B66E205FD38C1D84542D0F1B578F58C9BF5A0FAEAB6AB6494893053165EA\
FD465FC64A0C5F8F3F9003489415899D59A543D8208C54A3166529B53922"};
/* Label contribution from B */
const char *strLB1[] = {
    "0202030405060708090A0B0C0D0E0F10",
    "1202030405060708090A0B0C0D0E0F100202030405060708",
    "2202030405060708090A0B0C0D0E0F100202030405060709",
    "3202030405060708090A0B0C0D0E0F100202030405060708090A0B0C0D0E0F10",
    "10202030405060708090A0B0C0D0E0F1",
    "21202030405060708090A0B0C0D0E0F10020203040506070",
    "33202030405060708090A0B0C0D0E0F100202030405060708090A0B0C0D0E0F1"};
const char *strLB2[] = {
    "4202030405060708090A0B0C0D0E0F11",
    "5202030405060708090A0B0C0D0E0F110202030405060708",
    "6202030405060708090A0B0C0D0E0F110202030405060709",
    "7202030405060708090A0B0C0D0E0F110202030405060708090A0B0C0D0E0F11",
    "84202030405060708090A0B0C0D0E0F1",
    "95202030405060708090A0B0C0D0E0F11020203040506070",
    "A7202030405060708090A0B0C0D0E0F110202030405060708090A0B0C0D0E0F1"};

const char *strPB1[] = {
    /* ECDH (NIST P-256, NIST P-384, NIST P-521) public keys from NIST CAVP SP
     * 800-56A ECCCDH Primitive Test Vectors.
     * http://csrc.nist.gov/groups/STM/cavp/documents/components/ecccdhtestvectors.zip
     */
    "809F04289C64348C01515EB03D5CE7AC1A8CB9498F5CAA50197E58D43A86A7AEB29D84E811\
197F25EBA8F5194092CB6FF440E26D4421011372461F579271CDA3",
    "A7C76B970C3B5FE8B05D2838AE04AB47697B9EAF52E764592EFDA27FE7513272734466B400\
091ADBF2D68C58E0C50066AC68F19F2E1CB879AED43A9969B91A0839C4C38A49749B661EFEDF243\
451915ED0905A32B060992B468C64766FC8437A",
    "30F43FCF2B6B00DE53F624F1543090681839717D53C7C955D1D69EFAF0349B7363ACB44724\
0101CBB3AF6641CE4B88E025E46C0C54F0162A77EFCC27B6EA792002AE2BA82714299C860857A68\
153AB62E525EC0530D81B5AA15897981E858757",
    "00685A48E86C79F0F0875F7BC18D25EB5FC8C0B07E5DA4F4370F3A9490340854334B1E1B87\
FA395464C60626124A4E70D0F785601D37C09870EBF176666877A2046D01BA52C56FC8776D9E8F5\
DB4F0CC27636D0B741BBE05400697942E80B739884A83BDE99E0F6716939E632BC8986FA18DCCD4\
43A348B6C3E522497955A4F3C302F676",
    "700C48F77F56584C5CC632CA65640DB91B6BACCE3A4DF6B42CE7CC838833D287DB71E509E3\
FD9B060DDB20BA5C51DCC5948D46FBF640DFE0441782CAB85FA4AC",
    "1AEFBFA2C6C8C855A1A216774550B79A24CDA37607BB1F7CC906650EE4B3816D68F6A9C75D\
A6E4242CEBFB6652F65180419D28B723EBADB7658FCEBB9AD9B7ADEA674F1DA3DC6B6397B55DA0F\
61A3EDDACB4ACDB14441CB214B04A0844C02FA3",
    "01DF277C152108349BC34D539EE0CF06B24F5D3500677B4445453CCC21409453AAFB8A72A0\
BE9EBE54D12270AA51B3AB7F316AA5E74A951C5E53F74CD95FC29AEE7A013D52F33A9F3C14384D1\
587FA8ABE7AED74BC33749AD9C570B471776422C7D4505D9B0A96B3BFAC041E4C6A6990AE7F700E\
5B4A6640229112DEAFA0CD8BB0D089B0"};

const char *strPB2[] = {
    /* SIKE (sikep434, sikep503, sikep610 and sikep751) ciphertexts - Round 2 KATs,
     * https://csrc.nist.gov/CSRC/media/Projects/Post-Quantum-Cryptography/documents
     * /round-2/submissions/SIKE-Round2.zip
     */
    "0FDEB26DBD96E0CD272283CA5BDD1435BC9A7F9AB7FC24F83CA926DEED038AE4E47F39F988\
6E0BD7EEBEAACD12AB435CC92AA3383B2C01E6B9E02BC3BEF9C6C2719014562A96A0F3E784E3FA4\
4E5C62ED8CEA79E1108B6FECD5BF8836BF2DAE9FEB1863C4C8B3429220E2797F601FB4B8EBAFDD4\
F17355508D259CA60721D167F6E5480B5133E824F76D3240E97F31325DBB9A53E9A3EEE2E071273\
4825615A027857E2000D4D00E11988499A738452C93DA895BFA0E10294895CCF25E3C261CBE38F5\
D7E19ABE4E322094CB8DEC5BF7484902BABDE33CC69595F6013B20AABA9698C1DEA2BC6F65D5751\
9294E6FEEA3B549599D480948374D2D21B643573C276E1A5B0745301F648D7982AB46A306563996\
0182BF365819EFC0D4E61E87D2820DBC0E849E99E875B21501D1CA7588A1D458CD70C7DF793D499\
3B9B1679886CAE8013A8DD854F010A100C9933FA642DC0AEA9985786ED36B98D3",
    "100692A8BD30F01BE8AC6B1AF8D93A060D3821B2587F4038D64B72426A194BEDE63CA60B75\
A5C3C15532CE307115AA9D77AC232E14D99C1E1AFEF1EB2D6321AE342F990023466E683A2568D59\
A14325C2C6C272029741D8E36976D1804059BC06B802F3A495EA50D0DBBA93FD263F4CF30BDB5F7\
83BA6A0775715B05F700C85B316F7AA1A1624973885941DBFF91316BF47AC698E11D6B2418F5533\
79D67A00F784B8643FB8A94029584391D488775EB4414A5E6E8122B0F282D900F3D05775F1DD994\
FB232ED826106203CD3433967F60FF925DF9E86CB376CAB5FD90B132E425682741F6AF078E75792\
CB4CE085D44993CFB6A4ED5AA3541640A0A67687922B92382CAC47C6AD358011A269CC7C17CE651\
CA2E2393F7DFE19D7054FEF69610A353D676B1F076549510590D406AD13F4A3292CCF206DBDAE47\
F08D448CC006449F27C1FB54E9C9E6F16ED2F3D120DD5AA2620D76690F00E31904C601310C76A84\
3A58E1AEB9C5F515FCEC482C08205FDE99A89E64485EBBD43EEFE2E24D18EEE8F20DF6E113C6667\
512E28396862C98F5C0",
    "FB75E7D835313132AC0B29D8732F1F62E6DD10BBF30375B4A50C7B153431BAE6259E1C5526\
C07164E87EDC70E4F0D8331D73285661D1F639D216372D05B4583C1302932B03FF184D115D0B250\
297FF26AE81DFA0DE01A1DFB237C8008B22285A289C06BF4BC89C0BD77576932A14B1FEB9CE6D7F\
8816D710F1B043C8E58DCE1B32EF4EC8FB67E10CD23B6D4CC653DD8CD83B5F4DB0B5B741D30125C\
F842EE13EB940650E1E34E4666935B178F2351553F0822C8B354C70E47350E74A08F16D4F39F8AA\
80C3F4E0083C4BA1F31F5F1D04FD4CF835AEA688885E85509133FFE557A7892A0161AC01BBCC8A2\
7CE37E8CB9C1916A0F62BCF1E82C3F9213275B10CA272BFABCA2713CEEAECD0007C9FB6B562AFA2\
231FF7FD2C1D20D8ED28C11A840FEE931FE7A0E3BB925D88A852C2EE9BF606AD4000FA27643155A\
6FECAD9D4BABA8DE8F8D767AEC7A770D007ADB0D9F76E521DE6EF8D3567A32047688E2E8130AAF3\
EB594A366F3C534E335A3E9EDA326E60394CA10A44340CC78995742E48994002CEE1049870D14C2\
3C9FF2E5899DD7E3A1516D2F6E70B3DE1D79987379296E99EBCCAC43DA9A475CA3FE756D4649934\
BADA6DFA8C8F8BB21136172798BDA13E247B2F27874AFE13CCCA31F53D01A94B9520C3CBCDD1B1E\
B9BBBD6B83C76F64FC5D7C1DCF33A",
    "66D24BC4630B2EE312F01B26EC1D3EC1F583795EC93B90FD9B5453E0BEDA2A70FB6181C9B9\
EA86A9866F1468E62CE853C6C65AA5D0E4535828B3A97E2D1D31DC4372E6A36DA7B0C4733574B91\
CCE215086C59B54F2364F61298004C8410B17658B4021CD36859C94210DE338869CACF0E2DC1141\
2FA5172E1663AAEBEF4B5EB0ED9175D6C86C5107DA92B8772342A2F44C93EFFE61F6C76AB8ABA19\
4E862543EDB707E9D2EE884995B1062FE2F60627D5C7673C7AC0D15B08C2F8510DC239463B1B32A\
D46873F6D1CB5A8579457386FD75700989BEED2CA547FE505C581B6B436AABC0F75AD6373A08CEC\
1504258A972C64EC4A1FEB86BFE32ACF3A73ECF815CBC883F39B42C6429A5875BD0BD6A94CEAD58\
7AF49AC8EFB43E1A447D2D8555CB0ADFBC9F335F1C599BC9FEAB3E4FE5F2D06D930A58C2FFEEDE0\
E2726EBD85EC890D1CB0E6870DD784AE30286F1A336D57FC41D2F2E2F89765C6A110853BB63E478\
A64D54A31A18FB4BA44FF58A3662F4D82D544BD9B0E94FC88ABC4E4D27D5F6084B5F2162B357A04\
A1A28C8938834ECC987E50C0A2CACA442850493CE16C047DF677097D3F7EA034BC3D25355042760\
03DBBEB12F1949C3D369E7EFBA09831E83D622AB2D9277F523946FBAB1DDE14015857EA47663C5C\
CF30BDF261CCBF31DBE2A560E96CE87FBA80B783350A42C837EB36B2F39A9FED1B649B8ECCE3D32\
35825F7C800834740546E0CF42C9C2C8B12495225F991B14547E5EEDB22858B26EE6E0AE13DBE3D\
50C6C1EF79C4B97DAD1B0239C4037C1AAC29EE1505E0E527EC81348900E7C216A1A1B34B8D2753A\
F2693647C412",
    /* KYBER (kyber512, kyber768, kyber1024) ciphertexts from Round 3 KATs,
     * https://csrc.nist.gov/CSRC/media/Projects/post-quantum-cryptography
     * /documents/round-3/submissions/Kyber-Round3.zip
     */
    "EDF24145E43B4F6DC6BF8332F54E02CAB02DBF3B5605DDC90A15C886AD3ED489462699E4AB\
ED44350BC3757E2696FBFB2534412E8DD201F1E4540A3970B055FE3B0BEC3A71F9E115B3F9F3910\
2065B1CCA8314DCC795E3C0E8FA98EE83CA6628457028A4D09E839E554862CF0B7BF56C5C0A829E\
8657947945FE9C22564FBAEBC1B3AF350D7955508A26D8A8EB547B8B1A2CF03CCA1AABCE6C34977\
83B6465BA0B6E7ACBA821195124AEF09E628382A1F914043BE7096E952CBC4FB4AFED1360904611\
7C011FD741EE286C83771690F0AEB50DA0D71285A179B215C6036DEB780F4D16769F72DE16FDADA\
C73BEFA5BEF8943197F44C59589DC9F4973DE1450BA1D0C3290D6B1D683F294E759C954ABE8A7DA\
5B1054FD6D21329B8E73D3756AFDA0DCB1FC8B1582D1F90CF275A102ABC6AC699DF0C5870E50A1F\
989E4E6241B60AAA2ECF9E8E33E0FFCF40FE831E8FDC2E83B52CA7AB6D93F146D29DCA53C7DA1DB\
4AC4F2DB39EA120D90FA60F4D437C6D00EF483BC94A3175CDA163FC1C2828BE4DBD6430507B584B\
B5177E171B8DDA9A4293C3200295C803A865D6D2166F66BA5401FB7A0E853168600A2948437E036\
E3BF19E12FD3F2A2B8B343F784248E8D685EB0AFDE6315338730E7A1001C27D8D2A76FA69D157BA\
1AC7AD56DA5A8C70FE4B5B8D786DC6FC0566BA8E1B8816334D32A3FB1CE7D4D5E4C332AF7B003D0\
91741A3D5C965292255DFF8ED2BBF1F9116BE50C17B8E548748AD4B2E957BBD1953482A2E1718CE\
C66CD2C81F572D552B7187885E6B8943D6431413C59EBB7E036048490BE5289E95B20A89E8B159F\
61A9A9886E147568F4C9021F362F02688A1C8C3BB0D24086880E55B6EDB43F3745D2C166DC1CB74\
3C76FE6BE523A893CC764D16435C37851252A81E2FFBA0F18971A3DEE37D4877CB928E36E523503\
7A6B2057897D518A5F0E348E3AB6D5B52DFC60757F3B41A4FEC7828F1DEEAF4587CCC8EADF647F4\
D203B2FAA05A649B582340CB4CACE57A30711BE752FACF0227D0A80C4128442DDC544BE805B9CFE\
8FE9B1237C80F96787CD9281CCF270C1AFC0670D",
    "B52C56B92A4B7CE9E4CB7C5B1B163167A8A1675B2FDEF84A5B67CA15DB694C9F11BD027C30\
AE22EC921A1D911599AF0585E48D20DA70DF9F39E32EF95D4C8F44BFEFDAA5DA64F1054631D04D6\
D3CFD0A540DD7BA3886E4B5F13E878788604C95C096EAB3919F427521419A946C26CC041475D712\
4CDC01D0373E5B09C7A70603CFDB4FB3405023F2264DC3F983C4FC02A2D1B268F2208A1F6E2A620\
9BFF12F6F465F0B069C3A7F84F606D8A94064003D6EC114C8E808D3053884C1D5A142FBF20112EB\
360FDA3F0F28B172AE50F5E7D83801FB3F0064B687187074BD7FE30EDDAA334CF8FC04FA8CED899\
CEADE4B4F28B68372BAF98FF482A415B731155B75CEB976BE0EA0285BA01A27F1857A8FB377A3AE\
0C23B2AA9A079BFABFF0D5B2F1CD9B718BEA03C42F343A39B4F142D01AD8ACBB50E38853CF9A50C\
8B44C3CF671A4A9043B26DDBB24959AD6715C08521855C79A23B9C3D6471749C40725BDD5C2776D\
43AED20204BAA141EFB3304917474B7F9F7A4B08B1A93DAED98C67495359D37D67F7438BEE5E435\
85634B26C6B3810D7CDCBC0F6EB877A6087E68ACB8480D3A8CF6900447E49B417F15A53B607A0E2\
16B855970D37406870B4568722DA77A4084703816784E2F16BED18996532C5D8B7F5D214464E5F3\
F6E905867B0CE119E252A66713253544685D208E1723908A0CE97834652E08AE7BDC881A131B73C\
71E84D20D68FDEFF4F5D70CD1AF57B78E3491A9865942321800A203C05ED1FEEB5A28E584E19F65\
35E7F84E4A24F84A72DCAF5648B4A4235DD664464482F03176E888C28BFC6C1CB238CFFA35A321E\
71791D9EA8ED0878C61121BF8D2A4AB2C1A5E120BC40ABB1892D1715090A0EE48252CA297A99AA0\
E510CF26B1ADD06CA543E1C5D6BDCD3B9C585C8538045DB5C252EC3C8C3C954D9BE5907094A894E\
60EAB43538CFEE82E8FFC0791B0D0F43AC1627830A61D56DAD96C62958B0DE780B78BD47A604550\
DAB83FFF227C324049471F35248CFB849B25724FF704D5277AA352D550958BE3B237DFF473EC2AD\
BAEA48CA2658AEFCC77BBD4264AB374D70EAE5B964416CE8226A7E3255A0F8D7E2ADCA062BCD6D7\
8D60D1B32E11405BE54B66EF0FDDD567702A3BCCFEDE3C584701269ED14809F06F8968356BB9267\
FE86E514252E88BB5C30A7ECB3D0E621021EE0FBF7871B09342BF84F55C97EAF86C48189C7FF4DF\
389F077E2806E5FA73B3E9458A16C7E275F4F602275580EB7B7135FB537FA0CD95D6EA58C108CD8\
943D70C1643111F4F01CA8A8276A902666ED81B78D168B006F16AAA3D8E4CE4F4D0FB0997E41AEF\
FB5B3DAA838732F357349447F387776C793C0479DE9E99498CC356FDB0075A703F23C55D47B550E\
C89B02ADE89329086A50843456FEDC3788AC8D97233C54560467EE1D0F024B18428F0D73B30E19F\
5C63B9ABF11415BEA4D0170130BAABD33C05E6524E5FB5581B22B0433342248266D0F1053B245CC\
2462DC44D34965102482A8ED9E4E964D5683E5D45D0C8269",
    "A6AF29D5F5B80BD130F518BADDD6C8F17545413D860FB3DE451979EBFA5E4E3112C7C0ADF9\
9824BB526F2C3550748ED0E134F0457A7C61F9F526F002BAADC03FC13E38131219513C3EDE06166\
1E74F603C4FCF7951C8E52C9C213B0D22D9293663D669A6B58ED8FCEFCF8249D7BB5298F5576144\
5B2B83CE7F005CB04248AEC8BDA22FD2D42AA766322014EA038CC32C55C8E4B9E28EC9119F52734\
1E4F66A035121073B85DE6706DA19E0838A9F33B719A68F039B664DC002659EABFC398679AA7009\
CE0CD01CDAFB6CD2A26FE4101672C98FF58F7C47D5BDA2906653B3A6F9651F7A121EA77EA74723F\
AE5B873F9BB7B664F0C8A93831EF9D51C7CC1EF44AC0E55A55CA76D137FE9B75F40509CEF156E5A\
D18F9FB999680008E547D55EECD5B4D1CB1D9F076CEC21501C7402509ECB77AFB2CB9A61340A8BD\
1514C6E71B4AA45E47EC37512271B911F8FB46C9082C9DF07204ABB5A50E6E3647A8AD4D8D5D7BF\
F19C8A509308BCFB895536D045CA2B97CB16A29BB7181CAD0509DDB91735028EBA8C31D74BD275E\
AA65B5340B3A43FBFE0B3061D6BAE7E75B7098CDABE91D4B31E36C9AA7A8298862AD63C8FD282E0\
3B460B3AB464CE0F27B1C3D11155ACAA011EB9E2AE3E6DDA07D6F491737CBCE9B05F9BC56BE20E8\
D326BA132C57FB235161144519CDF40560FBE279BDE411E112531F826D6AB10D4547350ADD2A9DE\
8D62C2AC82CABE6815646F4DC9742BB0C2A3F77EC7B46C6B537605FA31798CD89281221A33DFB97\
96E644305630332C2CB931408AB481A16D953F6BEAE3891D6D9AC1FAB38222D9271872D9D0CADB9\
1ABE9B4E265F75C6E5E829E146C3D8CE1E9D12E0D129801957F46B0D2DBE1F749B1D08E2345F623\
9A731342EB75B0CF1BF411749BC2CAF2810B788C6B7238B4D3DA2D6315CE9542E24404F145755A3\
0AB851E4445841BD33F716A586884888ECC6BC6498AA32919AE81D20C26973C2BD54582A0F6AD98\
ABFD2627E15690A727E69F581DD2A7127982A90E33E2D4A03FE339142C7E44C326AC46ED395A225\
D3033389917328B45316B1585A01B2C304B2944E903ABBB3EC5619441CFC8965A446DF75DEFA80C\
6E15ADBD506B7AB2DE12DDA9BC81441CFC89052E2E5808F7126C6FD3AC6AC8081258A84A09AE50F\
6CD7CC0F4AF336FD1D643E99079996268C2D32D909F22E3504F07FBB563196D4312FDDB9335D5C1\
D36E8C5EEA2278DBA23B94D193C947CC41CA993DC7DB1396340AD9C4FE687DD7B8D0C7A5120AE02\
04F2C665BD5F473D644C7FF26BFFBA7A36980830702128A7E661D677A092A36E7428A4139FB29B0\
095CC11086F447D2A9EF6C9B161F189C6299E084CB7AA00FAF787797BFB069FBC087FDE26252A16\
64F19C5A8A22EC5EE1AEB076357B7DC37E6B0F1520F958F7851BACB92C89FD114A72FEAC54652D4\
5B09E1AE7651ABD164BCD537D58FA39D3EC8ACDCDF98425005862FA59692DE162B77E6297C66233\
348408A8AB695CE2F2728DB9FBE27E958967EC5974767C5A66023074B4A71AFD264AD2890E970A1\
F31D6E3311B736F9F9488793DDC88F23458064254C82A1D9E59EAD2FCEC40B430687C4B7E289609\
26AFCACC9BD756A71088C78450E20A2E980AEDE9EBEDFE7FABD6ABFE96F934C4B02C01CA194D01B\
73C25D5997039D3FCD0F099521F70CAEE69110AC1FC5A99917AD752FC96ADFAD7186D0A7C9CFE56\
01C07514EA6448D661C57AA20242103C4276A070A489A4CB6BCA0F9ECC4379FB220215FD91F8101\
9D5B0AE619358B52468F272C178E3A74CF6775AA924FE329C3175D9E4C3E21AB9EC836EDC3ACAB2\
E3891EE8DEDA515D39AF9B8DDD0EE7B0164F805C3835F6D2BABDB30EAB4756E7EC7F829ECE01E8E\
ADFBBED12FC283B3D4C69F575E7F80417689FDFCFC7BE27EE3B8CDF57AAEBEC4A95B7E5BB585B85\
227F7C32BE30DB3E65E42E30DCF5A5FA073DBA399D942F2222ADB9B9898102AFE5432EDC7F04AE3\
4A8FEC2D81CB49A9A9B43814CE71D97F726E2B1E8F64B50E65DFB4816E12E82A3197484A4E9BBA4\
D2D69E3F19D0B75C21E2BFFE9FC0C98CF48A3AAF08D467F72687DF0178174B7897F734349B181EC\
A86A598A0C5E8C25946F24DC5572BD324A40458A788E5137F3C7A7C97FC9F12A3C463A8FE944910\
1CCE966D7C009323932998D56EF430C73BC24F5D95F737858DDC4F32C013"};

const char *strk1[] = {
    /* ECDH (NIST P-256, NIST P-384 and NIST P-521) shared secrets from
     * NIST CAVP SP 800-56A ECCCDH Primitive Test Vectors.
     * http://csrc.nist.gov/groups/STM/cavp/documents/components
     * /ecccdhtestvectors.zip
     */
    "057D636096CB80B67A8C038C890E887D1ADFA4195E9B3CE241C8A778C59CDA67",
    "5F9D29DC5E31A163060356213669C8CE132E22F57C9A04F40BA7FCEAD493B457E5621E766C\
40A2E3D4D6A04B25E533F1",
    "A23742A2C267D7425FDA94B93F93BBCC24791AC51CD8FD501A238D40812F4CBFC59AAC9520\
D758CF789C76300C69D2FF",
    "005FC70477C3E63BC3954BD0DF3EA0D1F41EE21746ED95FC5E1FDF90930D5E136672D72CC7\
70742D1711C3C3A4C334A0AD9759436A4D3C5BF6E74B9578FAC148C831",
    "46FC62106420FF012E54A434FBDD2D25CCC5852060561E68040DD7778997BD7B",
    "3D2E640F350805EED1FF43B40A72B2ABED0A518BCEBE8F2D15B111B6773223DA3C3489121D\
B173D414B5BD5AD7153435",
    "000B3920AC830ADE812C8F96805DA2236E002ACBBF13596A9AB254D44D0E91B6255EBF1229\
F366FB5A05C5884EF46032C26D42189273CA4EFA4C3DB6BD12A6853759"};
const char *strk2[] = {
    /* SIKE (sikep434, sikep503, sikep610, sikep751) secrets keys from Round 3 KATs,
     * https://csrc.nist.gov/CSRC/media/Projects/post-quantum-cryptography
     * /documents/round-3/submissions/SIKE-Round3.zip
     */
    "35F7F8FF388714DEDC41F139078CEDC9",
    "AF1280151C2C59B4D4150B18BA7F71590523CEA83C9BDDDA",
    "0A5CFC45865775D0CC10F89EFAD9FFD33A6C8A7AB868309D",
    "FEE94595E8A05C50113C044D4D8558DA101035EBBF604AA41D0AAA75B8A7F786",
    /* KYBER (kyber512, kyber768, kyber1024) secret keys from Round 3 KATs,
     * https://csrc.nist.gov/CSRC/media/Projects/post-quantum-cryptography
     * /documents/round-3/submissions/Kyber-Round3.zip
     */
    "0A6925676F24B22C286F4C81A4224CEC506C9B257D480E02E3B49F44CAA3237F",
    "914CB67FE5C38E73BF74181C0AC50428DEDF7750A98058F7D536708774535B29",
    "B10F7394926AD3B49C5D62D5AEB531D5757538BCC0DA9E550D438F1B61BD7419"};
/*  Derived key_material  */
const char *strkmcat[] = {
    "5C366F23281D33EB85CAB026D3D9A35A",
    "A808667CC0275AEB4A646E442F13E8D0",
    "F79E08982B3D2B7C2D0B27F921CCF71C0E3EE98877DA225A",
    "6D3F348DAB7EB85F043288112862AD16A10A70F65E73F171",
    "933C64D06F5F3FB02D74530BCB093072682FAFDD64D42933",
    "05C6E23AF74E3A8E1B1AA9F599C517468B86AB2068591AF8",
    "4DB991EEDA686DDA8CB5F128C16267CC7516643FE5EBD8C6176405C9AB4600FA",
    "270EF22EEDE80D0100872E307388027E8E12AE331AB81A27D6D166F7241B7BAF",
    "1ACFDC16EEC7C8A669EC6007A4CAAB1A",
    "9B8DD1A341ADFB821E32980DA418A114",
    "1529723880E3E1777F73F8D89724D323572E56858D3DD732",
    "7A954FCF3DE3C845B65E1D6F3CCD6186BD2F10FF83525B7A",
    "6A4D002A9B310D6E73D87BF36224E2F41EDD8F75CFF448D59493D65D61C07D8B",
    "203F634CFF7D7EC462281A65706E50342F025B7145E03A976640137C89F919A1"};
const char *strkmcas[] = {
    "C1EF63BE6F26F6EBE2078C940AC47E59",
    "5C0D6EE9B187EEE46C22EB3C3F36349A",
    "CA58B87E512FCAEF2B11B1A1A8550FB4B1C1E2ACF7C364D5",
    "007B8F4E6C95851C3598F59484D75A2A6FCFC931B42D74E4",
    "DBC55D5F607CF981CB9774E920288ED030C3A13CADD4E594",
    "944DC1D2AD91B40E3B3B6D2B19981D8571E27A0C49EE4767",
    "109654A3D902FC17C83AD709D436E10FDA9A901CF6A1BDF9A02CB087171A8732",
    "70D1D1AF1A736B6C648B7410F728D0AC689E2D8CD54F8B1DDA410CAD566F5CFE",
    "A324764EEA9EBA338F0596D6D64DFA55",
    "07930E42BCA7B1C60F530DA08A1C85F1",
    "45F0EA18A6B22D94139BAF38F3C932277B48B4113A6B4E27",
    "220D8F8C87FC3CD29FB1E9D24E5C6FAE4A6E42E085EB2D5F",
    "EC38A129393864C9A949365155FD6A6943E30459DB80304EFEB83047B903D117",
    "C2D57FA0CF6175F31581634849703060E203346823D091C182B5520F3714A487"};
/*
 * print_array( )  - this function is intended only for use within this
 *                   application.
 *
 *   WARNING:        Users should be aware that using this with non-null
 *                   terminated strings may result in unknown behaviors.
 */
void print_array(const char *label, const uint8_t *array, const uint32_t alength)
{
    uint32_t i;
    if ((label == NULL) || ((alength) && (array == NULL))) {
        return;
    }
    printf("%.*s", MAX_LABEL_LEN, label);
    for (i = 0; i < alength; i++) {
        printf("%02X", array[i]);
    }
    printf("\n");
    return;
}

/* Helper functions to loads a set of ascii hex strings to an internal
 * buffers.  For example, if called with (array, alength, 3, "0ABC", "0132",
 * "015AF4"), it wiill set array = { 0x0A, 0xBC, 0x01, 0x32, 0x01, 0x5A, 0xF4},
 * and *alength = 6
 */
void ascii_hex_strings_to_uint8(uint8_t *array, uint32_t *alength, const uint32_t scount, ...)
{
    const char *pos;
    uint32_t    str_length, remaining_length, i, j;
    uint8_t *   ptr;
    va_list     args;

    if ((array == NULL) || (alength == NULL)) {
        exit(0);
    }
    ptr              = array;
    remaining_length = *alength;
    *alength         = 0;
    va_start(args, scount);
    for (i = 0; i < scount; i++) {
        if ((pos = va_arg(args, const char *)) == NULL) {
            va_end(args);
            exit(0);
        }
        str_length = (uint32_t)strlen(pos);
        if (str_length % 2) {
            va_end(args);
            exit(0);
        }
        *alength = *alength + (str_length / 2);
        if (remaining_length < str_length / 2) {
            va_end(args);
            exit(0);
        }
        for (j = 0; j < str_length / 2; j++) {
            sscanf(pos, "%2hhx", ptr++);
            pos += 2;
        }
        remaining_length -= str_length / 2;
    }
    va_end(args);
    return;
}
/* Testing harness for concatenation KDF with two exchanges */
int test_hkex_concatenate()
{
    uint8_t       k1[MAX_KEY_BYTE_LEN], k2[MAX_KEY_BYTE_LEN];
    uint8_t       km[MAX_KEY_BYTE_LEN], key_material[MAX_KEY_BYTE_LEN];
    uint8_t       MA[MAX_MSG_BYTE_LEN], MB[MAX_MSG_BYTE_LEN];
    uint8_t       label[MAX_LABEL_BYTE_LEN];
    uint32_t      alength, blength, llength, klength, kmlength, k1length;
    uint32_t      k2length, r, j;
    const EVP_MD *md_type;

    for (r = 0; r < TEST_VECTOR_CNT; r++) {
        k1length = sizeof(k1);
        ascii_hex_strings_to_uint8(k1, &k1length, 1, strk1[r]);
        k2length = sizeof(k2);
        ascii_hex_strings_to_uint8(k2, &k2length, 1, strk2[r]);
        for (j = 0; j < HASH_FUNCTIONS_CNT; j++) {
            md_type = (*evp_hash[r][j])();
            /* MA = LA1 || PA1 || PA2
             NOTE: for test vector purposes ONLY not a suggested message format
            */
            alength = sizeof(MA);
            ascii_hex_strings_to_uint8(MA, &alength, 3, strLA1[r], strPA1[r], strPA2[r]);
            /* MB = LB1 || PB1 || PB2
             NOTE: for test vector purposes ONLY not a suggested message format
            */
            blength = sizeof(MB);
            ascii_hex_strings_to_uint8(MB, &blength, 3, strLB1[r], strPB1[r], strPB2[r]);
            /* label = LA || LB */
            llength = sizeof(label);
            ascii_hex_strings_to_uint8(label, &llength, 2, strLA1[r], strLB1[r]);
            klength = key_length[r];
            if (hkex_concat(md_type, key_material, klength, NULL, 0, k1, k1length, k2, k2length, MA, alength, MB,
                            blength, (uint8_t *)CONCAT_CONTEXT_TEST_VECTOR,
                            (uint32_t)strlen(CONCAT_CONTEXT_TEST_VECTOR), label, llength)) {
                return FAILURE;
            }
            kmlength = sizeof(km);
            ascii_hex_strings_to_uint8(km, &kmlength, 1, strkmcat[j + HASH_FUNCTIONS_CNT * r]);
            printf("LA = %s\n", strLA1[r]);
            printf("PA1 = %s\n", strPA1[r]);
            printf("PA2 = %s\n", strPA2[r]);
            printf("LB = %s\n", strLB1[r]);
            printf("PB1 = %s\n", strPB1[r]);
            printf("PB2 = %s\n", strPB2[r]);
            printf("k1 = %s\n", strk1[r]);
            printf("k2 = %s\n", strk2[r]);
            printf("context = %s\n", CONCAT_CONTEXT_TEST_VECTOR);
            printf("label = LA || LB\n");
            printf("MA = LA || PA1 || PA2\n");
            printf("MB = LB || PB1 || PB2\n\n");
            print_array("key material = ", key_material, klength);
            if (memcmp(km, key_material, key_length[r]) != 0) {
                printf("Test vector %d failed\n", j + r * HASH_FUNCTIONS_CNT);
                return FAILURE;
            }
        }
    }
    return SUCCESS;
}
/* Testing harness for the cascade KDF using two rounds */
int test_hkex_cascade()
{
    uint8_t       k1[MAX_KEY_BYTE_LEN], k2[MAX_KEY_BYTE_LEN];
    uint8_t       km[MAX_KEY_BYTE_LEN], round_secret[MAX_KEY_BYTE_LEN];
    uint8_t       MA[MAX_MSG_BYTE_LEN], MB[MAX_MSG_BYTE_LEN];
    uint8_t       label[MAX_LABEL_BYTE_LEN], chain_secret1[MAX_KEY_BYTE_LEN];
    uint8_t       chain_secret2[MAX_KEY_BYTE_LEN];
    uint32_t      alength, blength, llength, clength, rlength, kmlength;
    uint32_t      k1length, k2length, r, j;
    const EVP_MD *md_type;

    for (r = 0; r < TEST_VECTOR_CNT; r++) {
        k1length = sizeof(k1);
        ascii_hex_strings_to_uint8(k1, &k1length, 1, strk1[r]);
        k2length = sizeof(k2);
        ascii_hex_strings_to_uint8(k2, &k2length, 1, strk2[r]);
        for (j = 0; j < HASH_FUNCTIONS_CNT; j++) {
            md_type = (*evp_hash[r][j])();
            /* MAi = LAi || PAi
             NOTE: for test vector purposes ONLY not a suggested message format
            */
            alength = sizeof(MA);
            ascii_hex_strings_to_uint8(MA, &alength, 2, strLA1[r], strPA1[r]);
            /* MBi = LBi || PBi
             NOTE: for test vector purposes ONLY not a suggested message format
            */
            blength = sizeof(MB);
            ascii_hex_strings_to_uint8(MB, &blength, 2, strLB1[r], strPB1[r]);
            /* labeli = LAi || LBi */
            llength = sizeof(label);
            ascii_hex_strings_to_uint8(label, &llength, 2, strLA1[r], strLB1[r]);
            rlength = key_length[r];
            clength = EVP_MD_size(md_type);
            if (hkex_cascade(md_type, chain_secret1, clength, round_secret, rlength, NULL, 0, k1, k1length, MA, alength,
                             MB, blength, (uint8_t *)CASCADE_CONTEXT_TEST_VECTOR,
                             (uint32_t)strlen(CASCADE_CONTEXT_TEST_VECTOR), label, llength)) {
                return FAILURE;
            }
            printf("LA1 = %s\n", strLA1[r]);
            printf("PA1 = %s\n", strPA1[r]);
            printf("LB1 = %s\n", strLB1[r]);
            printf("PB1 = %s\n", strPB1[r]);
            printf("previous_chain_secret = psk = <NULL>\n");
            printf("k1 = %s\n", strk1[r]);
            printf("context = %s\n", CASCADE_CONTEXT_TEST_VECTOR);
            printf("label = LA1 || LB1\n");
            printf("MA = LA1 || PA1\n");
            printf("MB = LB1 || PB1\n\n");

            print_array("chain_secret1 = ", chain_secret1, clength);
            print_array("key_material1 = ", round_secret, rlength);
            /* MA = LA2 || PA2
             NOTE: for test vector purposes ONLY not a suggested message format
            */
            alength = sizeof(MA);
            ascii_hex_strings_to_uint8(MA, &alength, 2, strLA2[r], strPA2[r]);
            /* MB = LB1 || PB1
             NOTE: for test vector purposes ONLY not a suggested message format
            */
            blength = sizeof(MB);
            ascii_hex_strings_to_uint8(MB, &blength, 2, strLB2[r], strPB2[r]);
            /* label = LA2 || LB2 */
            llength = sizeof(label);
            ascii_hex_strings_to_uint8(label, &llength, 2, strLA2[r], strLB2[r]);
            if (hkex_cascade(md_type, chain_secret2, clength, round_secret, rlength, chain_secret1, clength, k2,
                             k2length, MA, alength, MB, blength, (uint8_t *)CASCADE_CONTEXT_TEST_VECTOR,
                             (uint32_t)strlen(CASCADE_CONTEXT_TEST_VECTOR), label, llength)) {
                return FAILURE;
            }
            printf("LA2 = %s\n", strLA2[r]);
            printf("PA2 = %s\n", strPA2[r]);
            printf("LB2 = %s\n", strLB2[r]);
            printf("PB2 = %s\n", strPB2[r]);
            print_array("previous_chain_secret = ", chain_secret1, clength);
            printf("k2 = %s\n", strk1[r]);
            printf("context = %s\n", CASCADE_CONTEXT_TEST_VECTOR);
            printf("label = LA2 || LB2\n");
            printf("MA2 = LA2 || PA2\n");
            printf("MB2 = LB2 || PB2\n\n");

            print_array("chain_secret2 = ", chain_secret1, clength);
            print_array("key_material2 = ", round_secret, rlength);
            kmlength = sizeof(km);
            ascii_hex_strings_to_uint8(km, &kmlength, 1, strkmcas[j + HASH_FUNCTIONS_CNT * r]);
            if (memcmp(km, round_secret, key_length[r]) != 0) {
                printf("Test vector %d failed\n", j + r * HASH_FUNCTIONS_CNT);
                return FAILURE;
            }
        }
    }
    return SUCCESS;
}

int main(void)
{
    if (test_hkex_concatenate()) {
        return FAILURE;
    }
    if (test_hkex_cascade()) {
        return FAILURE;
    }
    printf("All tests passed\n");
    return SUCCESS;
}
