/* crypto/ec/ec_curve.c */
/*
 * Written by Nils Larsch for the OpenSSL project.
 */
/* ====================================================================
 * Copyright (c) 1998-2002 The OpenSSL Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit. (http://www.openssl.org/)"
 *
 * 4. The names "OpenSSL Toolkit" and "OpenSSL Project" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For written permission, please contact
 *    openssl-core@openssl.org.
 *
 * 5. Products derived from this software may not be called "OpenSSL"
 *    nor may "OpenSSL" appear in their names without prior written
 *    permission of the OpenSSL Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit (http://www.openssl.org/)"
 *
 * THIS SOFTWARE IS PROVIDED BY THE OpenSSL PROJECT ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE OpenSSL PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This product includes cryptographic software written by Eric Young
 * (eay@cryptsoft.com).  This product includes software written by Tim
 * Hudson (tjh@cryptsoft.com).
 *
 */
/* ====================================================================
 * Copyright 2002 Sun Microsystems, Inc. ALL RIGHTS RESERVED.
 *
 * Portions of the attached software ("Contribution") are developed by 
 * SUN MICROSYSTEMS, INC., and are contributed to the OpenSSL project.
 *
 * The Contribution is licensed pursuant to the OpenSSL open source
 * license provided above.
 *
 * The elliptic curve binary polynomial software is originally written by 
 * Sheueling Chang Shantz and Douglas Stebila of Sun Microsystems Laboratories.
 *
 */

#include "ec_lcl.h"
#include <openssl/err.h>
#include <openssl/obj_mac.h>

typedef struct ec_curve_data_st {
	int	field_type;	/* either NID_X9_62_prime_field or
				 * NID_X9_62_characteristic_two_field */
	const char *p;		/* either a prime number or a polynomial */
	const char *a;
	const char *b;
	const char *x;		/* the x coordinate of the generator */
	const char *y;		/* the y coordinate of the generator */
	const char *order;	/* the order of the group generated by the
				 * generator */
	const BN_ULONG cofactor;/* the cofactor */
	const unsigned char *seed;/* the seed (optional) */
	size_t	seed_len;
	const char *comment;	/* a short (less than 80 characters)
				 * description of the curve */
} EC_CURVE_DATA;

/* the nist prime curves */
static const unsigned char _EC_NIST_PRIME_192_SEED[] = {
	0x30,0x45,0xAE,0x6F,0xC8,0x42,0x2F,0x64,0xED,0x57,
	0x95,0x28,0xD3,0x81,0x20,0xEA,0xE1,0x21,0x96,0xD5};
static const EC_CURVE_DATA _EC_NIST_PRIME_192 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC",
	"64210519E59C80E70FA7E9AB72243049FEB8DEECC146B9B1",
	"188DA80EB03090F67CBF20EB43A18800F4FF0AFD82FF1012",
	"07192b95ffc8da78631011ed6b24cdd573f977a11e794811",
	"FFFFFFFFFFFFFFFFFFFFFFFF99DEF836146BC9B1B4D22831",1,
	_EC_NIST_PRIME_192_SEED, 20,
	"192 bit prime curve from X9.62 and SECG"
	};

static const unsigned char _EC_NIST_PRIME_224_SEED[] = {
	0xBD,0x71,0x34,0x47,0x99,0xD5,0xC7,0xFC,0xDC,0x45,
	0xB5,0x9F,0xA3,0xB9,0xAB,0x8F,0x6A,0x94,0x8B,0xC5};
static const EC_CURVE_DATA _EC_NIST_PRIME_224 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000000000000000000001",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFE",
	"B4050A850C04B3ABF54132565044B0B7D7BFD8BA270B39432355FFB4",
	"B70E0CBD6BB4BF7F321390B94A03C1D356C21122343280D6115C1D21",
	"bd376388b5f723fb4c22dfe6cd4375a05a07476444d5819985007e34",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFF16A2E0B8F03E13DD29455C5C2A3D",1,
	_EC_NIST_PRIME_224_SEED, 20,
	"SECG/NIST recommended curve over a 224 bit prime field"
	};

static const unsigned char _EC_NIST_PRIME_384_SEED[] = {
	0xA3,0x35,0x92,0x6A,0xA3,0x19,0xA2,0x7A,0x1D,0x00,
	0x89,0x6A,0x67,0x73,0xA4,0x82,0x7A,0xCD,0xAC,0x73};
static const EC_CURVE_DATA _EC_NIST_PRIME_384 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFF"
	"FFF0000000000000000FFFFFFFF",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFF"
	"FFF0000000000000000FFFFFFFC",
	"B3312FA7E23EE7E4988E056BE3F82D19181D9C6EFE8141120314088F5013875AC6563"
	"98D8A2ED19D2A85C8EDD3EC2AEF",
	"AA87CA22BE8B05378EB1C71EF320AD746E1D3B628BA79B9859F741E082542A385502F"
	"25DBF55296C3A545E3872760AB7",
	"3617de4a96262c6f5d9e98bf9292dc29f8f41dbd289a147ce9da3113b5f0b8c00a60b"
	"1ce1d7e819d7a431d7c90ea0e5f",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC7634D81F4372DDF581A0"
	"DB248B0A77AECEC196ACCC52973",1,
	_EC_NIST_PRIME_384_SEED, 20,
	"SECG/NIST recommended curve over a 384 bit prime field"
	};

static const unsigned char _EC_NIST_PRIME_521_SEED[] = {
	0xD0,0x9E,0x88,0x00,0x29,0x1C,0xB8,0x53,0x96,0xCC,
	0x67,0x17,0x39,0x32,0x84,0xAA,0xA0,0xDA,0x64,0xBA};
static const EC_CURVE_DATA _EC_NIST_PRIME_521 = {
	NID_X9_62_prime_field,
	"1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
	"1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC",
	"051953EB9618E1C9A1F929A21A0B68540EEA2DA725B99B315F3B8B489918EF109E156"
	"193951EC7E937B1652C0BD3BB1BF073573DF883D2C34F1EF451FD46B503F00",
	"C6858E06B70404E9CD9E3ECB662395B4429C648139053FB521F828AF606B4D3DBAA14"
	"B5E77EFE75928FE1DC127A2FFA8DE3348B3C1856A429BF97E7E31C2E5BD66",
	"011839296a789a3bc0045c8a5fb42c7d1bd998f54449579b446817afbd17273e662c9"
	"7ee72995ef42640c550b9013fad0761353c7086a272c24088be94769fd16650",
	"1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFA51"
	"868783BF2F966B7FCC0148F709A5D03BB5C9B8899C47AEBB6FB71E91386409",1,
	_EC_NIST_PRIME_521_SEED, 20,
	"SECG/NIST recommended curve over a 521 bit prime field"
	};
/* the x9.62 prime curves (minus the nist prime curves) */
static const unsigned char _EC_X9_62_PRIME_192V2_SEED[] = {
	0x31,0xA9,0x2E,0xE2,0x02,0x9F,0xD1,0x0D,0x90,0x1B,
	0x11,0x3E,0x99,0x07,0x10,0xF0,0xD2,0x1A,0xC6,0xB6};
static const EC_CURVE_DATA _EC_X9_62_PRIME_192V2 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC",
	"CC22D6DFB95C6B25E49C0D6364A4E5980C393AA21668D953",
	"EEA2BAE7E1497842F2DE7769CFE9C989C072AD696F48034A",
	"6574d11d69b6ec7a672bb82a083df2f2b0847de970b2de15",
	"FFFFFFFFFFFFFFFFFFFFFFFE5FB1A724DC80418648D8DD31",1,
	_EC_X9_62_PRIME_192V2_SEED, 20,
	"192 bit prime curve from X9.62"
	};

static const unsigned char _EC_X9_62_PRIME_192V3_SEED[] = {
	0xC4,0x69,0x68,0x44,0x35,0xDE,0xB3,0x78,0xC4,0xB6,
	0x5C,0xA9,0x59,0x1E,0x2A,0x57,0x63,0x05,0x9A,0x2E};
static const EC_CURVE_DATA _EC_X9_62_PRIME_192V3 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC",
	"22123DC2395A05CAA7423DAECCC94760A7D462256BD56916",
	"7D29778100C65A1DA1783716588DCE2B8B4AEE8E228F1896",
	"38a90f22637337334b49dcb66a6dc8f9978aca7648a943b0",
	"FFFFFFFFFFFFFFFFFFFFFFFF7A62D031C83F4294F640EC13",1,
	_EC_X9_62_PRIME_192V3_SEED, 20,
	"192 bit prime curve from X9.62"
	};

static const unsigned char _EC_X9_62_PRIME_239V1_SEED[] = {
	0xE4,0x3B,0xB4,0x60,0xF0,0xB8,0x0C,0xC0,0xC0,0xB0,
	0x75,0x79,0x8E,0x94,0x80,0x60,0xF8,0x32,0x1B,0x7D};
static const EC_CURVE_DATA _EC_X9_62_PRIME_239V1 = {
	NID_X9_62_prime_field,
	"7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF",
	"7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC",
	"6B016C3BDCF18941D0D654921475CA71A9DB2FB27D1D37796185C2942C0A",
	"0FFA963CDCA8816CCC33B8642BEDF905C3D358573D3F27FBBD3B3CB9AAAF",
	"7debe8e4e90a5dae6e4054ca530ba04654b36818ce226b39fccb7b02f1ae",
	"7FFFFFFFFFFFFFFFFFFFFFFF7FFFFF9E5E9A9F5D9071FBD1522688909D0B",1,
	_EC_X9_62_PRIME_239V1_SEED, 20,
	"239 bit prime curve from X9.62"
	};

static const unsigned char _EC_X9_62_PRIME_239V2_SEED[] = {
	0xE8,0xB4,0x01,0x16,0x04,0x09,0x53,0x03,0xCA,0x3B,
	0x80,0x99,0x98,0x2B,0xE0,0x9F,0xCB,0x9A,0xE6,0x16};
static const EC_CURVE_DATA _EC_X9_62_PRIME_239V2 = {
	NID_X9_62_prime_field,
	"7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF",
	"7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC",
	"617FAB6832576CBBFED50D99F0249C3FEE58B94BA0038C7AE84C8C832F2C",
	"38AF09D98727705120C921BB5E9E26296A3CDCF2F35757A0EAFD87B830E7",
	"5b0125e4dbea0ec7206da0fc01d9b081329fb555de6ef460237dff8be4ba",
	"7FFFFFFFFFFFFFFFFFFFFFFF800000CFA7E8594377D414C03821BC582063",1,
	_EC_X9_62_PRIME_239V2_SEED, 20,
	"239 bit prime curve from X9.62"
	};

static const unsigned char _EC_X9_62_PRIME_239V3_SEED[] = {
	0x7D,0x73,0x74,0x16,0x8F,0xFE,0x34,0x71,0xB6,0x0A,
	0x85,0x76,0x86,0xA1,0x94,0x75,0xD3,0xBF,0xA2,0xFF};
static const EC_CURVE_DATA _EC_X9_62_PRIME_239V3 = {
	NID_X9_62_prime_field,
	"7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF",
	"7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC",
	"255705FA2A306654B1F4CB03D6A750A30C250102D4988717D9BA15AB6D3E",
	"6768AE8E18BB92CFCF005C949AA2C6D94853D0E660BBF854B1C9505FE95A",
	"1607e6898f390c06bc1d552bad226f3b6fcfe48b6e818499af18e3ed6cf3",
	"7FFFFFFFFFFFFFFFFFFFFFFF7FFFFF975DEB41B3A6057C3C432146526551",1,
	_EC_X9_62_PRIME_239V3_SEED, 20,
	"239 bit prime curve from X9.62"
	};

static const unsigned char _EC_X9_62_PRIME_256V1_SEED[] = {
	0xC4,0x9D,0x36,0x08,0x86,0xE7,0x04,0x93,0x6A,0x66,
	0x78,0xE1,0x13,0x9D,0x26,0xB7,0x81,0x9F,0x7E,0x90};
static const EC_CURVE_DATA _EC_X9_62_PRIME_256V1 = {
	NID_X9_62_prime_field,
	"FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF",
	"FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFC",
	"5AC635D8AA3A93E7B3EBBD55769886BC651D06B0CC53B0F63BCE3C3E27D2604B",
	"6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296",
	"4fe342e2fe1a7f9b8ee7eb4a7c0f9e162bce33576b315ececbb6406837bf51f5",
	"FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632551",1,
	_EC_X9_62_PRIME_256V1_SEED, 20,
	"256 bit prime curve from X9.62 and SECG"
	};
/* the secg prime curves (minus the nist and x9.62 prime curves) */
static const unsigned char _EC_SECG_PRIME_112R1_SEED[] = {
	0x00,0xF5,0x0B,0x02,0x8E,0x4D,0x69,0x6E,0x67,0x68,
	0x75,0x61,0x51,0x75,0x29,0x04,0x72,0x78,0x3F,0xB1};
static const EC_CURVE_DATA _EC_SECG_PRIME_112R1 = {
	NID_X9_62_prime_field,
	"DB7C2ABF62E35E668076BEAD208B",
	"DB7C2ABF62E35E668076BEAD2088",
	"659EF8BA043916EEDE8911702B22",
	"09487239995A5EE76B55F9C2F098",
	"a89ce5af8724c0a23e0e0ff77500",
	"DB7C2ABF62E35E7628DFAC6561C5",1,
	_EC_SECG_PRIME_112R1_SEED, 20,
	"SECG/WTLS recommended curve over a 112 bit prime field"
	};

static const unsigned char _EC_SECG_PRIME_112R2_SEED[] = {
	0x00,0x27,0x57,0xA1,0x11,0x4D,0x69,0x6E,0x67,0x68,
	0x75,0x61,0x51,0x75,0x53,0x16,0xC0,0x5E,0x0B,0xD4};
static const EC_CURVE_DATA _EC_SECG_PRIME_112R2 = {
	NID_X9_62_prime_field,
	"DB7C2ABF62E35E668076BEAD208B",
	"6127C24C05F38A0AAAF65C0EF02C",
	"51DEF1815DB5ED74FCC34C85D709",
	"4BA30AB5E892B4E1649DD0928643",
	"adcd46f5882e3747def36e956e97",
	"36DF0AAFD8B8D7597CA10520D04B",4, 
	_EC_SECG_PRIME_112R2_SEED, 20,
	"SECG recommended curve over a 112 bit prime field"
	};

static const unsigned char _EC_SECG_PRIME_128R1_SEED[] = {
	0x00,0x0E,0x0D,0x4D,0x69,0x6E,0x67,0x68,0x75,0x61,
	0x51,0x75,0x0C,0xC0,0x3A,0x44,0x73,0xD0,0x36,0x79};
static const EC_CURVE_DATA _EC_SECG_PRIME_128R1 = {
	NID_X9_62_prime_field,
	"FFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFF",
	"FFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFC",
	"E87579C11079F43DD824993C2CEE5ED3",
	"161FF7528B899B2D0C28607CA52C5B86",
	"cf5ac8395bafeb13c02da292dded7a83",
	"FFFFFFFE0000000075A30D1B9038A115",1,
	_EC_SECG_PRIME_128R1_SEED, 20,
	"SECG recommended curve over a 128 bit prime field"
	};

static const unsigned char _EC_SECG_PRIME_128R2_SEED[] = {
	0x00,0x4D,0x69,0x6E,0x67,0x68,0x75,0x61,0x51,0x75,
	0x12,0xD8,0xF0,0x34,0x31,0xFC,0xE6,0x3B,0x88,0xF4};
static const EC_CURVE_DATA _EC_SECG_PRIME_128R2 = {
	NID_X9_62_prime_field,
	"FFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFF",
	"D6031998D1B3BBFEBF59CC9BBFF9AEE1",
	"5EEEFCA380D02919DC2C6558BB6D8A5D",
	"7B6AA5D85E572983E6FB32A7CDEBC140",
	"27b6916a894d3aee7106fe805fc34b44",
	"3FFFFFFF7FFFFFFFBE0024720613B5A3",4,
	_EC_SECG_PRIME_128R2_SEED, 20,
	"SECG recommended curve over a 128 bit prime field"
	};

static const EC_CURVE_DATA _EC_SECG_PRIME_160K1 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC73",
	"0",
	"7",
	"3B4C382CE37AA192A4019E763036F4F5DD4D7EBB",
	"938cf935318fdced6bc28286531733c3f03c4fee",
	"0100000000000000000001B8FA16DFAB9ACA16B6B3",1,
	NULL, 0,
	"SECG recommended curve over a 160 bit prime field"
	};

static const unsigned char _EC_SECG_PRIME_160R1_SEED[] = {
	0x10,0x53,0xCD,0xE4,0x2C,0x14,0xD6,0x96,0xE6,0x76,
	0x87,0x56,0x15,0x17,0x53,0x3B,0xF3,0xF8,0x33,0x45};
static const EC_CURVE_DATA _EC_SECG_PRIME_160R1 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFF",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFC",
	"1C97BEFC54BD7A8B65ACF89F81D4D4ADC565FA45",
	"4A96B5688EF573284664698968C38BB913CBFC82",
	"23a628553168947d59dcc912042351377ac5fb32",
	"0100000000000000000001F4C8F927AED3CA752257",1,
	_EC_SECG_PRIME_160R1_SEED, 20,
	"SECG recommended curve over a 160 bit prime field"
	};

static const unsigned char _EC_SECG_PRIME_160R2_SEED[] = {
	0xB9,0x9B,0x99,0xB0,0x99,0xB3,0x23,0xE0,0x27,0x09,
	0xA4,0xD6,0x96,0xE6,0x76,0x87,0x56,0x15,0x17,0x51};
static const EC_CURVE_DATA _EC_SECG_PRIME_160R2 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC73",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC70",
	"B4E134D3FB59EB8BAB57274904664D5AF50388BA",
	"52DCB034293A117E1F4FF11B30F7199D3144CE6D",
	"feaffef2e331f296e071fa0df9982cfea7d43f2e",
	"0100000000000000000000351EE786A818F3A1A16B",1,
	_EC_SECG_PRIME_160R2_SEED, 20,
	"SECG/WTLS recommended curve over a 160 bit prime field"
	};

static const EC_CURVE_DATA _EC_SECG_PRIME_192K1 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFEE37",
	"0",
	"3",
	"DB4FF10EC057E9AE26B07D0280B7F4341DA5D1B1EAE06C7D",
	"9b2f2f6d9c5628a7844163d015be86344082aa88d95e2f9d",
	"FFFFFFFFFFFFFFFFFFFFFFFE26F2FC170F69466A74DEFD8D",1,
	NULL, 20,
	"SECG recommended curve over a 192 bit prime field"
	};

static const EC_CURVE_DATA _EC_SECG_PRIME_224K1 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFE56D",
	"0",
	"5",
	"A1455B334DF099DF30FC28A169A467E9E47075A90F7E650EB6B7A45C",
	"7e089fed7fba344282cafbd6f7e319f7c0b0bd59e2ca4bdb556d61a5",
	"010000000000000000000000000001DCE8D2EC6184CAF0A971769FB1F7",1,
	NULL, 20,
	"SECG recommended curve over a 224 bit prime field"
	};

static const EC_CURVE_DATA _EC_SECG_PRIME_256K1 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFC2F",
	"0",
	"7",
	"79BE667EF9DCBBAC55A06295CE870B07029BFCDB2DCE28D959F2815B16F81798",
	"483ada7726a3c4655da4fbfc0e1108a8fd17b448a68554199c47d08ffb10d4b8",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEBAAEDCE6AF48A03BBFD25E8CD0364141",1,
	NULL, 20,
	"SECG recommended curve over a 256 bit prime field"
	};

/* some wap/wtls curves */
static const EC_CURVE_DATA _EC_WTLS_8 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFDE7",
	"0",
	"3",
	"1",
	"2",
	"0100000000000001ECEA551AD837E9",1,
	NULL, 20,
	"112 bit prime curve from the WTLS standard"
	};

static const EC_CURVE_DATA _EC_WTLS_9 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC808F",
	"0",
	"3",
	"1",
	"2",
	"0100000000000000000001CDC98AE0E2DE574ABF33",1,
	NULL, 20,
	"160 bit prime curve from the WTLS standard"
	};

static const EC_CURVE_DATA _EC_WTLS_12 = {
	NID_X9_62_prime_field,
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000000000000000000001",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFE",
	"B4050A850C04B3ABF54132565044B0B7D7BFD8BA270B39432355FFB4",
	"B70E0CBD6BB4BF7F321390B94A03C1D356C21122343280D6115C1D21",
	"bd376388b5f723fb4c22dfe6cd4375a05a07476444d5819985007e34",
	"FFFFFFFFFFFFFFFFFFFFFFFFFFFF16A2E0B8F03E13DD29455C5C2A3D", 1,
	NULL, 0,
	"224 bit prime curve from the WTLS standard"
	};

/* characteristic two curves */
static const unsigned char _EC_SECG_CHAR2_113R1_SEED[] = {
	0x10,0xE7,0x23,0xAB,0x14,0xD6,0x96,0xE6,0x76,0x87,
	0x56,0x15,0x17,0x56,0xFE,0xBF,0x8F,0xCB,0x49,0xA9};
static const EC_CURVE_DATA _EC_SECG_CHAR2_113R1 = {
	NID_X9_62_characteristic_two_field,
	"020000000000000000000000000201",
	"003088250CA6E7C7FE649CE85820F7",
	"00E8BEE4D3E2260744188BE0E9C723",
	"009D73616F35F4AB1407D73562C10F",
	"00A52830277958EE84D1315ED31886",
	"0100000000000000D9CCEC8A39E56F", 2,
	_EC_SECG_CHAR2_113R1_SEED, 20,
	"SECG recommended curve over a 113 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_113R2_SEED[] = {
	0x10,0xC0,0xFB,0x15,0x76,0x08,0x60,0xDE,0xF1,0xEE,
	0xF4,0xD6,0x96,0xE6,0x76,0x87,0x56,0x15,0x17,0x5D};
static const EC_CURVE_DATA _EC_SECG_CHAR2_113R2 = {
	NID_X9_62_characteristic_two_field,
	"020000000000000000000000000201",
	"00689918DBEC7E5A0DD6DFC0AA55C7",
	"0095E9A9EC9B297BD4BF36E059184F",
	"01A57A6A7B26CA5EF52FCDB8164797",
	"00B3ADC94ED1FE674C06E695BABA1D",
	"010000000000000108789B2496AF93", 2,
	_EC_SECG_CHAR2_113R2_SEED, 20,
	"SECG recommended curve over a 113 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_131R1_SEED[] = {
	0x4D,0x69,0x6E,0x67,0x68,0x75,0x61,0x51,0x75,0x98,
	0x5B,0xD3,0xAD,0xBA,0xDA,0x21,0xB4,0x3A,0x97,0xE2};
static const EC_CURVE_DATA _EC_SECG_CHAR2_131R1 = {
	NID_X9_62_characteristic_two_field,
	"080000000000000000000000000000010D",
	"07A11B09A76B562144418FF3FF8C2570B8",
	"0217C05610884B63B9C6C7291678F9D341",
	"0081BAF91FDF9833C40F9C181343638399",
	"078C6E7EA38C001F73C8134B1B4EF9E150",
	"0400000000000000023123953A9464B54D", 2,
	_EC_SECG_CHAR2_131R1_SEED, 20,
	"SECG/WTLS recommended curve over a 131 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_131R2_SEED[] = {
	0x98,0x5B,0xD3,0xAD,0xBA,0xD4,0xD6,0x96,0xE6,0x76,
	0x87,0x56,0x15,0x17,0x5A,0x21,0xB4,0x3A,0x97,0xE3};
static const EC_CURVE_DATA _EC_SECG_CHAR2_131R2 = {
	NID_X9_62_characteristic_two_field,
	"080000000000000000000000000000010D",
	"03E5A88919D7CAFCBF415F07C2176573B2",
	"04B8266A46C55657AC734CE38F018F2192",
	"0356DCD8F2F95031AD652D23951BB366A8",
	"0648F06D867940A5366D9E265DE9EB240F",
	"0400000000000000016954A233049BA98F", 2,
	_EC_SECG_CHAR2_131R2_SEED, 20,
	"SECG recommended curve over a 131 bit binary field"
	};

static const EC_CURVE_DATA _EC_SECG_CHAR2_163K1 = {
	NID_X9_62_characteristic_two_field,
	"0800000000000000000000000000000000000000C9",
	"1",
	"1",
	"02FE13C0537BBC11ACAA07D793DE4E6D5E5C94EEE8",
	"0289070FB05D38FF58321F2E800536D538CCDAA3D9",
	"04000000000000000000020108A2E0CC0D99F8A5EF", 2,
	NULL, 0,
	"SECG/NIST/WTLS recommended curve over a 163 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_163R1_SEED[] = {
	0x24,0xB7,0xB1,0x37,0xC8,0xA1,0x4D,0x69,0x6E,0x67,
	0x68,0x75,0x61,0x51,0x75,0x6F,0xD0,0xDA,0x2E,0x5C};
static const EC_CURVE_DATA _EC_SECG_CHAR2_163R1 = {
	NID_X9_62_characteristic_two_field,
	"0800000000000000000000000000000000000000C9",
	"07B6882CAAEFA84F9554FF8428BD88E246D2782AE2",
	"0713612DCDDCB40AAB946BDA29CA91F73AF958AFD9",
	"0369979697AB43897789566789567F787A7876A654",
	"00435EDB42EFAFB2989D51FEFCE3C80988F41FF883",
	"03FFFFFFFFFFFFFFFFFFFF48AAB689C29CA710279B", 2,
	_EC_SECG_CHAR2_163R1_SEED, 20,
	"SECG recommended curve over a 163 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_163R2_SEED[] = {
	0x85,0xE2,0x5B,0xFE,0x5C,0x86,0x22,0x6C,0xDB,0x12,
	0x01,0x6F,0x75,0x53,0xF9,0xD0,0xE6,0x93,0xA2,0x68};
static const EC_CURVE_DATA _EC_SECG_CHAR2_163R2 ={
	NID_X9_62_characteristic_two_field,
	"0800000000000000000000000000000000000000C9",
	"1",
	"020A601907B8C953CA1481EB10512F78744A3205FD",
	"03F0EBA16286A2D57EA0991168D4994637E8343E36",
	"00D51FBC6C71A0094FA2CDD545B11C5C0C797324F1",
	"040000000000000000000292FE77E70C12A4234C33", 2,
	_EC_SECG_CHAR2_163R2_SEED, 20,
	"SECG/NIST recommended curve over a 163 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_193R1_SEED[] = {
	0x10,0x3F,0xAE,0xC7,0x4D,0x69,0x6E,0x67,0x68,0x75,
	0x61,0x51,0x75,0x77,0x7F,0xC5,0xB1,0x91,0xEF,0x30};
static const EC_CURVE_DATA _EC_SECG_CHAR2_193R1 = {
	NID_X9_62_characteristic_two_field,
	"02000000000000000000000000000000000000000000008001",
	"0017858FEB7A98975169E171F77B4087DE098AC8A911DF7B01",
	"00FDFB49BFE6C3A89FACADAA7A1E5BBC7CC1C2E5D831478814",
	"01F481BC5F0FF84A74AD6CDF6FDEF4BF6179625372D8C0C5E1",
	"0025E399F2903712CCF3EA9E3A1AD17FB0B3201B6AF7CE1B05",
	"01000000000000000000000000C7F34A778F443ACC920EBA49", 2,
	_EC_SECG_CHAR2_193R1_SEED, 20,
	"SECG recommended curve over a 193 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_193R2_SEED[] = {
	0x10,0xB7,0xB4,0xD6,0x96,0xE6,0x76,0x87,0x56,0x15,
	0x17,0x51,0x37,0xC8,0xA1,0x6F,0xD0,0xDA,0x22,0x11};
static const EC_CURVE_DATA _EC_SECG_CHAR2_193R2 = {
	NID_X9_62_characteristic_two_field,
	"02000000000000000000000000000000000000000000008001",
	"0163F35A5137C2CE3EA6ED8667190B0BC43ECD69977702709B",
	"00C9BB9E8927D4D64C377E2AB2856A5B16E3EFB7F61D4316AE",
	"00D9B67D192E0367C803F39E1A7E82CA14A651350AAE617E8F",
	"01CE94335607C304AC29E7DEFBD9CA01F596F927224CDECF6C",
	"010000000000000000000000015AAB561B005413CCD4EE99D5", 2,
	_EC_SECG_CHAR2_193R2_SEED, 20,
	"SECG recommended curve over a 193 bit binary field"
	};

static const EC_CURVE_DATA _EC_SECG_CHAR2_233K1 = {
	NID_X9_62_characteristic_two_field,
	"020000000000000000000000000000000000000004000000000000000001",
	"0",
	"1",
	"017232BA853A7E731AF129F22FF4149563A419C26BF50A4C9D6EEFAD6126",
	"01DB537DECE819B7F70F555A67C427A8CD9BF18AEB9B56E0C11056FAE6A3",
	"008000000000000000000000000000069D5BB915BCD46EFB1AD5F173ABDF", 4,
	NULL, 0,
	"SECG/NIST/WTLS recommended curve over a 233 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_233R1_SEED[] = {
	0x74,0xD5,0x9F,0xF0,0x7F,0x6B,0x41,0x3D,0x0E,0xA1,
	0x4B,0x34,0x4B,0x20,0xA2,0xDB,0x04,0x9B,0x50,0xC3};
static const EC_CURVE_DATA _EC_SECG_CHAR2_233R1 = {
	NID_X9_62_characteristic_two_field,
	"020000000000000000000000000000000000000004000000000000000001",
	"000000000000000000000000000000000000000000000000000000000001",
	"0066647EDE6C332C7F8C0923BB58213B333B20E9CE4281FE115F7D8F90AD",
	"00FAC9DFCBAC8313BB2139F1BB755FEF65BC391F8B36F8F8EB7371FD558B",
	"01006A08A41903350678E58528BEBF8A0BEFF867A7CA36716F7E01F81052",
	"01000000000000000000000000000013E974E72F8A6922031D2603CFE0D7", 2,
	_EC_SECG_CHAR2_233R1_SEED, 20,
	"SECG/NIST/WTLS recommended curve over a 233 bit binary field"
	};

static const EC_CURVE_DATA _EC_SECG_CHAR2_239K1 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000004000000000000000000000000000000000000001",
	"0",
	"1",
	"29A0B6A887A983E9730988A68727A8B2D126C44CC2CC7B2A6555193035DC",
	"76310804F12E549BDB011C103089E73510ACB275FC312A5DC6B76553F0CA",
	"2000000000000000000000000000005A79FEC67CB6E91F1C1DA800E478A5", 4,
	NULL, 0,
	"SECG recommended curve over a 239 bit binary field"
	};

static const EC_CURVE_DATA _EC_SECG_CHAR2_283K1 = {
	NID_X9_62_characteristic_two_field,
	"080000000000000000000000000000000000000000000000000000000000000000001"
	"0A1",
	"0",
	"1",
	"0503213F78CA44883F1A3B8162F188E553CD265F23C1567A16876913B0C2AC2458492"
	"836",
	"01CCDA380F1C9E318D90F95D07E5426FE87E45C0E8184698E45962364E34116177DD2"
	"259",
	"01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE9AE2ED07577265DFF7F94451E061E163"
	"C61", 4,
	NULL, 20,
	"SECG/NIST recommended curve over a 283 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_283R1_SEED[] = {
	0x77,0xE2,0xB0,0x73,0x70,0xEB,0x0F,0x83,0x2A,0x6D,
	0xD5,0xB6,0x2D,0xFC,0x88,0xCD,0x06,0xBB,0x84,0xBE};
static const EC_CURVE_DATA _EC_SECG_CHAR2_283R1 = {
	NID_X9_62_characteristic_two_field,
	"080000000000000000000000000000000000000000000000000000000000000000001"
	"0A1",
	"000000000000000000000000000000000000000000000000000000000000000000000"
	"001",
	"027B680AC8B8596DA5A4AF8A19A0303FCA97FD7645309FA2A581485AF6263E313B79A"
	"2F5",
	"05F939258DB7DD90E1934F8C70B0DFEC2EED25B8557EAC9C80E2E198F8CDBECD86B12"
	"053",
	"03676854FE24141CB98FE6D4B20D02B4516FF702350EDDB0826779C813F0DF45BE811"
	"2F4",
	"03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEF90399660FC938A90165B042A7CEFADB"
	"307", 2,
	_EC_SECG_CHAR2_283R1_SEED, 20,
	"SECG/NIST recommended curve over a 283 bit binary field"
	};

static const EC_CURVE_DATA _EC_SECG_CHAR2_409K1 = {
	NID_X9_62_characteristic_two_field,
	"020000000000000000000000000000000000000000000000000000000000000000000"
	"00000000000008000000000000000000001",
	"0",
	"1",
	"0060F05F658F49C1AD3AB1890F7184210EFD0987E307C84C27ACCFB8F9F67CC2C4601"
	"89EB5AAAA62EE222EB1B35540CFE9023746",
	"01E369050B7C4E42ACBA1DACBF04299C3460782F918EA427E6325165E9EA10E3DA5F6"
	"C42E9C55215AA9CA27A5863EC48D8E0286B",
	"007FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE5F83B2D4EA20400"
	"EC4557D5ED3E3E7CA5B4B5C83B8E01E5FCF", 4,
	NULL, 0,
	"SECG/NIST recommended curve over a 409 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_409R1_SEED[] = {
	0x40,0x99,0xB5,0xA4,0x57,0xF9,0xD6,0x9F,0x79,0x21,
	0x3D,0x09,0x4C,0x4B,0xCD,0x4D,0x42,0x62,0x21,0x0B};
static const EC_CURVE_DATA _EC_SECG_CHAR2_409R1 = {
	NID_X9_62_characteristic_two_field,
	"020000000000000000000000000000000000000000000000000000000000000000000"
	"00000000000008000000000000000000001",
	"000000000000000000000000000000000000000000000000000000000000000000000"
	"00000000000000000000000000000000001",
	"0021A5C2C8EE9FEB5C4B9A753B7B476B7FD6422EF1F3DD674761FA99D6AC27C8A9A19"
	"7B272822F6CD57A55AA4F50AE317B13545F",
	"015D4860D088DDB3496B0C6064756260441CDE4AF1771D4DB01FFE5B34E59703DC255"
	"A868A1180515603AEAB60794E54BB7996A7",
	"0061B1CFAB6BE5F32BBFA78324ED106A7636B9C5A7BD198D0158AA4F5488D08F38514"
	"F1FDF4B4F40D2181B3681C364BA0273C706",
	"010000000000000000000000000000000000000000000000000001E2AAD6A612F3330"
	"7BE5FA47C3C9E052F838164CD37D9A21173", 2,
	_EC_SECG_CHAR2_409R1_SEED, 20,
	"SECG/NIST recommended curve over a 409 bit binary field"
	};

static const EC_CURVE_DATA _EC_SECG_CHAR2_571K1 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000000000000000000000000000"
	"000000000000000000000000000000000000000000000000000000000000000000000"
	"00425",
	"0",
	"1",
	"026EB7A859923FBC82189631F8103FE4AC9CA2970012D5D46024804801841CA443709"
	"58493B205E647DA304DB4CEB08CBBD1BA39494776FB988B47174DCA88C7E2945283A0"
	"1C8972",
	"0349DC807F4FBF374F4AEADE3BCA95314DD58CEC9F307A54FFC61EFC006D8A2C9D497"
	"9C0AC44AEA74FBEBBB9F772AEDCB620B01A7BA7AF1B320430C8591984F601CD4C143E"
	"F1C7A3",
	"020000000000000000000000000000000000000000000000000000000000000000000"
	"000131850E1F19A63E4B391A8DB917F4138B630D84BE5D639381E91DEB45CFE778F63"
	"7C1001", 4,
	NULL, 0,
	"SECG/NIST recommended curve over a 571 bit binary field"
	};

static const unsigned char _EC_SECG_CHAR2_571R1_SEED[] = {
	0x2A,0xA0,0x58,0xF7,0x3A,0x0E,0x33,0xAB,0x48,0x6B,
	0x0F,0x61,0x04,0x10,0xC5,0x3A,0x7F,0x13,0x23,0x10};
static const EC_CURVE_DATA _EC_SECG_CHAR2_571R1 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000000000000000000000000000"
	"000000000000000000000000000000000000000000000000000000000000000000000"
	"00425",
	"000000000000000000000000000000000000000000000000000000000000000000000"
	"000000000000000000000000000000000000000000000000000000000000000000000"
	"000001",
	"02F40E7E2221F295DE297117B7F3D62F5C6A97FFCB8CEFF1CD6BA8CE4A9A18AD84FFA"
	"BBD8EFA59332BE7AD6756A66E294AFD185A78FF12AA520E4DE739BACA0C7FFEFF7F29"
	"55727A",
	"0303001D34B856296C16C0D40D3CD7750A93D1D2955FA80AA5F40FC8DB7B2ABDBDE53"
	"950F4C0D293CDD711A35B67FB1499AE60038614F1394ABFA3B4C850D927E1E7769C8E"
	"EC2D19",
	"037BF27342DA639B6DCCFFFEB73D69D78C6C27A6009CBBCA1980F8533921E8A684423"
	"E43BAB08A576291AF8F461BB2A8B3531D2F0485C19B16E2F1516E23DD3C1A4827AF1B"
	"8AC15B",
	"03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
	"FFFE661CE18FF55987308059B186823851EC7DD9CA1161DE93D5174D66E8382E9BB2F"
	"E84E47", 2,
	_EC_SECG_CHAR2_571R1_SEED, 20,
	"SECG/NIST recommended curve over a 571 bit binary field"
	};

static const unsigned char _EC_X9_62_CHAR2_163V1_SEED[] = {
	0xD2,0xC0,0xFB,0x15,0x76,0x08,0x60,0xDE,0xF1,0xEE,
	0xF4,0xD6,0x96,0xE6,0x76,0x87,0x56,0x15,0x17,0x54};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_163V1 = {
	NID_X9_62_characteristic_two_field,
	"080000000000000000000000000000000000000107",
	"072546B5435234A422E0789675F432C89435DE5242",
	"00C9517D06D5240D3CFF38C74B20B6CD4D6F9DD4D9",
	"07AF69989546103D79329FCC3D74880F33BBE803CB",
	"01EC23211B5966ADEA1D3F87F7EA5848AEF0B7CA9F",
	"0400000000000000000001E60FC8821CC74DAEAFC1", 2,
	_EC_X9_62_CHAR2_163V1_SEED, 20,
	"163 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_163V2_SEED[] = {
	0x53,0x81,0x4C,0x05,0x0D,0x44,0xD6,0x96,0xE6,0x76,
	0x87,0x56,0x15,0x17,0x58,0x0C,0xA4,0xE2,0x9F,0xFD};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_163V2 = {
	NID_X9_62_characteristic_two_field,
 	"080000000000000000000000000000000000000107",
	"0108B39E77C4B108BED981ED0E890E117C511CF072",
	"0667ACEB38AF4E488C407433FFAE4F1C811638DF20",
	"0024266E4EB5106D0A964D92C4860E2671DB9B6CC5",
	"079F684DDF6684C5CD258B3890021B2386DFD19FC5",
	"03FFFFFFFFFFFFFFFFFFFDF64DE1151ADBB78F10A7", 2,
	_EC_X9_62_CHAR2_163V2_SEED, 20,
	"163 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_163V3_SEED[] = {
	0x50,0xCB,0xF1,0xD9,0x5C,0xA9,0x4D,0x69,0x6E,0x67,
	0x68,0x75,0x61,0x51,0x75,0xF1,0x6A,0x36,0xA3,0xB8};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_163V3 = {
	NID_X9_62_characteristic_two_field,
	"080000000000000000000000000000000000000107",
	"07A526C63D3E25A256A007699F5447E32AE456B50E",
	"03F7061798EB99E238FD6F1BF95B48FEEB4854252B",
	"02F9F87B7C574D0BDECF8A22E6524775F98CDEBDCB",
	"05B935590C155E17EA48EB3FF3718B893DF59A05D0",
	"03FFFFFFFFFFFFFFFFFFFE1AEE140F110AFF961309", 2,
	_EC_X9_62_CHAR2_163V3_SEED, 20,
	"163 bit binary curve from X9.62"
	};

static const EC_CURVE_DATA _EC_X9_62_CHAR2_176V1 = {
	NID_X9_62_characteristic_two_field,
	"0100000000000000000000000000000000080000000007",
	"E4E6DB2995065C407D9D39B8D0967B96704BA8E9C90B",
	"5DDA470ABE6414DE8EC133AE28E9BBD7FCEC0AE0FFF2",
	"8D16C2866798B600F9F08BB4A8E860F3298CE04A5798",
	"6FA4539C2DADDDD6BAB5167D61B436E1D92BB16A562C",
	"00010092537397ECA4F6145799D62B0A19CE06FE26AD", 0xFF6E,
	NULL, 0,
	"176 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_191V1_SEED[] = {
	0x4E,0x13,0xCA,0x54,0x27,0x44,0xD6,0x96,0xE6,0x76,
	0x87,0x56,0x15,0x17,0x55,0x2F,0x27,0x9A,0x8C,0x84};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_191V1 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000201",
	"2866537B676752636A68F56554E12640276B649EF7526267",
	"2E45EF571F00786F67B0081B9495A3D95462F5DE0AA185EC",
	"36B3DAF8A23206F9C4F299D7B21A9C369137F2C84AE1AA0D",
	"765BE73433B3F95E332932E70EA245CA2418EA0EF98018FB",
	"40000000000000000000000004A20E90C39067C893BBB9A5", 2,
	_EC_X9_62_CHAR2_191V1_SEED, 20,
	"191 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_191V2_SEED[] = {
	0x08,0x71,0xEF,0x2F,0xEF,0x24,0xD6,0x96,0xE6,0x76,
	0x87,0x56,0x15,0x17,0x58,0xBE,0xE0,0xD9,0x5C,0x15};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_191V2 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000201",
	"401028774D7777C7B7666D1366EA432071274F89FF01E718",
	"0620048D28BCBD03B6249C99182B7C8CD19700C362C46A01",
	"3809B2B7CC1B28CC5A87926AAD83FD28789E81E2C9E3BF10",
	"17434386626D14F3DBF01760D9213A3E1CF37AEC437D668A",
	"20000000000000000000000050508CB89F652824E06B8173", 4,
	_EC_X9_62_CHAR2_191V2_SEED, 20,
	"191 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_191V3_SEED[] = {
	0xE0,0x53,0x51,0x2D,0xC6,0x84,0xD6,0x96,0xE6,0x76,
	0x87,0x56,0x15,0x17,0x50,0x67,0xAE,0x78,0x6D,0x1F};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_191V3 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000201",
	"6C01074756099122221056911C77D77E77A777E7E7E77FCB",
	"71FE1AF926CF847989EFEF8DB459F66394D90F32AD3F15E8",
	"375D4CE24FDE434489DE8746E71786015009E66E38A926DD",
	"545A39176196575D985999366E6AD34CE0A77CD7127B06BE",
	"155555555555555555555555610C0B196812BFB6288A3EA3", 6,
	_EC_X9_62_CHAR2_191V3_SEED, 20,
	"191 bit binary curve from X9.62"
	};

static const EC_CURVE_DATA _EC_X9_62_CHAR2_208W1 = {
	NID_X9_62_characteristic_two_field,
	"010000000000000000000000000000000800000000000000000007",
	"0000000000000000000000000000000000000000000000000000",
	"C8619ED45A62E6212E1160349E2BFA844439FAFC2A3FD1638F9E",
	"89FDFBE4ABE193DF9559ECF07AC0CE78554E2784EB8C1ED1A57A",
	"0F55B51A06E78E9AC38A035FF520D8B01781BEB1A6BB08617DE3",
	"000101BAF95C9723C57B6C21DA2EFF2D5ED588BDD5717E212F9D", 0xFE48,
	NULL, 0,
	"208 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_239V1_SEED[] = {
	0xD3,0x4B,0x9A,0x4D,0x69,0x6E,0x67,0x68,0x75,0x61,
	0x51,0x75,0xCA,0x71,0xB9,0x20,0xBF,0xEF,0xB0,0x5D};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_239V1 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000000001000000001",
	"32010857077C5431123A46B808906756F543423E8D27877578125778AC76",
	"790408F2EEDAF392B012EDEFB3392F30F4327C0CA3F31FC383C422AA8C16",
	"57927098FA932E7C0A96D3FD5B706EF7E5F5C156E16B7E7C86038552E91D",
	"61D8EE5077C33FECF6F1A16B268DE469C3C7744EA9A971649FC7A9616305",
	"2000000000000000000000000000000F4D42FFE1492A4993F1CAD666E447", 4,
	_EC_X9_62_CHAR2_239V1_SEED, 20,
	"239 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_239V2_SEED[] = {
	0x2A,0xA6,0x98,0x2F,0xDF,0xA4,0xD6,0x96,0xE6,0x76,
	0x87,0x56,0x15,0x17,0x5D,0x26,0x67,0x27,0x27,0x7D};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_239V2 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000000001000000001",
	"4230017757A767FAE42398569B746325D45313AF0766266479B75654E65F",
	"5037EA654196CFF0CD82B2C14A2FCF2E3FF8775285B545722F03EACDB74B",
	"28F9D04E900069C8DC47A08534FE76D2B900B7D7EF31F5709F200C4CA205",
	"5667334C45AFF3B5A03BAD9DD75E2C71A99362567D5453F7FA6E227EC833",
	"1555555555555555555555555555553C6F2885259C31E3FCDF154624522D", 6,
	_EC_X9_62_CHAR2_239V2_SEED, 20,
	"239 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_239V3_SEED[] = {
	0x9E,0x07,0x6F,0x4D,0x69,0x6E,0x67,0x68,0x75,0x61,
	0x51,0x75,0xE1,0x1E,0x9F,0xDD,0x77,0xF9,0x20,0x41};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_239V3 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000000001000000001",
	"01238774666A67766D6676F778E676B66999176666E687666D8766C66A9F",
	"6A941977BA9F6A435199ACFC51067ED587F519C5ECB541B8E44111DE1D40",
	"70F6E9D04D289C4E89913CE3530BFDE903977D42B146D539BF1BDE4E9C92",
	"2E5A0EAF6E5E1305B9004DCE5C0ED7FE59A35608F33837C816D80B79F461",
	"0CCCCCCCCCCCCCCCCCCCCCCCCCCCCCAC4912D2D9DF903EF9888B8A0E4CFF", 0xA,
	_EC_X9_62_CHAR2_239V3_SEED, 20,
	"239 bit binary curve from X9.62"
	};

static const EC_CURVE_DATA _EC_X9_62_CHAR2_272W1 = {
	NID_X9_62_characteristic_two_field,
	"010000000000000000000000000000000000000000000000000000010000000000000"
	"B",
	"91A091F03B5FBA4AB2CCF49C4EDD220FB028712D42BE752B2C40094DBACDB586FB20",
	"7167EFC92BB2E3CE7C8AAAFF34E12A9C557003D7C73A6FAF003F99F6CC8482E540F7",
	"6108BABB2CEEBCF787058A056CBE0CFE622D7723A289E08A07AE13EF0D10D171DD8D",
	"10C7695716851EEF6BA7F6872E6142FBD241B830FF5EFCACECCAB05E02005DDE9D23",
	"000100FAF51354E0E39E4892DF6E319C72C8161603FA45AA7B998A167B8F1E629521",
	0xFF06,
	NULL, 0,
	"272 bit binary curve from X9.62"
	};

static const EC_CURVE_DATA _EC_X9_62_CHAR2_304W1 = {
	NID_X9_62_characteristic_two_field,
	"010000000000000000000000000000000000000000000000000000000000000000000"
	"000000807",
	"FD0D693149A118F651E6DCE6802085377E5F882D1B510B44160074C1288078365A039"
	"6C8E681",
	"BDDB97E555A50A908E43B01C798EA5DAA6788F1EA2794EFCF57166B8C14039601E558"
	"27340BE",
	"197B07845E9BE2D96ADB0F5F3C7F2CFFBD7A3EB8B6FEC35C7FD67F26DDF6285A644F7"
	"40A2614",
	"E19FBEB76E0DA171517ECF401B50289BF014103288527A9B416A105E80260B549FDC1"
	"B92C03B",
	"000101D556572AABAC800101D556572AABAC8001022D5C91DD173F8FB561DA6899164"
	"443051D", 0xFE2E,
	NULL, 0,
	"304 bit binary curve from X9.62"
	};

static const unsigned char _EC_X9_62_CHAR2_359V1_SEED[] = {
	0x2B,0x35,0x49,0x20,0xB7,0x24,0xD6,0x96,0xE6,0x76,
	0x87,0x56,0x15,0x17,0x58,0x5B,0xA1,0x33,0x2D,0xC6};
static const EC_CURVE_DATA _EC_X9_62_CHAR2_359V1 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000000000000000000000000000"
	"000100000000000000001",
	"5667676A654B20754F356EA92017D946567C46675556F19556A04616B567D223A5E05"
	"656FB549016A96656A557",
	"2472E2D0197C49363F1FE7F5B6DB075D52B6947D135D8CA445805D39BC34562608968"
	"7742B6329E70680231988",
	"3C258EF3047767E7EDE0F1FDAA79DAEE3841366A132E163ACED4ED2401DF9C6BDCDE9"
	"8E8E707C07A2239B1B097",
	"53D7E08529547048121E9C95F3791DD804963948F34FAE7BF44EA82365DC7868FE57E"
	"4AE2DE211305A407104BD",
	"01AF286BCA1AF286BCA1AF286BCA1AF286BCA1AF286BC9FB8F6B85C556892C20A7EB9"
	"64FE7719E74F490758D3B", 0x4C,
	_EC_X9_62_CHAR2_359V1_SEED, 20,
	"359 bit binary curve from X9.62"
	};

static const EC_CURVE_DATA _EC_X9_62_CHAR2_368W1 = {
	NID_X9_62_characteristic_two_field,
	"010000000000000000000000000000000000000000000000000000000000000000000"
	"0002000000000000000000007",
	"E0D2EE25095206F5E2A4F9ED229F1F256E79A0E2B455970D8D0D865BD94778C576D62"
	"F0AB7519CCD2A1A906AE30D",
	"FC1217D4320A90452C760A58EDCD30C8DD069B3C34453837A34ED50CB54917E1C2112"
	"D84D164F444F8F74786046A",
	"1085E2755381DCCCE3C1557AFA10C2F0C0C2825646C5B34A394CBCFA8BC16B22E7E78"
	"9E927BE216F02E1FB136A5F",
	"7B3EB1BDDCBA62D5D8B2059B525797FC73822C59059C623A45FF3843CEE8F87CD1855"
	"ADAA81E2A0750B80FDA2310",
	"00010090512DA9AF72B08349D98A5DD4C7B0532ECA51CE03E2D10F3B7AC579BD87E90"
	"9AE40A6F131E9CFCE5BD967", 0xFF70,
	NULL, 0,
	"368 bit binary curve from X9.62"
	};

static const EC_CURVE_DATA _EC_X9_62_CHAR2_431R1 = {
	NID_X9_62_characteristic_two_field,
	"800000000000000000000000000000000000000000000000000000000000000000000"
	"000000001000000000000000000000000000001",
	"1A827EF00DD6FC0E234CAF046C6A5D8A85395B236CC4AD2CF32A0CADBDC9DDF620B0E"
	"B9906D0957F6C6FEACD615468DF104DE296CD8F",
	"10D9B4A3D9047D8B154359ABFB1B7F5485B04CEB868237DDC9DEDA982A679A5A919B6"
	"26D4E50A8DD731B107A9962381FB5D807BF2618",
	"120FC05D3C67A99DE161D2F4092622FECA701BE4F50F4758714E8A87BBF2A658EF8C2"
	"1E7C5EFE965361F6C2999C0C247B0DBD70CE6B7",
	"20D0AF8903A96F8D5FA2C255745D3C451B302C9346D9B7E485E7BCE41F6B591F3E8F6"
	"ADDCBB0BC4C2F947A7DE1A89B625D6A598B3760",
	"0340340340340340340340340340340340340340340340340340340323C313FAB5058"
	"9703B5EC68D3587FEC60D161CC149C1AD4A91", 0x2760,
	NULL, 0,
	"431 bit binary curve from X9.62"
	};

static const EC_CURVE_DATA _EC_WTLS_1 = {
	NID_X9_62_characteristic_two_field,
	"020000000000000000000000000201",
	"1",
	"1",
	"01667979A40BA497E5D5C270780617",
	"00F44B4AF1ECC2630E08785CEBCC15",
	"00FFFFFFFFFFFFFFFDBF91AF6DEA73", 2,
	NULL, 0,
	"113 bit binary curve from the WTLS standard"
	};

typedef struct _ec_list_element_st {
	int	nid;
	const EC_CURVE_DATA *data;
	} ec_list_element;

static const ec_list_element curve_list[] = {
	/* prime field curves */	
	/* secg curves */
	{ NID_secp112r1, &_EC_SECG_PRIME_112R1},
	{ NID_secp112r2, &_EC_SECG_PRIME_112R2},
	{ NID_secp128r1, &_EC_SECG_PRIME_128R1},
	{ NID_secp128r2, &_EC_SECG_PRIME_128R2},
	{ NID_secp160k1, &_EC_SECG_PRIME_160K1},
	{ NID_secp160r1, &_EC_SECG_PRIME_160R1},
	{ NID_secp160r2, &_EC_SECG_PRIME_160R2},
	/* SECG secp192r1 is the same as X9.62 prime192v1 and hence omitted */
	{ NID_secp192k1, &_EC_SECG_PRIME_192K1},
	{ NID_secp224k1, &_EC_SECG_PRIME_224K1},
	{ NID_secp224r1, &_EC_NIST_PRIME_224},
	{ NID_secp256k1, &_EC_SECG_PRIME_256K1},
	/* SECG secp256r1 is the same as X9.62 prime256v1 and hence omitted */
	{ NID_secp384r1, &_EC_NIST_PRIME_384},
	{ NID_secp521r1, &_EC_NIST_PRIME_521},
	/* X9.62 curves */
	{ NID_X9_62_prime192v1, &_EC_NIST_PRIME_192},
	{ NID_X9_62_prime192v2, &_EC_X9_62_PRIME_192V2},
	{ NID_X9_62_prime192v3, &_EC_X9_62_PRIME_192V3},
	{ NID_X9_62_prime239v1, &_EC_X9_62_PRIME_239V1},
	{ NID_X9_62_prime239v2, &_EC_X9_62_PRIME_239V2},
	{ NID_X9_62_prime239v3, &_EC_X9_62_PRIME_239V3},
	{ NID_X9_62_prime256v1, &_EC_X9_62_PRIME_256V1},
	/* characteristic two field curves */
	/* secg curves */
	{ NID_sect113r1, &_EC_SECG_CHAR2_113R1},
	{ NID_sect113r2, &_EC_SECG_CHAR2_113R2},
	{ NID_sect131r1, &_EC_SECG_CHAR2_131R1},
	{ NID_sect131r2, &_EC_SECG_CHAR2_131R2},
	{ NID_sect163k1, &_EC_SECG_CHAR2_163K1},
	{ NID_sect163r1, &_EC_SECG_CHAR2_163R1},
	{ NID_sect163r2, &_EC_SECG_CHAR2_163R2},
	{ NID_sect193r1, &_EC_SECG_CHAR2_193R1},
	{ NID_sect193r2, &_EC_SECG_CHAR2_193R2},
	{ NID_sect233k1, &_EC_SECG_CHAR2_233K1},
	{ NID_sect233r1, &_EC_SECG_CHAR2_233R1},
	{ NID_sect239k1, &_EC_SECG_CHAR2_239K1},
	{ NID_sect283k1, &_EC_SECG_CHAR2_283K1},
	{ NID_sect283r1, &_EC_SECG_CHAR2_283R1},
	{ NID_sect409k1, &_EC_SECG_CHAR2_409K1},
	{ NID_sect409r1, &_EC_SECG_CHAR2_409R1},
	{ NID_sect571k1, &_EC_SECG_CHAR2_571K1},
	{ NID_sect571r1, &_EC_SECG_CHAR2_571R1},
	/* X9.62 curves */
	{ NID_X9_62_c2pnb163v1, &_EC_X9_62_CHAR2_163V1},
	{ NID_X9_62_c2pnb163v2, &_EC_X9_62_CHAR2_163V2},
	{ NID_X9_62_c2pnb163v3, &_EC_X9_62_CHAR2_163V3},
	{ NID_X9_62_c2pnb176v1, &_EC_X9_62_CHAR2_176V1},
	{ NID_X9_62_c2tnb191v1, &_EC_X9_62_CHAR2_191V1},
	{ NID_X9_62_c2tnb191v2, &_EC_X9_62_CHAR2_191V2},
	{ NID_X9_62_c2tnb191v3, &_EC_X9_62_CHAR2_191V3},
	{ NID_X9_62_c2pnb208w1, &_EC_X9_62_CHAR2_208W1},
	{ NID_X9_62_c2tnb239v1, &_EC_X9_62_CHAR2_239V1},
	{ NID_X9_62_c2tnb239v2, &_EC_X9_62_CHAR2_239V2},
	{ NID_X9_62_c2tnb239v3, &_EC_X9_62_CHAR2_239V3},
	{ NID_X9_62_c2pnb272w1, &_EC_X9_62_CHAR2_272W1},
	{ NID_X9_62_c2pnb304w1, &_EC_X9_62_CHAR2_304W1},
	{ NID_X9_62_c2tnb359v1, &_EC_X9_62_CHAR2_359V1},
	{ NID_X9_62_c2pnb368w1, &_EC_X9_62_CHAR2_368W1},
	{ NID_X9_62_c2tnb431r1, &_EC_X9_62_CHAR2_431R1},
	/* the WAP/WTLS curves */
	{ NID_wap_wsg_idm_ecid_wtls1, &_EC_WTLS_1},
	{ NID_wap_wsg_idm_ecid_wtls3, &_EC_SECG_CHAR2_163K1},
	{ NID_wap_wsg_idm_ecid_wtls4, &_EC_SECG_CHAR2_113R1},
	{ NID_wap_wsg_idm_ecid_wtls5, &_EC_X9_62_CHAR2_163V1},
	{ NID_wap_wsg_idm_ecid_wtls6, &_EC_SECG_PRIME_112R1},
	{ NID_wap_wsg_idm_ecid_wtls7, &_EC_SECG_PRIME_160R2},
	{ NID_wap_wsg_idm_ecid_wtls8, &_EC_WTLS_8},
	{ NID_wap_wsg_idm_ecid_wtls9, &_EC_WTLS_9 },
	{ NID_wap_wsg_idm_ecid_wtls10, &_EC_SECG_CHAR2_233K1},
	{ NID_wap_wsg_idm_ecid_wtls11, &_EC_SECG_CHAR2_233R1},
	{ NID_wap_wsg_idm_ecid_wtls12, &_EC_WTLS_12},
};

static size_t curve_list_length = sizeof(curve_list)/sizeof(ec_list_element);

static EC_GROUP *ec_group_new_from_data(const EC_CURVE_DATA *data)
	{
	EC_GROUP *group=NULL;
	EC_POINT *P=NULL;
	BN_CTX	 *ctx=NULL;
	BIGNUM 	 *p=NULL, *a=NULL, *b=NULL, *x=NULL, *y=NULL, *order=NULL;
	int	 ok=0;

	if ((ctx = BN_CTX_new()) == NULL)
		{
		ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_MALLOC_FAILURE);
		goto err;
		}
	if ((p = BN_new()) == NULL || (a = BN_new()) == NULL || 
		(b = BN_new()) == NULL || (x = BN_new()) == NULL ||
		(y = BN_new()) == NULL || (order = BN_new()) == NULL)
		{
		ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_MALLOC_FAILURE);
		goto err;
		}
	
	if (!BN_hex2bn(&p, data->p) || !BN_hex2bn(&a, data->a)
		|| !BN_hex2bn(&b, data->b))
		{
		ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_BN_LIB);
		goto err;
		}

	if (data->field_type == NID_X9_62_prime_field)
		{
		if ((group = EC_GROUP_new_curve_GFp(p, a, b, ctx)) == NULL)
			{
			ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_EC_LIB);
			goto err;
			}
		}
		else
		{ /* field_type == NID_X9_62_characteristic_two_field */
		if ((group = EC_GROUP_new_curve_GF2m(p, a, b, ctx)) == NULL)
			{
			ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_EC_LIB);
			goto err;
			}
		}

	if ((P = EC_POINT_new(group)) == NULL)
		{
		ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_EC_LIB);
		goto err;
		}
	
	if (!BN_hex2bn(&x, data->x) || !BN_hex2bn(&y, data->y))
		{
		ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_BN_LIB);
		goto err;
		}
	if (!EC_POINT_set_affine_coordinates_GF2m(group, P, x, y, ctx))
		{
		ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_EC_LIB);
		goto err;
		}
	if (!BN_hex2bn(&order, data->order) || !BN_set_word(x, data->cofactor))
		{
		ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_BN_LIB);
		goto err;
		}
	if (!EC_GROUP_set_generator(group, P, order, x))
		{
		ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_EC_LIB);
		goto err;
		}
	if (data->seed)
		{
		if (!EC_GROUP_set_seed(group, data->seed, data->seed_len))
			{
			ECerr(EC_F_EC_GROUP_NEW_FROM_DATA, ERR_R_EC_LIB);
			goto err;
			}
		}
	ok=1;
err:
	if (!ok)
		{
		EC_GROUP_free(group);
		group = NULL;
		}
	if (P)
		EC_POINT_free(P);
	if (ctx)
		BN_CTX_free(ctx);
	if (p)
		BN_free(p);
	if (a)
		BN_free(a);
	if (b)
		BN_free(b);
	if (order)
		BN_free(order);
	if (x)
		BN_free(x);
	if (y)
		BN_free(y);
	return group;
	}

EC_GROUP *EC_GROUP_new_by_nid(int nid)
	{
	size_t i;
	EC_GROUP *ret = NULL;

	if (nid <= 0)
		return NULL;

	for (i=0; i<curve_list_length; i++)
		if (curve_list[i].nid == nid)
			{
			ret = ec_group_new_from_data(curve_list[i].data);
			break;
			}

	if (ret == NULL)
		{
		ECerr(EC_F_EC_GROUP_NEW_BY_NID, EC_R_UNKNOWN_GROUP);
		return NULL;
		}

	EC_GROUP_set_nid(ret, nid);

	return ret;
	}

size_t EC_get_builtin_curves(EC_builtin_curve *r, size_t nitems)
	{
	size_t	i, min;

	if (r == NULL || nitems == 0)
		return curve_list_length;

	min = nitems < curve_list_length ? nitems : curve_list_length;

	for (i = 0; i < min; i++)
		{
		r[i].nid = curve_list[i].nid;
		r[i].comment = curve_list[i].data->comment;
		}

	return curve_list_length;
	}
