package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import com.google.inject.Inject;
import com.mysql.jdbc.Statement;

import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Authenticator;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthenticatorProvider;

/**
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 */
public class JDBCAuthenticatorProvider extends AbstractMysqlConnection implements AuthenticatorProvider {

	@Inject
	public JDBCAuthenticatorProvider() {
	}

	@Override
	public String getName() {
		return "Authenticator provider";
	}

	@Override
	public boolean save(Authenticator authenticator) throws ProviderException {
		String sql = String
				.format("INSERT INTO %s (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?);",
						Constants.AUTHENTICATORS__TABLE, Constants.AUTHENTICATORS__COL__ADDRESS,
						Constants.AUTHENTICATORS__COL__CERTIFICATE, Constants.AUTHENTICATORS__COL__COMPANY,
						Constants.AUTHENTICATORS__COL__CREATION_TIME, Constants.AUTHENTICATORS__COL__EMAIL,
						Constants.AUTHENTICATORS__COL__NAME, Constants.AUTHENTICATORS__COL__ITS_AID_LIST,
						Constants.AUTHENTICATORS__COL__ITS_AID_SSP_LIST,
						Constants.AUTHENTICATORS__COL__PRIORITY_ITS_AID_LIST,
						Constants.AUTHENTICATORS__COL__PRIORITY_SSP_LIST,
						Constants.AUTHENTICATORS__COL__IDENTIFIED_REGION,
						Constants.AUTHENTICATORS__COL__CIRCULAR_REGION,
						Constants.AUTHENTICATORS__COL__SUBJECT_ASSURANCE,
						Constants.AUTHENTICATORS__COL__TELEPHONE, Constants.HTTP_USER__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setString(1, authenticator.getAddress());
			st.setString(2, authenticator.getCertificate());
			st.setString(3, authenticator.getCompany());
			st.setTimestamp(4, new Timestamp(System.currentTimeMillis()));
			st.setString(5, authenticator.getEmail());
			st.setString(6, authenticator.getName());
			st.setBytes(7, authenticator.getItsAidList());
			st.setBytes(8, authenticator.getItsAidSspList());
			st.setBytes(9, authenticator.getPriorityItsAidList());
			st.setBytes(10, authenticator.getPrioritySspList());
			st.setBytes(11, authenticator.getIdRegion());
			st.setBytes(12, authenticator.getCircularRegion());
			st.setBytes(13, authenticator.getSubjectAssurance());
			st.setString(14, authenticator.getTelephone());
			st.setString(15, authenticator.getHttpUserId());
			st.execute();
			return true;
		} catch (SQLException e) {
			logger.error("AuthorizedDevice not stored: " + e.getMessage());
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public Authenticator get(String emailAddress) throws ProviderException {

		String sql = String.format("SELECT * FROM %s WHERE %s = ?;", Constants.AUTHENTICATORS__TABLE,
				Constants.AUTHENTICATORS__COL__EMAIL);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setString(1, emailAddress);
			st.execute();
			result = st.getResultSet();

			if (!result.first()) {
				logger.warn("Entry not found");
			} else {
				Authenticator info = new Authenticator();
				info.setAddress(result.getString(Constants.AUTHENTICATORS__COL__ADDRESS));
				info.setCertificate(result.getString(Constants.AUTHENTICATORS__COL__CERTIFICATE));
				info.setCompany(result.getString(Constants.AUTHENTICATORS__COL__COMPANY));
				info.setCreationTime(result.getTimestamp(Constants.AUTHENTICATORS__COL__CREATION_TIME));
				info.setEmail(result.getString(Constants.AUTHENTICATORS__COL__EMAIL));
				info.setName(result.getString(Constants.AUTHENTICATORS__COL__NAME));
				info.setItsAidList(result.getBytes(Constants.AUTHENTICATORS__COL__ITS_AID_LIST));
				info.setItsAidSspList(result.getBytes(Constants.AUTHENTICATORS__COL__ITS_AID_SSP_LIST));
				info.setPriorityItsAidList(result
						.getBytes(Constants.AUTHENTICATORS__COL__PRIORITY_ITS_AID_LIST));
				info.setPrioritySspList(result.getBytes(Constants.AUTHENTICATORS__COL__PRIORITY_SSP_LIST));
				info.setIdRegion(result.getBytes(Constants.AUTHENTICATORS__COL__IDENTIFIED_REGION));
				info.setCircularRegion(result.getBytes(Constants.AUTHENTICATORS__COL__CIRCULAR_REGION));
				info.setSubjectAssurance(result.getBytes(Constants.AUTHENTICATORS__COL__SUBJECT_ASSURANCE));
				info.setTelephone(result.getString(Constants.AUTHENTICATORS__COL__TELEPHONE));
				info.setHttpUserId(result.getString(Constants.HTTP_USER__COL__ID));
				return info;
			}
		} catch (SQLException | IOException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
		return null;
	}

	@Override
	public Authenticator[] getAll(int offset, int limit) throws ProviderException {
		String sql = String.format("SELECT * FROM %s ORDER BY %s DESC LIMIT %d, %d",
				Constants.AUTHENTICATORS__TABLE, Constants.AUTHENTICATORS__COL__CREATION_TIME, offset, limit);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<Authenticator> tmp = new ArrayList<>();
			while (result.next()) {
				Authenticator info = new Authenticator();
				try {

					info.setAddress(result.getString(Constants.AUTHENTICATORS__COL__ADDRESS));
					info.setCompany(result.getString(Constants.AUTHENTICATORS__COL__COMPANY));
					info.setCreationTime(result.getTimestamp(Constants.AUTHENTICATORS__COL__CREATION_TIME));
					info.setEmail(result.getString(Constants.AUTHENTICATORS__COL__EMAIL));
					info.setName(result.getString(Constants.AUTHENTICATORS__COL__NAME));
					info.setItsAidList(result.getBytes(Constants.AUTHENTICATORS__COL__ITS_AID_LIST));
					info.setItsAidSspList(result.getBytes(Constants.AUTHENTICATORS__COL__ITS_AID_SSP_LIST));
					info.setPriorityItsAidList(result
							.getBytes(Constants.AUTHENTICATORS__COL__PRIORITY_ITS_AID_LIST));
					info.setPrioritySspList(result.getBytes(Constants.AUTHENTICATORS__COL__PRIORITY_SSP_LIST));
					info.setIdRegion(result.getBytes(Constants.AUTHENTICATORS__COL__IDENTIFIED_REGION));
					info.setCircularRegion(result.getBytes(Constants.AUTHENTICATORS__COL__CIRCULAR_REGION));
					info.setSubjectAssurance(result
							.getBytes(Constants.AUTHENTICATORS__COL__SUBJECT_ASSURANCE));
					info.setTelephone(result.getString(Constants.AUTHENTICATORS__COL__TELEPHONE));
					info.setHttpUserId(result.getString(Constants.HTTP_USER__COL__ID));

					info.setCertificate(result.getString(Constants.AUTHENTICATORS__COL__CERTIFICATE));

					tmp.add(info);
				} catch (IOException e) {
					logger.warn("Unable to load authenticator \"" + info.getEmail() + "\" from DB. "
							+ e.getMessage());
				}
			}
			return tmp.toArray(new Authenticator[tmp.size()]);
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public Authenticator[] getRegisteredByUser(String httpUser) throws ProviderException {

		String sql = String.format("SELECT * FROM %s WHERE %s = ?;", Constants.AUTHENTICATORS__TABLE,
				Constants.HTTP_USER__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setString(1, httpUser);
			st.execute();
			result = st.getResultSet();

			List<Authenticator> tmp = new ArrayList<>();
			while (result.next()) {

				Authenticator info = new Authenticator();
				info.setAddress(result.getString(Constants.AUTHENTICATORS__COL__ADDRESS));
				info.setCertificate(result.getString(Constants.AUTHENTICATORS__COL__CERTIFICATE));
				info.setCompany(result.getString(Constants.AUTHENTICATORS__COL__COMPANY));
				info.setCreationTime(result.getTimestamp(Constants.AUTHENTICATORS__COL__CREATION_TIME));
				info.setEmail(result.getString(Constants.AUTHENTICATORS__COL__EMAIL));
				info.setName(result.getString(Constants.AUTHENTICATORS__COL__NAME));
				info.setItsAidList(result.getBytes(Constants.AUTHENTICATORS__COL__ITS_AID_LIST));
				info.setItsAidSspList(result.getBytes(Constants.AUTHENTICATORS__COL__ITS_AID_SSP_LIST));
				info.setPriorityItsAidList(result
						.getBytes(Constants.AUTHENTICATORS__COL__PRIORITY_ITS_AID_LIST));
				info.setPrioritySspList(result.getBytes(Constants.AUTHENTICATORS__COL__PRIORITY_SSP_LIST));
				info.setIdRegion(result.getBytes(Constants.AUTHENTICATORS__COL__IDENTIFIED_REGION));
				info.setCircularRegion(result.getBytes(Constants.AUTHENTICATORS__COL__CIRCULAR_REGION));
				info.setSubjectAssurance(result.getBytes(Constants.AUTHENTICATORS__COL__SUBJECT_ASSURANCE));
				info.setTelephone(result.getString(Constants.AUTHENTICATORS__COL__TELEPHONE));
				info.setHttpUserId(result.getString(Constants.HTTP_USER__COL__ID));

				tmp.add(info);
			}
			return tmp.toArray(new Authenticator[tmp.size()]);
		} catch (SQLException | IOException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public int removeAuthenticator(String email) throws ProviderException {
		String sql = String.format("DELETE FROM %s WHERE %s = ?;", Constants.AUTHENTICATORS__TABLE,
				Constants.AUTHENTICATORS__COL__EMAIL);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
			st.setString(1, email);
			return st.executeUpdate();
		} catch (SQLException e) {
			if (logger.isDebugEnabled())
				logger.debug("Cannot delete Authenticator: " + email, e);
			throw new ProviderException("Cannot delete Authenticator: " + email, e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

}
