package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.io.IOException;
import java.util.NoSuchElementException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;

import com.google.inject.Inject;
import com.google.inject.name.Named;

import de.fraunhofer.sit.c2x.pki.ca.certificates.CertificateBuilder;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.CaCertificate;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.CaInfoProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate;

/**
 * Certificate revocation list service (HTTP).
 * 
 * <p>
 * Allows to request the current RCA Certificate via HTTP(S).
 * </p>
 * <p>
 * Supported HTTP request types: GET and POST
 * </p>
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class CACertificateDownloadServlet extends AbstractServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	/**
	 * Base url for retrieving the RCA certificate.
	 * <p>
	 * Example: <code>/rootcert</code>
	 * </p>
	 */
	public static final String URL = "/getcert";
	public static final String TITLE = "Download CA certificate";

	@InjectLogger
	private Logger logger;
	private final CaInfoProvider<CaCertificate> caInfoProvider;

	@Inject
	@Named(value = "caName")
	private String caName;

	@Inject
	CertificateBuilder certificateBuilder;

	@Inject
	public CACertificateDownloadServlet(HtmlProvider htmlProvider,
			CaInfoProvider<CaCertificate> caInfoProvider) {
		super(htmlProvider);
		this.caInfoProvider = caInfoProvider;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		StringBuilder sb = new StringBuilder();

		try {
			Certificate[] certs = caInfoProvider.getAll(0, 100);
			Certificate cert = null;
			byte[] certId = null;
			if (req.getParameter("certId") != null && req.getParameter("certId").length() > 0) {
				certId = Hex.decodeHex(req.getParameter("certId").toCharArray());
				cert = certificateBuilder.readCertificateFromBytes(caInfoProvider.getCaCertificate(certId)
						.getCertificate());
			} else if (certs.length > 0) {
				cert = certificateBuilder.readCertificateFromBytes(certs[0].getCertificate());
			}

			if (certs.length > 1) {
				sb.append("Multiple certificates of the CA are available: \n");
				sb.append("<form name=\"knwonCaForm\" action=\"\" method=\"get\">\n");
				sb.append("<select id=\"certId\" name=\"certId\" style=\"width:200px\">");
				for (Certificate c : certs) {
					if (certId != null
							&& Hex.encodeHexString(c.getCertId()).equals(Hex.encodeHexString(certId)))
						sb.append("<option selected=\"selected\">" + Hex.encodeHexString(c.getCertId())
								+ "</option>\n");
					else
						sb.append("<option>" + Hex.encodeHexString(c.getCertId()) + "</option>\n");
				}
				sb.append("</select>\n");
				sb.append("<input type=\"submit\" value=\"Show Selected\" />");
				sb.append("</form><br /><br />\n");
			}

			if (cert == null)
				throw new ProviderException("no cert available");

			sb.append("Certificate of " + caName + " from type " + cert.getCertTypeAndVersion()
					+ " encoded as HEX string:\n");
			sb.append("<table><tr><td><textarea id=\"caCert\" name=\"caCert\" style=\"width:600px; height:400px;\" readonly=\"readonly\">"
					+ Hex.encodeHexString(cert.getCertificate()) + "</textarea></td></tr>\n");
			sb.append("</table>\n");
			if (logger.isDebugEnabled())
				logger.debug("CA certificate with ID " + Hex.encodeHexString(cert.getCertId())
						+ " is requested from the webpage");
		} catch (IOException | ProviderException | NoSuchElementException | HandlerException e) {
			sb.append("<p class=\"error\">Unable to get certificates from key store</p>\n");
			logger.debug("Unable to get certificates from key store", e);
		} catch (DecoderException e) {
			sb.append("<p class=\"error\">Unable to decode cert ID of requested certificate</p>\n");
			logger.debug("Unable to decode cert ID of requested certificate", e);
		}
		return sb.toString();
	}

	@Override
	public boolean isProtected() {
		return false;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] {};
	}

	@Override
	public Category getCategory() {
		return Category.CA_INFO;
	}
}
