package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.math.BigInteger;

import de.fraunhofer.sit.c2x.pki.ca.validator.region.utils.ISO31661CountryCode;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.utils.UnStatsCountryCode;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RegionDictionaryImpl.RegionDictionary;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class IdentifiedRegion extends WaveElement implements
		de.fraunhofer.sit.c2x.pki.ca.validator.region.interfaces.IdentifiedRegion {

	// ---- fields ----

	@Internal(order = 0)
	private RegionDictionary regionDictionary;

	@Internal(order = 1)
	private UInt16 regionIdentifier;

	@Internal(order = 2)
	private IntX localRegion;

	// ---- constructors ----

	public IdentifiedRegion() {
	}

	public IdentifiedRegion(BigInteger dictionary, BigInteger regionId, BigInteger localId) {

		regionDictionary = RegionDictionaryImpl.getInstance().getEnumType(dictionary.intValue());
		setRegionIdentifier(new UInt16(regionId.intValue()));
		localRegion = new IntX(localId.longValue());

	}

	public IdentifiedRegion(RegionDictionary regionDictionary, int regionId, long localId) {

		this.regionDictionary = regionDictionary;
		setRegionIdentifier(new UInt16(regionId));
		this.localRegion = new IntX(localId);

	}

	public IdentifiedRegion(DataInputStream in) throws IOException {

		byte[] regionBytes = new byte[1];
		in.readFully(regionBytes);
		regionDictionary = RegionDictionaryImpl.getInstance().getEnumType(regionBytes[0]);
		regionIdentifier = new UInt16(in);
		localRegion = new IntX(in);
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public RegionDictionary getGlobalRegion() {
		return this.regionDictionary;
	}

	public IntX getLocalRegion() {
		return this.localRegion;
	}

	// ---- setter ----

	public UInt16 getRegionIdentifier() {
		return regionIdentifier;
	}

	public void setRegionIdentifier(UInt16 regionIdentifier) {
		switch (regionDictionary) {
		case ISO_3166_1:
			if (ISO31661CountryCode.getByCode(regionIdentifier.get()) == null)
				throw new IllegalArgumentException("Given global region code " + regionIdentifier.get()
						+ " not part of ISO 3166-1");
			break;
		case UN_STATS:
			if (UnStatsCountryCode.getByCode(regionIdentifier.get()) == null)
				throw new IllegalArgumentException("Given global region code " + regionIdentifier.get()
						+ " not part of UN states");
			break;
		default:
			throw new IllegalArgumentException(
					"Unable to check whether region ID is correct because dictionary not set.");
		}

		this.regionIdentifier = regionIdentifier;
	}

	public void setGlobalRegion(RegionDictionary regionDictionary) {
		this.regionDictionary = regionDictionary;
	}

	public void setLocalRegion(IntX localRegion) {
		this.localRegion = localRegion;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		if (regionDictionary == null || localRegion == null || regionIdentifier == null) {
			throw new IllegalArgumentException("args may not be null");
		}
		int written = RegionDictionaryImpl.getInstance().writeData(out, regionDictionary);
		written += regionIdentifier.writeData(out);
		written += localRegion.writeData(out);
		return written;
	}

	@Override
	public long getGlobalRegionId() {
		return regionIdentifier.get();
	}

	public String getGlobalRegionString() {
		switch (regionDictionary) {
		case ISO_3166_1:
			return ISO31661CountryCode.getByCode(regionIdentifier.get()).toString();
		case UN_STATS:
			return UnStatsCountryCode.getByCode(regionIdentifier.get()).toString();
		default:
			throw new RuntimeException("Region dictionary not implemented");
		}
	}

	@Override
	public long getLocalRegionId() {
		return localRegion.getValue();
	}

	@Override
	public int getRegionDictionary() {
		return RegionDictionaryImpl.getInstance().getCode(regionDictionary);
	}

	public String getRegionDictionaryString() {
		return regionDictionary.toString();
	}
}