#pragma once

#include <memory>
#include <map>
#include <string>
#include <ctime>
#include <cstdarg>

/**
class Base_Type;
class OCTETSTRING;
class TTCN_Buffer;
class TTCN_Logger;
enum  TTCN_Logger::Severity;
extern void TTCN_error(const char *err_msg, ...) __attribute__ ((__format__ (__printf__, 1, 2), __noreturn__));
extern void TTCN_error_begin(const char *err_msg, ...) __attribute__ ((__format__ (__printf__, 1, 2)));
extern void TTCN_error_end() __attribute__ ((__noreturn__));
void TTCN_warning(const char *warning_msg, ...) __attribute__ ((__format__ (__printf__, 1, 2)));
extern void TTCN_warning_begin(const char *warning_msg, ...) __attribute__ ((__format__ (__printf__, 1, 2)));
extern void TTCN_warning_end();
**/
using namespace std; // Require for isnan()
#include <TTCN3.hh>


class loggers {
  static std::unique_ptr<loggers> _instance;
  std::map<std::string, std::clock_t> _times;
  loggers() : _times() {};

  inline void log_time_exec(const char *fmt, ...);

public:
  virtual ~loggers() {};

  static inline loggers& get_instance() { return *_instance.get(); };
    
  inline void log_to_hexa(const char *prompt, const TTCN_Buffer & buffer);
  inline void log_to_hexa(const char *prompt, const OCTETSTRING& msg);
  inline void log_msg(const char *prompt, const Base_Type& type);
  inline void log(const char *fmt, ...);

  inline void user_msg(const char *prompt, const Base_Type& type);
  inline void user(const char *fmt, ...);
  
  inline void warning(const char *fmt, ...);
  inline void warning_msg(const char *prompt, const Base_Type& type);
  
  inline void error(const char *fmt, ...);
    
  inline void set_start_time(std::string& p_time_key);
  inline void set_stop_time(std::string& p_time_key, float& p_time);
};

void loggers::log_to_hexa(const char *prompt, const TTCN_Buffer & buffer)
{
  TTCN_Logger::begin_event(TTCN_Logger::DEBUG_UNQUALIFIED);
  TTCN_Logger::log_event_str(prompt);
  buffer.log();
  TTCN_Logger::end_event();
}

void loggers::log_to_hexa(const char *prompt, const OCTETSTRING& msg)
{
  TTCN_Logger::begin_event(TTCN_Logger::DEBUG_UNQUALIFIED);
  TTCN_Logger::log_event_str(prompt);
  TTCN_Logger::log_event("Size: %d,\nMsg: ", msg.lengthof());

  for(int i = 0; i < msg.lengthof(); i++) {
    TTCN_Logger::log_event(" %02x", ((const unsigned char*)msg)[i]);
  }
  TTCN_Logger::log_event("\n");
  TTCN_Logger::end_event();
}

void loggers::log_msg(const char *prompt, const Base_Type& type)
{
  TTCN_Logger::begin_event(TTCN_Logger::DEBUG_UNQUALIFIED);
  TTCN_Logger::log_event_str(prompt);
  type.log();
  TTCN_Logger::end_event();
}

void loggers::log(const char *fmt, ...)
{
  TTCN_Logger::begin_event(TTCN_Logger::DEBUG_UNQUALIFIED);
  va_list args;
  va_start(args, fmt);
  TTCN_Logger::log_event_va_list(fmt, args);
  va_end(args);
  TTCN_Logger::end_event();
}

void loggers::user_msg(const char *prompt, const Base_Type& type)
{
  TTCN_Logger::begin_event(TTCN_Logger::USER_UNQUALIFIED);
  TTCN_Logger::log_event_str(prompt);
  type.log();
  TTCN_Logger::end_event();
}

void loggers::user(const char *fmt, ...)
{
  TTCN_Logger::begin_event(TTCN_Logger::USER_UNQUALIFIED);
  va_list args;
  va_start(args, fmt);
  TTCN_Logger::log_event_va_list(fmt, args);
  va_end(args);
  TTCN_Logger::end_event();
}

void loggers::warning(const char *fmt, ...)
{
  va_list args;
  va_start(args, fmt);
  TTCN_warning(fmt, args);
  va_end(args);
}

void loggers::warning_msg(const char *prompt, const Base_Type& type)
{
  TTCN_warning_begin("%s", prompt);
  type.log();
  TTCN_warning_end();
}

void loggers::error(const char *fmt, ...)
{
  va_list args;
  va_start(args, fmt);
  TTCN_error(fmt, args);
  va_end(args);
}

void loggers::set_start_time(std::string& p_time_key)
{
  _times[p_time_key] = std::clock();
}
  
void loggers::set_stop_time(std::string& p_time_key, float& p_time)
{
  std::map<std::string, std::clock_t>::iterator it = _times.find(p_time_key);
  if (it != loggers::_times.end()) {
    p_time = (std::clock() - _times[p_time_key]) * 1000.0 / CLOCKS_PER_SEC; // in milliseconds
    _times.erase(it);
    loggers::get_instance().log_time_exec("%s: Execution duration: %f ms", p_time_key.c_str(), p_time);
    
  }
}

void loggers::log_time_exec(const char *fmt, ...)
{
  TTCN_Logger::begin_event(TTCN_Logger::EXECUTOR_RUNTIME);
  va_list args;
  va_start(args, fmt);
  TTCN_Logger::log_event_va_list(fmt, args);
  va_end(args);
  TTCN_Logger::end_event();
}

