package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.codec.binary.Hex;

import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.PsidSspPriorityDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Authenticator;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.AuthorizedDevice;
import de.fraunhofer.sit.c2x.pki.ca.utils.ByteUtils;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.GeorgraphicRegionValidator;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.CircularRegion;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.GeographicRegion;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.IdentifiedRegion;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.IntX;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidPriority;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidPrioritySsp;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidSsp;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Opaque;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RegionTypeImpl.RegionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttribute;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Time32;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.UInt8;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestriction;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestrictionTypeImpl.ValidityRestrictionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.validator.time.TimeUtils;

public class EtsiPerrmissionUtils {

	public static ArrayList<SubjectAttribute> psidSetToSubjectAttributeArray(
			List<PsidSspPriorityDataContainer> psidSet) {
		ArrayList<SubjectAttribute> list = new ArrayList<SubjectAttribute>();

		if (psidSet == null)
			return list;

		ArrayList<IntX> its_aid_list = new ArrayList<IntX>();

		ArrayList<ItsAidSsp> its_aid_ssp_list = new ArrayList<ItsAidSsp>();

		ArrayList<ItsAidPriority> its_aid_priority_list = new ArrayList<ItsAidPriority>();

		ArrayList<ItsAidPrioritySsp> its_aid_priority_ssp_list = new ArrayList<ItsAidPrioritySsp>();

		for (Iterator<PsidSspPriorityDataContainer> iterator = psidSet.iterator(); iterator.hasNext();) {
			PsidSspPriorityDataContainer psidSspPriorityDataContainer = (PsidSspPriorityDataContainer) iterator
					.next();

			if (psidSspPriorityDataContainer == null || psidSspPriorityDataContainer.isNull()) {
				continue;
			} else if (psidSspPriorityDataContainer.isPsidOnly()) {
				IntX temp = new IntX();
				temp.setValue(psidSspPriorityDataContainer.getPsid());
				its_aid_list.add(temp);
			} else if (psidSspPriorityDataContainer.isPsidSsp()) {
				IntX tempPsid = new IntX();
				tempPsid.setValue(psidSspPriorityDataContainer.getPsid());
				ItsAidSsp temp = new ItsAidSsp();
				temp.setItsAid(tempPsid);
				temp.setServiceSpecificPermissions(new Opaque(psidSspPriorityDataContainer
						.getServiceSpecificPermissions()));
				its_aid_ssp_list.add(temp);
			} else if (psidSspPriorityDataContainer.isPsidMaxPriority()) {
				IntX tempPsid = new IntX();
				tempPsid.setValue(psidSspPriorityDataContainer.getPsid());
				ItsAidPriority temp = new ItsAidPriority();
				temp.setItsAid(tempPsid);
				temp.setMaxPriority(new UInt8(psidSspPriorityDataContainer.getMaxPriority()));
				its_aid_priority_list.add(temp);
			} else if (psidSspPriorityDataContainer.isPsidMaxPrioritySsp()) {
				IntX tempPsid = new IntX();
				tempPsid.setValue(psidSspPriorityDataContainer.getPsid());
				ItsAidPrioritySsp temp = new ItsAidPrioritySsp();
				temp.setItsAid(tempPsid);
				temp.setMaxPriority(new UInt8(psidSspPriorityDataContainer.getMaxPriority()));
				temp.setServiceSpecificPermissions(new Opaque(psidSspPriorityDataContainer
						.getServiceSpecificPermissions()));
				its_aid_priority_ssp_list.add(temp);
			}
		}

		if (its_aid_list.size() > 0) {
			SubjectAttribute its_aid = new SubjectAttribute(
					its_aid_list.toArray(new IntX[its_aid_list.size()]));
			list.add(its_aid);
		}

		if (its_aid_ssp_list.size() > 0) {
			SubjectAttribute its_aid_ssp = new SubjectAttribute(
					its_aid_ssp_list.toArray(new ItsAidSsp[its_aid_ssp_list.size()]));
			list.add(its_aid_ssp);
		}

		if (its_aid_priority_list.size() > 0) {
			SubjectAttribute its_aid_priority = new SubjectAttribute(
					its_aid_priority_list.toArray(new ItsAidPriority[its_aid_priority_list.size()]));
			list.add(its_aid_priority);
		}

		if (its_aid_priority_ssp_list.size() > 0) {
			SubjectAttribute its_aid_priority_ssp = new SubjectAttribute(
					its_aid_priority_ssp_list.toArray(new ItsAidPrioritySsp[its_aid_priority_ssp_list.size()]));
			list.add(its_aid_priority_ssp);
		}

		return list;
	}

	public static boolean checkItsAidPermission(IntX requested, IntX[] permittedItsAids,
			ItsAidSsp[] permittedItsAidSsps, ItsAidPriority[] permittedItsAidPriorities,
			ItsAidPrioritySsp[] permittedItsAidPrioritySsps) throws HandlerException {

		if (requested == null)
			throw new IllegalArgumentException("The argument 'null' is not allowed (ItsAid check)!");

		if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
				&& permittedItsAidPrioritySsps == null) {
			// no restrictions
			return true;
		}

		if (permittedItsAids != null)
			for (IntX itsAid : permittedItsAids) {
				if (requested.equals(itsAid))
					return true;
			}

		throw new HandlerException("ItsAid check failed. ItsAid " + requested.getValue()
				+ " is not permitted!");
	}

	public static boolean checkItsAidSspPermission(ItsAidSsp requested, IntX[] permittedItsAids,
			ItsAidSsp[] permittedItsAidSsps, ItsAidPriority[] permittedItsAidPriorities,
			ItsAidPrioritySsp[] permittedItsAidPrioritySsps) throws HandlerException {

		if (requested == null || requested.getItsAid() == null
				|| requested.getServiceSpecificPermissions() == null)
			throw new IllegalArgumentException("The argument 'null' is not allowed (ItsAidSsp check)!");

		if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
				&& permittedItsAidPrioritySsps == null) {
			// no restrictions
			return true;
		}

		if (permittedItsAids != null)
			for (IntX itsAid : permittedItsAids) {
				if (requested.getItsAid().equals(itsAid)) {
					return true;
				}
			}

		if (permittedItsAidSsps != null)
			for (ItsAidSsp itsAidSsp : permittedItsAidSsps) {
				if (requested.getItsAid().equals(itsAidSsp.getItsAid())) {
					// Aid exists -> check Ssp
					if (isSspPermitted(requested.getServiceSpecificPermissions(),
							itsAidSsp.getServiceSpecificPermissions())) {
						return true;
					}
				}
			}

		throw new HandlerException("ItsAid check failed. ItsAid " + requested.getItsAid().getValue()
				+ " (with Ssp) is not permitted!");
	}

	public static boolean checkItsAidPriorityPermission(ItsAidPriority requested, IntX[] permittedItsAids,
			ItsAidSsp[] permittedItsAidSsps, ItsAidPriority[] permittedItsAidPriorities,
			ItsAidPrioritySsp[] permittedItsAidPrioritySsps) throws HandlerException {

		if (requested == null || requested.getItsAid() == null || requested.getMaxPriority() == null)
			throw new IllegalArgumentException("The argument 'null' is not allowed (ItsAidPriority check)!");

		if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
				&& permittedItsAidPrioritySsps == null) {
			// no restrictions
			return true;
		}

		if (permittedItsAids != null)
			for (IntX itsAid : permittedItsAids) {
				if (requested.getItsAid().equals(itsAid)) {
					return true;
				}
			}

		if (permittedItsAidPriorities != null)
			for (ItsAidPriority itsAidPriority : permittedItsAidPriorities) {
				if (requested.getItsAid().equals(itsAidPriority.getItsAid())) {
					// Aid exists -> check Priority
					if (isPriorityPermitted(requested.getMaxPriority(), itsAidPriority.getMaxPriority())) {
						return true;
					}
				}
			}

		throw new HandlerException("ItsAid check failed. ItsAid " + requested.getItsAid().getValue()
				+ " (with Priority) is not permitted!");
	}

	public static boolean checkItsAidPrioritySspPermission(ItsAidPrioritySsp requested,
			IntX[] permittedItsAids, ItsAidSsp[] permittedItsAidSsps,
			ItsAidPriority[] permittedItsAidPriorities, ItsAidPrioritySsp[] permittedItsAidPrioritySsps)
			throws HandlerException {

		if (requested == null || requested.getItsAid() == null || requested.getMaxPriority() == null
				|| requested.getServiceSpecificPermissions() == null)
			throw new IllegalArgumentException(
					"The argument 'null' is not allowed (ItsAidPrioritySsp check)!");

		if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
				&& permittedItsAidPrioritySsps == null) {
			// no restrictions
			return true;
		}

		if (permittedItsAids != null)
			for (IntX itsAid : permittedItsAids) {
				if (requested.getItsAid().equals(itsAid))
					return true;
			}

		if (permittedItsAidSsps != null)
			for (ItsAidSsp itsAidSsp : permittedItsAidSsps) {
				if (requested.getItsAid().equals(itsAidSsp.getItsAid())) {
					// Aid exists -> check Ssp
					if (isSspPermitted(requested.getServiceSpecificPermissions(),
							itsAidSsp.getServiceSpecificPermissions())) {
						return true;
					}
				}
			}

		if (permittedItsAidPriorities != null)
			for (ItsAidPriority itsAidPriority : permittedItsAidPriorities) {
				if (requested.getItsAid().equals(itsAidPriority.getItsAid())) {
					// Aid exists -> check Priority
					if (isPriorityPermitted(requested.getMaxPriority(), itsAidPriority.getMaxPriority())) {
						return true;
					}
				}
			}

		if (permittedItsAidPrioritySsps != null)
			for (ItsAidPrioritySsp itsAidPrioritySsp : permittedItsAidPrioritySsps) {
				if (requested.getItsAid().equals(itsAidPrioritySsp.getItsAid())) {

					// Aid exists -> check Priority and Ssp
					if (isPriorityPermitted(requested.getMaxPriority(), itsAidPrioritySsp.getMaxPriority())) {
						if (isSspPermitted(requested.getServiceSpecificPermissions(),
								itsAidPrioritySsp.getServiceSpecificPermissions())) {
							return true;
						}
					}
				}
			}

		throw new HandlerException("ItsAid check failed. ItsAid " + requested.getItsAid().getValue()
				+ " (with Ssp and Priority) is not permitted!");
	}

	public static boolean checkSubjectAttributes(SubjectAttribute[] requestedAttributes,
			SubjectAttribute[] permittedAttributes) throws HandlerException {

		try {

			IntX[] permittedItsAids = null;
			ItsAidSsp[] permittedItsAidSsps = null;
			ItsAidPriority[] permittedItsAidPriorities = null;
			ItsAidPrioritySsp[] permittedItsAidPrioritySsps = null;
			Integer permittedAssuranceLevel = null;

			for (SubjectAttribute sa : permittedAttributes) {

				switch (sa.getType()) {
				case ASSURANCE_LEVEL:
					if (permittedAssuranceLevel != null)
						throw new HandlerException("Error - The CA has multiple assurance levels!");
					if (sa.getAssuranceLevel().getSubjectAssurance().get().length != 1)
						throw new HandlerException("Error - The CA assurance level with invalid length: "
								+ sa.getAssuranceLevel().getSubjectAssurance().get().length);
					permittedAssuranceLevel = subjectAssuranceByteToInt(sa.getAssuranceLevel()
							.getSubjectAssurance().get());
					break;

				case ITS_AID_LIST:

					if (permittedItsAids != null)
						throw new HandlerException("Error - The CA has multiple ItsAid lists!");

					permittedItsAids = sa.getItsAidList();

					break;

				case ITS_AID_SSP_LIST:

					if (permittedItsAidSsps != null)
						throw new HandlerException("Error - The CA has multiple ItsAidSsp lists!");

					permittedItsAidSsps = sa.getItsAidSspList();

					break;

				case PRIORITY_ITS_AID_LIST:

					if (permittedItsAidPriorities != null)
						throw new HandlerException("Error - The CA has multiple ItsAidPriority lists!");

					permittedItsAidPriorities = sa.getItsAidPriorityList();

					break;

				case PRIORITY_SSP_LIST:

					if (permittedItsAidPrioritySsps != null)
						throw new HandlerException("Error - The CA has multiple ItsAidPrioritySsp lists!");

					permittedItsAidPrioritySsps = sa.getItsAidPrioritySspList();

					break;
				default:
					break;
				}

			}

			/*
			 * If no subject attributes are given at all, there are no
			 * restrictions and all possibly requested subject attributes are
			 * allowed.
			 */
			if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
					&& permittedItsAidPrioritySsps == null && permittedAssuranceLevel == null)
				return true;

			boolean itsAidPermissionsRequested = false;

			for (SubjectAttribute sa : requestedAttributes) {

				switch (sa.getType()) {
				case ASSURANCE_LEVEL:
					if (sa.getAssuranceLevel().getSubjectAssurance().get().length != 1)
						throw new HandlerException("Requested assurance level has wrong size: "
								+ sa.getAssuranceLevel().getSubjectAssurance().get().length);
					int requestedAssuranceLevel = subjectAssuranceByteToInt(sa.getAssuranceLevel()
							.getSubjectAssurance().get());
					if (requestedAssuranceLevel > permittedAssuranceLevel)
						throw new HandlerException("Requested assurance level " + requestedAssuranceLevel
								+ " larger than permitted assurance level " + permittedAssuranceLevel);

					break;
				case ITS_AID_LIST:
					if (sa.getItsAidList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (IntX requested : sa.getItsAidList()) {
						checkItsAidPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case ITS_AID_SSP_LIST:
					if (sa.getItsAidSspList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidSsp requested : sa.getItsAidSspList()) {
						checkItsAidSspPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case PRIORITY_ITS_AID_LIST:
					if (sa.getItsAidPriorityList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidPriority requested : sa.getItsAidPriorityList()) {
						checkItsAidPriorityPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case PRIORITY_SSP_LIST:
					if (sa.getItsAidPrioritySspList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidPrioritySsp requested : sa.getItsAidPrioritySspList()) {
						checkItsAidPrioritySspPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;
				default:
					break;
				}

			}

			if (itsAidPermissionsRequested == false) {
				/*
				 * If no ItsAids are requested (= max. Permissions) but there
				 * are ItsAids associated with the registered device: Abort
				 */
				if (permittedItsAids != null || permittedItsAidSsps != null
						|| permittedItsAidPriorities != null || permittedItsAidPrioritySsps != null) {
					// if (logger != null)
					// logger.debug("Maximum permissions requested although limitations exist!");
					throw new HandlerException("Maximum permissions requested although limitations exist!");
				}
			}

		} catch (Exception e) {
			// if (logger != null)
			// logger.warn("Checking SubjectAttributes failed: " +
			// e.getMessage());
			throw new HandlerException("Checking SubjectAttributes failed: " + e.getMessage(), e);
		}

		return true;
	}

	public static boolean checkSubjectAttributes(ArrayList<SubjectAttribute> requestedAttributes,
			SubjectAttribute[] permittedAttributes) throws HandlerException {

		try {

			IntX[] permittedItsAids = null;
			ItsAidSsp[] permittedItsAidSsps = null;
			ItsAidPriority[] permittedItsAidPriorities = null;
			ItsAidPrioritySsp[] permittedItsAidPrioritySsps = null;
			Integer permittedAssuranceLevel = null;

			for (SubjectAttribute sa : permittedAttributes) {

				switch (sa.getType()) {
				case ASSURANCE_LEVEL:
					if (permittedAssuranceLevel != null)
						throw new HandlerException("Error - The CA has multiple assurance levels!");
					if (sa.getAssuranceLevel().getSubjectAssurance().get().length != 1)
						throw new HandlerException("Error - The CA assurance level with invalid length: "
								+ sa.getAssuranceLevel().getSubjectAssurance().get().length);
					permittedAssuranceLevel = subjectAssuranceByteToInt(sa.getAssuranceLevel()
							.getSubjectAssurance().get());
					break;

				case ITS_AID_LIST:

					if (permittedItsAids != null)
						throw new HandlerException("Error - The CA has multiple ItsAid lists!");

					permittedItsAids = sa.getItsAidList();

					break;

				case ITS_AID_SSP_LIST:

					if (permittedItsAidSsps != null)
						throw new HandlerException("Error - The CA has multiple ItsAidSsp lists!");

					permittedItsAidSsps = sa.getItsAidSspList();

					break;

				case PRIORITY_ITS_AID_LIST:

					if (permittedItsAidPriorities != null)
						throw new HandlerException("Error - The CA has multiple ItsAidPriority lists!");

					permittedItsAidPriorities = sa.getItsAidPriorityList();

					break;

				case PRIORITY_SSP_LIST:

					if (permittedItsAidPrioritySsps != null)
						throw new HandlerException("Error - The CA has multiple ItsAidPrioritySsp lists!");

					permittedItsAidPrioritySsps = sa.getItsAidPrioritySspList();

					break;
				default:
					break;
				}

			}

			/*
			 * If no subject attributes are given at all, there are no
			 * restrictions and all possibly requested subject attributes are
			 * allowed.
			 */
			if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
					&& permittedItsAidPrioritySsps == null && permittedAssuranceLevel == null)
				return true;

			boolean itsAidPermissionsRequested = false;
//			ArrayList<IntX> doubleAidCheck = new ArrayList<>();
			for (SubjectAttribute sa : requestedAttributes) {

				switch (sa.getType()) {
				case ASSURANCE_LEVEL:
					if (sa.getAssuranceLevel().getSubjectAssurance().get().length != 1)
						throw new HandlerException("Requested assurance level has wrong size: "
								+ sa.getAssuranceLevel().getSubjectAssurance().get().length);
					int requestedAssuranceLevel = subjectAssuranceByteToInt(sa.getAssuranceLevel()
							.getSubjectAssurance().get());
					if (requestedAssuranceLevel > permittedAssuranceLevel)
						throw new HandlerException("Requested assurance level " + requestedAssuranceLevel
								+ " larger than permitted assurance level " + permittedAssuranceLevel);

					break;
				case ITS_AID_LIST:
					if (sa.getItsAidList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (IntX requested : sa.getItsAidList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested))
//							throw new HandlerException("AID " + requested.getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested);

						checkItsAidPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case ITS_AID_SSP_LIST:
					if (sa.getItsAidSspList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidSsp requested : sa.getItsAidSspList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidSspPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case PRIORITY_ITS_AID_LIST:
					if (sa.getItsAidPriorityList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidPriority requested : sa.getItsAidPriorityList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidPriorityPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case PRIORITY_SSP_LIST:
					if (sa.getItsAidPrioritySspList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidPrioritySsp requested : sa.getItsAidPrioritySspList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidPrioritySspPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;
				default:
					break;
				}

			}

			if (itsAidPermissionsRequested == false) {
				/*
				 * If no ItsAids are requested (= max. Permissions) but there
				 * are ItsAids associated with the registered device: Abort
				 */
				if (permittedItsAids != null || permittedItsAidSsps != null
						|| permittedItsAidPriorities != null || permittedItsAidPrioritySsps != null) {
					// if (logger != null)
					// logger.debug("Maximum permissions requested although limitations exist!");
					throw new HandlerException("Maximum permissions requested although limitations exist!");
				}
			}

		} catch (Exception e) {
			// if (logger != null)
			// logger.warn("Checking SubjectAttributes failed: " +
			// e.getMessage());
			throw new HandlerException("Checking SubjectAttributes failed: " + e.getMessage(), e);
		}

		return true;
	}

	public static boolean checkSubjectAttributes(ArrayList<SubjectAttribute> requestedAttributes,
			Authenticator authenticator) throws HandlerException {

		try {

			if (authenticator == null) {
				throw new HandlerException("Error - The given authenticator is null!");
			}

			IntX[] permittedItsAids = null;
			ItsAidSsp[] permittedItsAidSsps = null;
			ItsAidPriority[] permittedItsAidPriorities = null;
			ItsAidPrioritySsp[] permittedItsAidPrioritySsps = null;

			if (authenticator.getSubjectAssurance().length != 1)
				throw new HandlerException("Error - Authenticator assurance level has invalid length: "
						+ authenticator.getSubjectAssurance().length);
			Integer permittedAssuranceLevel = subjectAssuranceByteToInt(authenticator.getSubjectAssurance());

			byte[] permittedItsAidsBytes = authenticator.getItsAidList();
			byte[] permittedItsAidSspsBytes = authenticator.getItsAidSspList();
			byte[] permittedItsAidPrioritiesBytes = authenticator.getPriorityItsAidList();
			byte[] permittedItsAidPrioritySspsBytes = authenticator.getPrioritySspList();

			if (permittedItsAidsBytes != null) {
				ByteArrayInputStream bais = new ByteArrayInputStream(permittedItsAidsBytes);
				DataInputStream dis = new DataInputStream(bais);
				permittedItsAids = WaveUtils.getArrayFromStream(dis, IntX.class);
				dis.close();
				bais.close();
			}
			if (permittedItsAidSspsBytes != null) {
				ByteArrayInputStream bais = new ByteArrayInputStream(permittedItsAidSspsBytes);
				DataInputStream dis = new DataInputStream(bais);
				permittedItsAidSsps = WaveUtils.getArrayFromStream(dis, ItsAidSsp.class);
				dis.close();
				bais.close();
			}
			if (permittedItsAidPrioritiesBytes != null) {
				ByteArrayInputStream bais = new ByteArrayInputStream(permittedItsAidPrioritiesBytes);
				DataInputStream dis = new DataInputStream(bais);
				permittedItsAidPriorities = WaveUtils.getArrayFromStream(dis, ItsAidPriority.class);
				dis.close();
				bais.close();
			}
			if (permittedItsAidPrioritySspsBytes != null) {
				ByteArrayInputStream bais = new ByteArrayInputStream(permittedItsAidPrioritySspsBytes);
				DataInputStream dis = new DataInputStream(bais);
				permittedItsAidPrioritySsps = WaveUtils.getArrayFromStream(dis, ItsAidPrioritySsp.class);
				dis.close();
				bais.close();
			}

			/*
			 * If no subject attributes are given at all, there are no
			 * restrictions and all possibly requested subject attributes are
			 * allowed.
			 */
			if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
					&& permittedItsAidPrioritySsps == null && permittedAssuranceLevel == null)
				return true;

			boolean itsAidPermissionsRequested = false;
//			ArrayList<IntX> doubleAidCheck = new ArrayList<>();
			for (SubjectAttribute sa : requestedAttributes) {

				switch (sa.getType()) {
				case ASSURANCE_LEVEL:
					if (sa.getAssuranceLevel().getSubjectAssurance().get().length != 1)
						throw new HandlerException("Requested assurance level has wrong size: "
								+ sa.getAssuranceLevel().getSubjectAssurance().get().length);
					int requestedAssuranceLevel = subjectAssuranceByteToInt(sa.getAssuranceLevel()
							.getSubjectAssurance().get());
					if (requestedAssuranceLevel > permittedAssuranceLevel)
						throw new HandlerException("Requested assurance level " + requestedAssuranceLevel
								+ " larger than permitted assurance level " + permittedAssuranceLevel);
					break;

				case ITS_AID_LIST:
					if (sa.getItsAidList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (IntX requested : sa.getItsAidList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested))
//							throw new HandlerException("AID " + requested.getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested);

						checkItsAidPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case ITS_AID_SSP_LIST:
					if (sa.getItsAidSspList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidSsp requested : sa.getItsAidSspList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidSspPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case PRIORITY_ITS_AID_LIST:
					if (sa.getItsAidPriorityList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidPriority requested : sa.getItsAidPriorityList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidPriorityPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case PRIORITY_SSP_LIST:
					if (sa.getItsAidPrioritySspList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidPrioritySsp requested : sa.getItsAidPrioritySspList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidPrioritySspPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;
				default:
					break;
				}

			}

			if (itsAidPermissionsRequested == false) {
				/*
				 * If no ItsAids are requested (= max. Permissions) but there
				 * are ItsAids associated with the registered device: Abort
				 */
				if (permittedItsAids != null || permittedItsAidSsps != null
						|| permittedItsAidPriorities != null || permittedItsAidPrioritySsps != null) {
					// if (logger != null)
					// logger.debug("Maximum permissions requested although limitations exist!");
					throw new HandlerException("Maximum permissions requested although limitations exist!");
				}
			}

		} catch (Exception e) {
			// if (logger != null)
			// logger.warn("Checking SubjectAttributes failed: " +
			// e.getMessage());
			throw new HandlerException("Checking SubjectAttributes with Authenticator failed: "
					+ e.getMessage(), e);
		}

		return true;
	}

	public static boolean checkSubjectAttributes(SubjectAttribute[] requestedAttributes,
			AuthorizedDevice permittedAttributes) throws HandlerException {

		try {
			Integer permittedAssuranceLevel = subjectAssuranceByteToInt(permittedAttributes
					.getSubjectAssurance());

			IntX[] permittedItsAids = null;
			if (permittedAttributes.getItsAidList() != null) {
				permittedItsAids = WaveUtils.getArrayFromStream(
						ByteUtils.bytesAsStream(permittedAttributes.getItsAidList()), IntX.class);
			}

			ItsAidSsp[] permittedItsAidSsps = null;
			if (permittedAttributes.getItsAidSspList() != null) {
				permittedItsAidSsps = WaveUtils.getArrayFromStream(
						ByteUtils.bytesAsStream(permittedAttributes.getItsAidSspList()), ItsAidSsp.class);
			}

			ItsAidPriority[] permittedItsAidPriorities = null;
			if (permittedAttributes.getPriorityItsAidList() != null) {
				permittedItsAidPriorities = WaveUtils.getArrayFromStream(
						ByteUtils.bytesAsStream(permittedAttributes.getPriorityItsAidList()),
						ItsAidPriority.class);
			}

			ItsAidPrioritySsp[] permittedItsAidPrioritySsps = null;
			if (permittedAttributes.getPrioritySspList() != null) {
				permittedItsAidPrioritySsps = WaveUtils.getArrayFromStream(
						ByteUtils.bytesAsStream(permittedAttributes.getPrioritySspList()),
						ItsAidPrioritySsp.class);
			}

			/*
			 * If no subject attributes are given at all, there are no
			 * restrictions and all possibly requested subject attributes are
			 * allowed.
			 */
			if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
					&& permittedItsAidPrioritySsps == null && permittedAssuranceLevel == null)
				return true;

			boolean itsAidPermissionsRequested = false;
//			ArrayList<IntX> doubleAidCheck = new ArrayList<>();
			for (SubjectAttribute sa : requestedAttributes) {

				switch (sa.getType()) {
				case ASSURANCE_LEVEL:
					if (sa.getAssuranceLevel().getSubjectAssurance().get().length != 1)
						throw new HandlerException("Requested assurance level has wrong size: "
								+ sa.getAssuranceLevel().getSubjectAssurance().get().length);
					int requestedAssuranceLevel = subjectAssuranceByteToInt(sa.getAssuranceLevel()
							.getSubjectAssurance().get());
					if (requestedAssuranceLevel > permittedAssuranceLevel)
						throw new HandlerException("Requested assurance level " + requestedAssuranceLevel
								+ " larger than permitted assurance level " + permittedAssuranceLevel);
					break;

				case ITS_AID_LIST:
					if (sa.getItsAidList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (IntX requested : sa.getItsAidList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested))
//							throw new HandlerException("AID " + requested.getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested);

						checkItsAidPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case ITS_AID_SSP_LIST:
					if (sa.getItsAidSspList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidSsp requested : sa.getItsAidSspList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidSspPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case PRIORITY_ITS_AID_LIST:
					if (sa.getItsAidPriorityList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidPriority requested : sa.getItsAidPriorityList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidPriorityPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;

				case PRIORITY_SSP_LIST:
					if (sa.getItsAidPrioritySspList() == null)
						continue;
					else
						itsAidPermissionsRequested = true;

					for (ItsAidPrioritySsp requested : sa.getItsAidPrioritySspList()) {
//						// check whether a AID is requested multiple times
//						if (doubleAidCheck.contains(requested.getItsAid()))
//							throw new HandlerException("AID " + requested.getItsAid().getValue()
//									+ " multiple times requested");
//						else
//							doubleAidCheck.add(requested.getItsAid());

						checkItsAidPrioritySspPermission(requested, permittedItsAids, permittedItsAidSsps,
								permittedItsAidPriorities, permittedItsAidPrioritySsps);
					}

					break;
				default:
					break;
				}

			}

			if (itsAidPermissionsRequested == false) {
				/*
				 * If no ItsAids are requested (= max. Permissions) but there
				 * are ItsAids associated with the registered device: Abort
				 */
				if (permittedItsAids != null || permittedItsAidSsps != null
						|| permittedItsAidPriorities != null || permittedItsAidPrioritySsps != null) {
					// if (logger != null)
					// logger.debug("Maximum permissions requested although limitations exist!");
					throw new HandlerException("Maximum permissions requested although limitations exist!");
				}
			}

		} catch (Exception e) {
			// if (logger != null)
			// logger.warn("Checking SubjectAttributes failed: " +
			// e.getMessage());
			throw new HandlerException("Checking SubjectAttributes against device "
					+ Hex.encodeHexString(permittedAttributes.getCanonicalId()) + " failed: "
					+ e.getMessage(), e);
		}

		return true;
	}

	public static boolean checkSubjectAttributes(AuthorizedDevice requestedAttributes,
			SubjectAttribute[] permittedAttributes) throws HandlerException {

		try {

			IntX[] permittedItsAids = null;
			ItsAidSsp[] permittedItsAidSsps = null;
			ItsAidPriority[] permittedItsAidPriorities = null;
			ItsAidPrioritySsp[] permittedItsAidPrioritySsps = null;
			Integer permittedAssuranceLevel = null;

			for (SubjectAttribute sa : permittedAttributes) {

				switch (sa.getType()) {
				case ASSURANCE_LEVEL:
					if (permittedAssuranceLevel != null)
						throw new HandlerException("Error - The CA has multiple assurance levels!");
					if (sa.getAssuranceLevel().getSubjectAssurance().get().length != 1)
						throw new HandlerException("Error - The CA assurance level with invalid length: "
								+ sa.getAssuranceLevel().getSubjectAssurance().get().length);
					permittedAssuranceLevel = subjectAssuranceByteToInt(sa.getAssuranceLevel()
							.getSubjectAssurance().get());
					break;

				case ITS_AID_LIST:

					if (permittedItsAids != null)
						throw new HandlerException("Error - The CA has multiple ItsAid lists!");

					permittedItsAids = sa.getItsAidList();

					break;

				case ITS_AID_SSP_LIST:

					if (permittedItsAidSsps != null)
						throw new HandlerException("Error - The CA has multiple ItsAidSsp lists!");

					permittedItsAidSsps = sa.getItsAidSspList();

					break;

				case PRIORITY_ITS_AID_LIST:

					if (permittedItsAidPriorities != null)
						throw new HandlerException("Error - The CA has multiple ItsAidPriority lists!");

					permittedItsAidPriorities = sa.getItsAidPriorityList();

					break;

				case PRIORITY_SSP_LIST:

					if (permittedItsAidPrioritySsps != null)
						throw new HandlerException("Error - The CA has multiple ItsAidPrioritySsp lists!");

					permittedItsAidPrioritySsps = sa.getItsAidPrioritySspList();

					break;
				default:
					break;
				}

			}

			/*
			 * If no subject attributes are given at all by the CA, there are no
			 * restrictions and all possibly requested subject attributes are
			 * allowed.
			 */
			if (permittedItsAids == null && permittedItsAidSsps == null && permittedItsAidPriorities == null
					&& permittedItsAidPrioritySsps == null && permittedAssuranceLevel == null) {
				// logger.debug("There are no permission restrictions given in the CA certificate -> no limitations for the request!");
				return true;
			}

			// else: check all subject attributes
			if (requestedAttributes.getSubjectAssurance() != null) {
				if (requestedAttributes.getSubjectAssurance().length != 1)
					throw new HandlerException("Error - Authenticator assurance level has invalid length: "
							+ requestedAttributes.getSubjectAssurance().length);
				Integer requestedAssuranceLevel = subjectAssuranceByteToInt(requestedAttributes
						.getSubjectAssurance());
				if (requestedAssuranceLevel > permittedAssuranceLevel)
					throw new HandlerException("Requested assurance level " + requestedAssuranceLevel
							+ " larger than permitted assurance level " + permittedAssuranceLevel);
			}

			IntX[] requestedItsAids = null;
			if (requestedAttributes.getItsAidList() != null) {
				requestedItsAids = WaveUtils.getArrayFromStream(
						ByteUtils.bytesAsStream(requestedAttributes.getItsAidList()), IntX.class);
			}

			ItsAidSsp[] requestedItsAidSsps = null;
			if (requestedAttributes.getItsAidSspList() != null) {
				requestedItsAidSsps = WaveUtils.getArrayFromStream(
						ByteUtils.bytesAsStream(requestedAttributes.getItsAidSspList()), ItsAidSsp.class);
			}

			ItsAidPriority[] requestedItsAidPriorities = null;
			if (requestedAttributes.getPriorityItsAidList() != null) {
				requestedItsAidPriorities = WaveUtils.getArrayFromStream(
						ByteUtils.bytesAsStream(requestedAttributes.getPriorityItsAidList()),
						ItsAidPriority.class);
			}

			ItsAidPrioritySsp[] requestedItsAidPrioritySsps = null;
			if (requestedAttributes.getPrioritySspList() != null) {
				requestedItsAidPrioritySsps = WaveUtils.getArrayFromStream(
						ByteUtils.bytesAsStream(requestedAttributes.getPrioritySspList()),
						ItsAidPrioritySsp.class);
			}

			if (requestedItsAids == null && requestedItsAidSsps == null && requestedItsAidPriorities == null
					&& requestedItsAidPrioritySsps == null) {
				/*
				 * If no ItsAids are requested (= max. Permissions) but there
				 * are ItsAids associated with the registered device: Abort
				 */
				if (requestedItsAids != null || requestedItsAidSsps != null
						|| requestedItsAidPriorities != null || requestedItsAidPrioritySsps != null) {
					// if(logger != null)
					// logger.debug("Maximum permissions requested although limitations exist!");
					throw new HandlerException("Maximum permissions requested although limitations exist!");
				}
			}
//			ArrayList<IntX> doubleAidCheck = new ArrayList<>();
			if (requestedItsAids != null)
				for (IntX requested : requestedItsAids) {
//					// check whether a AID is requested multiple times
//					if (doubleAidCheck.contains(requested))
//						throw new HandlerException("AID " + requested.getValue()
//								+ " multiple times requested");
//					else
//						doubleAidCheck.add(requested);

					checkItsAidPermission(requested, permittedItsAids, permittedItsAidSsps,
							permittedItsAidPriorities, permittedItsAidPrioritySsps);
				}

			if (requestedItsAidSsps != null)
				for (ItsAidSsp requested : requestedItsAidSsps) {
//					// check whether a AID is requested multiple times
//					if (doubleAidCheck.contains(requested.getItsAid()))
//						throw new HandlerException("AID " + requested.getItsAid().getValue()
//								+ " multiple times requested");
//					else
//						doubleAidCheck.add(requested.getItsAid());

					checkItsAidSspPermission(requested, permittedItsAids, permittedItsAidSsps,
							permittedItsAidPriorities, permittedItsAidPrioritySsps);
				}

			if (requestedItsAidPriorities != null)
				for (ItsAidPriority requested : requestedItsAidPriorities) {
//					// check whether a AID is requested multiple times
//					if (doubleAidCheck.contains(requested.getItsAid()))
//						throw new HandlerException("AID " + requested.getItsAid().getValue()
//								+ " multiple times requested");
//					else
//						doubleAidCheck.add(requested.getItsAid());

					checkItsAidPriorityPermission(requested, permittedItsAids, permittedItsAidSsps,
							permittedItsAidPriorities, permittedItsAidPrioritySsps);
				}

			if (requestedItsAidPrioritySsps != null)
				for (ItsAidPrioritySsp requested : requestedItsAidPrioritySsps) {
//					// check whether a AID is requested multiple times
//					if (doubleAidCheck.contains(requested.getItsAid()))
//						throw new HandlerException("AID " + requested.getItsAid().getValue()
//								+ " multiple times requested");
//					else
//						doubleAidCheck.add(requested.getItsAid());

					checkItsAidPrioritySspPermission(requested, permittedItsAids, permittedItsAidSsps,
							permittedItsAidPriorities, permittedItsAidPrioritySsps);
				}

		} catch (Exception e) {
			// if(logger != null)
			// logger.warn("Checking SubjectAttributes failed: "+e.getMessage());
			throw new HandlerException("Checking SubjectAttributes failed: " + e.getMessage(), e);
		}

		// if(logger != null)
		// logger.debug("Requested ItsAids successfully verified!");

		return true;
	}

	public static boolean checkValidityRestrictions(ValidityRestriction[] requestedValidityRestrictions,
			AuthorizedDevice permittedAttributes, GeorgraphicRegionValidator regionValidator)
			throws HandlerException {

		// read device region data into format specific structure
		GeographicRegion permittedGeographicRegion = new GeographicRegion();

		if (permittedAttributes.getCircularRegion() != null && permittedAttributes.getIdRegion() != null) {
			throw new HandlerException(
					"A device must not contain more than one region but two regions are given - aborting region checks!");
		}
		try {
			if (permittedAttributes.getCircularRegion() != null) {
				CircularRegion circularRegion = WaveUtils.getElementFromBytes(
						permittedAttributes.getCircularRegion(), CircularRegion.class);
				permittedGeographicRegion.setCircularRegion(circularRegion);
			} else if (permittedAttributes.getIdRegion() != null) {
				IdentifiedRegion idRegion = WaveUtils.getElementFromBytes(permittedAttributes.getIdRegion(),
						IdentifiedRegion.class);
				permittedGeographicRegion.setIdRegion(idRegion);
			} else {
				// no restrictions
				return true;
			}
		} catch (Exception e) {
			throw new HandlerException(
					"LTC region check against the registered device failed due to corrupt data", e);
		}

		GeographicRegion requestedRegion = null;

		for (ValidityRestriction v : requestedValidityRestrictions) {
			if (v.getType() == ValidityRestrictionType.REGION) {
				requestedRegion = v.getRegion();
				break;
			}
		}

		if (requestedRegion == null)
			throw new HandlerException(
					"The request is invalid as it contains no GeographicRegion information - aborting region check!");
		if (permittedGeographicRegion.getRegionType() != RegionType.NONE) {
			// validate region vs. registered device
			regionValidator.setRegion(permittedGeographicRegion);
			if (!regionValidator.validate(requestedRegion)) {
				throw new HandlerException(
						"Region check failed! The device restrictions do not allow the requested region");
			}
			// check passed
			return true;
		} else {
			// no restrictions
			return true;
		}
	}

	public static boolean checkValidityRestrictions(ValidityRestriction[] requestedValidityRestrictions,
			ValidityRestriction[] permittedValidityRestrictions, GeorgraphicRegionValidator regionValidator)
			throws HandlerException {

		// check time

		Time32[] permittedTime = TimeUtils.extractTime(permittedValidityRestrictions);
		Time32[] requestedTime = TimeUtils.extractTime(requestedValidityRestrictions);

		if (permittedTime[1].before(requestedTime[1])) {
			throw new HandlerException("Invalid Request: The requested validity end time "
					+ requestedTime[1].getDateTime() + " is after the permitted validity end time "
					+ permittedTime[1].getDateTime());
		}

		if (requestedTime[0].before(permittedTime[0])) {
			throw new HandlerException("Invalid Request: The requested validity start time "
					+ requestedTime[0].getDateTime() + " is before the permitted validity start time "
					+ permittedTime[0].getDateTime());
		}

		// FIXME check the maximum allowed certificate validity period (as/if
		// defined in the policies)

		// check region
		GeographicRegion permittedRegion = null;

		for (ValidityRestriction v : permittedValidityRestrictions) {
			if (v.getType() == ValidityRestrictionType.REGION) {
				permittedRegion = v.getRegion();
				break;
			}
		}

		if (permittedRegion == null)
			throw new HandlerException(
					"The CA certificate is invalid as it contains no GeographicRegion information - aborting region check!");

		if (permittedRegion.getRegionType() == RegionType.NONE) {
			// no restrictions
			return true;
		}

		GeographicRegion requestedRegion = null;

		for (ValidityRestriction v : requestedValidityRestrictions) {
			if (v.getType() == ValidityRestrictionType.REGION) {
				requestedRegion = v.getRegion();
				break;
			}
		}

		if (requestedRegion == null)
			throw new HandlerException(
					"The request is invalid as it contains no GeographicRegion information - aborting region check!");

		regionValidator.setRegion(permittedRegion);
		if (!regionValidator.validate(requestedRegion)) {
			throw new HandlerException(
					"RegionValidation failed! The LTCA certificate does not allow the requested region!");
		}

		return true;
	}

	public static boolean checkRegionRestrictions(GeographicRegion requestedRegion,
			ValidityRestriction[] permittedValidityRestrictions, GeorgraphicRegionValidator regionValidator)
			throws HandlerException {

		GeographicRegion permittedRegion = null;

		for (ValidityRestriction v : permittedValidityRestrictions) {
			if (v.getType() == ValidityRestrictionType.REGION) {
				permittedRegion = v.getRegion();
				break;
			}
		}

		if (permittedRegion == null)
			throw new HandlerException(
					"The CA certificate is invalid as it contains no GeographicRegion information - aborting region check!");

		if (permittedRegion.getRegionType() == RegionType.NONE) {
			// no restrictions
			return true;
		}

		if (requestedRegion == null)
			throw new HandlerException(
					"The request is invalid as it contains no GeographicRegion information - aborting region check!");

		regionValidator.setRegion(permittedRegion);
		if (!regionValidator.validate(requestedRegion)) {
			throw new HandlerException("RegionValidation failed! The requested region is not allowed!");
		}

		return true;
	}

	public static boolean checkRegionRestrictions(GeographicRegion requestedRegion,
			GeographicRegion permittedRegion, GeorgraphicRegionValidator regionValidator)
			throws HandlerException {

		if (permittedRegion == null)
			throw new HandlerException(
					"The CA certificate is invalid as it contains no GeographicRegion information - aborting region check!");

		if (permittedRegion.getRegionType() == RegionType.NONE) {
			// no restrictions
			return true;
		}

		if (requestedRegion == null)
			throw new HandlerException(
					"The request is invalid as it contains no GeographicRegion information - aborting region check!");

		// Perform check with regionValidator
		regionValidator.setRegion(permittedRegion);
		if (!regionValidator.validate(requestedRegion)) {
			throw new HandlerException("RegionValidation failed! The requested region is not allowed!");
		}

		return true;
	}

	public static boolean checkRegionRestrictions(GeographicRegion requestedRegion,
			Authenticator authenticator, GeorgraphicRegionValidator validator) throws HandlerException {

		try {

			if (authenticator == null) {
				throw new HandlerException("Error - The given authenticator is null!");
			}

			CircularRegion permittedCircularRegion = null;
			IdentifiedRegion permittedIdentifiedRegion = null;

			byte[] permittedCircularRegionBytes = authenticator.getCircularRegion();
			byte[] permittedIdentifiedRegionBytes = authenticator.getIdRegion();

			if (permittedCircularRegionBytes != null) {
				permittedCircularRegion = WaveUtils.getElementFromBytes(permittedCircularRegionBytes,
						CircularRegion.class);
			}
			if (permittedIdentifiedRegionBytes != null) {
				permittedIdentifiedRegion = WaveUtils.getElementFromBytes(permittedIdentifiedRegionBytes,
						IdentifiedRegion.class);
			}

			/*
			 * If no ItsAids are given at all, there are no restrictions and all
			 * possibly requested ItsAids are allowed.
			 */
			if (permittedCircularRegion == null && permittedIdentifiedRegion == null)
				return true;

			if (permittedCircularRegion != null && permittedIdentifiedRegion != null) {
				throw new HandlerException("Error: The given authenticator contains more than one region!");
			}

			GeographicRegion permittedGeographicRegion = null;

			if (permittedCircularRegion != null) {
				permittedGeographicRegion = new GeographicRegion(permittedCircularRegion);
			} else if (permittedIdentifiedRegion != null) {
				permittedGeographicRegion = new GeographicRegion(permittedIdentifiedRegion);
			} else {
				// no region restriction
				return true;
			}

			validator.setRegion(permittedGeographicRegion);

			if (!validator.validate(requestedRegion)) {
				throw new HandlerException("Region Validation failed!");
			}

		} catch (Exception e) {
			// e.printStackTrace();
			throw new HandlerException("Checking GeographicRegion restrictions with Authenticator failed: "
					+ e.getMessage(), e);
		}

		return true;
	}

	public static boolean isPriorityPermitted(UInt8 requestedPriority, UInt8 permittedPriority) {

		if (requestedPriority.get() <= permittedPriority.get()) {
			return true;
		} else {
			return false;
		}
	}

	public static boolean isSspPermitted(Opaque requestedServiceSpecificPermissions,
			Opaque permittedServiceSpecificPermissions) {

		if (requestedServiceSpecificPermissions == null || permittedServiceSpecificPermissions == null) {
			throw new IllegalArgumentException("The parameters must not be null");
		}

		byte requested[] = requestedServiceSpecificPermissions.get();
		byte permitted[] = permittedServiceSpecificPermissions.get();

		if (requested.length != permitted.length) {
			throw new IllegalArgumentException(
					"The byte array lengths of the provided ServiceSpecificPermissions do not match!");
		}

		for (int i = 0; i < permitted.length; i++) {
			if (permitted[i] != (requested[i] | permitted[i])) {
				// there exists at least one bit within the request that is not
				// permitted
				return false;
			}
		}

		// check passed
		return true;
	}

	public static int subjectAssuranceByteToInt(byte[] subjectAssurance) throws HandlerException {
		if (subjectAssurance.length != 1)
			throw new HandlerException("Assurance level has invalid length: " + subjectAssurance.length);
		byte b = subjectAssurance[0];
		for (int i = 0; i < 5; i++)
			b = (byte) (b >>> 1);
		b &= 0x00000007;
		if (b > 7 || b < 0)
			throw new HandlerException("Assurance level has invalid value: " + b);
		return b;
	}

	public static byte[] subjectAssuranceInttoByte(int subjectAssurance) throws HandlerException {
		if (subjectAssurance == 0)
			return new byte[] { (byte) 0x00 };
		else if (subjectAssurance == 1)
			return new byte[] { (byte) 0x20 };
		else if (subjectAssurance == 2)
			return new byte[] { (byte) 0x40 };
		else if (subjectAssurance == 3)
			return new byte[] { (byte) 0x60 };
		else if (subjectAssurance == 4)
			return new byte[] { (byte) 0x80 };
		else if (subjectAssurance == 5)
			return new byte[] { (byte) 0xA0 };
		else if (subjectAssurance == 6)
			return new byte[] { (byte) 0xC0 };
		else if (subjectAssurance == 7)
			return new byte[] { (byte) 0xE0 };
		else
			throw new HandlerException("Unsupported subjectAssurance: " + subjectAssurance);
	}

	public static SubjectAttribute[] cleanSubjectAttributeList(SubjectAttribute[] subjectAttributes) {
		ArrayList<SubjectAttribute> tmp = new ArrayList<SubjectAttribute>();
		for (SubjectAttribute sa : subjectAttributes) {
			switch (sa.getType()) {
			case ITS_AID_LIST:
				if (sa.getItsAidList().length > 0)
					tmp.add(sa);
				break;
			case PRIORITY_SSP_LIST:
				if (sa.getItsAidPrioritySspList().length > 0)
					tmp.add(sa);
				break;
			case ITS_AID_SSP_LIST:
				if (sa.getItsAidSspList().length > 0)
					tmp.add(sa);
				break;
			case PRIORITY_ITS_AID_LIST:
				if (sa.getItsAidPriorityList().length > 0)
					tmp.add(sa);
				break;
			default:
				tmp.add(sa);
			}
		}
		SubjectAttribute[] tmpArray = new SubjectAttribute[tmp.size()];
		return tmp.toArray(tmpArray);
	}
}
