package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils;

import java.io.IOException;
import java.math.BigInteger;
import java.security.Key;
import java.security.Provider;
import java.util.Random;

import javax.crypto.spec.SecretKeySpec;

import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.signers.ECDSASigner;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.provider.JCEECPrivateKey;
import org.bouncycastle.jce.provider.JCEECPublicKey;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPrivateKeySpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;

import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.WaveType;
import de.fraunhofer.sit.c2x.pki.ca.crypto.Digest;
import de.fraunhofer.sit.c2x.pki.ca.crypto.KeyGenerator;
import de.fraunhofer.sit.c2x.pki.ca.crypto.encryption.AESwithCCM;
import de.fraunhofer.sit.c2x.pki.ca.crypto.encryption.ECIES;
import de.fraunhofer.sit.c2x.pki.ca.crypto.encryption.ECIESCipher;
import de.fraunhofer.sit.c2x.pki.ca.crypto.signer.ECDSA;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPoint;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EcdsaSignature;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EciesEncryptedKey;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Opaque;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Signature;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttribute;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttributeTypeImpl.SubjectAttributeType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.UInt8;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 */
public class EtsiCryptoUtils{

	private final static Random RANDOM = new Random();
	public static final Provider BC = new BouncyCastleProvider();

	// private static final Logger logger = Logger.getLogger(CryptoUtils.class);

	public static EcdsaSignature signToECDSA(byte[] messageToSign, CipherParameters privateKey, BigInteger k) {

		ECDSA engine = new ECDSA();
		engine.init(privateKey);
		BigInteger[] signature = engine.generateSignature(messageToSign, k);

		int fieldSize = ((ECPrivateKeyParameters) privateKey).getParameters().getCurve().getFieldSize();
		return new EcdsaSignature(signature[0], signature[1], new UInt8(fieldSize / 8));
	}

	/**
	 * @param spec
	 * @return
	 */
	public static AsymmetricCipherKeyPair generateCompressedECKeyPair(String spec) {
		return KeyGenerator.getInstance().generateCompressedECKeyPair(spec);
	}

	/**
	 * @param string
	 * @return
	 */
	public static AsymmetricCipherKeyPair generateECKeyPair(String spec) {
		return KeyGenerator.getInstance().generateECKeyPair(spec);
	}

	public static EcdsaSignature signToECDSA(byte[] messageToSign, CipherParameters privateKey) {
		return signToECDSA(messageToSign, privateKey, new BigInteger(((ECPrivateKeyParameters) privateKey)
				.getParameters().getN().bitLength(), new Random()));
	}

	public static boolean verifyECDSA(byte[] messageToVerify, EcdsaSignature signature,
			ECPublicKeyParameters publicKey) {

		ECDSA engine = new ECDSA();
		engine.init(publicKey);
		boolean signatureIsValid = engine.verifySignature(messageToVerify, signature.toBigIntegers());

		return signatureIsValid;
	}

	public static boolean verifyECDSAWithoutHashing(byte[] messageHash, EcdsaSignature signature,
			ECPublicKeyParameters publicKey) {

		if (messageHash.length != 32)
			throw new IllegalArgumentException("Given hash has wrong length: " + messageHash.length
					+ " != 32");

		ECDSA engine = new ECDSA();
		engine.init(publicKey);
		boolean signatureIsValid = engine.verifySignatureUsingHash(messageHash, signature.toBigIntegers());

		return signatureIsValid;
	}

	public static BigInteger[] sign(byte[] msg, JCEECPrivateKey params) {
		ECDSASigner signer = new ECDSASigner();

		// logger.trace("------------------------------------");
		// logger.trace("k: " + params);
		// logger.trace("------------------------------------");
		// logger.trace("P: " + params.getParams());
		// logger.trace("C: " + params.getParams().getCurve());
		// logger.trace("C: " + params.getParams().getCofactor());
		// logger.trace("C: " + params.getParams().getGenerator());
		// logger.trace("C: " + params.getParams().getOrder());
		// logger.trace("P: " + params.getParameters());
		// logger.trace("C: " + params.getParameters().getCurve());
		// logger.trace("G: " + params.getParameters().getG());
		// logger.trace("H: " + params.getParameters().getH());
		// logger.trace("N: " + params.getParameters().getN());
		// logger.trace("S: " + params.getParameters().getSeed());
		ECCurve curve = params.getParameters().getCurve();
		ECPoint G = params.getParameters().getG();
		BigInteger n = params.getParameters().getN();
		BigInteger h = params.getParameters().getH();
		byte[] seed = params.getParameters().getSeed();
		ECPrivateKeyParameters params2 = new ECPrivateKeyParameters(params.getD(), new ECDomainParameters(
				curve, G, n, h, seed));
		signer.init(true, params2);

		return signer.generateSignature(msg);
	}
	public static boolean verify(byte[] msg, BigInteger[] signature, JCEECPublicKey params) {
		ECDSASigner signer = new ECDSASigner();

		ECCurve curve = params.getParameters().getCurve();
		ECPoint G = params.getParameters().getG();
		BigInteger n = params.getParameters().getN();
		BigInteger h = params.getParameters().getH();
		byte[] seed = params.getParameters().getSeed();

		ECPublicKeyParameters params2 = new ECPublicKeyParameters(params.getQ(), new ECDomainParameters(
				curve, G, n, h, seed));
		signer.init(false, params2);

		return signer.verifySignature(msg, signature[0], signature[1]);
	}

	public static byte[] randomBytes(int size) {
		byte[] rbytes = new byte[size];
		RANDOM.nextBytes(rbytes);
		return rbytes;
	}

	public static Opaque randomOpaqeArray(int size) {
		Opaque out = new Opaque();
		out.set(randomBytes(size));
		out.setArray(true);
		return out;
	}

	/**
	 * @param unsignedCsr
	 * @return
	 * @throws CryptoUtilsException
	 */
	public static byte[] sha256(byte[] bytes) {
		return Digest.getInstance().sha256(bytes);
	}

	public static EcdsaSignature signToECDSA(WaveType toSign, PublicKeyAlgorithm algorithm,
			JCEECPrivateKey privateKey) {

		BigInteger[] sig = new BigInteger[2];
		sig = sign(WaveUtils.getBytesFromWaveType(toSign), privateKey);
		return new EcdsaSignature(sig[0], sig[1], new UInt8(algorithm.getKeyLength() / 8));
	}

	public static ECDomainParameters getDomain(byte[] bytes) throws Exception {
		ECNamedCurveParameterSpec spec = null;
		if (bytes.length == 57) {
			spec = ECNamedCurveTable.getParameterSpec("P-224");
		} else if (bytes.length == 65) {
			spec = ECNamedCurveTable.getParameterSpec("P-256");
		} else {
			throw new Exception("Invalid key size: " + bytes.length);
			// spec = ECNamedCurveTable.getParameterSpec("P-256"); // FIXME
			// System.out.println("Error");
		}
		return new ECDomainParameters(spec.getCurve(), spec.getG(), spec.getN(), spec.getH(), spec.getSeed());
	}

	public static ECPublicKeyParameters bytesToEcPublicKeyParameters(byte[] bytes) throws Exception {

		ECDomainParameters domain = getDomain(bytes);

		ECPoint q = domain.getCurve().decodePoint(bytes);
		return new ECPublicKeyParameters(q, domain);
	}

	public static ECPrivateKeyParameters bytesToEcPrivateKeyParameters(byte[] bytes) {

		ECNamedCurveParameterSpec spec = ECNamedCurveTable.getParameterSpec("P-256");
		ECDomainParameters domain = new ECDomainParameters(spec.getCurve(), spec.getG(), spec.getN(),
				spec.getH(), spec.getSeed());

		return new ECPrivateKeyParameters(new BigInteger(1, bytes), domain);
	}

	//
	// /**
	// * @param unsignedCertificate
	// * @param ecdsaNistp256WithSha256
	// * @param private1
	// * @return
	// */
	//
	// /**
	// * @param unsignedCertificate
	// * @param ecdsaNistp256WithSha256
	// * @param private1
	// * @return
	// * @throws IOException
	// * @throws CryptoUtilsException
	// */
	public static Signature sign(WaveType dataToSign, PublicKeyAlgorithm algorithm,
			CipherParameters privateKey, BigInteger k) throws IOException {
		if (WaveType.class.isInstance(dataToSign) == false)
			throw new IllegalArgumentException("Argument \"dataToSign\" is not implementing WaveType");

		return sign(WaveUtils.getBytesFromWaveType(dataToSign), algorithm, privateKey, k);
	}

	/**
	 * Sign with ecdsa (P-256, P-224)
	 * 
	 * @param unsignedCertificate
	 * @param ecdsaNistp256WithSha256
	 * @param private1
	 * @return a ecdsa-signature wrapped in a signature
	 * @throws CryptoUtilsException
	 */
	public static Signature sign(byte[] messageToSign, PublicKeyAlgorithm algorithm,
			CipherParameters privateKey, BigInteger k) {

		EcdsaSignature ecdsaSignature = signToECDSA(messageToSign, privateKey, k);
		Signature signature = new Signature();
		signature.setEcdsaSignature(ecdsaSignature);
		signature.setAlgorithm(algorithm);
		return signature;
	}

	public static Signature sign(WaveType dataToSign, PublicKeyAlgorithm algorithm,
			CipherParameters privateKey) throws IOException {
		if (WaveType.class.isInstance(dataToSign) == false)
			throw new IllegalArgumentException("Argument \"dataToSign\" is not implementing WaveType");

		return sign(WaveUtils.getBytesFromWaveType(dataToSign), algorithm, privateKey);
	}

	/**
	 * Sign with ecdsa (P-256, P-224)
	 * 
	 * @param unsignedCertificate
	 * @param ecdsaNistp256WithSha256
	 * @param private1
	 * @return a ecdsa-signature wrapped in a signature
	 * @throws CryptoUtilsException
	 */
	public static Signature sign(byte[] messageToSign, PublicKeyAlgorithm algorithm,
			CipherParameters privateKey) {

		EcdsaSignature ecdsaSignature = signToECDSA(messageToSign, privateKey);
		Signature signature = new Signature();
		signature.setEcdsaSignature(ecdsaSignature);
		signature.setAlgorithm(algorithm);
		return signature;
	}

	public static Signature sign(Certificate cert, PublicKeyAlgorithm algorithm, CipherParameters privateKey)
			throws IOException {

		byte[] messageToSign = cert.getUnsignedBytes();

		EcdsaSignature ecdsaSignature = signToECDSA(messageToSign, privateKey);
		Signature signature = new Signature();
		signature.setEcdsaSignature(ecdsaSignature);
		signature.setAlgorithm(algorithm);
		return signature;
	}

	public static boolean verify(WaveType dataToVerify, Signature signature, CipherParameters publicKey)
			throws IOException, Exception {
		if (WaveType.class.isInstance(dataToVerify) == false)
			throw new IllegalArgumentException("Argument \"dataToVerify\" is not implementing WaveType");

		return verifyECDSA(WaveUtils.getBytesFromWaveType(dataToVerify), signature.getEcdsaSignature(),
				(ECPublicKeyParameters) publicKey);
	}

	public static boolean verify(byte[] messageToSign, Signature signature, CipherParameters publicKey) {

		return verifyECDSA(messageToSign, signature.getEcdsaSignature(), (ECPublicKeyParameters) publicKey);
	}

	public static boolean verifyWithoutHashing(byte[] messageHash, Signature signature,
			CipherParameters publicKey) {

		return verifyECDSAWithoutHashing(messageHash, signature.getEcdsaSignature(),
				(ECPublicKeyParameters) publicKey);
	}

	//
	public static ECPublicKeyParameters compressECPublicKeyParameters(CipherParameters pubKey) {
		ECPublicKeyParameters publicKey = (ECPublicKeyParameters) pubKey;

		return new ECPublicKeyParameters(new ECPoint.Fp(publicKey.getQ().getCurve(), publicKey.getQ().getX(),
				publicKey.getQ().getY(), true), publicKey.getParameters());
	}

	/**
	 * @param encoded
	 * @param private1
	 * @param pubKey
	 * @return
	 * @throws CryptoException
	 * @throws CryptoUtilsException
	 */
	public static ECIESCipher encryptECIES(Key symmetricKeyToEncrypt, ECPrivateKeyParameters ownPrivateKey,
			ECPublicKeyParameters publicKeyOfRecipient) throws CryptoException {

		ECIES engine = new ECIES();
		engine.init(ownPrivateKey, publicKeyOfRecipient);

		byte[] cipherBytes = engine.encrypt(symmetricKeyToEncrypt.getEncoded(), EciesEncryptedKey.TAG_LENGTH);
		return new ECIESCipher(cipherBytes, EciesEncryptedKey.TAG_LENGTH);
	}

	/**
	 * @param encoded
	 * @param private1
	 * @param pubKey
	 * @return
	 * @throws CryptoException
	 * @throws CryptoUtilsException
	 */
	public static Key decryptECIES(byte[] encryptedMessage, CipherParameters ownPrivateKey,
			CipherParameters publicKeyOfSender) throws CryptoException {

		ECIES engine = new ECIES();
		engine.init(ownPrivateKey, publicKeyOfSender);

		byte[] cipherBytes = engine.decrypt(encryptedMessage, EciesEncryptedKey.TAG_LENGTH);
		return new SecretKeySpec(cipherBytes, "AES");
	}

	public static Key decryptECIES(ECIESCipher eciesCipher, ECPrivateKeyParameters ownPrivateKey,
			ECPublicKeyParameters publicKeyOfSender) throws CryptoException {

		return decryptECIES(eciesCipher.getBytes(), ownPrivateKey, publicKeyOfSender);
	}

	public static Key decryptECIES(ECIESCipher eciesCipher, CipherParameters ownPrivateKey,
			CipherParameters publicKeyOfSender) throws CryptoException {

		return decryptECIES(eciesCipher.getBytes(), (ECPrivateKeyParameters) ownPrivateKey,
				(ECPublicKeyParameters) publicKeyOfSender);
	}

	/**
	 * @param encoded
	 * @param private1
	 * @param pubKey
	 * @return
	 * @throws CryptoException
	 * @throws CryptoUtilsException
	 */
	public static byte[] encryptAES(Key symmKey, Opaque nonce, byte[] messageToEncrypt)
			throws CryptoException {

		AESwithCCM aesEngine = new AESwithCCM();
		aesEngine.init(symmKey.getEncoded(), nonce.get());

		byte[] ccmCipher = aesEngine.encrypt(messageToEncrypt, EciesEncryptedKey.TAG_LENGTH);
		return ccmCipher;
	}

	/**
	 * @param encoded
	 * @param private1
	 * @param pubKey
	 * @return
	 * @throws CryptoException
	 * @throws CryptoUtilsException
	 */
	public static Opaque decryptAES(Key symmKey, byte[] aesCcmCiphertext, Opaque nonce)
			throws CryptoException {

		AESwithCCM aesEngine = new AESwithCCM();
		aesEngine.init(symmKey.getEncoded(), nonce.get());

		byte[] message = aesEngine.decrypt(aesCcmCiphertext, EciesEncryptedKey.TAG_LENGTH);
		return new Opaque(message);
	}

	/**
	 * @param encoded
	 * @param private1
	 * @param pubKey
	 * @return
	 * @throws CryptoUtilsException
	 * @throws CryptoException
	 */
	public static ECIESCipher encryptECIES(Key symmetricKeyToEncrypt, CipherParameters ownPrivateKey,
			CipherParameters publicKeyOfRecipient) throws CryptoException {

		return encryptECIES(symmetricKeyToEncrypt, (ECPrivateKeyParameters) ownPrivateKey,
				(ECPublicKeyParameters) publicKeyOfRecipient);
	}

	public static ECPrivateKeyParameters convertJCEECPrivateKeyToECECPrivateKeyParameters(
			JCEECPrivateKey privateKey) {

		ECParameterSpec spec = privateKey.getParameters();
		ECDomainParameters domain = new ECDomainParameters(spec.getCurve(), spec.getG(), spec.getN(),
				spec.getH(), spec.getSeed());
		return new ECPrivateKeyParameters(privateKey.getD(), domain);
	}

	public static JCEECPrivateKey convertECECPrivateKeyParametersToJCEECPrivateKey(
			ECPrivateKeyParameters privateKey) {
		ECDomainParameters domain = privateKey.getParameters();
		ECParameterSpec spec = new ECParameterSpec(domain.getCurve(), domain.getG(), domain.getN(),
				domain.getH(), domain.getSeed());
		ECPrivateKeySpec keySpec = new ECPrivateKeySpec(privateKey.getD(), spec);
		return new JCEECPrivateKey("EC", keySpec);
	}

	public static JCEECPublicKey convertECECPublicKeyParametersToJCEECPublicKey(
			ECPublicKeyParameters publicKey) {
		ECDomainParameters domain = publicKey.getParameters();
		ECParameterSpec spec = new ECParameterSpec(domain.getCurve(), domain.getG(), domain.getN(),
				domain.getH(), domain.getSeed());
		ECParameterSpec paramECParameterSpec = new ECParameterSpec(spec.getCurve(), spec.getG(), spec.getN(),
				spec.getH(), spec.getSeed());
		return new JCEECPublicKey("EC", publicKey, paramECParameterSpec);
	}

	public static ECPublicKeyParameters getVerificationKey(Certificate cert) {
		EccPoint verificationKey = null;
		for (SubjectAttribute b : cert.getSubjectAttributes()) {
			if (b.getType() == SubjectAttributeType.VERIFICATION_KEY) {
				verificationKey = b.getKey().getPublicKey();
				break;
			}
		}
		return verificationKey.toECPublicKeyParameters();
	}

	public static boolean verify(Certificate cert, ECPublicKeyParameters verificationKey) {

		try {
			return verify(cert.getUnsignedBytes(), cert.getSignature(), verificationKey);
		} catch (Exception e) {
			return false;
		}
	}
}
