package de.fraunhofer.sit.c2x.pki.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.junit.Before;
import org.junit.Test;

import de.fraunhofer.sit.c2x.CertChain;
import de.fraunhofer.sit.c2x.CryptoLib;
import de.fraunhofer.sit.c2x.pki.CertChainGenerator;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.crypto.CryptoUtils;
import de.fraunhofer.sit.c2x.pki.ca.crypto.KeyGenerator;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RegionTypeImpl.RegionType;

public class CertChainGeneratorTests {
	private static final DateFormat df = new SimpleDateFormat("dd.MM.yyyy");
	private List<byte[]> signingKeysX;
	private List<byte[]> signingKeysY;
	private List<byte[]> encryptionKeysX;
	private List<byte[]> encryptionKeysY;

	@Before
	public void createKeyPair() {
		signingKeysX = new ArrayList<>();
		signingKeysY = new ArrayList<>();
		encryptionKeysX = new ArrayList<>();
		encryptionKeysY = new ArrayList<>();

		for (int i = 0; i < 10; i++) {
			// generate signing key pair
			AsymmetricCipherKeyPair signingKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");

			byte[] signingKeyX = CryptoUtils.getPublicKeyX(signingKeyPair.getPublic(), 32);
			signingKeysX.add(signingKeyX);

			byte[] signingKeyY = CryptoUtils.getPublicKeyY(signingKeyPair.getPublic(), 32);
			signingKeysY.add(signingKeyY);

			// generate encryption key pair
			AsymmetricCipherKeyPair encryptionKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");

			byte[] encryptionKeyX = CryptoUtils.getPublicKeyX(encryptionKeyPair.getPublic(), 32);
			encryptionKeysX.add(encryptionKeyX);

			byte[] encryptionKeyY = CryptoUtils.getPublicKeyY(encryptionKeyPair.getPublic(), 32);
			encryptionKeysY.add(encryptionKeyY);
		}
	}

	private void verifyCertChain(CertChain certChain) throws Exception {
		assertFalse(certChain.getRootCert().isEmpty());

		// root
		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(2, root.getVersionAndType().get());
		assertEquals("ROOT_CA", root.getSubjectInfo().getSubjectType().toString());
		assertEquals("SELF", root.getIssuerType());
		assertEquals(root.getHashedId8().getCertIdHex(), root.getIssuer().getCertIdHex());
		assertNotNull(root.getSubjectAssurance().toInt());
		assertTrue(root.getSubjectAssurance().toInt() >= 0);
		assertNotNull(root.getItsAidListString());
		assertTrue(root.getItsAidListString().length() > 0);
		System.out.println("Unsigned Bytes: " + Hex.encodeHexString(root.getUnsignedBytes()));
		System.out.println("Certificate   : " + Hex.encodeHexString(root.getBytes()));
		System.out.println("Signature     : "
				+ Hex.encodeHexString(WaveUtils.getBytesFromWaveType(root.getSignature())));
		byte[] toBeVerifiedData = root.getUnsignedBytes();
		byte[] ts103097SignatureEncodedAsByteArray = WaveUtils.getBytesFromWaveType(root.getSignature());
		byte[] ecdsaNistp256PublicKeyX = root.getVerificationKey().getPublicKey().getX().get();
		byte[] ecdsaNistp256PublicKeyY = root.getVerificationKey().getPublicKey().getY().get();
		boolean verificationOK = CryptoLib.verifyWithEcdsaNistp256WithSha256(toBeVerifiedData,
				ts103097SignatureEncodedAsByteArray, ecdsaNistp256PublicKeyX, ecdsaNistp256PublicKeyY);
		assertTrue(verificationOK);

		// aa cert
		for (String aaCert : certChain.getAuthorizationAuthorityCerts()) {
			Certificate aa = WaveUtils.getElementFromBytes(Hex.decodeHex(aaCert.toCharArray()),
					Certificate.class);
			assertEquals(2, aa.getVersionAndType().get());
			assertEquals("AUTHORIZATION_AUTHORITY", aa.getSubjectInfo().getSubjectType().toString());
			assertEquals("CERTIFICATE_DIGEST_WITH_ECDSAP256", aa.getIssuerType());
			assertEquals(root.getHashedId8().getCertIdHex(), aa.getIssuer().getCertIdHex());
			assertNotNull(aa.getSubjectAssurance().toInt());
			assertTrue(aa.getSubjectAssurance().toInt() >= 0);
			assertNotNull(aa.getItsAidListString());
			assertTrue(aa.getItsAidListString().length() > 0);
			toBeVerifiedData = aa.getUnsignedBytes();
			ts103097SignatureEncodedAsByteArray = WaveUtils.getBytesFromWaveType(aa.getSignature());
			ecdsaNistp256PublicKeyX = root.getVerificationKey().getPublicKey().getX().get();
			ecdsaNistp256PublicKeyY = root.getVerificationKey().getPublicKey().getY().get();
			verificationOK = CryptoLib.verifyWithEcdsaNistp256WithSha256(toBeVerifiedData,
					ts103097SignatureEncodedAsByteArray, ecdsaNistp256PublicKeyX, ecdsaNistp256PublicKeyY);
			assertTrue(verificationOK);

			for (String atCert : certChain.getAuthorizationTicketCerts(aaCert)) {
				// pc cert
				Certificate pc = WaveUtils.getElementFromBytes(Hex.decodeHex(atCert.toCharArray()),
						Certificate.class);
				assertEquals(2, pc.getVersionAndType().get());
				assertEquals("AUTHORIZATION_TICKET", pc.getSubjectInfo().getSubjectType().toString());
				assertEquals("CERTIFICATE_DIGEST_WITH_ECDSAP256", pc.getIssuerType());
				assertEquals(aa.getHashedId8().getCertIdHex(), pc.getIssuer().getCertIdHex());
				assertNotNull(pc.getSubjectAssurance().toInt());
				assertTrue(pc.getSubjectAssurance().toInt() >= 0);
				assertNotNull(pc.getItsAidSspListString());
				assertTrue(pc.getItsAidSspListString().length() > 0);
				toBeVerifiedData = pc.getUnsignedBytes();
				ts103097SignatureEncodedAsByteArray = WaveUtils.getBytesFromWaveType(pc.getSignature());
				ecdsaNistp256PublicKeyX = aa.getVerificationKey().getPublicKey().getX().get();
				ecdsaNistp256PublicKeyY = aa.getVerificationKey().getPublicKey().getY().get();
				verificationOK = CryptoLib
						.verifyWithEcdsaNistp256WithSha256(toBeVerifiedData,
								ts103097SignatureEncodedAsByteArray, ecdsaNistp256PublicKeyX,
								ecdsaNistp256PublicKeyY);
				assertTrue(verificationOK);
			}
		}
	}

	@Test
	public void testMinimalCertChain() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChain(df.format(cal.getTime()), 0, aidList, sspList,
				signingKeysX, signingKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate pc = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain
						.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.get(0).toCharArray()), Certificate.class);
		assertNotNull(pc.getVerificationKey());
		assertNull(pc.getEncryptionKey());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.NONE, root.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testMinimalCertChain2() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChain(df.format(cal.getTime()), 0, aidList, sspList,
				signingKeysX, signingKeysY, null, null, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate pc = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain
						.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.get(0).toCharArray()), Certificate.class);
		assertNotNull(pc.getVerificationKey());
		assertNull(pc.getEncryptionKey());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.NONE, root.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWith10PseudonymsPerPCA() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChain(df.format(cal.getTime()), 0, aidList, sspList,
				signingKeysX, signingKeysY, 1, 10);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(10,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate pc = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain
						.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.get(0).toCharArray()), Certificate.class);
		assertNotNull(pc.getVerificationKey());
		assertNull(pc.getEncryptionKey());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWith5PCAsHavingEach10Pseudonyms() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChain(df.format(cal.getTime()), 0, aidList, sspList,
				signingKeysX, signingKeysY, 5, 10);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(5, certChain.getAuthorizationAuthorityCerts().size());
		for (String pcaCert : certChain.getAuthorizationAuthorityCerts()) {
			assertEquals(10, certChain.getAuthorizationTicketCerts(pcaCert).size());
		}

		Certificate pc = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain
						.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.get(0).toCharArray()), Certificate.class);
		assertNotNull(pc.getVerificationKey());
		assertNull(pc.getEncryptionKey());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithEncrpytionKey() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChain(df.format(cal.getTime()), 0, aidList, sspList,
				signingKeysX, signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate pc = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain
						.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.get(0).toCharArray()), Certificate.class);
		assertNotNull(pc.getVerificationKey());
		assertNotNull(pc.getEncryptionKey());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.NONE, root.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithCircularRegionRestrictionMinLatLongValues() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		double circularRegionRestrictionCenterLatitude = -90.0000000;
		double circularRegionRestrictionCenterLongitude = -180.00000000;
		int circularRegionRestrictionRadius = 1000;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithCircularRegionRestriction(
				df.format(cal.getTime()), 0, aidList, sspList, circularRegionRestrictionCenterLatitude,
				circularRegionRestrictionCenterLongitude, circularRegionRestrictionRadius, true,
				signingKeysX, signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.CIRCLE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.CIRCLE, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithCircularRegionRestrictionMaxLatLongValues() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		double circularRegionRestrictionCenterLatitude = 90.0000000;
		double circularRegionRestrictionCenterLongitude = 180.00000000;
		int circularRegionRestrictionRadius = 1000;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithCircularRegionRestriction(
				df.format(cal.getTime()), 0, aidList, sspList, circularRegionRestrictionCenterLatitude,
				circularRegionRestrictionCenterLongitude, circularRegionRestrictionRadius, true,
				signingKeysX, signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.CIRCLE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.CIRCLE, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithCircularRegionRestrictionMaxLatLongValuesNoRootRegionRestriction()
			throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		double circularRegionRestrictionCenterLatitude = 90.0000000;
		double circularRegionRestrictionCenterLongitude = 180.00000000;
		int circularRegionRestrictionRadius = 1000;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithCircularRegionRestriction(
				df.format(cal.getTime()), 0, aidList, sspList, circularRegionRestrictionCenterLatitude,
				circularRegionRestrictionCenterLongitude, circularRegionRestrictionRadius, false,
				signingKeysX, signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.NONE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.CIRCLE, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithIdentifiedRegionRestrictionISO() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		String dict = "iso_3166_1";
		long regionId = 276;
		long localId = 0;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithIdentifiedRegionRestriction("01.01.2018", 1,
				aidList, sspList, dict, regionId, localId, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.ID, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.ID, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithIdentifiedRegionRestrictionISONoRootRegionRestriction() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		String dict = "iso_3166_1";
		long regionId = 276;
		long localId = 0;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithIdentifiedRegionRestriction("01.01.2018", 1,
				aidList, sspList, dict, regionId, localId, false, signingKeysX, signingKeysY,
				encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.NONE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.ID, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithIdentifiedRegionRestrictionUN() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		String dict = "un_stats";
		long regionId = 150;
		long localId = 0;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithIdentifiedRegionRestriction("01.01.2018", 1,
				aidList, sspList, dict, regionId, localId, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.ID, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.ID, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithIdentifiedRegionRestrictionUNNoRootRegionRestriction() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		String dict = "un_stats";
		long regionId = 150;
		long localId = 0;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithIdentifiedRegionRestriction("01.01.2018", 1,
				aidList, sspList, dict, regionId, localId, false, signingKeysX, signingKeysY,
				encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.NONE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.ID, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithRectangularRegionRestriction() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] rectangularRegionRestrictionNorthWestPointsLatitude = new Double[] { 49.8716540 };
		Double[] rectangularRegionRestrictionNorthWestPointsLongitude = new Double[] { 8.638208 };
		Double[] rectangularRegionRestrictionSouthEastPointsLatitude = new Double[] { 49.862985 };
		Double[] rectangularRegionRestrictionSouthEastPointsLongitude = new Double[] { 8.667058 };

		CertChain certChain = CertChainGenerator.getCertChainWithRectangularRegionRestriction("01.01.2018",
				1, aidList, sspList, rectangularRegionRestrictionNorthWestPointsLatitude,
				rectangularRegionRestrictionNorthWestPointsLongitude,
				rectangularRegionRestrictionSouthEastPointsLatitude,
				rectangularRegionRestrictionSouthEastPointsLongitude, true, signingKeysX, signingKeysY,
				encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.RECTANGLE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.RECTANGLE, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithRectangularRegionRestrictionTwoRectangles() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] rectangularRegionRestrictionNorthWestPointsLatitude = new Double[] { 49.8716540, 49.905548 };
		Double[] rectangularRegionRestrictionNorthWestPointsLongitude = new Double[] { 8.638208, 8.597563 };
		Double[] rectangularRegionRestrictionSouthEastPointsLatitude = new Double[] { 49.862985, 49.885232 };
		Double[] rectangularRegionRestrictionSouthEastPointsLongitude = new Double[] { 8.667058, 8.631151 };

		CertChain certChain = CertChainGenerator.getCertChainWithRectangularRegionRestriction("01.01.2018",
				1, aidList, sspList, rectangularRegionRestrictionNorthWestPointsLatitude,
				rectangularRegionRestrictionNorthWestPointsLongitude,
				rectangularRegionRestrictionSouthEastPointsLatitude,
				rectangularRegionRestrictionSouthEastPointsLongitude, true, signingKeysX, signingKeysY,
				encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.RECTANGLE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.RECTANGLE, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithRectangularRegionRestrictionTwoRectanglesNoRootRegionRestriction()
			throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] rectangularRegionRestrictionNorthWestPointsLatitude = new Double[] { 49.8716540, 49.905548 };
		Double[] rectangularRegionRestrictionNorthWestPointsLongitude = new Double[] { 8.638208, 8.597563 };
		Double[] rectangularRegionRestrictionSouthEastPointsLatitude = new Double[] { 49.862985, 49.885232 };
		Double[] rectangularRegionRestrictionSouthEastPointsLongitude = new Double[] { 8.667058, 8.631151 };

		CertChain certChain = CertChainGenerator.getCertChainWithRectangularRegionRestriction("01.01.2018",
				1, aidList, sspList, rectangularRegionRestrictionNorthWestPointsLatitude,
				rectangularRegionRestrictionNorthWestPointsLongitude,
				rectangularRegionRestrictionSouthEastPointsLatitude,
				rectangularRegionRestrictionSouthEastPointsLongitude, false, signingKeysX, signingKeysY,
				encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.NONE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.RECTANGLE, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithPolygonalRegionRestrictionThreePoints() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] polygonalRegionRestrictionPointsLatitude = new Double[] { 49.8716540, 49.862985, 49.871546 };
		Double[] polygonalRegionRestrictionPointsLongitude = new Double[] { 8.638208, 8.667058, 8.667283 };

		CertChain certChain = CertChainGenerator.getCertChainWithPolygonalRegionRestriction("01.01.2018", 1,
				aidList, sspList, polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.POLYGON, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.POLYGON, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithPolygonalRegionRestrictionTwelvePoints() throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] polygonalRegionRestrictionPointsLatitude = new Double[] { 49.871654, 49.868555, 49.204040,
				49.866984, 49.864583, 49.862985, 49.864892, 49.869614, 49.871546, 49.876998, 49.883170,
				49.879561 };
		Double[] polygonalRegionRestrictionPointsLongitude = new Double[] { 8.638208, 8.641105, 8.647560,
				8.653751, 8.660072, 8.667058, 8.673761, 8.676889, 8.667283, 8.670312, 8.655515, 8.642972 };

		CertChain certChain = CertChainGenerator.getCertChainWithPolygonalRegionRestriction("01.01.2018", 1,
				aidList, sspList, polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.POLYGON, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.POLYGON, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test
	public void testCertChainWithPolygonalRegionRestrictionTwelvePointsNoRootRegionRestriction()
			throws Exception {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] polygonalRegionRestrictionPointsLatitude = new Double[] { 49.871654, 49.868555, 49.204040,
				49.866984, 49.864583, 49.862985, 49.864892, 49.869614, 49.871546, 49.876998, 49.883170,
				49.879561 };
		Double[] polygonalRegionRestrictionPointsLongitude = new Double[] { 8.638208, 8.641105, 8.647560,
				8.653751, 8.660072, 8.667058, 8.673761, 8.676889, 8.667283, 8.670312, 8.655515, 8.642972 };

		CertChain certChain = CertChainGenerator.getCertChainWithPolygonalRegionRestriction("01.01.2018", 1,
				aidList, sspList, polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude, false, signingKeysX, signingKeysY,
				encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());

		Certificate root = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getRootCert().toCharArray()), Certificate.class);
		assertEquals(RegionType.NONE, root.getRegionType());
		Certificate pca = WaveUtils.getElementFromBytes(
				Hex.decodeHex(certChain.getAuthorizationAuthorityCerts().get(0).toCharArray()),
				Certificate.class);
		assertEquals(RegionType.POLYGON, pca.getRegionType());

		verifyCertChain(certChain);
	}

	@Test(expected = HandlerException.class)
	public void testCertChainWithExpiredEndDate() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set yesterday's date
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, -1);

		CertChainGenerator.getCertChain(df.format(cal.getTime()), 0, aidList, sspList, signingKeysX,
				signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);
	}

	@Test(expected = HandlerException.class)
	public void testCertChainWithoutPermissions() throws HandlerException {
		long[] aidList = new long[] {};
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] {};

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChainGenerator.getCertChain(df.format(cal.getTime()), 0, aidList, sspList, signingKeysX,
				signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);
	}

	@Test(expected = HandlerException.class)
	public void testCertChainWithInvalidNegativeAssuranceLevel() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChainGenerator.getCertChain(df.format(cal.getTime()), -1, aidList, sspList, signingKeysX,
				signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);
	}

	@Test(expected = HandlerException.class)
	public void testCertChainWithInvalidPositiveAssuranceLevel() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChainGenerator.getCertChain(df.format(cal.getTime()), 8, aidList, sspList, signingKeysX,
				signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidMinCircularRegionRestriction() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		double circularRegionRestrictionCenterLatitude = -90.0000001;
		double circularRegionRestrictionCenterLongitude = -180.00000001;
		int circularRegionRestrictionRadius = 1000;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithCircularRegionRestriction(
				df.format(cal.getTime()), 0, aidList, sspList, circularRegionRestrictionCenterLatitude,
				circularRegionRestrictionCenterLongitude, circularRegionRestrictionRadius, true,
				signingKeysX, signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidNeutralCircularRegionRestriction() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		double circularRegionRestrictionCenterLatitude = 90.0000001;
		double circularRegionRestrictionCenterLongitude = 180.0000001;
		int circularRegionRestrictionRadius = 1000;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithCircularRegionRestriction(
				df.format(cal.getTime()), 0, aidList, sspList, circularRegionRestrictionCenterLatitude,
				circularRegionRestrictionCenterLongitude, circularRegionRestrictionRadius, true,
				signingKeysX, signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidMaxCircularRegionRestriction() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		double circularRegionRestrictionCenterLatitude = 90.0000002;
		double circularRegionRestrictionCenterLongitude = 180.0000002;
		int circularRegionRestrictionRadius = 1000;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithCircularRegionRestriction(
				df.format(cal.getTime()), 0, aidList, sspList, circularRegionRestrictionCenterLatitude,
				circularRegionRestrictionCenterLongitude, circularRegionRestrictionRadius, true,
				signingKeysX, signingKeysY, encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidIdentifiedRegionRestriction() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		String dict = "0";
		long regionId = 276;
		long localId = 0;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithIdentifiedRegionRestriction("01.01.2018", 1,
				aidList, sspList, dict, regionId, localId, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidIdentifiedRegionRestrictionISO() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		String dict = "iso_3166_1";
		long regionId = 150;
		long localId = 0;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithIdentifiedRegionRestriction("01.01.2018", 1,
				aidList, sspList, dict, regionId, localId, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidIdentifiedRegionRestrictionUN() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		String dict = "un_stats";
		long regionId = 999;
		long localId = 0;

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		CertChain certChain = CertChainGenerator.getCertChainWithIdentifiedRegionRestriction("01.01.2018", 1,
				aidList, sspList, dict, regionId, localId, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = HandlerException.class)
	public void testCertChainWithInvalidRectangularRegionRestrictionSamePoints() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] rectangularRegionRestrictionNorthWestPointsLatitude = new Double[] { 49.8716540 };
		Double[] rectangularRegionRestrictionNorthWestPointsLongitude = new Double[] { 8.638208 };
		Double[] rectangularRegionRestrictionSouthEastPointsLatitude = new Double[] { 49.8716540 };
		Double[] rectangularRegionRestrictionSouthEastPointsLongitude = new Double[] { 8.638208 };

		CertChain certChain = CertChainGenerator.getCertChainWithRectangularRegionRestriction("01.01.2018",
				1, aidList, sspList, rectangularRegionRestrictionNorthWestPointsLatitude,
				rectangularRegionRestrictionNorthWestPointsLongitude,
				rectangularRegionRestrictionSouthEastPointsLatitude,
				rectangularRegionRestrictionSouthEastPointsLongitude, true, signingKeysX, signingKeysY,
				encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = HandlerException.class)
	public void testCertChainWithInvalidRectangularRegionRestrictionDifferentZone() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] rectangularRegionRestrictionNorthWestPointsLongitude = new Double[] { 49.8716540 };
		Double[] rectangularRegionRestrictionNorthWestPointsLatitude = new Double[] { 8.638208 };
		Double[] rectangularRegionRestrictionSouthEastPointsLongitude = new Double[] { 49.204040 };
		Double[] rectangularRegionRestrictionSouthEastPointsLatitude = new Double[] { 12.043586 };

		CertChain certChain = CertChainGenerator.getCertChainWithRectangularRegionRestriction("01.01.2018",
				1, aidList, sspList, rectangularRegionRestrictionNorthWestPointsLatitude,
				rectangularRegionRestrictionNorthWestPointsLongitude,
				rectangularRegionRestrictionSouthEastPointsLatitude,
				rectangularRegionRestrictionSouthEastPointsLongitude, true, signingKeysX, signingKeysY,
				encryptionKeysX, encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidPolygonalRegionRestrictionOnePoint() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] polygonalRegionRestrictionPointsLatitude = new Double[] { 49.8716540 };
		Double[] polygonalRegionRestrictionPointsLongitude = new Double[] { 8.638208 };

		CertChain certChain = CertChainGenerator.getCertChainWithPolygonalRegionRestriction("01.01.2018", 1,
				aidList, sspList, polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidPolygonalRegionRestrictionTwoPoints() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] polygonalRegionRestrictionPointsLatitude = new Double[] { 49.8716540, 49.862985 };
		Double[] polygonalRegionRestrictionPointsLongitude = new Double[] { 8.638208, 8.667058 };

		CertChain certChain = CertChainGenerator.getCertChainWithPolygonalRegionRestriction("01.01.2018", 1,
				aidList, sspList, polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidPolygonalRegionRestrictionThreePointsDiffernetZone()
			throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] polygonalRegionRestrictionPointsLatitude = new Double[] { 49.8716540, 49.862985, 49.204040 };
		Double[] polygonalRegionRestrictionPointsLongitude = new Double[] { 8.638208, 8.667058, 12.043586 };

		CertChain certChain = CertChainGenerator.getCertChainWithPolygonalRegionRestriction("01.01.2018", 1,
				aidList, sspList, polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = HandlerException.class)
	public void testCertChainWithInvalidPolygonalRegionRestrictionFourPointsIntersecting()
			throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] polygonalRegionRestrictionPointsLatitude = new Double[] { 49.8716540, 49.862985, 49.871546,
				49.864896 };
		Double[] polygonalRegionRestrictionPointsLongitude = new Double[] { 8.638208, 8.667058, 8.667283,
				8.647560 };

		CertChain certChain = CertChainGenerator.getCertChainWithPolygonalRegionRestriction("01.01.2018", 1,
				aidList, sspList, polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCertChainWithInvalidPolygonalRegionRestrictionThirteenPoints() throws HandlerException {
		long[] aidList = new long[] { 16512l };
		byte[][] sspList = new byte[1][];
		sspList[0] = new byte[] { 0x00 };

		// set expiration date = today + 365 days
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +365);

		Double[] polygonalRegionRestrictionPointsLatitude = new Double[] { 49.871654, 49.868555, 49.204040,
				49.866984, 49.864583, 49.862985, 49.864892, 49.869614, 49.871546, 49.876998, 49.883170,
				49.879561, 49.876005 };
		Double[] polygonalRegionRestrictionPointsLongitude = new Double[] { 8.638208, 8.641105, 8.647560,
				8.653751, 8.660072, 8.667058, 8.673761, 8.676889, 8.667283, 8.670312, 8.655515, 8.642972,
				8.639486 };
		CertChain certChain = CertChainGenerator.getCertChainWithPolygonalRegionRestriction("01.01.2018", 1,
				aidList, sspList, polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude, true, signingKeysX, signingKeysY, encryptionKeysX,
				encryptionKeysY, 1, 1);

		assertFalse(certChain.getRootCert().isEmpty());
		assertEquals(1, certChain.getAuthorizationAuthorityCerts().size());
		assertEquals(1,
				certChain.getAuthorizationTicketCerts(certChain.getAuthorizationAuthorityCerts().get(0))
						.size());
	}
}
