#include "security_cache.hh"

#include "params.hh"

#include "converter.hh"

using namespace std; // Required for isnan()
#include "etsi_ts103097_certificate_codec.hh"

#include "loggers.hh"

security_cache::security_cache(): _certificates(), _hashed_id8s() {
  loggers::get_instance().log(">>> security_cache::security_cache");
} // End of ctor

security_cache::~security_cache() {
  loggers::get_instance().log(">>> security_cache::~security_cache");
  clear();
} // End of dtor

int security_cache::clear() {
  loggers::get_instance().log(">>> security_cache::clear");
  _certificates.clear(); // Smart pointers will do the job
  _hashed_id8s.clear();
  
  return 0;
} // End of clear method

int security_cache::get_certificate_id(const OCTETSTRING& p_hashed_id8, std::string& p_certifcate_id) const {
  loggers::get_instance().log_msg(">>> security_cache::get_certificate_id: ", p_hashed_id8);

  std::vector<unsigned char> key(static_cast<const unsigned char*>(p_hashed_id8), p_hashed_id8.lengthof() + static_cast<const unsigned char*>(p_hashed_id8));
  std::map<std::vector<unsigned char>, std::string>::const_iterator it = _hashed_id8s.find(key);
  if (it == _hashed_id8s.cend()) {
    p_certifcate_id = "";
    return -1;
  }
  p_certifcate_id = it->second;

  return 0;
}

int security_cache::get_certificate(const std::string& p_certificate_id, OCTETSTRING& p_certificate) const {
  loggers::get_instance().log(">>> security_cache::get_certificate: '%s'", p_certificate_id.c_str());

  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("security_cache::get_certificate: record not found");
    dump();
    return -1;
  }
  const std::vector<unsigned char> cert = it->second.get()->certificate();
  p_certificate = OCTETSTRING(cert.size(), cert.data());
  
  return 0;
}

int security_cache::get_certificate(const std::string& p_certificate_id, IEEE1609dot2::CertificateBase& p_certificate) const {
  loggers::get_instance().log(">>> security_cache::get_certificate: '%s'", p_certificate_id.c_str());

  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("security_cache::get_certificate: record not found");
    return -1;
  }
  p_certificate = it->second.get()->decoded_certificate();
  
  return 0;
}

int security_cache::get_issuer(const std::string& p_certificate_id, OCTETSTRING& p_hashed_id_issuer) const {
  loggers::get_instance().log(">>> security_cache::get_issuer: '%s'", p_certificate_id.c_str());

  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("security_cache::get_issuer: record not found");
    return -1;
  }
  const std::vector<unsigned char> issuer = it->second.get()->issuer();
  p_hashed_id_issuer = OCTETSTRING(issuer.size(), issuer.data());
  
  return 0;
}

int security_cache::get_hashed_id(const std::string& p_certificate_id, OCTETSTRING& p_hashed_id) const {
  loggers::get_instance().log(">>> security_cache::get_hashed_id: '%s'", p_certificate_id.c_str());

  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("security_cache::get_hashed_id: record not found");
    return -1;
  }
  const std::vector<unsigned char> hashed_id = it->second.get()->hashed_id();
  p_hashed_id = OCTETSTRING(hashed_id.size(), hashed_id.data());
  
  return 0;
}

int security_cache::get_private_key(const std::string& p_certificate_id, OCTETSTRING& p_private_key) const {
  loggers::get_instance().log(">>> security_cache::get_private_key: '%s'", p_certificate_id.c_str());

  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("security_cache::get_private_key: record not found");
    return -1;
  }
  const std::vector<unsigned char> private_key = it->second.get()->private_key();
  p_private_key = OCTETSTRING(private_key.size(), private_key.data());
  
  return 0;
}

int security_cache::get_public_keys(const std::string& p_certificate_id, OCTETSTRING& p_public_key_x, OCTETSTRING& p_public_key_y) const {
  loggers::get_instance().log(">>> security_cache::get_public_keys: '%s'", p_certificate_id.c_str());

  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("security_cache::get_public_keys: record not found");
    return -1;
  }
  const std::vector<unsigned char> public_key_x = it->second.get()->public_key_x();
  p_public_key_x = OCTETSTRING(public_key_x.size(), public_key_x.data());
  const std::vector<unsigned char> public_key_y = it->second.get()->public_key_y();
  p_public_key_y = OCTETSTRING(public_key_y.size(), public_key_y.data());
  
  return 0;
}

int security_cache::get_private_enc_key(const std::string& p_certificate_id, OCTETSTRING& p_private_enc_key) const {
  loggers::get_instance().log(">>> security_cache::get_private_enc_key: '%s'", p_certificate_id.c_str());

  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("security_cache::get_private_enc_key: record not found");
    return -1;
  }
  const std::vector<unsigned char> private_enc_key = it->second.get()->private_enc_key();
  p_private_enc_key = OCTETSTRING(private_enc_key.size(), private_enc_key.data());

  return 0;
}

int security_cache::get_public_enc_keys(const std::string& p_certificate_id, OCTETSTRING& p_public_enc_key_x, OCTETSTRING& p_public_enc_key_y) const {
  loggers::get_instance().log(">>> security_cache::get_public_enc_keys: '%s'", p_certificate_id.c_str());

  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("security_cache::get_public_enc_keys: record not found");
    return -1;
  }
  const std::vector<unsigned char> public_enc_key_x = it->second.get()->public_enc_key_x();
  p_public_enc_key_x = OCTETSTRING(public_enc_key_x.size(), public_enc_key_x.data());
  const std::vector<unsigned char> public_enc_key_y = it->second.get()->public_enc_key_y();
  p_public_enc_key_y = OCTETSTRING(public_enc_key_y.size(), public_enc_key_y.data());

  return 0;
}

int security_cache::store_certificate(const CHARSTRING& p_cert_id, const OCTETSTRING& p_cert, const OCTETSTRING& p_private_key, const OCTETSTRING& p_public_key_x, const OCTETSTRING& p_public_key_y, const OCTETSTRING& p_hashed_id8, const OCTETSTRING& p_issuer, const OCTETSTRING& p_private_enc_key, const OCTETSTRING& p_public_enc_key_x, const OCTETSTRING& p_public_enc_key_y) {
  loggers::get_instance().log_msg(">>> security_cache::store_certificate: ", p_cert_id);

  std::string key(static_cast<const char*>(p_cert_id));
  std::vector<unsigned char> cert(static_cast<const unsigned char*>(p_cert), static_cast<const unsigned char*>(p_cert) + p_cert.lengthof());
  std::vector<unsigned char> private_key(static_cast<const unsigned char*>(p_private_key), static_cast<const unsigned char*>(p_private_key) + p_private_key.lengthof());
  std::vector<unsigned char> public_key_x(static_cast<const unsigned char*>(p_public_key_x), static_cast<const unsigned char*>(p_public_key_x) + p_public_key_x.lengthof());
  std::vector<unsigned char> public_key_y(static_cast<const unsigned char*>(p_public_key_y), static_cast<const unsigned char*>(p_public_key_y) + p_public_key_y.lengthof());
  std::vector<unsigned char> hashed_id8(static_cast<const unsigned char*>(p_hashed_id8), static_cast<const unsigned char*>(p_hashed_id8) + p_hashed_id8.lengthof());
  std::vector<unsigned char> issuer(static_cast<const unsigned char*>(p_issuer), static_cast<const unsigned char*>(p_issuer) + p_issuer.lengthof());


  //, const OCTETSTRING& p_private_enc_key, const OCTETSTRING& p_public_enc_key_x, const OCTETSTRING& p_public_enc_key_y
  std::vector<unsigned char> private_enc_key;
  std::vector<unsigned char> public_enc_key_x;
  std::vector<unsigned char> public_enc_key_y;
  if (p_private_enc_key.lengthof() != 0) {
    private_enc_key.assign(static_cast<const unsigned char*>(p_private_enc_key), static_cast<const unsigned char*>(p_private_enc_key) + p_private_enc_key.lengthof());
    public_enc_key_x.assign(static_cast<const unsigned char*>(p_public_enc_key_x), static_cast<const unsigned char*>(p_public_enc_key_x) + p_public_enc_key_x.lengthof());
    public_enc_key_y.assign(static_cast<const unsigned char*>(p_public_enc_key_y), static_cast<const unsigned char*>(p_public_enc_key_y) + p_public_enc_key_y.lengthof());
  } // else, no encryption keys
  
  IEEE1609dot2::CertificateBase decoded_certificate;
  etsi_ts103097_certificate_codec codec;
  codec.decode(p_cert, decoded_certificate);
  loggers::get_instance().log_msg("security_cache::store_certificate: Decoded certificate: ", decoded_certificate);
  security_db_record* p = new security_db_record(
                                                 key,              // Certificate id
                                                 cert,             // Certificate
                                                 decoded_certificate,
                                                 issuer,           // Hashed ID for the issuer
                                                 hashed_id8,       // Hashed ID
                                                 private_key,      // Private key
                                                 public_key_x,     // Public key X-coordinate
                                                 public_key_y,     // Public key Y-coordinate
                                                 private_enc_key,  // Private encryption key
                                                 public_enc_key_x, // Public encryption key X-coordinate
                                                 public_enc_key_y  // Public encryption key Y-coordinate
                                                 );
  if (p == nullptr) {
    return -1;
  }
  std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator i = _certificates.find(key);
  if (i != _certificates.cend()) {
    _certificates.erase(i);
  }
  std::map<std::vector<unsigned char>, std::string>::const_iterator h = _hashed_id8s.find(hashed_id8);
  if (h != _hashed_id8s.cend()) {
    _hashed_id8s.erase(h);
  }
  _certificates.insert(std::pair<std::string, std::unique_ptr<security_db_record> >(key, std::unique_ptr<security_db_record>(p)));
  _hashed_id8s.insert(std::pair<std::vector<unsigned char>, std::string&>(hashed_id8, key));
  dump();

  return 0;
}

void security_cache::dump() const {
  loggers::get_instance().log("security_cache::dump_certificates: # items = %d", _certificates.size());
  for (std::map<std::string, std::unique_ptr<security_db_record> >::const_iterator it = _certificates.cbegin(); it != _certificates.cend(); ++it) {
    security_db_record* p = it->second.get();
    loggers::get_instance().log("security_cache::dump: certificate_id = %s", p->certificate_id().c_str());
    loggers::get_instance().log_to_hexa("security_cache::dump: certificate = ", p->certificate().data(), p->certificate().size());
    loggers::get_instance().log_msg("security_cache::dump: decoded certificate = ", p->decoded_certificate());
    loggers::get_instance().log_to_hexa("security_cache::dump: hashed_id = ", p->hashed_id().data(), p->hashed_id().size());
    loggers::get_instance().log_to_hexa("security_cache::dump: private_key = ", p->private_key().data(), p->private_key().size());
    loggers::get_instance().log_to_hexa("security_cache::dump: public_key_x = ", p->public_key_x().data(), p->public_key_x().size());
    loggers::get_instance().log_to_hexa("security_cache::dump: public_key_y = ", p->public_key_y().data(), p->public_key_y().size());
    loggers::get_instance().log_to_hexa("security_cache::dump: private_enc_key = ", p->private_enc_key().data(), p->private_enc_key().size());
    loggers::get_instance().log_to_hexa("security_cache::dump: public_enc_key_x = ", p->public_enc_key_x().data(), p->public_enc_key_x().size());
    loggers::get_instance().log_to_hexa("security_cache::dump: public_enc_key_y = ", p->public_enc_key_y().data(), p->public_enc_key_y().size());
  } // End of 'for' statement
  for (std::map<std::vector<unsigned char>, std::string>::const_iterator it = _hashed_id8s.cbegin(); it != _hashed_id8s.cend(); ++it) {
    loggers::get_instance().log("security_cache::dump: Hashedid8 idx = %s", it->second.c_str());
  } // End of 'for' statement
} // End of method dump
