/**
 * 
 */
package de.fraunhofer.sit.c2x.pki.ca.validator.region;

import javax.inject.Inject;

import org.apache.log4j.Logger;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.interfaces.PolygonalRegion;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.DoubleVector2d;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.LineSegment;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.Polygon;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.utils.LinearMath;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.utils.PolygonPoint;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.utils.PolygonUtils;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class PolygonPolygonValidator extends AbstractPolygonValidator<PolygonalRegion> {

	@Inject
	public PolygonPolygonValidator() {
	}

	@InjectLogger
	private Logger logger;

	/**
	 * @param polygonalRegion
	 */
	public PolygonPolygonValidator(PolygonalRegion polygonalRegion) {
		super(polygonalRegion);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.preserve.pki.ca.validator.Validator#validate(java.lang.Object)
	 */
	@Override
	public boolean validate(PolygonalRegion regionToValidate) {

		if (regionToValidate == null)
			throw new IllegalArgumentException("argument may not be null");

		Polygon polyToValidate = convertToPolygon(regionToValidate);

		// check that all points of the requested polygon is inside the polygon
		// of the issuer
		for (PolygonPoint p : polyToValidate.getLinkedPoints()) {
			if (!PolygonUtils.jordanTest(p, caPolygon, true)) {
				if (logger.isWarnEnabled())
					logger.warn("Point of requested polygon is outside of isser polygon");
				return false;
			}
		}

		for (LineSegment caSeg : caPolygon.getEdges()) {
			for (LineSegment polySegment : polyToValidate.getEdges()) {
				DoubleVector2d intersectionPoint = LinearMath.intersection(caSeg, polySegment);
				boolean invalid = false;
				if (intersectionPoint != null) {
					// line of polygon has no intersection with another line
					boolean startPointEqualToIntersection = (polySegment.getStart().getX()
							.equals(intersectionPoint.getX()) && polySegment.getStart().getY()
							.equals(intersectionPoint.getY()));
					if (!startPointEqualToIntersection) {
						// start of polygon line is not equal to found
						// intersection point
						boolean endPointEqualToIntersection = (polySegment.getEnd().getX()
								.equals(intersectionPoint.getX()) && polySegment.getEnd().getY()
								.equals(intersectionPoint.getY()));
						if (!endPointEqualToIntersection) {
							// end of polygon line is not equal to found
							// intersection point
							invalid = true;
							if (logger.isWarnEnabled()){
								logger.warn("Intersection in polygon found: " + intersectionPoint);
							}

						} else {
							if (logger.isTraceEnabled())
								logger.trace("Found intersection in polygon is end point: "
										+ polySegment.getEnd());
						}
					} else {
						if (logger.isTraceEnabled())
							logger.trace("Found intersection in polygon is start point: "
									+ polySegment.getStart());
					}
				}

				if (invalid) {
					return false;
				}
			}
		}

		return true;
	}
}
