package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeMap;
import java.util.Vector;

import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;

import com.google.inject.Inject;
import com.google.inject.name.Named;
import com.mysql.jdbc.Statement;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.measuring.ItsStationType;
import de.fraunhofer.sit.c2x.pki.ca.measuring.MeasuringKey;
import de.fraunhofer.sit.c2x.pki.ca.measuring.MeasuringStatistics;
import de.fraunhofer.sit.c2x.pki.ca.measuring.MeasuringStatisticsEntry;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Measuring;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.MeasuringProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.MeasuringWriter;

public class JDBCMeasuringProvider extends AbstractMysqlConnection implements MeasuringProvider {

	@Override
	public String getName() {
		return "Measuring provider";
	}

	@InjectLogger
	private Logger logger;

	@Inject
	@Named(value = "caType")
	private ItsStationType caType;
	@Inject
	private MeasuringWriter writerThread;

	public JDBCMeasuringProvider() {
	}

	@Override
	public boolean add(byte[] stationDigest, byte[] requestHash, MeasuringKey key, long value,
			int numberOfOperations) {
		Measuring measuring = new Measuring();
		measuring.setStationType(caType.toString());
		measuring.setStationDigest(stationDigest);

		if (requestHash.length > 10) {
			byte[] tmp = new byte[10];
			System.arraycopy(requestHash, 0, tmp, 0, tmp.length);
			requestHash = tmp;
		}

		measuring.setRequestHash(requestHash);
		measuring.setKey(key);
		measuring.setValue(new Long(value).intValue());
		measuring.setNumberOperations(numberOfOperations);

		boolean result = false;
//		try {
			result = writerThread.writeMeasuring(measuring);
			//result = save(measuring);
//		} catch (ProviderException e1) {
//			logger.warn("Time measurement cannot be stored in DB");
//		}

		return result;
	}

	@Override
	public boolean save(Measuring measuring) throws ProviderException {
		String sql = String.format("INSERT INTO %s (%s, %s, %s, %s, %s, %s,%s) VALUES (?,?,?,?,?,?,?);",
				Constants.MEASURING__TABLE, Constants.MEASURING__COL__TIME,
				Constants.MEASURING__COL__STATION_TYPE, Constants.MEASURING__COL__STATION_DIGEST,
				Constants.MEASURING__COL__REQUEST_HASH, Constants.MEASURING__COL__KEY,
				Constants.MEASURING__COL__VALUE, Constants.MEASURING__COL__NUMBER);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setTimestamp(1, measuring.getCreationTime());
			st.setString(2, measuring.getStationType());
			st.setBytes(3, measuring.getStationDigest());
			st.setBytes(4, measuring.getRequestHash());
			st.setString(5, measuring.getKey().toString());
			st.setInt(6, measuring.getValue());
			st.setInt(7, measuring.getNumberOperations());
			st.execute();
			
			return true;
		} catch (SQLException e) {
			logger.error("Measuring not stored: " + e.getMessage());
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public Measuring get(Timestamp time, byte[] stationDigest, MeasuringKey key) throws ProviderException {
		String sql = String.format("SELECT * FROM %s WHERE %s = ? AND %s = ? AND %s = ?;",
				Constants.MEASURING__TABLE, Constants.MEASURING__COL__TIME,
				Constants.MEASURING__COL__STATION_DIGEST, Constants.MEASURING__COL__KEY);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setTimestamp(1, time);
			st.setBytes(2, stationDigest);
			st.setString(3, key.toString());
			st.execute();
			result = st.getResultSet();

			if (!result.first()) {
				logger.warn("Entry not found");
			} else {
				Measuring info = new Measuring();
				info.setCreationTime(result.getTimestamp(Constants.MEASURING__COL__TIME));
				info.setRequestHash(result.getBytes(Constants.MEASURING__COL__REQUEST_HASH));
				info.setStationDigest(result.getBytes(Constants.MEASURING__COL__STATION_DIGEST));
				info.setStationType(result.getString(Constants.MEASURING__COL__STATION_TYPE));
				info.setKey(result.getString(Constants.MEASURING__COL__KEY));
				info.setValue(result.getInt(Constants.MEASURING__COL__VALUE));
				info.setNumberOperations(result.getInt(Constants.MEASURING__COL__NUMBER));
				return info;
			}
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
		return null;
	}

	@Override
	public Measuring[] getAll(int offset, int limit) throws ProviderException {
		String sql = String.format("SELECT * FROM %s ORDER BY %s DESC LIMIT %d, %d",
				Constants.MEASURING__TABLE, Constants.MEASURING__COL__TIME, offset, limit);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<Measuring> tmp = new ArrayList<>();
			while (result.next()) {

				Measuring info = new Measuring();
				info.setId(result.getLong(Constants.MEASURING__COL__ID));
				info.setCreationTime(result.getTimestamp(Constants.MEASURING__COL__TIME));
				info.setRequestHash(result.getBytes(Constants.MEASURING__COL__REQUEST_HASH));
				info.setStationDigest(result.getBytes(Constants.MEASURING__COL__STATION_DIGEST));
				info.setStationType(result.getString(Constants.MEASURING__COL__STATION_TYPE));
				info.setKey(result.getString(Constants.MEASURING__COL__KEY));
				info.setValue(result.getInt(Constants.MEASURING__COL__VALUE));
				info.setNumberOperations(result.getInt(Constants.MEASURING__COL__NUMBER));

				tmp.add(info);
			}
			return tmp.toArray(new Measuring[tmp.size()]);
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public Measuring[] getRange(Timestamp startTime, Timestamp endTime) throws ProviderException {
		String sql = String.format("SELECT * FROM %s WHERE %s >= ? AND %s <= ? ORDER BY %s DESC",
				Constants.MEASURING__TABLE, Constants.MEASURING__COL__TIME, startTime,
				Constants.MEASURING__COL__TIME, endTime, Constants.MEASURING__COL__TIME);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<Measuring> tmp = new ArrayList<>();
			while (result.next()) {

				Measuring info = new Measuring();
				info.setId(result.getLong(Constants.MEASURING__COL__ID));
				info.setCreationTime(result.getTimestamp(Constants.MEASURING__COL__TIME));
				info.setRequestHash(result.getBytes(Constants.MEASURING__COL__REQUEST_HASH));
				info.setStationDigest(result.getBytes(Constants.MEASURING__COL__STATION_DIGEST));
				info.setStationType(result.getString(Constants.MEASURING__COL__STATION_TYPE));
				info.setKey(result.getString(Constants.MEASURING__COL__KEY));
				info.setValue(result.getInt(Constants.MEASURING__COL__VALUE));
				info.setNumberOperations(result.getInt(Constants.MEASURING__COL__NUMBER));

				tmp.add(info);
			}
			return tmp.toArray(new Measuring[tmp.size()]);
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public Measuring[] getKeys(MeasuringKey key) throws ProviderException {
		String sql = String.format("SELECT * FROM %s WHERE %s = %s ORDER BY %s DESC",
				Constants.MEASURING__TABLE, Constants.MEASURING__COL__KEY, key,
				Constants.MEASURING__COL__TIME);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<Measuring> tmp = new ArrayList<>();
			while (result.next()) {

				Measuring info = new Measuring();
				info.setId(result.getLong(Constants.MEASURING__COL__ID));
				info.setCreationTime(result.getTimestamp(Constants.MEASURING__COL__TIME));
				info.setRequestHash(result.getBytes(Constants.MEASURING__COL__REQUEST_HASH));
				info.setStationDigest(result.getBytes(Constants.MEASURING__COL__STATION_DIGEST));
				info.setStationType(result.getString(Constants.MEASURING__COL__STATION_TYPE));
				info.setKey(result.getString(Constants.MEASURING__COL__KEY));
				info.setValue(result.getInt(Constants.MEASURING__COL__VALUE));
				info.setNumberOperations(result.getInt(Constants.MEASURING__COL__NUMBER));

				tmp.add(info);
			}
			return tmp.toArray(new Measuring[tmp.size()]);
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public int removeMeasuring(Timestamp time, byte[] stationDigest, MeasuringKey key)
			throws ProviderException {
		String sql = String.format("DELETE FROM %s WHERE %s = ? AND %s = ? AND %s = ?;",
				Constants.MEASURING__TABLE, Constants.MEASURING__COL__TIME,
				Constants.MEASURING__COL__STATION_DIGEST, Constants.MEASURING__COL__KEY);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
			st.setTimestamp(1, time);
			st.setBytes(2, stationDigest);
			st.setString(3, key.toString());
			return st.executeUpdate();
		} catch (SQLException e) {
			if (logger.isDebugEnabled())
				logger.debug(
						"Cannot delete Measuring: " + time + ", digest=" + Hex.encodeHexString(stationDigest)
								+ ", key=" + key, e);
			throw new ProviderException("Cannot delete Measuring: " + time + ", digest="
					+ Hex.encodeHexString(stationDigest) + ", key=" + key, e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public MeasuringStatistics getStatistics(Timestamp start, Timestamp end) throws ProviderException {
		// String sql = String
		// .format("SELECT %s, %s, COUNT(*) AS %s, MIN(%s) AS %s, AVG(%s) AS %s, MAX(%s) AS %s, %s FROM %s WHERE %s >= \"%s\" AND %s <= \"%s\" GROUP BY %s, %s ORDER BY %s ASC",
		// Constants.MEASURING__COL__KEY, Constants.MEASURING__COL__NUMBER,
		// Constants.MEASURING__COL__NUMBER_PROCESSES,
		// Constants.MEASURING__COL__VALUE,
		// Constants.MEASURING__COL__MIN_TIME, Constants.MEASURING__COL__VALUE,
		// Constants.MEASURING__COL__MEAN_TIME, Constants.MEASURING__COL__VALUE,
		// Constants.MEASURING__COL__MAX_TIME, Constants.MEASURING__COL__TIME,
		// Constants.MEASURING__TABLE, Constants.MEASURING__COL__TIME, start,
		// Constants.MEASURING__COL__TIME, end, Constants.MEASURING__COL__KEY,
		// Constants.MEASURING__COL__NUMBER, Constants.MEASURING__COL__KEY);
		String sql = String
				.format("SELECT %s, %s, COUNT(*) AS %s, MIN(%s) AS %s, AVG(%s) AS %s, MAX(%s) AS %s, %s FROM %s WHERE %s <= \"%s\" GROUP BY %s, %s ORDER BY %s ASC",
						Constants.MEASURING__COL__KEY, Constants.MEASURING__COL__NUMBER,
						Constants.MEASURING__COL__NUMBER_PROCESSES, Constants.MEASURING__COL__VALUE,
						Constants.MEASURING__COL__MIN_TIME, Constants.MEASURING__COL__VALUE,
						Constants.MEASURING__COL__MEAN_TIME, Constants.MEASURING__COL__VALUE,
						Constants.MEASURING__COL__MAX_TIME, Constants.MEASURING__COL__TIME,
						Constants.MEASURING__TABLE, Constants.MEASURING__COL__TIME, end,
						Constants.MEASURING__COL__KEY, Constants.MEASURING__COL__NUMBER,
						Constants.MEASURING__COL__KEY);

		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			MeasuringStatistics stats = new MeasuringStatistics(start, end);
			while (result.next()) {
				String key = result.getString(Constants.MEASURING__COL__KEY);
				int numberRequestedElements = result.getInt(Constants.MEASURING__COL__NUMBER);
				long numberProcesses = result.getLong(Constants.MEASURING__COL__NUMBER_PROCESSES);
				double minTime = result.getDouble(Constants.MEASURING__COL__MIN_TIME);
				double meanTime = result.getDouble(Constants.MEASURING__COL__MEAN_TIME);
				double maxTime = result.getDouble(Constants.MEASURING__COL__MAX_TIME);
				MeasuringStatisticsEntry entry = new MeasuringStatisticsEntry(numberRequestedElements,
						numberProcesses, minTime, meanTime, maxTime);

				Vector<MeasuringStatisticsEntry> statNumber = null;
				if(stats.getStatistics() == null)
					stats.setStatistics(new TreeMap<String, Vector<MeasuringStatisticsEntry>>());
				if (stats.getStatistics().containsKey(key) == false) {
					statNumber = new Vector<MeasuringStatisticsEntry>();
				} else {
					statNumber = stats.getStatistics().get(key);
				}
				statNumber.add(entry);
				stats.getStatistics().put(key, statNumber);

			}
			return stats;
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}
}
