package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.IssuedCertificateInfo;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.IssuedCertsInfoProvider;

public class JDBCIssuedCertsInfoProvider extends AbstractMysqlConnection
		implements IssuedCertsInfoProvider<IssuedCertificateInfo> {

	@Inject
	public JDBCIssuedCertsInfoProvider() {
	}

	@Override
	public String getName() {
		return "Issued Certs Provider";
	}

	@Override
	public IssuedCertificateInfo[] getAll(int offset, int limit)
			throws ProviderException {
		String sql = String.format(
				"SELECT * FROM %s ORDER BY %s DESC LIMIT %d, %d",
				Constants.ISSUED_CERTIFICATE_INFO__TABLE,
				Constants.ISSUED_CERTIFICATE_INFO__COL__CREATION_TIME, offset,
				limit);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<IssuedCertificateInfo> tmp = new ArrayList<>();
			while (result.next()) {
				IssuedCertificateInfo info = new IssuedCertificateInfo();
				info.setCertId(result
						.getBytes(Constants.ISSUED_CERTIFICATE_INFO__COL__ID));
				info.setCertificate(result
						.getBytes(Constants.ISSUED_CERTIFICATE_INFO__COL__CERTIFICATE));
				info.setCreationTime(result
						.getTimestamp(Constants.ISSUED_CERTIFICATE_INFO__COL__CREATION_TIME));
				info.setExpirationTime(result
						.getTimestamp(Constants.ISSUED_CERTIFICATE_INFO__COL__EXPIRATION_TIME));
				info.setStartTime(result
						.getTimestamp(Constants.ISSUED_CERTIFICATE_INFO__COL__START_TIME));
				info.setSubjectName(result.getString(Constants.ISSUED_CERTIFICATE_INFO__COL__SUBJECT_NAME));
				tmp.add(info);
			}
			return tmp.toArray(new IssuedCertificateInfo[tmp.size()]);
		} catch (SQLException e) {
			throw new ProviderException("c"
					+ "Can not get all issued certificates from DB", e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public IssuedCertificateInfo getIssuedCertificate(byte[] certId)
			throws ProviderException {
		String sql = String.format("SELECT * FROM %s WHERE %s = ?;",
				Constants.ISSUED_CERTIFICATE_INFO__TABLE,
				Constants.ISSUED_CERTIFICATE_INFO__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, certId);
			st.execute();
			result = st.getResultSet();

			if (!result.first()) {
				throw new ProviderException("Entry not found!");
			} else {
				IssuedCertificateInfo info = new IssuedCertificateInfo();
				info.setCertId(result
						.getBytes(Constants.ISSUED_CERTIFICATE_INFO__COL__ID));
				info.setCertificate(result
						.getBytes(Constants.ISSUED_CERTIFICATE_INFO__COL__CERTIFICATE));
				info.setCreationTime(result
						.getTimestamp(Constants.ISSUED_CERTIFICATE_INFO__COL__CREATION_TIME));
				info.setExpirationTime(result
						.getTimestamp(Constants.ISSUED_CERTIFICATE_INFO__COL__EXPIRATION_TIME));
				info.setStartTime(result
						.getTimestamp(Constants.ISSUED_CERTIFICATE_INFO__COL__START_TIME));
				return info;
			}
		} catch (SQLException e) {
			throw new ProviderException("c"
					+ "Can not issued certificates from DB", e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	public void save(IssuedCertificateInfo info) throws ProviderException {
		String sql = String
				.format("INSERT INTO %s (%s, %s, %s, %s, %s, %s) VALUES (?,?,?,?,?,?);",
						Constants.ISSUED_CERTIFICATE_INFO__TABLE,
						Constants.ISSUED_CERTIFICATE_INFO__COL__ID,
						Constants.ISSUED_CERTIFICATE_INFO__COL__CERTIFICATE,
						Constants.ISSUED_CERTIFICATE_INFO__COL__START_TIME,
						Constants.ISSUED_CERTIFICATE_INFO__COL__EXPIRATION_TIME,
						Constants.ISSUED_CERTIFICATE_INFO__COL__CREATION_TIME,
						Constants.ISSUED_CERTIFICATE_INFO__COL__SUBJECT_NAME);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, info.getCertId());
			st.setBytes(2, info.getCertificate());
			st.setTimestamp(3, info.getStartTime());
			st.setTimestamp(4, info.getExpirationTime());
			st.setTimestamp(5, new Timestamp(System.currentTimeMillis()));
			st.setString(6, info.getSubjectName());
			st.execute();
		} catch (SQLException e) {
			throw new ProviderException("Can not prepare statement", e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

}
