package de.fraunhofer.sit.c2x.pki.ca.module.webservice.service;

import java.lang.reflect.Field;
import java.math.BigInteger;
import java.net.InetSocketAddress;
import java.security.cert.X509Certificate;
import java.util.NoSuchElementException;

import javax.annotation.Resource;
import javax.inject.Inject;
import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebResult;
import javax.jws.WebService;
import javax.jws.soap.SOAPBinding;
import javax.xml.ws.Action;
import javax.xml.ws.WebServiceContext;

import org.apache.log4j.Logger;
import org.eclipse.jetty.http.spi.JettyHttpExchangeDelegate;
import org.eclipse.jetty.http.spi.JettyHttpsExchange;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.util.AttributesMap;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets.AuthenticatorRegistrationServlet;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets.interfaces.ItsStationRegistrationHandler;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.AuthorizationValidationRequest;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.AuthorizationValidationResponse;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.AuthorizationValidationResponseAck;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.ItsStationRegistrationRequest;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.ItsStationRegistrationResponse;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.ObjectFactory;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.RequestError;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.handler.interfaces.AuthorizationValidationAckHandler;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.handler.interfaces.AuthorizationValidationRequestHandler;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Authenticator;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthenticatorProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;

@WebService(serviceName = "LtcaWebWithClientAuth", portName = "LtcaWebWithClientAuthPort", targetNamespace = "http://http.ltca.pki.c2x.sit.fraunhofer.de/LtcaWebServiceWithClientAuth")
@SOAPBinding(style = SOAPBinding.Style.RPC)
public class LtcaWebServiceWithClientAuth implements
		de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.WebServiceWithClientAuth {

	@Resource
	WebServiceContext wsContext;
	private InetSocketAddress remoteAddress;

	@InjectLogger
	private Logger logger;

	@Inject
	private ItsStationRegistrationHandler registrationHandler;

	@Inject
	private AuthorizationValidationRequestHandler authorizationValidationRequestHandler;

	@Inject
	private AuthorizationValidationAckHandler authorizationValidationAckHandler;

	@Inject
	private ConfigProvider configProvider;

	@Inject
	private AuthenticatorProvider ap;

	/**
	 * 
	 * @param authorizationValidationRequest
	 * @return returns
	 *         de.fraunhofer.sit.c2x.pki.ltca.http.AuthorizationValidationResponse
	 */
	@WebMethod
	@WebResult(name = "processAuthorizationValidationRequestResponse", targetNamespace = "http://http.ltca.pki.c2x.sit.fraunhofer.de/", partName = "processAuthorizationValidationRequestResponse")
	@Action(input = "http://http.ltca.pki.c2x.sit.fraunhofer.de/LtcaWebServiceWithClientAuth/processAuthorizationValidationRequestRequest", output = "http://http.ltca.pki.c2x.sit.fraunhofer.de/LtcaWebServiceWithClientAuth/processAuthorizationValidationRequestResponse")
	public AuthorizationValidationResponse processAuthorizationValidationRequest(
			@WebParam(name = "authorizationValidationRequest", targetNamespace = "http://http.ltca.pki.c2x.sit.fraunhofer.de/", partName = "authorizationValidationRequest") AuthorizationValidationRequest authorizationValidationRequest) {

		if (logger.isDebugEnabled())
			logger.debug("START: handling of an Authorization Validation Request from " + getRemoteAddress());

		try {
			AuthorizationValidationResponse response = authorizationValidationRequestHandler
					.handle(authorizationValidationRequest);
			if (logger.isDebugEnabled())
				logger.debug("END: handling of a Authorization Validation Reques from " + getRemoteAddress()
						+ " (successful)");
			return response;
		} catch (Exception e) {
			e.printStackTrace();
			logger.error("Authorization validation request cannot be processed: " + e.getMessage());
			RequestError error = new ObjectFactory().createRequestError();
			error.setErrorCode(BigInteger.ONE);
			error.setErrorDetails(e.getMessage());
			AuthorizationValidationResponse response = new AuthorizationValidationResponse(error);

			if (logger.isDebugEnabled()) {
				logger.debug("Error handling an Authorization Validation Request from " + getRemoteAddress()
						+ ": " + e.getMessage());
				logger.debug("END: handling of an Authorization Validation Request from "
						+ getRemoteAddress() + " (failed)");
			}

			return response;
		}
	}

	/**
	 * 
	 * @param itsStationRegistrationRequest
	 * @return returns
	 *         de.fraunhofer.sit.c2x.pki.ltca.http.ItsStationRegistrationResponse
	 */
	@WebMethod
	@WebResult(name = "processItsStationRegistrationRequestResponse", targetNamespace = "http://http.ltca.pki.c2x.sit.fraunhofer.de/", partName = "processItsStationRegistrationRequestResponse")
	@Action(input = "http://http.ltca.pki.c2x.sit.fraunhofer.de/LtcaWebServiceWithClientAuth/processItsStationRegistrationRequestRequest", output = "http://http.ltca.pki.c2x.sit.fraunhofer.de/LtcaWebServiceWithClientAuth/processItsStationRegistrationRequestResponse")
	public ItsStationRegistrationResponse processItsStationRegistrationRequest(
			@WebParam(name = "itsStationRegistrationRequest", targetNamespace = "http://http.ltca.pki.c2x.sit.fraunhofer.de/", partName = "itsStationRegistrationRequest") ItsStationRegistrationRequest itsStationRegistrationRequest) {

		if (logger.isDebugEnabled())
			logger.debug("START: handling of a ITS-Station Registration from " + getRemoteAddress());

		try {
			Authenticator authenticator = null;
			if (configProvider.get("webserverWithClientAuthentication").equals("true")) {
				Object o = wsContext.getMessageContext().get("com.sun.xml.internal.ws.http.exchange");
				if (o == null || !o.getClass().equals(JettyHttpsExchange.class)) {
					throw new NoSuchElementException("Client X.509 certificate not accessible");
				}
				JettyHttpsExchange exchange = (JettyHttpsExchange) o;
				try {
					// dirty hack to get the client's X.509 certificate via
					// reflection
					Field f = exchange.getClass().getDeclaredField("_delegate");
					f.setAccessible(true);
					JettyHttpExchangeDelegate delegate = (JettyHttpExchangeDelegate) f.get(exchange);

					f = delegate.getClass().getDeclaredField("_req");
					f.setAccessible(true);
					Request request = (Request) f.get(delegate);

					f = request.getClass().getDeclaredField("_attributes");
					f.setAccessible(true);
					AttributesMap attributesMap = (AttributesMap) f.get(request);
					o = attributesMap.getAttribute("javax.servlet.request.X509Certificate");
					if (o == null || !o.getClass().equals(X509Certificate[].class)) {
						throw new NoSuchElementException("Client X.509 certificate not accessible");
					}
					X509Certificate[] clientX509Certs = (X509Certificate[]) o;
					if (clientX509Certs.length <= 0)
						throw new NoSuchElementException("Unexpected client X.509 certificate length: "
								+ clientX509Certs.length);
					String[] dnElements = clientX509Certs[0].getSubjectDN().getName().split(",");
					String emailAddress = "";
					for (String s : dnElements) {
						if (s.contains("EMAILADDRESS=")) {
							String[] str = s.split("=");
							emailAddress = str[1];
							if (AuthenticatorRegistrationServlet.validate(emailAddress) == false)
								throw new NoSuchElementException(
										"Unexpected email in client X.509 certificate: \"" + emailAddress
												+ "\"");
							break;
						}
					}
					authenticator = ap.get(emailAddress);
					if (authenticator == null)
						throw new NoSuchElementException("Authenticator with email address \"" + emailAddress
								+ "\" not found in DB");
					else if (logger.isInfoEnabled())
						logger.info("Authenticator with email address \"" + emailAddress
								+ "\" requests ITS-Station Registration");
				} catch (NoSuchFieldException | IllegalArgumentException | IllegalAccessException e) {
					throw new Exception("client X.509 certificate not accessible", e);
				}
			}

			// return either the number of processed ITS stations or an error

			BigInteger numberOfRegisteredStations = registrationHandler.handleItsStationRegistration(
					itsStationRegistrationRequest, authenticator);
			ItsStationRegistrationResponse response = new ObjectFactory()
					.createItsStationRegistrationResponse();
			response.setNumberOfRegisteredStations(numberOfRegisteredStations);

			if (logger.isDebugEnabled())
				logger.debug("END: handling of a ITS-Station Registration from " + getRemoteAddress()
						+ " (successful)");

			return response;
		} catch (Exception e) {
			// e.printStackTrace();
			logger.error("ITS-Station Registration cannot be processed: " + e.getMessage());
			RequestError error = new ObjectFactory().createRequestError();
			error.setErrorCode(BigInteger.ONE);
			error.setErrorDetails(e.getMessage());
			ItsStationRegistrationResponse response = new ObjectFactory()
					.createItsStationRegistrationResponse();
			response.setRequestError(error);

			if (logger.isDebugEnabled()) {
				logger.debug("Error handling a ITS-Station Registration from " + getRemoteAddress() + ": "
						+ e.getMessage());
				logger.debug("END: handling of a ITS-Station Registration from " + getRemoteAddress()
						+ " (failed)");
			}

			return response;
		}

	}

	/**
	 * 
	 * @param authorizationValidationResponseAck
	 */
	@WebMethod
	@Action(input = "http://http.ltca.pki.c2x.sit.fraunhofer.de/LtcaWebServiceWithClientAuth/processAuthorizationValidationResponseAckRequest", output = "http://http.ltca.pki.c2x.sit.fraunhofer.de/LtcaWebServiceWithClientAuth/processAuthorizationValidationResponseAckResponse")
	public void processAuthorizationValidationResponseAck(
			@WebParam(name = "authorizationValidationResponseAck", targetNamespace = "http://http.ltca.pki.c2x.sit.fraunhofer.de/", partName = "authorizationValidationResponseAck") AuthorizationValidationResponseAck authorizationValidationResponseAck) {

		if (logger.isDebugEnabled())
			logger.debug("START: handling of an Authorization Validation ACK from " + getRemoteAddress());

		try {
			authorizationValidationAckHandler.handle(authorizationValidationResponseAck);

			if (logger.isDebugEnabled())
				logger.debug("END: handling of an Authorization Validation ACK from " + getRemoteAddress()
						+ " (successful)");

		} catch (Exception e) {
			e.printStackTrace();
			logger.error("Authorization Validation ACK cannot be processed: " + e.getMessage());
			RequestError error = new ObjectFactory().createRequestError();
			error.setErrorCode(BigInteger.ONE);
			error.setErrorDetails(e.getMessage());
			ItsStationRegistrationResponse response = new ObjectFactory()
					.createItsStationRegistrationResponse();
			response.setRequestError(error);

			if (logger.isDebugEnabled()) {
				logger.debug("Error handling an Authorization Validation ACK from " + getRemoteAddress()
						+ ": " + e.getMessage());
				logger.debug("END: handling of an Authorization Validation ACK from " + getRemoteAddress()
						+ " (failed)");
			}
		}
	}

	@Override
	public String getPath() {
		return "/LtcaWebServiceWithClientAuth";
	}

	private InetSocketAddress getRemoteAddress() {
		if (remoteAddress != null) {
			return remoteAddress;
		} else {
			Object o = wsContext.getMessageContext().get("com.sun.xml.internal.ws.http.exchange");
			if (o == null || !o.getClass().equals(JettyHttpsExchange.class)) {
				throw new NoSuchElementException("No http exchange found to get remote IP address");
			}
			JettyHttpsExchange exchange = (JettyHttpsExchange) o;
			remoteAddress = exchange.getRemoteAddress();
			return remoteAddress;
		}
	}
}
