package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.text.SimpleDateFormat;
import java.util.Date;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.AuthorizedDevice;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.IssuedCertificateInfoLink;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthorizedDeviceProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.IssuedCertsWithIdkInfoProvider;

/**
 * A simple servlet that outputs the certificate content of issued certificates.
 * Following arguments are allowed per GET
 * 
 * <ul>
 * <li>CertId8 or CertId10: <b>/cacert?cert=HEX or Base64</b></li>
 * </ul>
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class RegisteredItsStationsServlet extends AbstractServlet {

	private static final long serialVersionUID = 1L;
	protected static final String URL = "/regItsS";
	protected static final String TITLE = "Search registered ITS-S";
	public static final String DATE_PATTERN_LOG = "yyyy-MM-dd HH:mm:ss";

	@InjectLogger
	private Logger logger;
	private AuthorizedDeviceProvider authorizedDeviceProvider;
	private IssuedCertsWithIdkInfoProvider certInfoLinkingProvide;

	@Inject
	public RegisteredItsStationsServlet(HtmlProvider htmlProvider,
			AuthorizedDeviceProvider authorizedDeviceProvider,
			IssuedCertsWithIdkInfoProvider certInfoLinkingProvide) {
		super(htmlProvider);
		this.authorizedDeviceProvider = authorizedDeviceProvider;
		this.certInfoLinkingProvide = certInfoLinkingProvide;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		String userInfo = "";
		StringBuilder sb = new StringBuilder();
		sb.append("<form name=\"certForm\" action=\"\" method=\"get\">Search ITS-S with canonical ID encoded as HEX string: ");
		sb.append("<input type=\"text\" id=\"moduleId\" name=\"moduleId\" value=\"\" maxlength=\"32\" style=\"width:300px;\" />&nbsp;<input type=\"submit\" value=\"Find\" /></form>");
		AuthorizedDevice[] registeredItsSs = new AuthorizedDevice[0];

		// get moduleId from GET variable
		byte[] moduleIDBytes = new byte[0];
		if (req.getParameter("moduleId") != null && req.getParameter("moduleId").length() > 0) {
			String moduleID = req.getParameter("moduleId");
			moduleID = moduleID.trim();
			if (moduleID.length() > 0) {
				try {
					moduleIDBytes = Hex.decodeHex(moduleID.toCharArray());
				} catch (DecoderException e) {
					logger.warn("Unable to decode hex values of authorized device ID " + moduleID, e);
					userInfo = "<div id=\"errorbox\">Unable to decode hex values of authorized device ID "
							+ moduleID + "</div>";
				}
			}
		}

		try {
			if (moduleIDBytes.length == 16) {
				AuthorizedDevice registeredItsS = authorizedDeviceProvider.get(moduleIDBytes);
				if (registeredItsS != null) {
					registeredItsSs = new AuthorizedDevice[1];
					registeredItsSs[0] = registeredItsS;
				} else {
					userInfo = "<div id=\"errorbox\">Authorized device with ID "
							+ req.getParameter("moduleId") + " not found in DB</div>";
				}
			}
		} catch (ProviderException e) {
			logger.warn(e);
			userInfo = "<div id=\"errorbox\">Unable to request authorized device info from DB</div>";
		}
		if (!userInfo.isEmpty()) {
			sb.append("<br/><br/>" + userInfo);
		} else {
			// output list of cert infos
			sb.append("<table>\n");
			for (AuthorizedDevice info : registeredItsSs) {
				boolean isActive = (info.getDeactivated() == 0);
				String targetURL;
				if (isActive) {
					targetURL = ItsStationRevocationServlet.URL;
				} else {
					targetURL = ItsStationReactivationServlet.URL;
				}
				sb.append("<tr><form action=\"" + targetURL
						+ "\" method=\"get\">\n<td style=\"width:450px;\"><ul><li id=\"id_"
						+ Hex.encodeHexString(info.getCanonicalId()) + "\">CanonicalId: "
						+ Hex.encodeHexString(info.getCanonicalId()) + "</li>\n");
				sb.append("<li id=\"creationTime_" + Hex.encodeHexString(info.getCanonicalId())
						+ "\">Creation time: " + formatedDate(info.getCreationTime()) + "</li>\n");

				try {
					IssuedCertificateInfoLink[] relaredLtcs = certInfoLinkingProvide.getCerts(
							info.getCanonicalId(), 0, 1000);
					sb.append("<li>Related LTCs:\n<ul>\n");
					for (IssuedCertificateInfoLink relatedLtc : relaredLtcs) {
						sb.append("<li>" + Hex.encodeHexString(relatedLtc.getCertId()));
//						sb.append(" <a href=\""
//								+ IssuedLongTermCertificatesServlet.URL + "?cert="
//								+ Hex.encodeHexString(relatedLtc.getCertId()) + "\">show details</a>");
						sb.append("</li>\n");
					}
					sb.append("</ul>\n</li>\n");
				} catch (ProviderException e) {
					logger.warn(e);
					sb.append("<div id=\"errorbox\">Unable to get related LTCs from DB</div>");
				}

				if (isActive) {
					sb.append("<li id=\"status_" + Hex.encodeHexString(info.getCanonicalId())
							+ "\">Active: true</li>\n");
					sb.append("</ul></td><td style=\"width:150px;\"><input type=\"hidden\" name=\"deactivationId\" value=\""
							+ Hex.encodeHexString(info.getCanonicalId())
							+ "\"/><input type=\"submit\" value=\"Deactivate\" /></td><td style=\"width:150px;\"></td>\n</form>\n</tr>\n");
				} else {
					sb.append("<li id=\"status_" + Hex.encodeHexString(info.getCanonicalId())
							+ "\">Active: false</li>\n");
					sb.append("</ul></td><td style=\"width:150px;\"></td><td style=\"width:150px;\"><input type=\"hidden\" name=\"moduleId\" value=\""
							+ Hex.encodeHexString(info.getCanonicalId())
							+ "\"/><input type=\"submit\" value=\"Re-activate\" /></td>\n</form>\n</tr>\n");
				}

			}
			sb.append("</table>\n");
		}

		return sb.toString();
	}

	private String formatedDate(Date date) {
		return new SimpleDateFormat(DATE_PATTERN_LOG).format(date);
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString(),
				UserRole.DEVELOPER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.ITS_S;
	}
}
