package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;

import javax.inject.Inject;
import javax.net.ssl.KeyManagerFactory;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;
import org.eclipse.jetty.security.authentication.SessionAuthentication;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.measuring.ItsStationType;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Host;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.KnownCA;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.KnownCaProvider;

/**
 * A simple servlet that outputs the content of the configuration stored in the
 * DB
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class KnownCaServlet extends AbstractServlet {
	private static final long serialVersionUID = 1L;

	public static final String URL = "/knownCAs";
	public static final String TITLE = "Add known CA";
	public static final String DATE_PATTERN_LOG = "yyyy-MM-dd HH:mm:ss,SSS";

	private final KnownCaProvider knownCaProvider;

	@InjectLogger
	private Logger logger;

	@Inject
	public KnownCaServlet(HtmlProvider htmlProvider, KnownCaProvider knownCaProvider) {
		super(htmlProvider);
		this.knownCaProvider = knownCaProvider;
	}

	@Override
	protected void doPost(HttpServletRequest req, HttpServletResponse response) throws ServletException,
			IOException {
		doGet(req, response);
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		StringBuilder sb = new StringBuilder();
		String userInfo = "";

		try {
			if (((req.getParameter("hostAddress") != null && req.getParameter("hostAddress").length() > 0) || (req
					.getParameter("wsdlUrl") != null && req.getParameter("wsdlUrl").length() > 0)
					&& req.getParameter("certId") != null
					&& req.getParameter("certId").length() > 0
					&& req.getParameter("cert") != null && req.getParameter("cert").length() > 0)
					&& req.getParameter("caType") != null && req.getParameter("cert").length() > 0) {
				Host host = new Host();

				String hostAddress = req.getParameter("hostAddress");
				if (req.getParameter("hostAddress") == null)
					hostAddress = "";
				host.setAdress(hostAddress);

				String httpPortString = req.getParameter("httpPort");
				Short httpPort = null;
				if (httpPortString != null && httpPortString.length() > 0)
					httpPort = new Short(httpPortString);
				host.setHttpPort(httpPort);

				String udpPortString = req.getParameter("udpPort");
				Short udpPort = null;
				if (udpPortString != null && udpPortString.length() > 0)
					udpPort = new Short(udpPortString);
				host.setHttpPort(udpPort);

				String wsdlUrl = req.getParameter("wsdlUrl");
				host.setWsdlUrl(wsdlUrl);

				String caType = req.getParameter("caType");

				String certIdString = req.getParameter("certId");
				byte[] certId = new byte[0];
				if (certIdString != null && certIdString.length() > 0)
					certId = Hex.decodeHex(certIdString.toCharArray());

				String certString = req.getParameter("cert");
				byte[] certificate = new byte[0];
				if (certString != null && certString.length() > 0)
					certificate = Hex.decodeHex(certString.toCharArray());

				String x509ClientCertString = req.getParameter("x509ClientCert");
				String rsaClientKeyString = "";
				String rsaClientKeyPassword = "";
				if (!x509ClientCertString.isEmpty()) {
					X509Certificate x509Cert = KnownCA.createX509FromString(x509ClientCertString);
					String x509Name = x509Cert.getSubjectDN().getName();
					if (x509Name.isEmpty())
						x509ClientCertString = "";

					rsaClientKeyString = req.getParameter("rsaClientKey");
					rsaClientKeyPassword = req.getParameter("rsaClientKeyPassword");
					RSAPrivateKey key = KnownCA.createKeyFromString(rsaClientKeyString, rsaClientKeyPassword);

					if (!x509Name.isEmpty() && !rsaClientKeyString.isEmpty()) {
						// try to load key and certificate into keystore
						KeyStore ks = KeyStore.getInstance("JKS");
						ks.load(null, "".toCharArray());
						ks.setKeyEntry(x509Cert.getSubjectDN().getName(), key,
								rsaClientKeyPassword.toCharArray(),
								new java.security.cert.Certificate[] { x509Cert });

						KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance("SunX509");
						keyManagerFactory.init(ks, rsaClientKeyPassword.toCharArray());

						if (logger.isDebugEnabled())
							logger.debug("X.509 cert and RSA private key successfully loaded into key store");
					}
				}

				knownCaProvider.save(host, caType, certId, certificate, x509ClientCertString,
						rsaClientKeyString, rsaClientKeyPassword);

				// get user from http session
				SessionAuthentication user = (SessionAuthentication) req.getSession().getAttribute(
						SessionAuthentication.__J_AUTHENTICATED);
				logger.info("USER: User " + user.getUserIdentity().getUserPrincipal().getName().toString()
						+ " registered a new known CA with certID: " + Hex.encodeHexString(certId));

				userInfo = "<div id=\"ackbox\">CA with host address " + hostAddress + " and certId "
						+ Hex.encodeHexString(certId) + " inserted in database</div>";
			} else if (req.getParameter("submit") != null) {
				if (userInfo.isEmpty())
					userInfo = "<div id=\"errorbox\">Please provide all required elements</div>";
			}
		} catch (NumberFormatException | DecoderException | ProviderException e) {
			logger.error("Unable to insert CA into database: " + e.getMessage(), e);
			userInfo = "<div id=\"errorbox\">Unable to insert CA into database: " + e.getMessage() + "</div>";
		} catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException
				| UnrecoverableKeyException e) {
			logger.error(
					"Unable to create keystore for given X.509 cert and RSA private key: " + e.getMessage(),
					e);
			userInfo = "<div id=\"errorbox\">Unable to create keystore for given X.509 cert and RSA private key: "
					+ e.getMessage() + "</div>";
		}

		if (req.getParameter("userInfo") != null && req.getParameter("userInfo").length() > 0) {
			userInfo = req.getParameter("userInfo");
		}

		if (userInfo != "")
			sb.append(userInfo);

		sb.append("<form name=\"knwonCaForm\" action=\"\" method=\"post\">\n");
		sb.append("<table>\n");

		// sb.append("<tr><td>Host IP address*:</td><td><input id=\"hostAddress\" name=\"hostAddress\" style=\"width:400px\" maxlength=\"200\" /></td></tr>\n");
		// sb.append("<tr><td>HTTP/S port:</td><td><input id=\"httpPort\" name=\"httpPort\" style=\"width:400px\" maxlength=\"200\" /></td></tr>\n");
		// sb.append("<tr><td>UDP port:</td><td><input id=\"udpPort\" name=\"udpPort\" style=\"width:400px\" maxlength=\"200\" /></td></tr>\n");
		sb.append("<tr><td>Webservice WSDL URL*:</td><td><input id=\"wsdlUrl\" name=\"wsdlUrl\" style=\"width:400px\" maxlength=\"200\" /></td></tr>\n");

		sb.append("<tr><td>CA type:</td><td><select id=\"caType\" name=\"caType\" style=\"width:400px\">");
		sb.append("<option>" + ItsStationType.RCA.toString() + "</option>\n");
		sb.append("<option>" + ItsStationType.LTCA.toString() + "</option>\n");
		sb.append("</select></td></tr>\n");

		sb.append("<tr><td>Cert ID*:<br />(hex encoded):</td><td><input id=\"certId\" name=\"certId\" style=\"width:400px\" maxlength=\"200\" /></td></tr>\n");
		sb.append("<tr><td>Certificate*:<br />(hex encoded):</td><td><textarea id=\"cert\" name=\"cert\" style=\"width:400px\"  rows=\"8\"></textarea></td></tr>\n");

		sb.append("<tr><td>X.509 client certificate:<br />(one PEM Base64 encoded DER cert):</td><td><textarea id=\"x509ClientCert\" name=\"x509ClientCert\" style=\"width:400px\"  rows=\"8\"></textarea></td></tr>\n");
		sb.append("<tr><td>RSA client key:<br />(one PEM Base64 encoded DER key):</td><td><textarea id=\"rsaClientKey\" name=\"rsaClientKey\" style=\"width:400px\"  rows=\"8\"></textarea></td></tr>\n");
		sb.append("<tr><td>RSA client key password:</td><td><input id=\"rsaClientKeyPassword\" name=\"rsaClientKeyPassword\" style=\"width:400px\" maxlength=\"30\" /></td></tr>\n");

		sb.append("<tr><td></td><td><br /><input type=\"submit\" value=\"Insert\" /></td></tr>\n"
				+ "</table>\n");
		sb.append("</form>\n");
		sb.append("<br/>\nRequired elements are marked by a *\n");

		return sb.toString();
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.SETUP;
	}
}
