/**
 * 
 */
package de.fraunhofer.sit.c2x.pki.ca.crypto.signer;

import java.math.BigInteger;
import java.util.Random;

import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECPoint;

import de.fraunhofer.sit.c2x.pki.ca.crypto.Digest;
import de.fraunhofer.sit.c2x.pki.ca.utils.ByteUtils;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class ECDSA {

	private static final boolean DEBUG = false;
	private CipherParameters key;

	public void init(CipherParameters key) {
		this.key = key;
	}

	public BigInteger[] generateSignature(byte[] messageToSign) {

		return generateSignature(messageToSign, randomBigInt(((ECPrivateKeyParameters) key).getParameters()
				.getN().bitLength()));

	}

	public static BigInteger randomBigInt(int numOfBits) {
		return new BigInteger(numOfBits, new Random());
	}

	public BigInteger[] generateSignature(byte[] messageToSign, BigInteger randomK) {

		ECPrivateKeyParameters privateKey = (ECPrivateKeyParameters) key;
		BigInteger n = privateKey.getParameters().getN();
		BigInteger k = randomK;
		ECPoint R;
		BigInteger r;
		do {

			R = privateKey.getParameters().getG().multiply(k);
			r = R.getX().toBigInteger().mod(n);
			if (r.equals(BigInteger.ZERO)) {
				k = randomBigInt(n.bitLength());
			}
		} while (r.equals(BigInteger.ZERO));

		byte[] hash = hash(messageToSign);
		// hash = ByteUtils.toLittleEndian(hash);
		BigInteger e = new BigInteger(1, hash);

		BigInteger s = k.modInverse(n).multiply(e.add(privateKey.getD().multiply(r))).mod(n);
		if (DEBUG) {
			System.out.println("*** Crypto Debug - Start ***");

			ByteUtils.print("K (BigEndian)", ByteUtils.trim(k.toByteArray()));
			ByteUtils.print("r (BigEndian)", ByteUtils.trim(r.toByteArray()));
			ByteUtils.print("d (BigEndian)", ByteUtils.trim(privateKey.getD().toByteArray()));
			ByteUtils.print("e = h(m).asBigInteger (BigEndian)", ByteUtils.trim(e.toByteArray()));

			// System.out.println("k = " +
			// Hex.encodeHexString(k.toByteArray()));
			// System.out.println("r = " +
			// Hex.encodeHexString(r.toByteArray()));
			// System.out.println("d = " +
			// Hex.encodeHexString(privateKey.getD().toByteArray()));
			// System.out.println("e = " + Hex.encodeHexString(hash));

			BigInteger kModInverse = k.modInverse(n);

			ByteUtils.print("(k^-1) mod n (BigEndian)", ByteUtils.trim(kModInverse.toByteArray()));
			// System.out.println("k^-1 mod n = "
			// +
			// Hex.encodeHexString(ByteUtils.toLittleEndian(kModInverse.toByteArray())));

			BigInteger dMulR_mod = (privateKey.getD().multiply(r)).mod(n);

			ByteUtils.print("(d * r) mod n (BigEndian)", ByteUtils.trim(dMulR_mod.toByteArray()));
			// System.out.println("d*r mod n = " +
			// Hex.encodeHexString(dMulR_mod.toByteArray()));

			BigInteger hashAddDMulR_mod = (e.add(dMulR_mod)).mod(n);

			ByteUtils.print("(h(m) + (d * r)) mod n (BigEndian)",
					ByteUtils.trim(hashAddDMulR_mod.toByteArray()));

			// System.out.println("h(m) + (d*r) mod n = " +
			// Hex.encodeHexString(hashAddDMulR_mod.toByteArray()));
			// System.out.println(Hex.encodeHexString(e.add(dMulR_mod).subtract(e).mod(n).toByteArray()));

			// try {
			// BigInteger tmp = new BigInteger(1, ByteUtils.toBigEndian(Hex
			// .decodeHex("2D76DD019B2C6F619DB9072EDDCBFD86B652310ED9B9FD92EB6F52F4D9AC27E2"
			// .toCharArray())));
			// TestUtils.print("Check",
			// ByteUtils.toBigEndian(tmp.subtract(dMulR_mod).toByteArray()));
			// } catch (DecoderException e1) {
			// // TODO Auto-generated catch block
			// e1.printStackTrace();
			// }

			BigInteger news = (kModInverse.multiply(hashAddDMulR_mod)).mod(n);
			// correct verified!!!!
			ByteUtils.print("s = (k^-1 * (h(m) + (d * r))) mod n (BigEndian)",
					ByteUtils.trim(s.toByteArray()));

			// System.out.println("s = (k^-1 * (h(m) + (d * r))) mod n = " +
			// Hex.encodeHexString(news.toByteArray()));
			System.out.println("*** Crypto Debug - Stop ***");

		}
		return new BigInteger[] { r, s };
	}

	protected byte[] hash(byte[] messageToSign) {
		return Digest.getInstance().sha256(messageToSign);
	}

	public boolean verifySignature(byte[] message, BigInteger[] signatureToVerify) {
		return verifySignatureUsingHash(hash(message), signatureToVerify);
	}

	public boolean verifySignatureUsingHash(byte[] hash, BigInteger[] signatureToVerify) {

		ECPublicKeyParameters pubKey = (ECPublicKeyParameters) key;
		BigInteger n = pubKey.getParameters().getN();
		BigInteger r = signatureToVerify[0];
		BigInteger s = signatureToVerify[1];

		if (r.compareTo(BigInteger.ONE) < 0 && r.compareTo(n) >= 0) {
			// throw new
			// CryptoUtilsException("r is invalid. r should be in the interval [1, n-1 ]");
			return false;
		}

		if (s.compareTo(BigInteger.ONE) < 0 && s.compareTo(n) >= 0) {
			// throw new
			// CryptoUtilsException("s is invalid. r should be in the interval [1, n-1 ]");
			return false;
		}

		// hash = ByteUtils.toLittleEndian(hash);
		BigInteger e = new BigInteger(1, hash);
		BigInteger w = s.modInverse(n);

		BigInteger u1 = (e.multiply(w)).mod(n);
		BigInteger u2 = (r.multiply(w)).mod(n);
		ECPoint G = pubKey.getParameters().getG();
		ECPoint Q = pubKey.getQ();

		ECPoint X = ECAlgorithms.sumOfTwoMultiplies(G, u1, Q, u2);
		if (X.isInfinity()) {
			return false;
		}

		BigInteger v = X.getX().toBigInteger().mod(n);

		return v.equals(r);
	}

	public static byte[] randomBytes(int size) {
		byte[] randomBytes = new byte[size];
		new Random().nextBytes(randomBytes);

		return randomBytes;
	}

	public static ECPoint computePublicKey(ECNamedCurveParameterSpec spec, byte[] d) {
		return spec.getG().multiply(new BigInteger(1, d));
	}

}
