/**
 * 
 */
package de.fraunhofer.sit.c2x.pki.ca.crypto.encryption;

import java.math.BigInteger;
import java.nio.charset.Charset;

import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.crypto.BasicAgreement;
import org.bouncycastle.crypto.BufferedBlockCipher;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.DerivationFunction;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.crypto.Mac;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.params.IESParameters;
import org.bouncycastle.crypto.params.IESWithCipherParameters;
import org.bouncycastle.crypto.params.KDFParameters;
import org.bouncycastle.crypto.params.KeyParameter;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class ECIESforIEEE1609Dot2 {

	private final static boolean DEBUG = false;

	BasicAgreement agree;
	DerivationFunction kdf;
	Mac mac;
	BufferedBlockCipher cipher;
	byte[] macBuf;

	boolean forEncryption;
	CipherParameters privParam, pubParam;
	IESParameters param;

	/**
	 * set up for use with stream mode, where the key derivation function is
	 * used to provide a stream of bytes to xor with the message.
	 * 
	 * @param agree
	 *            the key agreement used as the basis for the encryption
	 * @param kdf
	 *            the key derivation function used for byte generation
	 * @param mac
	 *            the message authentication code generator for the message
	 */
	public ECIESforIEEE1609Dot2(BasicAgreement agree, DerivationFunction kdf, Mac mac) {
		this.agree = agree;
		this.kdf = kdf;
		this.mac = mac;
		this.macBuf = new byte[mac.getMacSize()];
		this.cipher = null;
	}

	/**
	 * set up for use in conjunction with a block cipher to handle the message.
	 * 
	 * @param agree
	 *            the key agreement used as the basis for the encryption
	 * @param kdf
	 *            the key derivation function used for byte generation
	 * @param mac
	 *            the message authentication code generator for the message
	 * @param cipher
	 *            the cipher to used for encrypting the message
	 */
	public ECIESforIEEE1609Dot2(BasicAgreement agree, DerivationFunction kdf, Mac mac,
			BufferedBlockCipher cipher) {
		this.agree = agree;
		this.kdf = kdf;
		this.mac = mac;
		this.macBuf = new byte[mac.getMacSize()];
		this.cipher = cipher;
	}

	/**
	 * Initialise the encryptor.
	 * 
	 * @param forEncryption
	 *            whether or not this is encryption/decryption.
	 * @param privParam
	 *            our private key parameters
	 * @param pubParam
	 *            the recipient's/sender's public key parameters
	 * @param param
	 *            encoding and derivation parameters.
	 */
	public void init(boolean forEncryption, CipherParameters privParam, CipherParameters pubParam,
			CipherParameters param) {
		this.forEncryption = forEncryption;
		this.privParam = privParam;
		this.pubParam = pubParam;
		this.param = (IESParameters) param;
	}

	private byte[] decryptBlock(byte[] ieeeInput, int inOff, int inLen, byte[] z)
			throws InvalidCipherTextException {
		if (DEBUG) {
			System.out.println("============================");
			System.out.println("Decryption: Use receiver's secret key and senders's public key");
			printLine();
			print("secret-key", ((ECPrivateKeyParameters) privParam).getD().toByteArray());
			print("public-key", ((ECPublicKeyParameters) pubParam).getQ().getEncoded());
			printLine();
			print("Message to decrypt", ieeeInput);
			printLine();
		}

		byte[] in_enc = new byte[mac.getMacSize() + inLen - 20]; // extend input
																	// from 32
																	// bytes to
																	// 48 bytes
		inLen = in_enc.length; // set length to 48
		System.arraycopy(ieeeInput, 0, in_enc, 0, ieeeInput.length);
		byte[] M = null;
		KeyParameter macKey = null;
		KDFParameters kParam = new KDFParameters(z, param.getDerivationV());
		int macKeySize = param.getMacKeySize();
		if (DEBUG) {
			printLine();
			print("Agreement", z);
		}

		kdf.init(kParam);

		inLen -= mac.getMacSize();

		if (cipher == null) // stream mode
		{
			byte[] buf = generateKdfBytes(kParam, inLen + (macKeySize / 8));

			M = new byte[inLen];

			for (int i = 0; i != inLen; i++) {
				M[i] = (byte) (in_enc[inOff + i] ^ buf[i]);
			}
			macKey = new KeyParameter(buf, inLen, (macKeySize / 8));
			if (DEBUG) {
				print("Message (without MAC)", M);
				print("MAC Key", macKey.getKey());
			}
		} else {
			int cipherKeySize = ((IESWithCipherParameters) param).getCipherKeySize();
			byte[] buf = generateKdfBytes(kParam, (cipherKeySize / 8) + (macKeySize / 8));

			cipher.init(false, new KeyParameter(buf, 0, (cipherKeySize / 8)));

			byte[] tmp = new byte[cipher.getOutputSize(inLen)];

			int len = cipher.processBytes(in_enc, inOff, inLen, tmp, 0);

			len += cipher.doFinal(tmp, len);

			M = new byte[len];

			System.arraycopy(tmp, 0, M, 0, len);

			macKey = new KeyParameter(buf, (cipherKeySize / 8), (macKeySize / 8));
		}

		byte[] macIV = param.getEncodingV();

		mac.init(macKey);
		mac.update(in_enc, inOff, inLen);
		mac.update(macIV, 0, macIV.length);
		mac.doFinal(macBuf, 0);
		if (DEBUG) {
			print("Tag", macBuf);
			printLine();
			print("Message (" + (new String(M, Charset.forName("UTF-8"))) + ")", M);
			System.out.println("============================");
		}
		inOff += inLen;

		for (int t = 0; t < macBuf.length - 12; t++) { // check only the first
														// 20 Bytes of Tag
			if (macBuf[t] != in_enc[inOff + t]) {
				throw (new InvalidCipherTextException("Mac codes failed to equal."));
			}
		}

		return M;
	}

	private byte[] encryptBlock(byte[] in, int inOff, int inLen, byte[] z) throws InvalidCipherTextException {
		if (DEBUG) {
			System.out.println("============================");
			System.out.println("Encryption: Use sender's secret key and receiver's public key");
			printLine();
			print("secret-key", ((ECPrivateKeyParameters) privParam).getD().toByteArray());
			print("public-key", ((ECPublicKeyParameters) pubParam).getQ().getEncoded());
			printLine();
			print("Message to encrypt (" + new String(in, Charset.forName("UTF-8")) + ")", in);
			printLine();
		}
		byte[] C = null;
		KeyParameter macKey = null;
		KDFParameters kParam = new KDFParameters(z, param.getDerivationV());
		int c_text_length = 0;
		int macKeySize = param.getMacKeySize();

		if (DEBUG) {
			print("Agreement", z);
		}

		if (cipher == null) // stream mode
		{
			byte[] buf = generateKdfBytes(kParam, inLen + (macKeySize / 8));

			C = new byte[inLen + mac.getMacSize()];
			c_text_length = inLen;

			for (int i = 0; i != inLen; i++) {
				C[i] = (byte) (in[inOff + i] ^ buf[i]);
			}
			macKey = new KeyParameter(buf, inLen, (macKeySize / 8));
			if (DEBUG) {
				print("Cipher (without MAC)", C);
				print("MAC-Key", macKey.getKey());
				System.out.println("------");
			}
		} else {
			int cipherKeySize = ((IESWithCipherParameters) param).getCipherKeySize();
			byte[] buf = generateKdfBytes(kParam, (cipherKeySize / 8) + (macKeySize / 8));

			cipher.init(true, new KeyParameter(buf, 0, (cipherKeySize / 8)));

			c_text_length = cipher.getOutputSize(inLen);

			byte[] tmp = new byte[c_text_length];

			int len = cipher.processBytes(in, inOff, inLen, tmp, 0);

			len += cipher.doFinal(tmp, len);

			C = new byte[len + mac.getMacSize()];
			c_text_length = len;

			System.arraycopy(tmp, 0, C, 0, len);

			macKey = new KeyParameter(buf, (cipherKeySize / 8), (macKeySize / 8));
		}

		byte[] macIV = param.getEncodingV();

		mac.init(macKey);
		mac.update(C, 0, c_text_length);
		mac.update(macIV, 0, macIV.length);
		//
		// return the message and it's MAC
		//

		mac.doFinal(C, c_text_length);

		byte[] ieeeReturn = new byte[inLen + 20];
		System.arraycopy(C, 0, ieeeReturn, 0, ieeeReturn.length);
		if (DEBUG) {
			print("Cipher (final)", C);
			print("Cipher (IEEE)", ieeeReturn);
			byte[] c = new byte[ieeeReturn.length - 20];
			byte[] t = new byte[20];
			System.arraycopy(ieeeReturn, 0, c, 0, c.length);
			System.arraycopy(ieeeReturn, c.length, t, 0, t.length);
			print(" - c", c);
			print(" - t", t);
			System.out.println("============================");
		}
		return ieeeReturn;
	}

	private void print(String name, byte[] bytes) {
		System.out.println(name + ": " + Hex.encodeHexString(bytes) + ", length= " + bytes.length + "Bytes");
	}

	private void printLine() {
		System.out.println("-------");
	}

	private byte[] generateKdfBytes(KDFParameters kParam, int length) {
		byte[] buf = new byte[length];

		kdf.init(kParam);

		kdf.generateBytes(buf, 0, buf.length);

		return buf;
	}

	public byte[] processBlock(byte[] in, int inOff, int inLen) throws InvalidCipherTextException {
		agree.init(privParam);

		BigInteger z = agree.calculateAgreement(pubParam);

		byte[] newZ = new byte[32];

		if (z.toByteArray().length < 32) {
			System.arraycopy(z.toByteArray(), 0, newZ, 32 - z.toByteArray().length, z.toByteArray().length);
		} else {

			if (z.toByteArray().length > 32) {
				System.arraycopy(z.toByteArray(), z.toByteArray().length - 32, newZ, 0, 32);
			} else {
				newZ = z.toByteArray();
			}
		}

		if (forEncryption) {
			return encryptBlock(in, inOff, inLen, newZ);
		} else {
			return decryptBlock(in, inOff, inLen, newZ);
		}
	}

}
