package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.provider;

import java.io.IOException;
import java.sql.Timestamp;

import javax.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.CaCertificate;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.CaInfoProvider;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SignerInfo;

public class EtsiCaInfoProvider implements CaInfoProvider<Certificate> {

	@Inject
	private CaInfoProvider<CaCertificate> provider;

	@Override
	public void startService() {
		provider.startService();
	}

	@Override
	public void stopService() {
		provider.stopService();
	}

	@Override
	public String getName() {
		return provider.getName();
	}

	@Override
	public boolean isStarted() {
		return provider.isStarted();
	}

	@Override
	public long getUptime() {
		return provider.getUptime();
	}

	@Override
	public Certificate getCaCertificate() throws ProviderException {
		try {
			return WaveUtils.getElementFromBytes(provider.getCaCertificate().getCertificate(),
					Certificate.class);
		} catch (IOException e) {
			e.printStackTrace();
			throw new ProviderException("Unable to create certificate from DB bytes: "+e.getMessage());
		}
	}
	
	@Override
	public Certificate getCaCertificate(byte[] certId) throws ProviderException {
		try {
			return WaveUtils.getElementFromBytes(provider.getCaCertificate(certId).getCertificate(),
					Certificate.class);
		} catch (IOException e) {
			e.printStackTrace();
			throw new ProviderException("Unable to create certificate from DB bytes: "+e.getMessage());
		}
	}
	
	@Override
	public Certificate[] getAll(int offset, int limit) throws ProviderException {
		CaCertificate[] caCerts = provider.getAll(0, 1000);
		Certificate[] certs = new Certificate[caCerts.length];
		for(int i=0; i < caCerts.length; i++){
			CaCertificate caCert = caCerts[i];
			try {
				Certificate cert = WaveUtils.getElementFromBytes(caCert.getCertificate(),Certificate.class);
				certs[i] = cert;
			} catch (IOException e) {
				throw new ProviderException("Unable to create certificate from DB bytes: "+e.getMessage());
			}
		}
		return certs;
	}

	@Override
	public void setCaCertificate(Certificate cert) throws ProviderException {

		CaCertificate caCert = new CaCertificate();

		try {

			caCert.setCertId(cert.getHashedId8().getCertID().get());
			caCert.setCertificate(cert.getBytes());
			caCert.setCreationTime(new Timestamp(System.currentTimeMillis()));
			caCert.setExpirationTime(cert.getValidityRestrictions()[0].getEndValidity().toSqlTimestamp());
			
			SignerInfo signerInfo = cert.getSignerInfo();

			switch (signerInfo.getType()) {
			case CERTIFICATE:
				caCert.setSignerCertId(signerInfo.getCertificate().getHashedId8().getCertID().get());
				break;
			case CERTIFICATE_DIGEST_WITH_ECDSAP256:
				caCert.setSignerCertId(signerInfo.getDigest().getCertID().get());
				break;
			case SELF:
				caCert.setSignerCertId(cert.getHashedId8().getCertID().get());
				break;
			default:
				throw new ProviderException("SignerInfo type " + signerInfo.getType()
						+ " currently not implemented");
			}
			
			caCert.setStartTime(cert.getValidityRestrictions()[0].getStartValidity().toSqlTimestamp());
			caCert.setSubjectName(new String(cert.getSubjectInfo().getSubjectName().get()));
			caCert.setSubjectType(cert.getSubjectInfo().getSubjectType().name());

		} catch (Exception e) {
			throw new ProviderException(
					"Setting CA certificate failed: Problem parsing certificate parameters for caCertificate!",
					e);
		}

		provider.setCaCertificate(caCert);

	}

	@Override
	public void sortUp(byte[] certID) throws ProviderException {
		throw new ProviderException("Not implemented");
	}

	@Override
	public void sortDown(byte[] certID) throws ProviderException {
		throw new ProviderException("Not implemented");
	}

}
