package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;

import javax.inject.Inject;

import com.mysql.jdbc.Statement;

import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Crl;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.CrlCertificate;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.CrlSeries;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.RevokedCertificate;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.UpdateableCrlProvider;
import de.fraunhofer.sit.c2x.pki.ca.utils.ByteUtils;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class JDBCCrlProvider extends AbstractMysqlConnection implements
		UpdateableCrlProvider<Crl, RevokedCertificate, CrlCertificate> {

	@Inject
	private ConfigProvider configProvider;
	private CrlCertificate cert;

	private Crl crl = null;

	@Override
	public String getName() {
		return "Crl Provider";
	}

	@Override
	public void save(Crl rev) {

		String sql = String.format("INSERT INTO %s (%s, %s, %s, %s, %s) VALUES(?,?,?,?,?)"
				+ "ON DUPLICATE KEY UPDATE %s=VALUES(%s), %s=VALUES(%s), %s=VALUES(%s);",
				Constants.CRL__TABLE, Constants.CRL__COL__CRL_SERIES, Constants.CRL__COL__CURRENT_SERIAL,
				Constants.CRL__COL__START_PERIOD, Constants.CRL__COL__NEXT_CRL,
				Constants.CRL__COL__CRL_BYTES, Constants.CRL__COL__START_PERIOD,
				Constants.CRL__COL__START_PERIOD, Constants.CRL__COL__NEXT_CRL, Constants.CRL__COL__NEXT_CRL,
				Constants.CRL__COL__CRL_BYTES, Constants.CRL__COL__CRL_BYTES);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setLong(1, rev.getCrlSeries().getCrlSeries());
			st.setLong(2, rev.getCrlSeries().getCurrentSerial());
			st.setTimestamp(3, rev.getStartPeriod());
			st.setTimestamp(4, rev.getNextCrl());
			st.setBytes(5, rev.getCrl());
			st.execute();
			clearCache();
		} catch (SQLException e) {
			logger.error(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public Crl get(boolean verifyCrl) throws ProviderException {
		return get();
	}

	@Override
	public Crl get(long crlSeries) throws ProviderException {
		if (crl == null) {
			String sql = String.format("SELECT * FROM %s WHERE %s = ? ORDER BY %s DESC LIMIT 1;",
					Constants.CRL__TABLE, Constants.CRL__COL__CRL_SERIES, Constants.CRL__COL__NEXT_CRL);
			Connection con = getConnection();
			PreparedStatement st = null;
			ResultSet result = null;
			try {
				st = con.prepareStatement(sql);
				st.setLong(1, crlSeries);
				st.execute();
				result = st.getResultSet();
				if (!result.first()) {
					throw new ProviderException("No crl found in database");
				} else {

					Crl crl = new Crl();
					CrlSeries crlS = new CrlSeries();
					crlS.setCrlSeries(crlSeries);
					crlS.setCurrentSerial(result.getLong(Constants.CRL__COL__CURRENT_SERIAL));
					crl.setCrlSeries(crlS);
					crl.setNextCrl(result.getTimestamp(Constants.CRL__COL__NEXT_CRL));
					crl.setStartPeriod(result.getTimestamp(Constants.CRL__COL__START_PERIOD));
					crl.setCrl(result.getBytes(Constants.CRL__COL__CRL_BYTES));

					this.crl = crl;
				}
			} catch (SQLException e) {
				throw new ProviderException("Illegal sql" + e.getMessage() + " " + sql, e);
			} finally {
				closeStatement(st);
				closeConnection(con);
				closeResultSet(result);
			}
		}
		return crl;
	}

	@Override
	public RevokedCertificate[] getRevokedCerts() {
		String sql = String.format("SELECT * FROM %s;", Constants.REVOKED__TABLE);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<RevokedCertificate> tmp = new ArrayList<>();
			while (result.next()) {
				RevokedCertificate info = new RevokedCertificate();
				info.setCertId(result.getBytes(Constants.REVOKED__COL__ID));
				info.setRevocationTime(result.getTimestamp(Constants.REVOKED__COL__TIME));
				tmp.add(info);
			}
			return tmp.toArray(new RevokedCertificate[tmp.size()]);
		} catch (SQLException e) {
			logger.error(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
		return new RevokedCertificate[0];

	}

	@Override
	public void revoke(RevokedCertificate cert) {
		String sql = String.format("INSERT INTO %s (%s, %s) VALUES(?,?);", Constants.REVOKED__TABLE,
				Constants.REVOKED__COL__ID, Constants.REVOKED__COL__TIME);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, cert.getCertId());
			st.setTimestamp(2, cert.getRevocationTime());
			st.execute();
			clearCache();
		} catch (SQLException e) {
			logger.error(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public boolean removeRevokedCertificate(byte[] certId) {
		String sql = String.format("DELETE FROM %s WHERE %s = ?;", Constants.REVOKED__TABLE,
				Constants.REVOKED__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
			st.setBytes(1, certId);
			st.execute();
			clearCache();
			return true;
		} catch (SQLException e) {
			logger.warn("Can not prepare statement", e);
			return false;
		} finally {
			closeStatement(st);
			closeConnection(con);
		}

	}

	@Override
	public Crl get() throws ProviderException {
		return get(configProvider.getInt("crlSeries", 1));
	}

	@Override
	public CrlCertificate getCertificate() {
		if (cert == null) {
			logger.debug("Try to fetch certificate from database");
			String sql = String.format("SELECT * FROM %s;", Constants.CRL_CERTIFICATE__TABLE);
			Connection con = getConnection();
			PreparedStatement st = null;
			ResultSet result = null;
			try {
				st = con.prepareStatement(sql);
				st.execute();
				result = st.getResultSet();

				if (!result.first()) {
					logger.error("No CA certificate on DB found!");
					throw new NoSuchElementException("No CA certificate on DB found!");
				} else {
					cert = new CrlCertificate();
					cert.setCertId(result.getBytes(Constants.CA_CERTIFICATE__COL__ID));
					cert.setCertificate(result.getBytes(Constants.CA_CERTIFICATE__COL__CERTIFICATE));
					cert.setCreationTime(result.getTimestamp(Constants.CA_CERTIFICATE__COL__CREATION_TIME));
					cert.setExpirationTime(result
							.getTimestamp(Constants.CA_CERTIFICATE__COL__EXPIRATION_TIME));
					cert.setSignerCertId(result.getBytes(Constants.CA_CERTIFICATE__COL__SIGNER_CERT_ID));
					cert.setStartTime(result.getTimestamp(Constants.CA_CERTIFICATE__COL__START_TIME));
					cert.setSubjectName(result.getString(Constants.CA_CERTIFICATE__COL__SUBJECT_NAME));
				}
			} catch (SQLException e) {
				logger.debug("invalid sql statement" + sql, e);
			} finally {
				closeStatement(st);
				closeConnection(con);
				closeResultSet(result);
			}
		}
		return cert;
	}

	public void setCertificate(CrlCertificate cert) {
		logger.debug("Try to store certificate into database");
		String sql = String.format("INSERT INTO %s (%s, %s, %s, %s, %s, %s, %s) VALUES(?,?, ?, ?, ?, ?,?);",
				Constants.CRL_CERTIFICATE__TABLE, Constants.CRL_CERTIFICATE__COL__ID,
				Constants.CRL_CERTIFICATE__COL__CERTIFICATE, Constants.CRL_CERTIFICATE__COL__CREATION_TIME,
				Constants.CRL_CERTIFICATE__COL__EXPIRATION_TIME,
				Constants.CRL_CERTIFICATE__COL__SIGNER_CERT_ID, Constants.CRL_CERTIFICATE__COL__START_TIME,
				Constants.CRL_CERTIFICATE__COL__SUBJECT_NAME);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, cert.getCertId());
			st.setBytes(2, cert.getCertificate());
			st.setTimestamp(3, new Timestamp(System.currentTimeMillis()));
			st.setTimestamp(4, cert.getExpirationTime());
			st.setBytes(5, cert.getSignerCertId());
			st.setTimestamp(6, cert.getStartTime());
			st.setString(7, cert.getSubjectName());
			st.execute();
		} catch (SQLException e) {
			logger.error(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
		this.cert = cert;
	}

	private void clearCache() {
		crl = null;
	}

	@Override
	public boolean isRevoked(byte[] certId) {

		String sql = String.format("SELECT * FROM %s WHERE %s = ?;", Constants.REVOKED__TABLE,
				Constants.REVOKED__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, certId);
			st.execute();
			result = st.getResultSet();

			if (!result.first()) {
				if (logger.isTraceEnabled())
					logger.trace("Certificate with certId=" + ByteUtils.getHex(certId) + " is not revoked");
				return false;
			} else {
				if (logger.isTraceEnabled())
					logger.trace("Certificate with certId=" + ByteUtils.getHex(certId) + " is revoked");
				return true;
			}
		} catch (SQLException e) {
			logger.error(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
		return false;
	}

}
