package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttributeTypeImpl.SubjectAttributeType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

public class SubjectAttribute extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private SubjectAttributeType type;

	@Internal(order = 1)
	private PublicKey key;

	@Internal(order = 2)
	private EccPoint rv;

	@Internal(order = 3)
	private SubjectAssurance assuranceLevel;

	@Internal(order = 4)
	private IntX[] itsAidList;

	@Internal(order = 5)
	private ItsAidSsp[] itsAidSspList;

	@Internal(order = 6)
	private ItsAidPriority[] itsAidPriorityList;

	@Internal(order = 7)
	private ItsAidPrioritySsp[] itsAidPrioritySspList;

	@Internal(order = 8)
	private Opaque otherAttribute;

	// ---- constructors ----

	public SubjectAttribute() {
	}

	public SubjectAttribute(PublicKey key, SubjectAttributeType type) {
		if (type != SubjectAttributeType.ENCRYPTION_KEY && type != SubjectAttributeType.VERIFICATION_KEY)
			throw new IllegalArgumentException("Unexpected type for provided key");
		setKey(key, type);
	}

	public SubjectAttribute(SubjectAssurance assuranceLevel) {
		this.type = SubjectAttributeType.ASSURANCE_LEVEL;
		this.assuranceLevel = assuranceLevel;
	}

	public SubjectAttribute(EccPoint reconstructionValue) {
		this.type = SubjectAttributeType.RECONSTRUCTION_VALUE;
		this.rv = reconstructionValue;
	}

	public SubjectAttribute(IntX[] itsAidList) {
		this.type = SubjectAttributeType.ITS_AID_LIST;
		this.itsAidList = itsAidList;
	}

	public SubjectAttribute(ItsAidSsp[] itsAidSspList) {
		this.type = SubjectAttributeType.ITS_AID_SSP_LIST;
		this.itsAidSspList = itsAidSspList;
	}

	public SubjectAttribute(ItsAidPriority[] itsAidPriorityList) {
		this.type = SubjectAttributeType.PRIORITY_ITS_AID_LIST;
		this.itsAidPriorityList = itsAidPriorityList;
	}

	public SubjectAttribute(ItsAidPrioritySsp[] itsAidPrioritySspList) {
		this.type = SubjectAttributeType.PRIORITY_SSP_LIST;
		this.itsAidPrioritySspList = itsAidPrioritySspList;
	}

	public SubjectAttribute(DataInputStream in) throws IOException {
		type = SubjectAttributeTypeImpl.getInstance().getEnumType(in.readByte());
		if (type == SubjectAttributeType.VERIFICATION_KEY || type == SubjectAttributeType.ENCRYPTION_KEY) {
			setKey(new PublicKey(in), type);
		} else if (type == SubjectAttributeType.ASSURANCE_LEVEL) {
			assuranceLevel = new SubjectAssurance(in);
		} else if (type == SubjectAttributeType.RECONSTRUCTION_VALUE) {
			// TODO: hardcoded algorithm correct?
			rv = new EccPoint(in, new UInt8(32), PublicKeyAlgorithm.ECIES_NISTP256);
		} else if (type == SubjectAttributeType.ITS_AID_LIST) {
			itsAidList = WaveUtils.getArrayFromStream(in, IntX.class);
		} else if (type == SubjectAttributeType.ITS_AID_SSP_LIST) {
			itsAidSspList = WaveUtils.getArrayFromStream(in, ItsAidSsp.class);
		} else if (type == SubjectAttributeType.PRIORITY_ITS_AID_LIST) {
			itsAidPriorityList = WaveUtils.getArrayFromStream(in, ItsAidPriority.class);
		} else if (type == SubjectAttributeType.PRIORITY_SSP_LIST) {
			itsAidPrioritySspList = WaveUtils.getArrayFromStream(in, ItsAidPrioritySsp.class);
		} else {
			otherAttribute = new Opaque(in);
		}
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public SubjectAttributeType getType() {
		return this.type;
	}

	public PublicKey getKey() {
		return this.key;
	}

	public SubjectAssurance getAssuranceLevel() {
		return this.assuranceLevel;
	}

	public IntX[] getItsAidList() {
		return this.itsAidList;
	}

	public ItsAidSsp[] getItsAidSspList() {
		return this.itsAidSspList;
	}

	public ItsAidPriority[] getItsAidPriorityList() {
		return this.itsAidPriorityList;
	}

	public ItsAidPrioritySsp[] getItsAidPrioritySspList() {
		return this.itsAidPrioritySspList;
	}

	public Opaque getOtherAttribute() {
		return this.otherAttribute;
	}

	// ---- setter ----

	public void setKey(PublicKey key, SubjectAttributeType keyType) {
		if (keyType != SubjectAttributeType.VERIFICATION_KEY
				&& keyType != SubjectAttributeType.ENCRYPTION_KEY)
			throw new IllegalArgumentException(
					"invalid subject attribute type -  use VerificationKey or EncryptionKey");

		if (type == SubjectAttributeType.VERIFICATION_KEY
				&& key.getAlgorithm() != PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256)
			throw new IllegalArgumentException("Unexpected public key algorithm of key: "
					+ key.getAlgorithm() + " != ECDSA_NISTP256_WITH_SHA256");

		if (type == SubjectAttributeType.ENCRYPTION_KEY
				&& key.getAlgorithm() != PublicKeyAlgorithm.ECIES_NISTP256)
			throw new IllegalArgumentException("Unexpected public key algorithm of key: "
					+ key.getAlgorithm() + " != ECIES_NISTP256");

		this.type = keyType;
		this.key = key;
	}

	public void setAssuranceLevel(SubjectAssurance assuranceLevel) {
		this.type = SubjectAttributeType.ASSURANCE_LEVEL;
		this.assuranceLevel = assuranceLevel;
	}

	public void setItsAidList(IntX[] itsAidList) {
		this.type = SubjectAttributeType.ITS_AID_LIST;
		this.itsAidList = itsAidList;
	}

	public void setItsAidSspList(ItsAidSsp[] itsAidSspList) {
		this.type = SubjectAttributeType.ITS_AID_SSP_LIST;
		this.itsAidSspList = itsAidSspList;
	}

	public void setItsAidPriorityList(ItsAidPriority[] itsAidPriorityList) {
		this.type = SubjectAttributeType.PRIORITY_ITS_AID_LIST;
		this.itsAidPriorityList = itsAidPriorityList;
	}

	public void setItsAidPrioritySspList(ItsAidPrioritySsp[] itsAidPrioritySspList) {
		this.type = SubjectAttributeType.PRIORITY_SSP_LIST;
		this.itsAidPrioritySspList = itsAidPrioritySspList;
	}

	public void setOtherAttribute(Opaque otherAttribute) {
		this.otherAttribute = otherAttribute;
	}

	@Override
	public String toString() {
		if (type == null)
			return "SubjectAttribute []";
		if (type == SubjectAttributeType.VERIFICATION_KEY || type == SubjectAttributeType.ENCRYPTION_KEY) {
			return "SubjectAttribute [" + key + "]";
		} else if (type == SubjectAttributeType.ASSURANCE_LEVEL) {
			return "SubjectAttribute [" + assuranceLevel + "]";
		} else if (type == SubjectAttributeType.RECONSTRUCTION_VALUE) {
			return "SubjectAttribute [" + rv + "]";
		} else if (type == SubjectAttributeType.ITS_AID_LIST) {
			return "SubjectAttribute [" + itsAidList + "]";
		} else if (type == SubjectAttributeType.ITS_AID_SSP_LIST) {
			return "SubjectAttribute [" + itsAidSspList + "]";
		} else if (type == SubjectAttributeType.PRIORITY_ITS_AID_LIST) {
			return "SubjectAttribute [" + itsAidPriorityList + "]";
		} else if (type == SubjectAttributeType.PRIORITY_SSP_LIST) {
			return "SubjectAttribute [" + itsAidPrioritySspList + "]";
		} else {
			return "SubjectAttribute [" + otherAttribute + "]";
		}

	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {

		if (type == null)
			throw new IllegalArgumentException("args may not be null");
		int written = SubjectAttributeTypeImpl.getInstance().writeData(out, type);
		if (type == SubjectAttributeType.VERIFICATION_KEY || type == SubjectAttributeType.ENCRYPTION_KEY) {
			if (key == null)
				throw new IllegalArgumentException("args may not be null");
			written += key.writeData(out);
		} else if (type == SubjectAttributeType.ASSURANCE_LEVEL) {
			if (assuranceLevel == null)
				throw new IllegalArgumentException("args may not be null");
			written += assuranceLevel.writeData(out);
		} else if (type == SubjectAttributeType.RECONSTRUCTION_VALUE) {
			if (rv == null)
				throw new IllegalArgumentException("args may not be null");
			written += rv.writeData(out);
		} else if (type == SubjectAttributeType.ITS_AID_LIST) {
			if (itsAidList == null)
				throw new IllegalArgumentException("args may not be null");
			written += WaveUtils.writeArrayToStream(out, itsAidList);
		} else if (type == SubjectAttributeType.ITS_AID_SSP_LIST) {
			if (itsAidSspList == null)
				throw new IllegalArgumentException("args may not be null");
			written += WaveUtils.writeArrayToStream(out, itsAidSspList);
		} else if (type == SubjectAttributeType.PRIORITY_ITS_AID_LIST) {
			if (itsAidPriorityList == null)
				throw new IllegalArgumentException("args may not be null");
			written += WaveUtils.writeArrayToStream(out, itsAidPriorityList);
		} else if (type == SubjectAttributeType.PRIORITY_SSP_LIST) {
			if (itsAidPrioritySspList == null)
				throw new IllegalArgumentException("args may not be null");
			written += WaveUtils.writeArrayToStream(out, itsAidPrioritySspList);
		} else {
			if (otherAttribute == null)
				throw new IllegalArgumentException("args may not be null");
			written += otherAttribute.writeData(out);
		}

		return written;
	}
}