#include "ivim_types.hh"

#include "ivim_layer_factory.hh"

#include "loggers.hh"

ivim_layer::ivim_layer(const std::string & p_type, const std::string & param) : t_layer<LibItsIvim__TestSystem::IvimPort>(p_type), _params(), _codec() {
  loggers::get_instance().log(">>> ivim_layer::ivim_layer: %s, %s", to_string().c_str(), param.c_str());
  // Setup parameters
  params::convert(_params, param);
}

void ivim_layer::sendMsg(const LibItsIvim__TestSystem::IvimReq& p, params& params){
  loggers::get_instance().log_msg(">>> ivim_layer::sendMsg: ", p);

  // Encode IVIM PDU
  OCTETSTRING data;
  if (_codec.encode(p.msgOut(), data) == -1) {
    loggers::get_instance().warning("ivim_layer::sendMsg: Encodeing failure");
    return;
  }
  send_data(data, params);
}

void ivim_layer::send_data(OCTETSTRING& data, params& params) {
  loggers::get_instance().log_msg(">>> ivim_layer::send_data: ", data);
  //params.log();
  send_to_all_layers(data, params);
}

void ivim_layer::receive_data(OCTETSTRING& data, params& params)
{
  loggers::get_instance().log_msg(">>> ivim_layer::receive_data: ", data);
  
  // Sanity check
  if (*(static_cast<const unsigned char*>(data)+ 1) != 0x06) { // Check that received packet has IVI message id - See ETSI TS 102 894
    // Not a IVIM message, discard it
    loggers::get_instance().warning("ivim_layer::receive_data: Wrong message id: 0x%02x", *(static_cast<const unsigned char*>(data)+ 1));
    return;
  }
  
  // Decode the payload
  LibItsIvim__TestSystem::IvimInd p;
  _codec.decode(data, p.msgIn());
  if (!p.msgIn().is_bound()) {
    // Discard it
    return;
  } // else, continue
  

  // Process lower layer data
  // gnNextHeader
  params::const_iterator it = params.find(params::gn_next_header);
  if (it != params.cend()) {
    p.gnNextHeader() = std::stoi(it->second);
  } else {
    p.gnNextHeader().set_to_omit();
  }
  // gnHeaderType
  it = params.find(params::gn_header_type);
  if (it != params.cend()) {
    p.gnHeaderType() = std::stoi(it->second);
  } else {
    p.gnHeaderType().set_to_omit();
  }
  // gnHeaderSubtype
  it = params.find(params::gn_header_sub_type);
  if (it != params.cend()) {
    p.gnHeaderSubtype() = std::stoi(it->second);
  } else {
    p.gnHeaderSubtype().set_to_omit();
  }
  // gnLifetime
  it = params.find(params::gn_lifetime);
  if (it != params.cend()) {
    p.gnLifetime() = std::stoi(it->second);
  } else {
    p.gnLifetime().set_to_omit();
  }
  // gnTrafficClass
  it = params.find(params::gn_traffic_class);
  if (it != params.cend()) {
    p.gnTrafficClass() = std::stoi(it->second);
  } else {
    p.gnTrafficClass().set_to_omit();
  }
  // btpDestinationPort
  it = params.find(params::btp_destination_port);
  if (it != params.cend()) {
    p.btpDestinationPort() = std::stoi(it->second);
  } else {
    p.btpDestinationPort().set_to_omit();
  }
  // btpInfo
  it = params.find(params::btp_info);
  if (it != params.cend()) {
    p.btpInfo() = std::stoi(it->second);
  } else {
    p.btpInfo().set_to_omit();
  }
  // ssp
  it = params.find(_params[params::its_aid]);
  if (it != params.cend()) {
    loggers::get_instance().log("denm_layer::receive_data: ssp=%s", it->second.c_str());
    p.ssp() = oct2bit(str2oct(CHARSTRING(it->second.c_str())));
  } else {
    p.ssp().set_to_omit();
  }
  // its_aid
  if (it != params.cend()) {
    p.its__aid() = std::stoi(it->second);
  } else {
    p.its__aid().set_to_omit();
  }
  
  // Pass it to the ports if amy
  to_all_upper_ports(p, params);
}

ivim_layer_factory ivim_layer_factory::_f;
