package de.fraunhofer.sit.c2x;

import java.io.IOException;
import java.math.BigInteger;

import org.bouncycastle.crypto.params.ECPrivateKeyParameters;

import de.fraunhofer.sit.c2x.pki.ca.crypto.CryptoUtils;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPoint;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPointTypeImpl.EccPointType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKey;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Signature;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils.EtsiCryptoUtils;

public class CryptoLib {

	public static byte[] hashWithSha256(byte[] toBeHashedData) {
		return CryptoUtils.sha256(toBeHashedData);
	}

	public static byte[] signWithEcdsaNistp256WithSha256(byte[] toBeSignedData, BigInteger privateKey)
			throws Exception {
		byte[] privateKeyBytes = privateKey.toByteArray();

		if(privateKeyBytes.length == 33 && privateKeyBytes[0] == 0x00){
			// remove first byte if 0x00
			byte[] privateKeyBytesTmp = new byte[32];
			for(int i=0; i < privateKeyBytesTmp.length; i++){
				privateKeyBytesTmp[i] = privateKeyBytes[i+1];
			}
			privateKeyBytes = privateKeyBytesTmp;
		}
		if (privateKeyBytes.length != 32)
			throw new Exception("Invalid length of private key: " + privateKeyBytes.length + " != 32");

		ECPrivateKeyParameters privateKeyParameters = CryptoUtils
				.bytesToEcPrivateKeyParameters(privateKeyBytes);

		Signature signature = EtsiCryptoUtils.sign(toBeSignedData,
				PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, privateKeyParameters);

		byte[] signatureBytes = WaveUtils.getBytesFromWaveType(signature);

		return signatureBytes;
	}

	public static boolean verifyWithEcdsaNistp256WithSha256(byte[] toBeVerifiedData,
			byte[] ts103097SignatureEncodedAsByteArray, byte[] ecdsaNistp256PublicKeyX,
			byte[] ecdsaNistp256PublicKeyY) throws Exception {

		Signature signature = WaveUtils.getElementFromBytes(ts103097SignatureEncodedAsByteArray,
				Signature.class);

		PublicKey pubKey = getPublicKeyFromBytes(ecdsaNistp256PublicKeyX, ecdsaNistp256PublicKeyY);

		return EtsiCryptoUtils.verify(toBeVerifiedData, signature, pubKey.getPublicKey()
				.toECPublicKeyParameters());
	}
	
	public static EcdsaP256KeyPair generateKeyPair(){
		return new EcdsaP256KeyPair();
	}

	private static PublicKey getPublicKeyFromBytes(byte[] pubKeyX, byte[] pubKeyY) throws Exception {
		if (pubKeyY != null && pubKeyY.length > 0) {

			if (pubKeyX.length < 32)
				throw new Exception("Unexpected length of public key X: " + pubKeyX.length + " < 32");

			if (pubKeyY.length < 32)
				throw new Exception("Unexpected length of public key Y: " + pubKeyY.length + " < 32");

			return getUncompressedVerificationPublicKeyFromBytes(pubKeyX, pubKeyY);
		} else {
			if (pubKeyX.length < 32)
				throw new Exception("Unexpected length of public key X: " + pubKeyX.length + " < 32");

			return getCompressedPublicKeyFromBytes(pubKeyX);
		}

	}

	private static PublicKey getUncompressedVerificationPublicKeyFromBytes(byte[] pubKeyX, byte[] pubKeyY)
			throws IOException {
		EccPoint eccPublicKey = new EccPoint(EccPointType.UNCOMPRESSED, pubKeyX, pubKeyY,
				PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, 32);
		return new PublicKey(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, eccPublicKey);
	}

	private static PublicKey getCompressedPublicKeyFromBytes(byte[] pubKeyX) throws IOException {
		EccPoint eccPublicKey = null;
		int lsb = (byte) (pubKeyX[31] & 0x01);
		if (lsb == 0)
			eccPublicKey = new EccPoint(EccPointType.COMPRESSED_LSB_Y_0, pubKeyX, null,
					PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, 32);
		else
			eccPublicKey = new EccPoint(EccPointType.COMPRESSED_LSB_Y_1, pubKeyX, null,
					PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, 32);
		return new PublicKey(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, eccPublicKey);
	}
}
