#include <string>
#include <memory>

#include "UpperTesterIvimCodec.hh"
#include "loggers.hh"

#include "UpperTesterTypes.hh"

#include "IVIMCodec.hh"

template <typename T_type> class OPTIONAL;
class TTCN_EncDec;

int UpperTesterIvimCodec::encode (const Record_Type& msg, OCTETSTRING& data)
{
  loggers::get_instance().log_msg(">>> UpperTesterIvimCodec::encode: ", (const Record_Type&)msg);
  
  TTCN_EncDec::clear_error();
  TTCN_Buffer encoding_buffer;
  if (std::string(msg.get_descriptor()->name).compare("@LibItsIvim_TypesAndValues.UtIvimInitialize") == 0) {
    encoding_buffer.put_c(static_cast<const unsigned char>(UpperTesterIvimCodec::c_utIvimInitialize));
  } else if (std::string(msg.get_descriptor()->name).compare("@LibItsIvim_TypesAndValues.UtIvimTrigger") == 0) {
    encoding_buffer.put_c(static_cast<const unsigned char>(UpperTesterIvimCodec::c_utIvimTrigger));
    // See ETSI TR 103 099 V1.4.1 Clause C.9.1 GenerateIvimEvent
    unsigned char flags = 0x00;
    const LibItsIvim__TypesAndValues::UtIvimTrigger &r = static_cast<const LibItsIvim__TypesAndValues::UtIvimTrigger &>(msg);
    if (r.validFrom().is_present()) { // V
      flags |= 0x80;
    }
    if (r.validTo().is_present()) { // R
      flags |= 0x40;
    }
    if (r.repetitionInterval().is_present()) { // T
      flags |= 0x20;
    }
    encoding_buffer.put_c(flags);
    loggers::get_instance().log_to_hexa("UpperTesterIvimCodec::encode: flags=", encoding_buffer);
    // Continue with encoding
  } else if (std::string(msg.get_descriptor()->name).compare("@LibItsIvim_TypesAndValues.UtIvimUpdate") == 0) {
    encoding_buffer.put_c(static_cast<const unsigned char>(UpperTesterIvimCodec::c_utIvimUpdate));
    // See ETSI TR 103 099 V1.4.1 Clause C.9.2	UpdateIvimEvent
    unsigned char flags = 0x00;
    const LibItsIvim__TypesAndValues::UtIvimUpdate &r = static_cast<const LibItsIvim__TypesAndValues::UtIvimUpdate &>(msg);
    if (r.timeStamp().is_present()) { // T
      flags |= 0x80;
    }
    if (r.validFrom().is_present()) { // V
      flags |= 0x40;
    }
    if (r.validTo().is_present()) { // R
      flags |= 0x20;
    }
    encoding_buffer.put_c(flags);
    loggers::get_instance().log_to_hexa("UpperTesterIvimCodec::encode: flags=", encoding_buffer);
    // Continue with encoding
  } else if (std::string(msg.get_descriptor()->name).compare("@LibItsIvim_TypesAndValues.UtIvimTermination") == 0) {
    // See ETSI TR 103 099 V1.4.1 Clause C.9.3	TerminateIvimEvent
    encoding_buffer.put_c(static_cast<const unsigned char>(UpperTesterIvimCodec::c_utIvimTermination));
  } else { // Error
    data = OCTETSTRING();
    loggers::get_instance().log("<<< UpperTesterIvimCodec::encode: Return -1");
    return -1;
  }
  encode_(msg, *msg.get_descriptor(), encoding_buffer);
  data = OCTETSTRING(encoding_buffer.get_len(), encoding_buffer.get_data());
  loggers::get_instance().log_msg("<<< UpperTesterIvimCodec::encode: data=", data);
  return 0;
}

int UpperTesterIvimCodec::encode_ (const Base_Type& type, const TTCN_Typedescriptor_t& field_descriptor, TTCN_Buffer& encoding_buffer)
{
  loggers::get_instance().log(">>> UpperTesterIvimCodec::encode_: processing %s/%s/%p", type.get_descriptor()->name, field_descriptor.name, dynamic_cast<const Record_Type*>(&type));
  loggers::get_instance().log_msg(">>> UpperTesterIvimCodec::encode_: ", type);
  
  if (dynamic_cast<const Record_Type*>(&type) != NULL) {
    const Record_Type& r = (const Record_Type &)type;
    loggers::get_instance().log("UpperTesterIvimCodec::encode_: processing Record_Type %s", r.get_descriptor()->name);
    for(int i = 0; i < r.get_count(); i++) {
      loggers::get_instance().log("UpperTesterIvimCodec::encode_: processing %s/%s/%s - %d (1 ==> use dynamic_cast<const OPTIONAL<...>) - %d", r.fld_name(i), r.fld_descr(i)->name, r.get_at(i)->get_descriptor()->name, r.get_at(i)->is_optional(), r.get_at(i)->is_present());
      if (r.get_at(i)->is_present()) {
        if (encode_(*r.get_at(i), *r.fld_descr(i), encoding_buffer) == -1) {
          loggers::get_instance().warning("UpperTesterIvimCodec::encode_: -1 result code returned");
          return -1;
        }
      } else {
        std::string s(r.fld_descr(i)->name);
        if (
            (s.rfind(".timeStamp") != string::npos) ||
            (s.rfind(".validFrom") != string::npos) ||
            (s.rfind(".validTo") != string::npos) ||
            (s.rfind(".repetitionInterval") != string::npos)
            ) { // Pad with six bytes
          loggers::get_instance().log("UpperTesterIvimCodec::encode_: Padding for %s", field_descriptor.name);
          encoding_buffer.put_c((unsigned char)0x00);
          encoding_buffer.put_c((unsigned char)0x00);
          encoding_buffer.put_c((unsigned char)0x00);
          encoding_buffer.put_c((unsigned char)0x00);
          encoding_buffer.put_c((unsigned char)0x00);
          encoding_buffer.put_c((unsigned char)0x00);
        }
      }
    } // End of 'for' statement
  } else {
    std::string s(field_descriptor.name);
    loggers::get_instance().log("UpperTesterIvimCodec::encode_: field to process %s", s.c_str());
    if (s.rfind(".iviIdentificationNumber") != string::npos) {
      const INTEGER& i = static_cast<const INTEGER&>(type);
      loggers::get_instance().log_msg("UpperTesterIvimCodec::encode_: i=", i);
      encoding_buffer.put_string(int2oct(i, 2));
    } else if (
               (s.rfind(".timeStamp") != string::npos) ||
               (s.rfind(".validTo") != string::npos) ||
               (s.rfind(".validFrom") != string::npos) ||
               (s.rfind(".repetitionInterval") != string::npos)
              ) {
      const INTEGER& i = static_cast<const INTEGER&>(type);
      loggers::get_instance().log_msg("UpperTesterIvimCodec::encode_: i=", i);
      loggers::get_instance().log_msg("UpperTesterIvimCodec::encode_: int2oct=", int2oct(i, 6));
      encoding_buffer.put_string(int2oct(i, 6));
    } else {
      loggers::get_instance().log("UpperTesterIvimCodec::encode_ (else): processing type %s/%s", type.get_descriptor()->name, field_descriptor.name);
      type.encode(field_descriptor, encoding_buffer, TTCN_EncDec::CT_RAW);
    }
  }
  
  loggers::get_instance().log_to_hexa("<<< UpperTesterIvimCodec::encode_: encoding_buffer=", encoding_buffer);
  return 0;
}

int UpperTesterIvimCodec::decode (const OCTETSTRING& data, Record_Type& msg, Params* params)
{
  TTCN_EncDec::clear_error();
  TTCN_Buffer decoding_buffer(data);
  _params = params;
  
  loggers::get_instance().log_to_hexa(">>> UpperTesterIvimCodec::decode: decoding_buffer=", decoding_buffer);
  //decode_(msg, *msg.get_descriptor(), decoding_buffer);
  
  loggers::get_instance().log_msg("<<< UpperTesterIvimCodec::decode: ", (const Record_Type&)msg);
  return 0;
}

std::unique_ptr<Base_Type> UpperTesterIvimCodec::decode (const OCTETSTRING& data, Params* params) {

  std::unique_ptr<Base_Type> result;
  
  const unsigned char* ptr = static_cast<const unsigned char*>(data);
  if (*ptr != UpperTesterIvimCodec::c_utIvimEventInd) {
    LibItsIvim__TypesAndValues::UtIvimResults res;
    decode(data, res, params);
    result.reset((Base_Type *)res.clone());
  } else {
    LibItsIvim__TypesAndValues::UtIvimEventInd ind;
    decode(data, ind, params);
    result.reset((Base_Type *)ind.clone());
  }

  return result;
}

int UpperTesterIvimCodec::decode (const OCTETSTRING& data, LibItsIvim__TypesAndValues::UtIvimResults& msg, Params* params)
{
  loggers::get_instance().log_to_hexa(">>> UpperTesterIvimCodec::decode: decoding_buffer=", data);
  
  const unsigned char* ptr = static_cast<const unsigned char*>(data);
  if (*ptr == UpperTesterIvimCodec::c_utIvimInitializeResult) {
    msg.utIvimInitializeResult() = BOOLEAN(*(ptr + 1) == 0x01);
  } else if (*ptr == UpperTesterIvimCodec::c_utIvimTerminationResult) { 
    msg.utIvimTerminationResult() = BOOLEAN(*(ptr + 1) == 0x01);
  } else if (*ptr == UpperTesterIvimCodec::c_utIvimTriggerResult) {
    LibItsIvim__TypesAndValues::UtIvimTriggerResult r;
    ptr += 1;
    r.result() = BOOLEAN(*ptr == 0x01);
    ptr += 1;
    OCTETSTRING os(2, ptr);
    ptr += 2;
    r.iviIdentificationNumber() = oct2int(os);
    msg.utIvimTriggerResult() = r;
  } else if (*ptr == UpperTesterIvimCodec::c_utIvimUpdateResult) {
    LibItsIvim__TypesAndValues::UtIvimUpdateResult r;
    ptr += 1;
    r.result() = BOOLEAN(*ptr == 0x01);
    ptr += 1;
    OCTETSTRING os(2, ptr);
    ptr += 2;
    r.iviIdentificationNumber() = oct2int(os);
    msg.utIvimUpdateResult() = r;
  } else {
    TTCN_warning("UpperTesterIvimCodec::decode: Unsupported result");
  }
  
  loggers::get_instance().log_msg("<<< UpperTesterIvimCodec::decode", msg);
  return 0;
}

int UpperTesterIvimCodec::decode (const OCTETSTRING& data, LibItsIvim__TypesAndValues::UtIvimEventInd& msg, Params* params)
{
  loggers::get_instance().log_to_hexa(">>> UpperTesterIvimCodec::decode: decoding_buffer=", data);

  const unsigned char* ptr = static_cast<const unsigned char*>(data) + 1;
  OCTETSTRING os(2, ptr);
  ptr += 2;
  unsigned int length = oct2int(os);
  loggers::get_instance().log("UpperTesterIvimCodec::decode: IVIM message length=%d", length);
  os = OCTETSTRING(length, ptr);
  loggers::get_instance().log_to_hexa("UpperTesterIvimCodec::decode: IVIM message=", os);
  IVIMCodec codec;
  codec.decode(os, msg.iviMsg(), params);
  
  loggers::get_instance().log_msg("<<< UpperTesterIvimCodec::decode", msg);
  return 0;
}
