#pragma once

#include <string>
#include <vector>

#include "security_ecc.hh"

namespace IEEE1609dot2 {
  class CertificateBase;
};

/*!
 * \class security_db_record
 * \brief This class provides security record description for security database
 */
class security_db_record {
  
  ec_elliptic_curves _algorithm;
  std::string _certificate_id;                    /*!< certificate storage*/
  std::vector<unsigned char> _certificate;        /*!< Certificate storage */
  std::vector<unsigned char> _hashed_id_issuer;   /*!< Certificate hash id 8 storage */
  std::vector<unsigned char> _hashed_id;          /*!< Certificate hash id 8 storage */
  std::vector<unsigned char> _pr_key;             /*!< Private key storage */
  std::vector<unsigned char> _pu_key_x;           /*!< Public key X-coordinate storage */
  std::vector<unsigned char> _pu_key_y;           /*!< Public key Y-coordinate storage */
  std::vector<unsigned char> _pr_enc_key;         /*!< Private encryption key storage */
  std::vector<unsigned char> _pu_enc_key_x;       /*!< Public encryption key X-coordinate storage */
  std::vector<unsigned char> _pu_enc_key_y;       /*!< Public encryption key Y-coordinate storage */
  bool _to_be_saved;                              /*!< Flag to indicate if the record shall be saved, i.e. has been modified */
  IEEE1609dot2::CertificateBase* _decoded_certificate;
  
public: /*! \publicsection */
  /*!
   * \brief Default private ctor
   */
  explicit security_db_record(): _algorithm{ec_elliptic_curves::nist_p_256}, _certificate_id(), _certificate(), _hashed_id_issuer(), _hashed_id(), _pr_key(), _pu_key_x(), _pu_key_y(), _pr_enc_key(), _pu_enc_key_x(), _pu_enc_key_y(), _to_be_saved(false), _decoded_certificate(nullptr) { };
  security_db_record(const std::string& p_certificate_id, const std::vector<unsigned char>& p_certificate, const IEEE1609dot2::CertificateBase& p_decoded_certificate, const std::vector<unsigned char>& p_hashed_id_issuer, const std::vector<unsigned char>& p_hashed_id, const std::vector<unsigned char>& p_pr_key, const std::vector<unsigned char>& p_pu_key_x, const std::vector<unsigned char>& p_pu_key_y, const std::vector<unsigned char>& p_pr_enc_key, const std::vector<unsigned char>& p_pu_enc_key_x, const std::vector<unsigned char>& p_pu_enc_key_y, const bool p_to_be_saved = true);
  
  /*!
   * \brief Default private dtor
   */
  ~security_db_record() { if (_decoded_certificate != nullptr) { delete _decoded_certificate; } };

  const IEEE1609dot2::CertificateBase& decoded_certificate() const;
  
  inline const std::string& certificate_id() const { return _certificate_id; };
  inline const std::vector<unsigned char>& certificate() const { return _certificate; };
  inline const std::vector<unsigned char>& hashed_id() const { return _hashed_id; };
  inline const std::vector<unsigned char>& issuer() const { return _hashed_id_issuer; };
  inline const std::vector<unsigned char>& private_key() const { return _pr_key; };
  inline const std::vector<unsigned char>& public_key_x() const { return _pu_key_x; };
  inline const std::vector<unsigned char>& public_key_y() const { return _pu_key_y; };
  inline const std::vector<unsigned char>& private_enc_key() const { return _pr_enc_key; };
  inline const std::vector<unsigned char>& public_enc_key_x() const { return _pu_enc_key_x; };
  inline const std::vector<unsigned char>& public_enc_key_y() const { return _pu_enc_key_y; };
}; // End of class security_db_record
