package de.fraunhofer.sit.c2x.pki.ca.validator.pseudonym.intervaltree;

import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class Node<DataType> {
	private SortedMap<Interval<DataType>, List<Interval<DataType>>> intervals;
	private long center;
	private Node<DataType> leftNode;
	private Node<DataType> rightNode;

	public Node() {
		intervals = new TreeMap<Interval<DataType>, List<Interval<DataType>>>();
		center = 0;
		leftNode = null;
		rightNode = null;
	}

	public Node(List<Interval<DataType>> intervalList) {

		intervals = new TreeMap<Interval<DataType>, List<Interval<DataType>>>();

		SortedSet<Long> endpoints = new TreeSet<Long>();

		for (Interval<DataType> interval : intervalList) {
			endpoints.add(interval.getStart());
			endpoints.add(interval.getEnd());
		}

		long median = getMedian(endpoints);
		center = median;

		List<Interval<DataType>> left = new ArrayList<Interval<DataType>>();
		List<Interval<DataType>> right = new ArrayList<Interval<DataType>>();

		for (Interval<DataType> interval : intervalList) {
			if (interval.getEnd() < median)
				left.add(interval);
			else if (interval.getStart() > median)
				right.add(interval);
			else {
				List<Interval<DataType>> posting = intervals.get(interval);
				if (posting == null) {
					posting = new ArrayList<Interval<DataType>>();
					intervals.put(interval, posting);
				}
				posting.add(interval);
			}
		}

		if (left.size() > 0)
			leftNode = new Node<DataType>(left);
		if (right.size() > 0)
			rightNode = new Node<DataType>(right);
	}

	/**
	 * Perform a stabbing query on the node
	 * 
	 * @param time
	 *            the time to query at
	 * @return all intervals containing time
	 */
	public List<Interval<DataType>> stab(long time) {
		List<Interval<DataType>> result = new ArrayList<Interval<DataType>>();

		for (Entry<Interval<DataType>, List<Interval<DataType>>> entry : intervals.entrySet()) {
			if (entry.getKey().contains(time))
				for (Interval<DataType> interval : entry.getValue())
					result.add(interval);
			else if (entry.getKey().getStart() > time)
				break;
		}

		if (time < center && leftNode != null)
			result.addAll(leftNode.stab(time));
		else if (time > center && rightNode != null)
			result.addAll(rightNode.stab(time));
		return result;
	}

	/**
	 * Perform an interval intersection query on the node
	 * 
	 * @param target
	 *            the interval to intersect
	 * @return all intervals containing time
	 */
	public List<Interval<DataType>> query(Interval<?> target) {
		List<Interval<DataType>> result = new ArrayList<Interval<DataType>>();

		for (Entry<Interval<DataType>, List<Interval<DataType>>> entry : intervals.entrySet()) {
			if (entry.getKey().intersects(target))
				for (Interval<DataType> interval : entry.getValue())
					result.add(interval);
			else if (entry.getKey().getStart() > target.getEnd())
				break;
		}

		if (target.getStart() < center && leftNode != null)
			result.addAll(leftNode.query(target));
		if (target.getEnd() > center && rightNode != null)
			result.addAll(rightNode.query(target));
		return result;
	}

	public long getCenter() {
		return center;
	}

	public void setCenter(long center) {
		this.center = center;
	}

	public Node<DataType> getLeft() {
		return leftNode;
	}

	public void setLeft(Node<DataType> left) {
		this.leftNode = left;
	}

	public Node<DataType> getRight() {
		return rightNode;
	}

	public void setRight(Node<DataType> right) {
		this.rightNode = right;
	}

	/**
	 * @param set
	 *            the set to look on
	 * @return the median of the set, not interpolated
	 */
	private Long getMedian(SortedSet<Long> set) {
		int i = 0;
		int middle = set.size() / 2;
		for (Long point : set) {
			if (i == middle)
				return point;
			i++;
		}
		return null;
	}

	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append(center + ": ");
		for (Entry<Interval<DataType>, List<Interval<DataType>>> entry : intervals.entrySet()) {
			sb.append("[" + entry.getKey().getStart() + "," + entry.getKey().getEnd() + "]:{");
			for (Interval<DataType> interval : entry.getValue()) {
				sb.append("(" + interval.getStart() + "," + interval.getEnd() + "," + interval.getData()
						+ ")");
			}
			sb.append("} ");
		}
		return sb.toString();
	}
}
