package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.CallBack;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Configuration;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;

public class JDBCConfigProvider extends AbstractMysqlConnection implements ConfigProvider {

	// List of call back interfaces that are called if the config id "String" is
	// updated
	private Map<String, ArrayList<CallBack>> callBacks = new HashMap<String, ArrayList<CallBack>>();

	@Override
	public String getName() {
		return "Config provider";
	}

	@Override
	public void registerCallBack(String configId, CallBack callBackClass) {
		boolean configIdFound = false;
		for (Configuration conf : getAll()) {
			if (conf.getId().equals(configId)) {
				configIdFound = true;
				break;
			}
		}
		if (configIdFound == false)
			throw new NoSuchElementException("ConfigId " + configId + " not available in DB");

		ArrayList<CallBack> list = callBacks.get(configId);
		if (list == null)
			list = new ArrayList<CallBack>();
		list.add(callBackClass);
		callBacks.put(configId, list);

		if (logger.isDebugEnabled())
			logger.debug("CallBack " + callBackClass.getClass().getName()
					+ " registered for changed on config id " + configId);
	}

	@Override
	public void deregisterCallBack(String configId, CallBack callBackClass) {
		if (callBacks.containsKey(configId)) {
			ArrayList<CallBack> list = callBacks.get(configId);
			if (list != null) {
				if (list.remove(callBackClass)) {
					if (logger.isDebugEnabled())
						logger.debug("CallBack " + callBackClass.getClass().getName()
								+ " deregistered for changed on config id " + configId);
					callBacks.put(configId, list);
				} else {
					if (logger.isDebugEnabled())
						logger.debug("CallBack " + callBackClass.getClass().getName() + "not found in list");
				}
			}
		}
	}

	public String get(String identifier) {
		String sql = String.format("SELECT * FROM %s WHERE %s = ?;", Constants.CONFIGURATION__TABLE,
				Constants.CONFIGURATION__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setString(1, identifier);
			st.execute();
			result = st.getResultSet();

			if (!result.first()) {
				return null;
			} else {
				return result.getString(Constants.CONFIGURATION__COL__VALUE);
			}
		} catch (SQLException e) {
			logger.error(e);
			return null;
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	public int getInt(String identifier) {
		return getInt(identifier, 0);
	}

	public int getInt(String identifier, int defaultValue) {
		String value = get(identifier);
		if (value == null || value.isEmpty())
			return defaultValue;
		try {
			return Integer.parseInt(value);
		} catch (Exception e) {
			return defaultValue;
		}

	}

	public void save(Configuration config) {

		boolean update = false;
		String value = get(config.getId());
		if (value != null)
			update = true;

		String sqlpattern = update ? "UPDATE %s SET %s = ? WHERE %s = ?;"
				: "INSERT INTO %s (%s, %s) VALUES (?,?);";
		String sql = String.format(sqlpattern, Constants.CONFIGURATION__TABLE,
				Constants.CONFIGURATION__COL__VALUE, Constants.CONFIGURATION__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setString(1, config.getValue());
			st.setString(2, config.getId());
			st.execute();

			// call registered functions if config id matches
			if (callBacks.containsKey(config.getId())) {
				// copy list of call backs because the list is manipulated while
				// the call backs are executed
				ArrayList<CallBack> callBacksCopy = new ArrayList<>();
				for (CallBack cb : callBacks.get(config.getId())) {
					callBacksCopy.add(cb);
				}
				for (CallBack cb : callBacksCopy) {
					cb.notifyEvent(config.getId() + "=" + config.getValue());
					if (logger.isDebugEnabled())
						logger.debug("CallBack " + cb.getClass().getName()
								+ " notified about changed configuration parameter");
				}
			}
		} catch (SQLException e) {
			logger.error(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public Configuration[] getAll() {
		String sql = String.format("SELECT * FROM %s;", Constants.CONFIGURATION__TABLE);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<Configuration> tmp = new ArrayList<>();
			while (result.next()) {
				Configuration conf = new Configuration(result.getString(Constants.CONFIGURATION__COL__ID),
						result.getString(Constants.CONFIGURATION__COL__VALUE));
				tmp.add(conf);
			}
			return tmp.toArray(new Configuration[tmp.size()]);
		} catch (SQLException e) {
			return new Configuration[0];
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public long getLong(String identifier, long defaultValue) {
		String value = get(identifier);
		if (value == null || value.isEmpty())
			return defaultValue;
		try {
			return Long.parseLong(value);
		} catch (Exception e) {
			return defaultValue;
		}
	}

	@Override
	public boolean getBoolean(String identifier, boolean defaultValue) {
		String value = get(identifier);
		if (value.isEmpty())
			return defaultValue;
		try {
			return Boolean.parseBoolean(value);
		} catch (Exception e) {
			return defaultValue;
		}
	}

}
