package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.math.BigInteger;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPointTypeImpl.EccPointType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.External;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class EcdsaSignature extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private EccPoint r;

	@Internal(order = 1)
	private Opaque s;

	// ---- fields ----

	@External
	private PublicKeyAlgorithm algorithm;

	@External
	private UInt8 fieldSize;

	// ---- constructors ----

	public EcdsaSignature() {
	}

	public EcdsaSignature(DataInputStream in, UInt8 fieldSize, PublicKeyAlgorithm algorithm)
			throws IOException {
		this.fieldSize = fieldSize;
		this.algorithm = algorithm;
		this.r = new EccPoint(in, fieldSize, algorithm);
		if (r.getType() != EccPointType.COMPRESSED_LSB_Y_0 && r.getType() != EccPointType.COMPRESSED_LSB_Y_1
				&& r.getType() != EccPointType.X_COORDINATE_ONLY) {
			throw new IllegalArgumentException("Invalid EccPointType for EcdsaSignature: " + r.getType());
		}
		this.s = new Opaque(in, fieldSize.get(), true);
	}

	public EcdsaSignature(EccPoint r, byte[] s, int fieldSize, PublicKeyAlgorithm algorithm)
			throws IOException {
		this.fieldSize = new UInt8(fieldSize);
		this.algorithm = algorithm;
		if (r.getType() != EccPointType.COMPRESSED_LSB_Y_0 && r.getType() != EccPointType.COMPRESSED_LSB_Y_1
				&& r.getType() != EccPointType.X_COORDINATE_ONLY) {
			throw new IllegalArgumentException("Invalid EccPointType for EcdsaSignature: " + r.getType());
		}
		this.r = r;
		this.s = new Opaque(s, true);
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public EccPoint getR() {
		return this.r;
	}

	public Opaque getS() {
		return this.s;
	}

	public PublicKeyAlgorithm getAlgorithm() {
		return this.algorithm;
	}

	public UInt8 getFieldSize() {
		return this.fieldSize;
	}

	// ---- setter ----

	public void setR(EccPoint r) {
		if (r.getType() != EccPointType.COMPRESSED_LSB_Y_0 && r.getType() != EccPointType.COMPRESSED_LSB_Y_1
				&& r.getType() != EccPointType.X_COORDINATE_ONLY) {
			throw new IllegalArgumentException("Invalid EccPointType for EcdsaSignature: " + r.getType());
		}
		this.r = r;
	}

	public void setS(Opaque s) {
		this.s = s;
	}

	public void setAlgorithm(PublicKeyAlgorithm algorithm) {
		this.algorithm = algorithm;
	}

	public void setFieldSize(UInt8 fieldSize) {
		this.fieldSize = fieldSize;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		int written = 0;
		if (r == null || s == null || !s.isArray()) {
			throw new IllegalArgumentException();
		}
		written += r.writeData(out);
		written += s.writeData(out);
		return written;
	}

	public EcdsaSignature(BigInteger r, BigInteger s, UInt8 fieldSize) {

		byte[] tmpR;
		if (r.toByteArray().length > fieldSize.get()) {
			tmpR = new byte[fieldSize.get()];
			System.arraycopy(r.toByteArray(), 1, tmpR, 0, fieldSize.get());
		} else {
			tmpR = r.toByteArray();
		}

		if (tmpR.length < fieldSize.get()) {
			byte[] tmp = new byte[fieldSize.get()];
			System.arraycopy(tmpR, 0, tmp, fieldSize.get() - tmpR.length, tmpR.length);
			tmpR = tmp;
		}

		byte[] tmpS;
		if (s.toByteArray().length > fieldSize.get()) {
			tmpS = new byte[fieldSize.get()];
			System.arraycopy(s.toByteArray(), 1, tmpS, 0, fieldSize.get());
		} else {
			tmpS = s.toByteArray();
		}

		if (tmpS.length < fieldSize.get()) {
			byte[] tmp = new byte[fieldSize.get()];
			System.arraycopy(tmpS, 0, tmp, fieldSize.get() - tmpS.length, tmpS.length);
			tmpS = tmp;
		}

		this.r = new EccPoint();
		this.r.setFieldSize(fieldSize);
		this.r.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		this.r.setType(EccPointType.X_COORDINATE_ONLY);
		Opaque x = new Opaque();
		x.set(tmpR);
		x.setArray(true);
		this.r.setX(x);

		Opaque sOpaque = new Opaque();
		sOpaque.set(tmpS);
		sOpaque.setArray(true);
		this.s = sOpaque;

		this.fieldSize = fieldSize;
	}

	public BigInteger[] toBigIntegers() {

		BigInteger r = new BigInteger(1, this.r.getX().get());
		BigInteger s = new BigInteger(1, this.s.get());

		return new BigInteger[] { r, s };
	}

}