package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.io.IOException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.Security;
import java.security.cert.CertificateException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Hashtable;
import java.util.NoSuchElementException;

import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import com.google.inject.Inject;
import com.mysql.jdbc.Statement;

import de.fraunhofer.sit.c2x.pki.ca.crypto.keystore.KeyStoreImpl;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Keystore;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.KeystoreProvider;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class JDBCKeystoreProvider extends AbstractMysqlConnection implements KeystoreProvider {

	// <certId as hex encoded string, keystore>
	private final Hashtable<String, KeyStoreImpl> keystores = new Hashtable<String, KeyStoreImpl>();

	@Inject
	private ConfigProvider configProvider;
	
	@Inject
	public JDBCKeystoreProvider() {
		super();
		Security.addProvider(new BouncyCastleProvider());
	}

	@Override
	public String getName() {
		return "Keystore provider";
	}

	private void initKeystore(byte[] certid) throws NoSuchAlgorithmException, CertificateException,
			IOException, KeyStoreException {
		String certIdHexEncoded = Hex.encodeHexString(certid);
		if (!keystores.containsKey(certIdHexEncoded)) {
			if (logger.isDebugEnabled())
				logger.debug("Try to load keystore from database for CertId10=" + certIdHexEncoded);
			String sql = String.format("SELECT * FROM %s WHERE %s = ?;", Constants.KEYSTORE__TABLE,
					Constants.KEYSTORE__COL__CERT_ID);
			Connection con = getConnection();
			PreparedStatement st = null;
			ResultSet result = null;
			try {
				st = con.prepareStatement(sql);
				st.setBytes(1, certid);
				st.execute();
				result = st.getResultSet();

				if (!result.first()) {
					if (logger.isDebugEnabled())
						logger.error("No Keystore found for CertId10=" + certIdHexEncoded);
				} else {
					if (logger.isDebugEnabled())
						logger.debug("Keystore found for CertId10=" + certIdHexEncoded);

					byte[] keyStoreBytes = result.getBytes(Constants.KEYSTORE__COL__KEYSTORE);
					KeyStoreImpl keystore = new KeyStoreImpl(keyStoreBytes, configProvider.get("keyStorePassword"));

					keystores.put(certIdHexEncoded, keystore);
					
					if (logger.isDebugEnabled())
						logger.debug("Keystore loaded");

				}
			} catch (SQLException e) {
				logger.error("Wrong SQL Statement", e);
			} finally {
				closeStatement(st);
				closeConnection(con);
				closeResultSet(result);
			}
		}

	}

	public ECPrivateKeyParameters getKey(byte[] certid, KeyType type) throws IOException {
		String certIdHexEncoded = Hex.encodeHexString(certid);
		try {
			if (!keystores.containsKey(certIdHexEncoded)) {
				initKeystore(certid);
			}
			KeyStoreImpl keys = keystores.get(certIdHexEncoded);
			if (keys != null) {
				return keystores.get(certIdHexEncoded).getECKey(configProvider.get("keyStorePassword"), type);
			} else {
				logger.error("No keystore found!");
				throw new NoSuchElementException("No keystore found!");
			}
		} catch (Exception e) {
			logger.error("Wrong passphrase for keystore or key alias not in keystore");
			throw new IOException("Wrong passphrase for keystore or key alias not in keystore", e);
		}
	}

	public long addKeystore(Keystore keystore) {

		String sql = String.format("INSERT INTO %s (%s) VALUES (?);", Constants.KEYSTORE__TABLE,
				Constants.KEYSTORE__COL__KEYSTORE);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
			st.setBytes(1, keystore.getKeystore());
			st.execute();

			ResultSet res = st.getGeneratedKeys();
			res.next();
			return res.getLong(1);
		} catch (SQLException e) {
			logger.warn("Can not prepare statement", e);
			return 0;
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	public boolean removeKeystore(long id) {

		String sql = String.format("DELETE FROM %s WHERE %s = ?;", Constants.KEYSTORE__TABLE,
				Constants.KEYSTORE__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
			st.setLong(1, id);
			st.execute();
			return true;
		} catch (SQLException e) {
			logger.warn("Can not prepare statement", e);
			return false;
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public void linkCaCertificate(long id, byte[] certid) {
		String sql = String.format("UPDATE %s SET %s = ? WHERE %s = ?;", Constants.KEYSTORE__TABLE,
				Constants.KEYSTORE__COL__CERT_ID, Constants.KEYSTORE__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, certid);
			st.setLong(2, id);
			st.execute();
		} catch (SQLException e) {
			logger.warn("Can not prepare statement", e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public boolean hasTemporaryCaCertificate(long id) {

		String sql = String.format("SELECT %s FROM %s WHERE %s = ?;", Constants.KEYSTORE__COL__CERT_ID,
				Constants.KEYSTORE__TABLE, Constants.KEYSTORE__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setLong(1, id);
			st.execute();

			ResultSet res = st.getResultSet();

			// is there any row associated with the given id?
			if (!res.first())
				return false;

			// is the certId field null ( = not yet linked)?
			if (res.getBytes(1) != null)
				return false;

			// the DB contains a temporary ca certificate slot! :-)
			return true;
		} catch (SQLException e) {
			logger.warn("Can not prepare statement", e);
			return false;
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}
}
