package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.io.IOException;
import java.util.ArrayList;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.eclipse.jetty.security.authentication.SessionAuthentication;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.GeographicRegionDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.PsidSspPriorityDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets.interfaces.ItsStationRegistrationHandler;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.AuthorizedDevice;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.HttpUser;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.PublicKey;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.HttpUserProvider;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class ItsStationRegistrationServlet extends AbstractServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	public static final String URL = "/registerItsS";
	public static final String TITLE = "Register ITS-S";
	private String info;
	private static final int NUM_PERMISSIONS = 5;

	@Inject
	private ItsStationRegistrationHandler itsStationRegistrationHandler;

	@Inject
	private HttpUserProvider httpUserProvider;

	@Inject
	public ItsStationRegistrationServlet(HtmlProvider htmlProvider) {
		super(htmlProvider);
	}

	public static String safeToString(Object o) {
		if (o == null) {
			return "";
		}
		return o.toString();
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException,
			IOException {
		doGet(req, resp);
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		StringBuilder sb = new StringBuilder();
		info = "";

		// get user from http session
		SessionAuthentication user = (SessionAuthentication) req.getSession().getAttribute(
				SessionAuthentication.__J_AUTHENTICATED);
		String userName = user.getUserIdentity().getUserPrincipal().getName().toString();

		if (req.getParameter("canonicalID") != null && req.getParameter("canonicalID").length() > 0
				&& req.getParameter("pubKey") != null && req.getParameter("pubKey").length() > 0
				&& req.getParameter("assuranceLevel") != null
				&& req.getParameter("assuranceLevel").length() > 0) {

			if (logger.isDebugEnabled())
				logger.debug("All required parameters given for ITS-S");

			byte[] canonicalId = getCanonicalId(req.getParameter("canonicalID"));
			if (canonicalId != null && canonicalId.length != 16) {
				if (info.isEmpty())
					info = "<div id=\"errorbox\">Canonical ID with 16 bytes required</div>";
				logger.error("Canonical id has not length of 16 bytes.");
			} else {

				if (logger.isDebugEnabled())
					logger.debug("Given canonical for ITS-S has valid size");

				// read module public key
				byte[] pubKey = getPublicKey(req.getParameter("pubKey"));

				// read assurance level
				if (req.getParameter("assuranceLevel").length() != 2) {
					if (info.isEmpty())
						info = "<div id=\"errorbox\">Assurance Level as byte required</div>";
					if (logger.isDebugEnabled())
						logger.debug("Assurance Level as byte required.");
				}
				byte[] assuranceLevelByte = null;
				try {
					assuranceLevelByte = Hex.decodeHex(req.getParameter("assuranceLevel").toCharArray());
				} catch (DecoderException e1) {
					if (info.isEmpty())
						info = "<div id=\"errorbox\">Given assurance Level as HEX string "
								+ req.getParameter("assuranceLevel") + " is cannot be decoded</div>";
					if (logger.isDebugEnabled())
						logger.debug("Given assurance Level as HEX string "
								+ req.getParameter("assuranceLevel") + " is cannot be decoded");
				}
				// read permissions if available
				ArrayList<PsidSspPriorityDataContainer> permissions = new ArrayList<PsidSspPriorityDataContainer>(
						NUM_PERMISSIONS);
				for (int i = 1; i <= NUM_PERMISSIONS; i++) {
					PsidSspPriorityDataContainer permissionContainer = null;
					try {
						if (req.getParameter("aid" + i) != null && req.getParameter("aid" + i).length() > 0) {
							permissionContainer = new PsidSspPriorityDataContainer();
							long psid = new Long(req.getParameter("aid" + i));
							permissionContainer.setPsid(psid);
							if (req.getParameter("ssp" + i) != null
									&& req.getParameter("ssp" + i).length() > 0) {
								permissionContainer
										.setServiceSpecificPermissions(req.getParameter("ssp" + i));
							}
							if (req.getParameter("priority" + i) != null
									&& req.getParameter("priority" + i).length() > 0) {
								int priority = new Integer(req.getParameter("priority" + i));
								permissionContainer.setMaxPriority(priority);
							}
						}
					} catch (NumberFormatException | DecoderException e) {
						if (info.isEmpty())
							info = "<div id=\"errorbox\">" + i + ". Permission cannot be  read: "
									+ e.getMessage() + "</div>";
						if (logger.isDebugEnabled())
							logger.debug(i + ". Permission cannot be  read: " + e.getMessage());
					}
					if (permissionContainer != null)
						permissions.add(permissionContainer);
				}
				// read regions
				GeographicRegionDataContainer region = null;
				if (req.getParameter("regionDictionary") != null
						&& req.getParameter("regionDictionary").length() > 0) {
					Long regionId = new Long(req.getParameter("regionId"));
					Long regionLocalId = new Long(req.getParameter("regionLocalId"));
					region = new GeographicRegionDataContainer(req.getParameter("regionDictionary"),
							regionId, regionLocalId);
				} else if (req.getParameter("regionCircleLatitude") != null
						&& req.getParameter("regionCircleLatitude").length() > 0
						&& req.getParameter("regionCircleLongitude") != null
						&& req.getParameter("regionCircleLongitude").length() > 0
						&& req.getParameter("regionCircleRadius") != null
						&& req.getParameter("regionCircleRadius").length() > 0) {
					Double centerLatitude = new Double(req.getParameter("regionCircleLatitude"));
					Double centerLongitude = new Double(req.getParameter("regionCircleLongitude"));
					Integer radius = new Integer(req.getParameter("regionCircleRadius"));
					region = new GeographicRegionDataContainer(centerLatitude, centerLongitude, radius);
				}

				if (pubKey != null && assuranceLevelByte != null) {
					String pubKeyAlgorithm = "ECDSA_NISTP256_WITH_SHA256";
					if (logger.isDebugEnabled())
						logger.debug("Given public key with algorithm " + pubKeyAlgorithm + " is valid");
					if (pubKeyAlgorithm != null) {
						PublicKey publicKey = new PublicKey();
						publicKey.setPublicKeyAlgorithm(pubKeyAlgorithm);
						publicKey.setPublicKey(pubKey);

						AuthorizedDevice device = new AuthorizedDevice();
						device.setCanonicalId(canonicalId);
						device.setPublicKey(publicKey);
						device.setSubjectAssurance(assuranceLevelByte);
						boolean res = false;
						try {
							res = itsStationRegistrationHandler.handleItsStationRegistration(canonicalId,
									publicKey, assuranceLevelByte, permissions, region, userName);
						} catch (HandlerException e) {
							if (info.isEmpty())
								info = "<div id=\"errorbox\">ITS-S cannot be registered: " + e.getMessage()
										+ "</div>";
							if (logger.isDebugEnabled())
								logger.debug("ITS-S cannot be registered: " + e.getMessage() + "");
						}
						if (res) {
							if (logger.isInfoEnabled())
								logger.info("USER: User "
										+ user.getUserIdentity().getUserPrincipal().getName().toString()
										+ " registered a new ITS-S with canonical ID: "
										+ Hex.encodeHexString(canonicalId));

							info = "<div id=\"ackbox\">New ITS-S stored in DB</div>";
							logger.debug("ITS-S \"" + Hex.encodeHexString(canonicalId) + "\" registered");
						} else {
							if (info.isEmpty())
								info = "<div id=\"errorbox\">ITS-S is not stored in DB</div>";
							if (logger.isDebugEnabled())
								logger.debug("ITS-S \"" + Hex.encodeHexString(canonicalId)
										+ "\" not registered");
						}
					}
				}
			}

		}

		HttpUser httpUser = httpUserProvider.getUser(userName);
		byte[] canonicalIdPrefix = httpUser.getItsSRegPrefix();
		if (canonicalIdPrefix == null)
			canonicalIdPrefix = new byte[0];

		sb.append("<form action=\"" + URL + "\" method=\"get\">\n");
		if (!info.isEmpty())
			sb.append(info);
		sb.append("<table>\n");
		sb.append("<tr><td style=\"width:170px; padding-right:10px;\">Canonical ID with 16 bytes as HEX value with 32 characters *:</td><td><input id=\"canonicalID\" name=\"canonicalID\" type=\"text\" value=\""
				+ Hex.encodeHexString(canonicalIdPrefix)
				+ "\" maxlength=\"32\" style=\"width:600px\" /></td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td>Public key as HEX value with 128 characters*:</td><td><textarea id=\"pubKey\" name=\"pubKey\" rows=\"5\" style=\"width:600px\"></textarea></td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td>Assurance level*:</td><td>\n");
		sb.append("<select id=\"assuranceLevel\" name=\"assuranceLevel\" style=\"width:600px\">");
		sb.append("<option value=\"00\">0</option>\n");
		sb.append("<option value=\"20\">1</option>\n");
		sb.append("<option value=\"40\">2</option>\n");
		sb.append("<option value=\"60\">3</option>\n");
		sb.append("<option value=\"80\">4</option>\n");
		sb.append("<option value=\"A0\">5</option>\n");
		sb.append("<option value=\"C0\">6</option>\n");
		sb.append("<option value=\"E0\">7</option>\n");
		sb.append("</select></td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td>Permissions:</td><td>");
		sb.append("<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">");
		for (int i = 1; i <= NUM_PERMISSIONS; i++) {
			sb.append("<tr>\n");
			sb.append("<td style=\"width:200px\" align=\"left\">" + i + ". AID (int):&nbsp;<input id=\"aid"
					+ i + "\" name=\"aid" + i
					+ "\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"8\" /></td>");
			sb.append("<td style=\"width:200px\" align=\"center\">" + i + ". SSP (hex):&nbsp;<input id=\"ssp"
					+ i + "\" name=\"ssp" + i
					+ "\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"4\" /></td>");
			sb.append("<td style=\"width:200px\" align=\"right\">" + i
					+ ". Priority (int):&nbsp;<input id=\"priority" + i + "\" name=\"priority" + i
					+ "\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"8\" /></td>");
			sb.append("</tr>");
		}
		sb.append("</table>\n");
		sb.append("</td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td>Region:</td><td>");
		sb.append("<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">");
		sb.append("<tr>\n");
		sb.append("<td style=\"width:100px\" align=\"left\">IdentifiedRegion:&nbsp;</td>");
		sb.append("<td style=\"width:170px\" align=\"right\">Dictionary:&nbsp;<select id=\"regionDictionary\" name=\"regionDictionary\" style=\"width:100px\"><option value=\"\"></option><option value=\"iso_3166_1\">iso_3166_1</option><option value=\"un_stats\">un_stats</option></select></td>");
		sb.append("<td style=\"width:170px\" align=\"right\">Region ID (int16):&nbsp;<input id=\"regionId\" name=\"regionId\" type=\"text\" value=\"\" style=\"width:50px\" maxlength=\"5\" /></td>");
		sb.append("<td style=\"width:160px\" align=\"right\">Local ID (int16):&nbsp;<input id=\"regionLocalId\" name=\"regionLocalId\" type=\"text\" value=\"0\" style=\"width:50px\" maxlength=\"5\" /></td>");
		sb.append("</tr>");
		sb.append("<tr>\n");
		sb.append("<td align=\"left\">CircularRegion:&nbsp;</td>");
		sb.append("<td align=\"right\">Lat. (int):&nbsp;<input id=\"regionCircleLatitude\" name=\"regionCircleLatitude\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"11\" /></td>");
		sb.append("<td align=\"right\">Long. (int):&nbsp;<input id=\"regionCircleLongitude\" name=\"regionCircleLongitude\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"11\" /></td>");
		sb.append("<td align=\"right\">Radius (uint16):&nbsp;<input id=\"regionCircleRadius\" name=\"regionCircleRadius\" type=\"text\" value=\"\" style=\"width:50px\" maxlength=\"5\" /></td>");
		sb.append("</tr>");
		sb.append("</table>\n");
		sb.append("</td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");

		sb.append("<tr><td></td><td><input type=\"submit\" value=\"Register ITS-S\" /></td></tr>\n");
		sb.append("</table>\n" + "</form>\n");
		sb.append("<br/>\nRequired elements are marked by a *\n");

		return sb.toString();
	}

	private byte[] getPublicKey(String pubKeyString) {

		byte[] pubKey = null;;
		if (pubKeyString.length() < 128) {
			if (logger.isDebugEnabled())
				logger.debug("2a! base64 " + pubKeyString);
			pubKey = Base64.decodeBase64(pubKeyString);
		} else if (pubKeyString.length() > 128) { 
			if (logger.isDebugEnabled())
				logger.debug("Decoder error of public key string: " + pubKeyString);
			info = "<div id=\"errorbox\">Unable to decode hex public key.</div>";
			logger.error("Unable to decode hex public key.");
		}else {
			try {
				pubKey = Hex.decodeHex(pubKeyString.toCharArray());
			} catch (DecoderException e) {
				if (logger.isDebugEnabled())
					logger.debug("Decoder error of public key string: " + pubKeyString);
				info = "<div id=\"errorbox\">Unable to decode hex public key.</div>";
				logger.error("Unable to decode hex public key.");
			}
		}
		
		if (pubKey != null && pubKey.length == 64) {
			byte[] tmp = new byte[65];
			tmp[0] = 0x04;
			System.arraycopy(pubKey, 0, tmp, 1, pubKey.length);
			return tmp;
		}

		return pubKey;
	}

	private byte[] getCanonicalId(String idString) {
		byte[] id = null;
		if (idString.length() < 32) {
			id = Base64.decodeBase64(idString);
		} else {
			try {
				id = Hex.decodeHex(idString.toCharArray());
			} catch (DecoderException e) {
				info = "<div id=\"errorbox\">Unable to decode hex canonical id.</div>";
				logger.error("Unable to decode hex canonical id.");
			}
		}
		return id;
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString(),
				UserRole.DEVELOPER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.ITS_S;
	}
}