/*!
 * \file      sha256.cc
 * \brief     Source file for SHA-256 helper methods.
 * \author    ETSI STF525
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#include <TTCN3.hh>

#include "sha256.hh"

int sha256::generate(const OCTETSTRING& p_buffer, OCTETSTRING& p_hash) {
  // Sanity check
  if (p_buffer.lengthof() == 0) {
    p_hash = get_sha256_empty_string();
    return 0;
  }

  return generate(static_cast<const unsigned char*>(p_buffer), p_buffer.lengthof(), p_hash);
}

int sha256::generate(const unsigned char* p_buffer, const size_t p_length, OCTETSTRING& p_hash) {
  // Sanity check
  if ((p_buffer == nullptr) || (p_length == 0)) {
    p_hash = get_sha256_empty_string();
    return 0;
  }

  // Resize data buffer
  p_hash = int2oct(0, SHA256_DIGEST_LENGTH);
  // Compute the hash value
  ::SHA256_Init(&_ctx);
  ::SHA256_Update(&_ctx, p_buffer, p_length);
  ::SHA256_Final((unsigned char*)static_cast<const unsigned char*>(p_hash), &_ctx);
  return 0;
};

  /*!
   * \fn const OCTETSTRING get_sha256_empty_string() const;
   * \brief Return the SHA-256 of an empty string
   * \return The SHA-256 of an empty string
   */
  inline const OCTETSTRING sha256::get_sha256_empty_string() const {
    static unsigned char sha256_empty_string[] = { 0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14, 0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24, 0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c, 0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55 }; //! SHA-256 of an empty string
    return OCTETSTRING(32, sha256_empty_string);
  };
