package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.WebServerWithClientAuth;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthenticatorProvider;

/**
 * A simple servlet that outputs the certificate content of issued certificates.
 * Following arguments are allowed per GET
 * 
 * <ul>
 * <li>CertId8 or CertId10: <b>/cacert?cert=HEX or Base64</b></li>
 * </ul>
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class AuthenticatorDeletionServlet extends AbstractServlet {

	private static final long serialVersionUID = 1L;
	protected static final String URL = "/deleteAuthenticator";
	protected static final String TITLE = "Delete registered authenticator";
	public static final Pattern VALID_EMAIL_ADDRESS_REGEX = Pattern.compile(
			"^[A-Z0-9._%+-]+@[A-Z0-9.-]+\\.[A-Z]{2,6}$", Pattern.CASE_INSENSITIVE);

	public static boolean validate(String emailStr) {
		Matcher matcher = VALID_EMAIL_ADDRESS_REGEX.matcher(emailStr);
		return matcher.find();
	}

	@InjectLogger
	private Logger logger;
	private AuthenticatorProvider authenticatorProvider;

	@Inject
	WebServerWithClientAuth webServerWithClientAuth;

	@Inject
	public AuthenticatorDeletionServlet(HtmlProvider htmlProvider, AuthenticatorProvider authenticatorProvider) {
		super(htmlProvider);
		this.authenticatorProvider = authenticatorProvider;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		String userInfo = "";
		StringBuilder sb = new StringBuilder();

		String emailAddress = null;
		if (req.getParameter("email") != null && req.getParameter("email").length() > 0) {
			emailAddress = req.getParameter("email");
			emailAddress = emailAddress.trim();
			if (emailAddress.length() > 0) {
				try {
					if (validate(emailAddress) == false) {
						if (logger.isDebugEnabled())
							logger.debug("Invalid email address");
						throw new Exception("Invalid email address");
					}
				} catch (Exception e) {
					userInfo = "<div id=\"errorbox\">Invalid email address: " + e.getMessage() + "</div>";
				}
			}
		}

		if (req.getParameter("deactivate") != null && emailAddress != null) {
			try {
				if (authenticatorProvider.removeAuthenticator(emailAddress) >= 1) {
					userInfo = "<div id=\"ackbox\">Authenticator " + emailAddress
							+ " deleted from DB if available</div>";
					// restart webserver with client authentication
					webServerWithClientAuth.restartWebServer();
				} else {
					throw new Exception("Authenticator " + emailAddress + " not found");
				}
			} catch (Exception e) {
				userInfo = "<div id=\"errorbox\">Authenticator " + emailAddress + " not deleted from DB: "
						+ e.getMessage() + "</div>";
			}
		}

		if (!userInfo.isEmpty()) {
			sb.append(userInfo);
		}

		// Prepare the form
		sb.append("<form name=\"certForm\" onsubmit=\"return askBeforeDelete('Do you really want delete the authenticator?');\" action=\"\" method=\"get\">Delete registered authenticator with its Email address: ");

		if (emailAddress == null)
			emailAddress = "";
		sb.append("<input type=\"hidden\" name=\"deactivate\" value=\"true\"/>\n<input type=\"text\" id=\"email\" name=\"email\" value=\""
				+ emailAddress
				+ "\" maxlength=\"80\" style=\"width:300px;\" />&nbsp;<input type=\"submit\" value=\"Delete\" /></form>");

		return sb.toString();
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.AUTHENTICATOR;
	}
}
