package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SignerInfoTypeImpl.SignerInfoType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class SignerInfo extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private SignerInfoType type;

	@Internal(order = 1)
	private HashedId8 digest;

	@Internal(order = 2)
	private Certificate certificate;

	@Internal(order = 3)
	private Certificate[] certificates;

	@Internal(order = 4)
	private PublicKeyAlgorithm algorithm;

	@Internal(order = 5)
	private Opaque info;

	// ---- constructors ----

	public SignerInfo() {
	}
	
	public SignerInfo(HashedId8 digest) {
		this.type = SignerInfoType.CERTIFICATE_DIGEST_WITH_ECDSAP256;
		this.digest = digest;
	}
	
	public SignerInfo(Certificate certificate) {
		this.type = SignerInfoType.CERTIFICATE;
		this.certificate = certificate;
	}
	
	public SignerInfo(Certificate[] certificates) {
		this.type = SignerInfoType.CERTIFICATE_CHAIN;
		this.certificates = certificates;
	}

	public SignerInfo(DataInputStream in) throws IOException {
		type = SignerInfoTypeImpl.getInstance().getEnumType(in.readByte());

		if (type == SignerInfoType.SELF) {
		} else if (type == SignerInfoType.CERTIFICATE_DIGEST_WITH_ECDSAP256) {
			digest = new HashedId8(in);
		} else if (type == SignerInfoType.CERTIFICATE) {
			certificate = new Certificate(in);
		} else if (type == SignerInfoType.CERTIFICATE_CHAIN) {
			certificates = WaveUtils.getArrayFromStream(in, Certificate.class);
		} else if (type == SignerInfoType.CERTIFICATE_DIGEST_WITH_OTHER_ALGORITHM) {
			algorithm = PublicKeyAlgorithmImpl.getInstance().getEnumType(
					in.readByte());
			digest = new HashedId8(in);
		} else {
			info = new Opaque(in);
		}
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public SignerInfoType getType() {
		return this.type;
	}

	public HashedId8 getDigest() {
		return this.digest;
	}

	public Certificate getCertificate() {
		return this.certificate;
	}

	public Certificate[] getCertificates() {
		return this.certificates;
	}

	public PublicKeyAlgorithm getAlgorithm() {
		return this.algorithm;
	}

	public Opaque getInfo() {
		return this.info;
	}

	// ---- setter ----

	public void setType(SignerInfoType type) {
		this.type = type;
	}

	public void setDigest(HashedId8 digest) {
		this.digest = digest;
	}

	public void setCertificate(Certificate certificate) {
		this.certificate = certificate;
	}

	public void setCertificates(Certificate[] certificates) {
		this.certificates = certificates;
	}

	public void setAlgorithm(PublicKeyAlgorithm algorithm) {
		this.algorithm = algorithm;
	}

	public void setInfo(Opaque info) {
		this.info = info;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {

		int written = SignerInfoTypeImpl.getInstance().writeData(out, type);

		if (type == SignerInfoType.SELF) {
		} else if (type == SignerInfoType.CERTIFICATE_DIGEST_WITH_ECDSAP256) {
			written += WaveUtils.writeWave(out, digest);
		} else if (type == SignerInfoType.CERTIFICATE) {
			written += WaveUtils.writeWave(out, certificate);
		} else if (type == SignerInfoType.CERTIFICATE_CHAIN) {
			written += WaveUtils.writeArrayToStream(out, certificates);
		} else if (type == SignerInfoType.CERTIFICATE_DIGEST_WITH_OTHER_ALGORITHM) {
			written += PublicKeyAlgorithmImpl.getInstance().writeData(out,
					algorithm);
			written += WaveUtils.writeWave(out, digest);
		} else {
			if(info == null) throw new IllegalArgumentException();
			written += WaveUtils.writeWave(out, info);
		}
		return written;
	}

	public Opaque getId() {
		return info;
	}

}